/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

# include "proc.h"

# define CPP_LINE	(_PATH_CPP " -DP11 " CPP_EXTRA " " CPP_DINCL _PATH_P11INCL " -DSYSTEM=" SYSTEMS " -DPROC=" PROCS " ")

Proc	proc;
char	verbose[128];
char	*conffile = P11CONF;
int  	confline;
jmp_buf	trap;
char	cpp_opts[200];
pid_t	cpp_pid;
int	enable_fpp = 0;
ushort	startloc = 0173000;

extern IOOps lp_ops, rl_ops, kl_ops, mr_ops;

IOConf ioconf[] = {
	{ "rl",	&rl_ops },
	{ "kl",	&kl_ops },
	{ "mr", &mr_ops },
	{ "lp", &lp_ops },
};

int	Setup(void);
void	LoadFile(char *);
void	catch_sigs(void);
void	verb(char, char *, ...);

FILE	*open_read_pipe(char *, pid_t *);
int	close_pipe(FILE *, pid_t *);

main(int argc, char *argv[])
{
	int	opt;
	char	*file = NULL, *p;
	int	error;
	extern int optind;
	extern char *optarg;

	while( ( opt = getopt( argc, argv, "fs:c:v:D:I:U:" ) ) != EOF ) {
		switch( opt ) {

		case 'c':
			conffile = optarg;
			break;

		case 'v':
			for(p = optarg; p && *p; p++)
				if(*p < sizeof(verbose))
					verbose[*p]++;
			break;

		case 'I':
		case 'U':
		case 'D':
			if(optarg)
				sprintf(cpp_opts + strlen(cpp_opts), " -%c%s", opt, optarg);
			break;

		case 's':
			if(optarg)
				 sscanf(optarg, "%hi", &startloc);
			break;

		case 'f':
			enable_fpp = 1;
			break;
		}
	}

	catch_sigs();

	if( ( error = Setup() ) != 0 )
		return error;

	argc -= optind;
	argv += optind;

	while(argc-- > 0)
		LoadFile(*argv++);

	interp();

	return 0;
}

int
Setup(void)
{
	FILE	*fp;
	char	buffer[1024];
	int	argc;
	char	*argv[20];
	int	i = 0;		/* keep gcc quiet */
	IODev	*iodev = NULL;
	char	*cmd;
	int	j;
	char	*p11;
	int	len;

	catch_io_sigs();

	verb('c', "reading configuration from %s ...\n", conffile);
	bzero(&proc, sizeof(proc));
	proc.physmem = PHYSMEM;
	proc.bp = 0177777;
	proc.halt = 1;
	proc.fpd = !enable_fpp;

	len = strlen(CPP_LINE) + strlen(conffile) + strlen(cpp_opts) + 3;

	if((p11 = getenv("PATH11")) != 0) {
		len += strlen(" -DPATH11=") + strlen(p11);
		len += strlen(" -I") + strlen(p11) + strlen("/emu");
	}

	cmd = xalloc(len , "Setup");

	strcpy(cmd, CPP_LINE);
	if(p11)
		sprintf(cmd + strlen(cmd), "-DPATH11=%s -I%s/emu", p11, p11);
	sprintf(cmd + strlen(cmd), " %s %s", cpp_opts, conffile);

	verb('c', "starting '%s'\n", cmd);

	if((fp = open_read_pipe(cmd, &cpp_pid) ) == NULL )
		conf_panic("pipe cpp: %s\n", strerror(errno) );

	(void)setfields(" \t\n");

	while( ( fgets( buffer, sizeof(buffer), fp ) ) != NULL ) {
		confline++;
		argc = getmfields( buffer, argv, ArraySize(argv) );
		if( argc == 0 )
			continue;	/* empty line */
		if( !strcmp( argv[0], "#" ) )
			continue;	/* comment line */
		if( iodev == NULL ) {
			if( strcmp( argv[0], "ctrl" ) )
				conf_panic("Syntax error, ctrl keyword expected");
			if( argc < 2 )
				conf_panic("ctrl keyword without arg");
			for( i = 0 ; i < ArraySize(ioconf) ; i++ )
				if( !strcmp( ioconf[i].name, argv[1] ) )
					break;
			if( i == ArraySize(ioconf) )
				conf_panic("No such ctrl \"%s\"", argv[1] );
			iodev = xalloc( sizeof(IODev), "Setup iodev" );
			iodev->ioops = ioconf[i].ioops;
			iodev->name = ioconf[i].name;
			if(verbose['c']) {
				verb('c', "create controller %s(", iodev->name);
				for(j = 2; j < argc; j++)
					fprintf(stderr, "%s%s", argv[j], (j==argc-1)?"":",");
				fprintf(stderr, ")\n");
			}
			(*ioconf[i].ioops->ctrl_create)(iodev, argc-2, &argv[2] );
		} else {
			if( !strcmp( "end", argv[0] ) ) {
				verb('c', "controller complete\n");
				(*ioconf[i].ioops->ctrl_complete)(iodev);
				iodev->next = proc.devices;
				proc.devices = iodev;
				iodev = NULL;
			} else {
				if(verbose['c']) {
					verb('c', "  create device %s(", iodev->name);
					for(j = 0; j < argc; j++)
						fprintf(stderr, "%s%s", argv[j], (j==argc-1)?"":",");
					fprintf(stderr, ")\n");
				}
				(*ioconf[i].ioops->dev_create)(iodev, argc, argv);
			}
		}
	}
	if( iodev != NULL )
		conf_panic("Missing end keyword");

	verb('c', "plug in memory\n");
	memsys.next = proc.devices;
	proc.devices = &memsys;
	(*memsys.ioops->ctrl_complete)(&memsys);

	verb('c', "plug in processor\n");
	internals.next = proc.devices;
	proc.devices = &internals;
	(*internals.ioops->ctrl_complete)(&internals);


	close_pipe(fp, &cpp_pid);
	Reset();

	free(cmd);

# ifndef FPP
	if(!proc.fpd)
		panic("no fpp but fpp enabled\n");
# endif

	return 0;
}

void
Reset(void)
{
	IODev *dev;

	/*
	 * reset outstanding requests and then call each device
	 * this gives them a chance to place a new request
	 */
	proc.reqpri = 0;
	for(dev = proc.devices; dev; dev = dev->next)
		if(dev->ioops->reset)
			(*dev->ioops->reset)(dev);
}

void
Warning(char *fmt, ...)
{
	va_list	ap;

	fprintf(stderr, "Warning: ");
	VA_START(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
}

volatile void
panic(char *fmt, ...)
{
	va_list	ap;

	fprintf(stderr, "Panic: ");
	VA_START(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	exit(27);
}

void
conf_warning(char *fmt, ...)
{
	va_list	ap;

	fprintf(stderr, "Warning: %s, line %d : ", conffile, confline );
	VA_START(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
}

volatile void
conf_panic(char *fmt, ...)
{
	va_list	ap;

	fprintf(stderr, "Panic: %s, line %d : ", conffile, confline );
	VA_START(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	exit(27);
}

unsigned
parse_csr(char *p, char *text)
{
	unsigned csr;
	char	*end;

	csr = strtol( p, &end, 8);
	if( p == end )
		goto bad;
	if( csr <= 017760000 || csr > 017777776 )
		goto bad;
	if( csr & 1 )
		goto bad;
	return csr;
bad:
	conf_panic("%s:bad csr", text);
}

ushort
parse_vec(char *p, char *text)
{
	ushort	vec;
	char	*end;

	vec = strtol( p, &end, 8);
	if( p == end )
		goto bad;
	if( vec >= 01000 )
		goto bad;
	if( vec & 3 )
		goto bad;
	return vec;
bad:
	conf_panic("%s:bad vec", text);
}

ushort
parse_irq(char *p, char *text)
{
	ushort	irq;
	char	*end;

	irq = strtol( p, &end, 8);
	if( p == end )
		goto bad;
	if( irq < 1 || irq >  7 )
		goto bad;
	return irq;
bad:
	conf_panic("%s:bad irq", text);
}


void *
xalloc(size_t s, char *n)
{
	void *p = malloc(s);

	if(!p)
		panic("out of mem in %s", n);
	bzero(p, s);
	return p;
}


void
onpipe()
{
	pid_t	pid;
	int	status;
	int	got = 0;

	while((pid = waitpid((pid_t)-1, &status, WNOHANG)) > 0) {
		if(pid == cpp_pid) 
			continue;

		if(!got) {
			printf("got SIGPIPE/SIGCHLD\n");
			printf("waiting for children:\n");
			got++;
		}
		printf("%4d: ", (int)pid);
		if(WIFEXITED(status)) 
			printf("exited %d\n", WEXITSTATUS(status));
		else if(WIFSIGNALED(status)) {
			printf("signaled %d", WTERMSIG(status));
			if(WCOREDUMP(status))
				printf(" core dumped");
			printf("\n");
		} else if(WIFSTOPPED(status))
			printf("stopped %d\n", WSTOPSIG(status));
		else
			printf("???\n");
	}
	proc.halt = 1;
}

void 
onintr() 
{ 
	proc.halt = 1;
}

void	
catch_sigs(void)
{
	SIGNAL(SIGPIPE, onpipe);
	SIGNAL(SIGCHLD, onpipe);
	SIGNAL(SIGINT, onintr);
}

void	
LoadFile(char *f)
{
	FILE *fp;
	unsigned long addr, cnt;
	unsigned v;

	if((fp = fopen(f, "r")) == 0) {
		perror(f);
		exit(1);
	}
	while(fscanf(fp, "%lx%lx", &addr, &cnt) == 2) {
		while(cnt-- && addr < PHYSMEM && fscanf(fp, "%x", &v) == 1) {
			ushort *p = proc.memops->addrphys(proc.mmg, addr);
			if(addr & 1)
				SETHB(*p, (v << 8));
			else
				SETLB(*p, v);
			addr++;
		}
	}
	fclose(fp);
}

/*
 * verbosity
 */
void	
verb(char c, char *fmt, ...)
{
	va_list	ap;

	if(!verbose[c])
		return;
	VA_START(ap, fmt);
	fprintf(stderr, "p11: ");
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}

/*
 * duplicate popen because we need to know cpp's pid
 */
FILE *
open_read_pipe(char *cmd, pid_t *ppid)
{
	int	pv[2];
	FILE	*fp;

	if(pipe(pv))
		return NULL;
	if((*ppid = fork()) == -1) {
		*ppid = 0;
		close(pv[0]);
		close(pv[1]);
		return NULL;
	} else if(*ppid == 0) {
		if(pv[1] != 1) {
			dup2(pv[1], 1);
			close(pv[1]);
		}
		execl(_PATH_BSHELL, "sh", "-c", cmd, NULL);
		_exit(127);
	}
	fp = fdopen(pv[0], "r");
	close(pv[1]);
	return fp;
}

int
close_pipe(FILE *fp, pid_t *ppid)
{
	int 	omask;
	int	status;
	pid_t	pid;
	sigset_t iset;
	sigset_t oset;

	if(*ppid == 0)
		return -1;
	fclose(fp);

	sigemptyset(&iset);
	sigaddset(&iset, SIGINT);
	sigaddset(&iset, SIGQUIT);
	sigaddset(&iset, SIGHUP);
	sigprocmask(SIG_BLOCK, &iset, &oset);

	do {
		pid = waitpid(*ppid, &status, 0);
	} while(pid == -1 && errno == EINTR);

	sigprocmask(SIG_SETMASK, &oset, 0);

	*ppid = 0;

	return WEXITSTATUS(status);
}
