/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

# include <math.h>

# define InitFp()		/* nothing to do */
# define EndFp()		/* nothing to do */

/*
 * 'generic' floating point
 * this format is reasonable near the pdp format, but:
 *   1.0 <= m < 2.0
 *   mantissa has 63 bits and has point after two msb
 *   exponent has more bits and offset of 0x3FFF
 * your compiler must have 64 bit signed and unsigned arithmetic (UQuad, SQuad)
 */
struct Float {
	int		s;	/* sign: 1 means neg */
	unsigned	e;	/* exponent with offset 0x3FFF, 0 for real 0 */
	UQuad 		m;	/* mantissa is 1.0 <= m < 2.0 or m == 0 */
};
# define EOFFS	0x3FFF
# define MBITS	63


# define AddF(A,S)	(void)((A) = F_add((A), (S)))
# define SubF(A,S)	(void)((A) = F_sub((A), (S)))
# define MulF(A,S)	(void)((A) = F_mul((A), (S)))
# define DivF(A,S)	(void)((A) = F_div((A), (S)))
# define ModF(A,F,I)	(void)((F) = F_mod((A), &(I)))
# define IsZ(A)		((A).e == 0)
# define IsN(A)		((A).s)
# define CnvL(A,L)	(void)((A) = F_int((L)))
# define CnvF2L(S,L)	(void)((L) = F_chop((S)))

# define FrExp(A)	((A).e - EOFFS)

# define GetMant(P)	(((P)->m & ~f_msb) << 1)
# define GetExp(P)	((int)(P)->e - EOFFS + 1)
# define GetSign(P)	((P)->s)

# define SetMant(H,M)	(void)((H)->m = ((M) >> 1) | f_msb)
# define SetExp(H,E)	(void)((H)->e = (E) - 1 + EOFFS)
# define SetSign(H,S)	(void)((H)->s = (S))

Float	F_add(Float a, Float b);
Float	F_sub(Float a, Float b);
Float	F_mul(Float a, Float b);
Float	F_div(Float a, Float b);
Float	F_mod(Float a, Float *ip);
int	F_cmp(Float a, Float b);
long	F_chop(Float a);
long	F_round(Float a);
Float	F_int(long);

typedef double ldouble;		/* should be long double */
# define XFREXP	frexp
# define XLDEXP	ldexp

Float 	F_quad(ldouble ld);
ldouble	F_double(Float a);

char *	F_dump(Float a);

static void		f_add(Float *a, Float *b, Float *c);
static void		f_sub(Float *a, Float *b, Float *c);
static void		f_norm(Float *a);
static const UQuad	f_msb = (UQuad)1 << 62;
static const UQuad	f_carry = (UQuad)1 << 63;
static const Float	f_null = {0, 0, 0};

/*
 * debugging
 */
static double
mkdouble(PDP_Float *a, int d) 
{
	Float h;

	unfold(a, &h, d);
	return (double)F_double(h);
}

static double
mkd(Float *f)
{
	return (double)F_double(*f);
}

static void
dmpflt(Float *f)
{
	printf("%s", F_dump(*f));
}

/*
 * make (long)double from Float
 */
ldouble
F_double(Float a)
{
	ldouble	v;
	ldouble p;
	int i;
	UQuad mask;

	if(a.e == 0)
		return 0;
	v = 0.0;
	for(i = 0, p = 1.0, mask = f_msb; i < MBITS; i++, mask >>= 1, p /= 2)
		if((a.m & mask) != 0)
			v += p;

	return a.s ? -XLDEXP(v, (int)a.e - EOFFS) : XLDEXP(v, (int)a.e - EOFFS);
}

/*
 * make float from (long)double
 * sorry, there is no frmant function in analogy to frexp so we
 * must get the mantissa bitwise
 */
Float
F_quad(ldouble ld)
{
	int i;
	UQuad mask;
	Float a;

	if(ld == 0)
		return f_null;
	if(ld < 0) {
		a.s = 1;
		ld = -ld;
	} else
		a.s = 0;
	ld = 2 * XFREXP(ld, &i);	/* ld now >= 1.0 && < 2.0 */
	a.e = i - 1 + EOFFS;
	a.m = 0;
	for(i = 0, mask = f_msb; i < MBITS; i++, mask >>= 1, ld *= 2)
		if(ld >= 1.0) {
			a.m |= mask;
			ld -= 1.0;
		}
	return a;
}

/*
 * dump a float
 */
char *
F_dump(Float a)
{
	static char buf[30];

	sprintf(buf, "%d %04x %08lx %08lx", a.s, a.e, (ulong)(a.m >> 32), (ulong)a.m);
	return buf;
}

/*
 * convert long to float
 */
Float
F_int(long v)
{
	Float c;

	if(v == 0)
		return f_null;
	c.s = (v < 0);
	c.e = EOFFS + MBITS - 1;
	if(v < 0)
		c.m = (UQuad)-(SQuad)v;
	else
		c.m = (UQuad)v;
	f_norm(&c);
	return c;
}


/*
 * chop Float to long
 */
long
F_chop(Float a)
{
	int e;

	if(a.e == 0)
		return 0;
	if((e = (int)a.e - EOFFS) >= 32)
		return a.s ? 0x80000000U : 0x7fffffff;
	a.m >>= MBITS - 1 - e;
	if(!a.s) {
		if(a.m >= (UQuad)0x80000000U)
			return 0x7fffffffU;
		return (long)a.m;
	} else {
		if(a.m >= (UQuad)0x80000000U)
			return 0x80000000;
		return -(long)a.m;
	}
}

/*
 * round Float to long
 */
long
F_round(Float a)
{
	int e, r;

	if(a.e == 0)
		return 0;
	if((e = (int)a.e - EOFFS) >= 32)
		return a.s ? 0x80000000U : 0x7fffffff;
	a.m >>= MBITS - 1 - e - 1;
	r = (int)a.m & 1;
	a.m >>= 1;
	if(r)
		a.m += 1;
	if(!a.s) {
		if(a.m >= (UQuad)0x80000000U)
			return 0x7fffffffU;
		return (long)a.m;
	} else {
		if(a.m >= (UQuad)0x80000000U)
			return 0x80000000;
		return -(long)a.m;
	}
}

/*
 * add two floats
 */
Float
F_add(Float a, Float b)
{
	Float c;

	if(a.s == b.s)
		f_add(&a, &b, &c);
	else
		f_sub(&a, &b, &c);

	return c;
}

Float
F_sub(Float a, Float b)
{
	Float c;

	b.s = 1 - b.s;
	if(a.s == b.s)
		f_add(&a, &b, &c);
	else
		f_sub(&a, &b, &c);

	return c;
}

/*
 * multiply a and b
 */
Float
F_mul(Float a, Float b)
{
	Float	c;
	UQuad	mask;
	int	i;

	/*
	 * multiplying with 0 gives 0
	 */
	if(a.e == 0 || b.e == 0)
		return f_null;

	/*
	 * sign and exponent are really simple
	 */
	c.s = a.s ^ b.s;
	c.e = (int)a.e - EOFFS + (int)b.e;

	c.m = 0;
	for(i = 0, mask = (UQuad)1; i < MBITS; i++, mask <<= 1) {
		c.m >>= 1;
		if((a.m & mask) != 0)
			c.m += b.m;
	}
	if((c.m & f_carry) != 0) {
		c.e++;
		c.m >>= 1;
	}
	return c;
}


/*
 * compute a/b
 * note: if a or b is 0 then 0 is returned
 */
Float
F_div(Float a, Float b)
{
	Float c;
	int i;

	/*
	 * look for simple and bad cases
 	 */
	if(a.e == 0 || b.e == 0)		/* what else can we do? */
		return f_null;

	/*
	 * sign is easy, exponent can drop to low
	 */
	c.s = a.s ^ b.s;
	if((int)(c.e = (int)a.e - ((int)b.e - EOFFS)) <= 0)
		return f_null;

	/*
	 * we generate 1 bit more and shift back later if we get carry
	 * this is for cases like 1.000/1.FFF where we otherwise could
	 * lose one bit
	 */
	c.m = 0;
	for(i = 0; i < MBITS+1; i++) {
		c.m <<= 1;
		if(a.m >= b.m) {
			c.m++;
			a.m -= b.m;
		}
		a.m <<= 1;
	}
	if((c.m & f_carry) != 0)
		c.m >>= 1;
	else if((int)--c.e == 0)
		return f_null;
	return c;
}


/*
 * a < b  -> +1
 * a == b ->  0
 * a > b  -> -1
 */
int
F_cmp(Float a, Float b)
{
	if(a.e == 0) {
		if(b.e == 0)
			return 0;			/* a == 0 && b == 0 */
		else if(b.s)
			return -1;			/* a == 0 && b < 0 */
		else
			return +1;			/* a == 0 && b > 0 */
	} else if(b.e == 0) {
		if(a.s)
			return +1;			/* a < 0 && b == 0 */
		else
			return -1;			/* a > 0 && b == 0 */
	} else if(a.s && !b.s)
		return +1;				/* a < 0 && b > 0 */
	else if(!a.s && b.s)
		return -1;				/* a > 0 && b < 0 */

	/* signs are equal and numbers not 0 */
	if(a.e > b.e)
		return a.s ? +1 : -1;
	if(a.e < b.e)
		return a.s ? -1 : +1;

	/* exponents and signs are equal */
	if(a.m > b.m)
		return a.s ? +1 : -1;
	if(a.m < b.m)
		return a.s ? -1 : +1;

	return 0;
}

/*
 * return 'broken' part, store integer part in *ip
 */
Float
F_mod(Float a, Float *ip)
{
	Float c;
	int e;

	/*
	 * a number bigger or equal 2^MBITS is integer anyway
	 */
	if(a.e == 0 || (int)a.e >= (int)EOFFS + MBITS) {
		*ip = a;
		return f_null;
	}

	/*
	 * if the (unoffsetted) exponent is lesser than 0 the number is < 1.0
	 */
	if(a.e < EOFFS) {									/* all 'broken' */
		*ip = f_null;
		return a;
	}

	/*
	 * compute number of bits that are before the dot
	 * we clear the other bits (behind the dot) by shifting them
	 * out
	 */
	e = (int)a.e - EOFFS + 1;
	ip->m = a.m;
	ip->m >>= MBITS - e;
	ip->m <<= MBITS - e;
	ip->e = a.e;
	ip->s = a.s;

	c = F_sub(a, *ip);
	return c;
}


/*
 * real add routine
 * add the absolute values |c|=|a|+|b|, 
 * result has sign of bigger operand
 */
static void
f_add(Float *a, Float *b, Float *c)
{
	Float *t;

	/*
	 * handle simple cases
	 */
	if(a->e == 0) {		/* a is 0: 0+b=b */
		*c = *b;
		return;
	}
	if(b->e == 0) {		/* b is 0: a+0=a */
		*c = *a;
		return;
	}

	/* 
 	 * a and b are normal and != 0
	 * now make exponent(a) >= exponent(b)
	 */
	if(a->e < b->e) {
		t = a;
		a = b;
		b = t;
	}

	/*
	 * now we now result sign and exponent
	 * mantissa needs alignement if exponents differ
	 */
	c->s = a->s;
	c->e = a->e;
	if(a->e != b->e)	/* could optimize c->e - b->e >= MBITS */
		b->m >>= c->e - b->e;

	/* 
	 * add and don't lose msb carry 
	 * need maximum one shift to normalize
	 */
	c->m = a->m + b->m;
	if(c->m & f_carry) {
		c->m >>= 1;
		c->e++;
	}
}


/*
 * real subtract
 * subtract absolute values |c|=||a|-|b||
 * result has sign of bigger operand 
 */
static void
f_sub(Float *a, Float *b, Float *c)
{
	Float *t;

	/*
	 * handle simple cases
	 */
	if(a->e == 0) {
		*c = *b;
		return;
	}
	if(b->e == 0) {
		*c = *a;
		return;
	}

	/*
	 * now a and b are normal and not 0
  	 * ensure exponent(a) >= exponent(b)
	 */
	if(b->e > a->e) {
		t = a;
		a = b;
		b = t;
	}

	/*
	 * sign and exponent
	 */
	c->s = a->s;
	c->e = a->e;

	if(b->e != c->e) {
		b->m >>= c->e - b->e;	/* exponent(a) > exponent(c) */
		c->m = a->m - b->m;	/* mant(a) > mant(b) here */
	} else if(a->m == b->m) {	/* gives exact 0 */
		*c = f_null;
		return;
	} else if(a->m > b->m) {	/* no sign change */
		c->m = a->m - b->m;
	} else {			/* b->m > a->m -> sign changes */
		c->s = b->s;
		c->m = b->m - a->m;
	}

	/*
	 * now need to normalize
	 */
	if((c->m & f_msb) == 0)
		f_norm(c);
}

/*
 * normalize number
 */
static void
f_norm(Float *a)
{
	UQuad m;
	int i;

	if(a->m == 0) {
		*a = f_null;
		return;
	}

	/*
	 * find the first 1 bit in mantissa
	 */
	for(i = -1, m = f_carry; (a->m & m) == 0; m >>= 1, i++)
		;

	/*
	 * loss of precision
	 */
	if((int)a->e - i <= 0) {
		*a = f_null;
		return;
	}
	a->m <<= i;
	a->e = (int)a->e - i;
}
