/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * PDP11
 *
 * this is an implementation of a memory that pre-translates
 * the page boundaries to host physical addresses
 */
# include "proc.h"

typedef struct Space	Space;
typedef struct SpaceReg	SpaceReg;

struct SpaceReg {
	unsigned	par;		/* page address 		*/
	ushort		pdr;		/* orig pdr			*/
	ushort		*ppdr;		/* pointer to real pdr		*/
	uchar		*base;		/* page base			*/
	uchar		*high;		/* upper limit			*/
	uchar		*low;		/* lower limit			*/
	int		w;		/* write access			*/
	int		r;		/* r/w access			*/
};
	
struct Space {
	SpaceReg	r[8];		/* registers			*/
};

struct MMG {
	Space		s[4][2];	/* address spaces		*/
	Space		*a[4][2];	/* access spaces		*/
	SpaceReg	*s_last;	/* last accessed space		*/
	int		mmr0;		/* copy of proc.mmr0		*/
	int		mmr3;		/* copy of proc.mmr3		*/
	ushort	mem[Off(PHYSMEM)];	/* real memory			*/
};

static void 	comp_all(MMG *);
static void 	comp_page(MMG *, int, int, int);

void		sm_reset(MMG *);		/* bus reset		*/
void		sm_mmr0(MMG *);			/* mmr0 written		*/
void		sm_mmr3(MMG *);			/* mmr3 written		*/
void		sm_par(MMG *, int, int, int);	/* par written		*/
void		sm_pdr(MMG *, int, int, int);	/* pdr written		*/
void		sm_ccr(MMG *);			/* ccr written		*/
void		sm_info(MMG *);			/* odt info		*/
void		sm_store16(unsigned, int, int, int, ushort);
void		sm_store18(unsigned, int, int, int, ushort);
void		sm_store22(unsigned, int, int, int, ushort);
unsigned	sm_fetch16(unsigned, int, int);
unsigned	sm_fetch18(unsigned, int, int);
unsigned	sm_fetch22(unsigned, int, int);
ushort  	*sm_addrphys(MMG *, unsigned);	/* for dma		*/

MMG		mmg;		/* can have only one		*/
Memops		memops = {
	sm_reset,		/* reset	*/
	sm_mmr0,		/* mmr0		*/
	sm_mmr3,		/* mmr3		*/
	sm_par,			/* par		*/
	sm_pdr,			/* pdr		*/
	sm_ccr,			/* ccr		*/
	sm_info,		/* info		*/
	sm_store16,		/* store16	*/
	sm_store18,		/* store18	*/
	sm_store22,		/* store22	*/
	sm_fetch16,		/* fetch16	*/
	sm_fetch18,		/* fetch18	*/
	sm_fetch22,		/* fetch22	*/
	sm_addrphys,		/* addrphys	*/
};


/*
 * 16-bit mapping.
 */
unsigned
sm_fetch16(unsigned a, int dspace, int mode)
{
	if(a >= 0160000)
		return iofetch(a | 017760000);
	if(a >= proc.physmem)
		Trap4(040);
	return GETW(mmg.mem[Off(a)]);
}

void
sm_store16(unsigned a, int type, int dspace, int mode, ushort v)
{
	if(a >= 0160000) {
		iostore(a | 017760000, type, v);
		return;
	}
	if(a >= proc.physmem)
		Trap4(040);

	if(type & M_Low)
		SETLB(mmg.mem[Off(a)], v);
	else if(type & M_High)
		SETHB(mmg.mem[Off(a)], v);
	else
		SETW(mmg.mem[Off(a)], v);
}


/* 
 * 18/22 bit mapping
 */
# define GEN(NAME, SIZE, OR)						\
/*									\
 * Translate logical to physical address with check.			\
 */									\
static inline uchar *							\
physaddr##NAME(unsigned a, int mode, int dspace, int store)		\
{									\
	int 		page = a >> 13;					\
	uchar 		*pa;						\
	Space		*s;						\
	SpaceReg	*r;						\
									\
	s = mmg.a[mode][dspace];					\
	r = mmg.s_last = &s->r[page];					\
									\
	/*								\
	 * no need to check on mode==2 because mode2 r field is always 0\
	 */								\
	if(!r->r)							\
		MemTrap(page, mode, dspace, 0100000);			\
	if(store && !r->w)						\
		MemTrap(page, mode, dspace, 020000);			\
									\
	pa = r->base + (a & 017777);					\
									\
	if(pa >= r->high || pa < r->low)				\
		MemTrap(page, mode, dspace, 040000);			\
									\
	/* 								\
	 * the following implements the wrap around at the end of the	\
	 * physical address space					\
	 */								\
	if(pa >= (uchar *)&mmg.mem[Off(SIZE)])				\
		pa -= SIZE;						\
									\
	return pa;							\
}									\
									\
									\
/*									\
 * Fetch from logical address a, in space 				\
 */									\
unsigned								\
sm_fetch##NAME(unsigned a, int dspace, int mode)			\
{									\
	uchar *pa = physaddr##NAME(a, mode, dspace, 0);			\
									\
	if(pa >= (uchar *)&mmg.mem[Off(SIZE-020000)])			\
		return iofetch((pa - (uchar *)&mmg.mem[Off(SIZE-020000)]) | OR);\
									\
	if(pa >= (uchar *)&mmg.mem[Off(PHYSMEM)])			\
		Trap4(040);						\
									\
	return GETW(*(ushort *)pa);						\
}									\
									\
									\
									\
void									\
sm_store##NAME(unsigned a, int type, int dspace, int mode, ushort v)	\
{									\
	uchar *pa = physaddr##NAME(a, mode, dspace, 1);			\
									\
	if(pa >= (uchar *)&mmg.mem[Off(SIZE-020000)]) {			\
		iostore((pa - (uchar *)&mmg.mem[Off(SIZE-020000)]) | OR, type, v);\
		*mmg.s_last->ppdr |= 0100;				\
		return;							\
	}								\
									\
	if(pa >= (uchar *)&mmg.mem[Off(PHYSMEM)])			\
		Trap4(040);						\
									\
	if(type & M_Low)						\
		SETLB(*(ushort *)pa, v);				\
	else if(type & M_High)						\
		SETHB(*(ushort *)pa, v);				\
	else								\
		SETW(*(ushort *)pa, v);					\
	*mmg.s_last->ppdr |= 0100;					\
}

GEN(18,  01000000, 017760000)
GEN(22, 020000000, 017760000)

/*
 * routines called from generic interface
 */
void
mem_init(void)
{
	int i, j, k;

	for(i = 0; i < 4; i++)
		mmg.a[i][0] = mmg.a[i][1] = &mmg.s[i][0];
	for(i = 0; i < 4; i++)
		for(j = 0; j < 2; j++)
			for(k = 0; k < 8; k++)
				mmg.s[i][j].r[k].ppdr = &proc.pdr[i][j][k];

	proc.mmg = &mmg;
	proc.memops = &memops;
	memsys.data = &mmg;
}

void	
sm_reset(MMG *m)
{
	sm_mmr0(m);
	sm_mmr3(m);
}

void
sm_mmr0(MMG *m)
{
	/*
	 * if mapping is switched on set funcs and compute all registers
 	 */
	if((m->mmr0 & 1) != (proc.mmr0 & 1)) {
		comp_all(m);
	}
	m->mmr0 = proc.mmr0 & 1;
}

void
sm_mmr3(MMG *m)
{
	if((m->mmr3 & 020) != (proc.mmr3 & 020))
		comp_all(m);

	/*
	 * recompute pointers to accessible address spaces 
	 */
	m->a[0][1] = &m->s[0][(proc.mmr3 & 4) != 0];
	m->a[1][1] = &m->s[1][(proc.mmr3 & 2) != 0];
	m->a[3][1] = &m->s[3][(proc.mmr3 & 1) != 0];
	m->mmr3 = proc.mmr3;
}


void
sm_par(MMG *m, int mode, int space, int reg)
{
	if(m->s[mode][space].r[reg].par != proc.par[mode][space][reg]) {
		m->s[mode][space].r[reg].par = proc.par[mode][space][reg];
		comp_page(m, mode, space, reg);
	}
}


void
sm_pdr(MMG *m, int mode, int space, int reg)
{
	if(m->s[mode][space].r[reg].pdr != (proc.pdr[mode][space][reg] & 077416)) {
		m->s[mode][space].r[reg].pdr = proc.pdr[mode][space][reg] & 077416;
		comp_page(m, mode, space, reg);
	}
}

void
sm_ccr(MMG *m)
{
	/* nothing to do */
}

void
sm_info(MMG *m)
{
	printf("Pointer translated memory\n");
}

/*************************************************************
 *
 * get an ushort* for a physical address for dma
 * functions for monitor and dma support
 */
ushort *
sm_addrphys(MMG *m, unsigned a)
{
	return &m->mem[Off(a & ~1)];
}

/*************************************************************
 *
 * something changed - recompute pointers
 */
static void
comp_all(MMG *m)
{
	int reg;

	for(reg = 0; reg < 8; reg++) {
		comp_page(m, 0, 0, reg);
		comp_page(m, 0, 1, reg);
		comp_page(m, 1, 0, reg);
		comp_page(m, 1, 1, reg);
		comp_page(m, 3, 0, reg);
		comp_page(m, 3, 1, reg);
	}
}

static void
comp_page(MMG *m, int mode, int space, int page)
{
	Space		*s = &m->s[mode][space];
	SpaceReg	*r = &s->r[page];
	unsigned	base, low, high, bno;

	if(!(r->r = r->pdr & 02))
		return;
	r->w = r->pdr & 04;

	base = r->par << 6;
	bno  = (r->pdr & 077400) >> 2;
	if(r->pdr & 010) {
		high = base + 020000;
		low  = base + bno;
	} else {
		high = base + bno + 0100;
		low  = base;
	}
	r->base = (uchar *)&m->mem[Off(base)];
	r->low  = (uchar *)&m->mem[Off(low)];
	r->high = (uchar *)&m->mem[Off(high)];
}

