/* Copyright (c)1994-1999 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

/*
 * PDP11
 *
 * Generic memory interface
 */
# include "proc.h"

/*
# define IODBG
*/

static void	set_mapfunc(void);
static ushort 	get_mmr1(void);
static void	freeze_mmr(void);


/********
 *
 * I/O page
 */
ushort 
iofetch(unsigned pa)
{
	IODev *dev;


	if(!(dev = proc.iopage[IOP(pa)]) || !dev->ioops->fetch) {
# ifdef IODBG
		printf("iofetch(%o) *******\n", pa);
# endif
		Trap4(020);
	}

# ifdef IODBG
	if(pa < 017765000 || (pa >= 017766000 && pa < 01777300) ||
	   (pa >= 017774000 && pa < 017777776))
		printf("iofetch(%o) %s\n", pa, dev->name);
# endif

	return (*dev->ioops->fetch)(dev, pa);
}


void 
iostore(unsigned pa, int type, ushort v)
{
	IODev *dev;

	if(!(dev = proc.iopage[IOP(pa)]) || !dev->ioops->store) {
# ifdef IODBG
		printf("iofetch(%o) *******\n", pa);
# endif
		Trap4(020);
	}

# ifdef IODBG
	if(pa < 017765000 || (pa >= 017766000 && pa < 01777300) ||
	   (pa >= 017774000 && pa < 017777776))
		printf("iofetch(%o) %s\n", pa, dev->name);
# endif
	(*dev->ioops->store)(dev, pa, type, v);
}

/************
 *
 * Trap
 */
volatile void
MemTrap(int page, int mode, int dspace, ushort err)
{
	if(!proc.mmfrozen) {
		proc.mmr0 &= 1;
		proc.mmr0 |= err;
		proc.mmr0 |= mode << 5;
		proc.mmr0 |= dspace << 4;
		proc.mmr0 |= page << 1;
		freeze_mmr();
	}
	Trap(0250);
}


/**********
 * 
 * Memory manager I/O
 */
enum {
	MMG_0		= 017777572,
	MMG_1		= 017777574,
	MMG_2		= 017777576,
	MMG_3		= 017772516,
	MMG_UIPDR0	= 017777600,
	MMG_SIPDR0	= 017772200,
	MMG_KIPDR0	= 017772300,
	MEM_ERROR	= 017777744,
	MEM_CACHE	= 017777746,
	MEM_HITMISS	= 017777752,
};

void	memsys_ctrl_complete(IODev *);
void	memsys_reset(IODev *);
ushort	memsys_fetch(IODev *, unsigned);
void	memsys_store(IODev *, unsigned, int, ushort);
void	memsys_info(IODev *);

static int memsysc_info(IODev *dev, int argc, char **argv);

IODevCmd memsys_cmds[] = {
	{ "?",		"[command ...]",	dev_help },
	{ "help",	"[command ...]",	dev_help },
	{ "info",	"",			memsysc_info },
	{ NULL }
};

IOOps memsys_ops = {
	0,			/* ctrl_create 		*/
	0,			/* dev_create		*/
	memsys_ctrl_complete,	/* ctrl_complete	*/
	memsys_reset,		/* reset		*/
	memsys_fetch,		/* fetch		*/
	memsys_store,		/* store		*/
	0,			/* vector		*/
	0,			/* dma			*/
	0,			/* async		*/
	memsys_info,		/* info			*/
	0,			/* flush 		*/
	memsys_cmds		/* cmds */
};


void	
memsys_ctrl_complete(IODev *dev)
{
	int i;

	mem_init();

	proc.iopage[IOP(MMG_0)] = dev;
	proc.iopage[IOP(MMG_1)] = dev;
	proc.iopage[IOP(MMG_2)] = dev;
	proc.iopage[IOP(MMG_3)] = dev;
	proc.iopage[IOP(MEM_CACHE  )] = dev;
	proc.iopage[IOP(MEM_HITMISS)] = dev;
	proc.iopage[IOP(MEM_ERROR  )] = dev;
	
	for(i = 0; i < 0100; i += 2) {
		proc.iopage[IOP(MMG_UIPDR0 + i)] = dev;
		proc.iopage[IOP(MMG_SIPDR0 + i)] = dev;
		proc.iopage[IOP(MMG_KIPDR0 + i)] = dev;
	}
	
	set_mapfunc();
}

void	
memsys_reset(IODev *dev UNUSED)
{
	proc.mmr0 = 0;
	proc.mmr3 = 0;

	set_mapfunc();
	proc.memops->reset(proc.mmg);
}



ushort	
memsys_fetch(IODev *dev UNUSED, unsigned a)
{
	ushort	v;
	int	amode, aspace;

	switch(a) {
	
	case MMG_0:	
		v = proc.mmfrozen ? ((proc.fmmr0 & 0177776) | (proc.mmr0 & 1))
				  : proc.mmr0; 
		break;

	case MMG_1:	
		v = proc.mmfrozen ? proc.fmmr1 : get_mmr1();
		break;
	case MMG_2:	
		v = proc.mmfrozen ? proc.fmmr2 : proc.mmr2;
		break;

	case MMG_3:	
		v = proc.mmr3; 
		break;

	case MEM_CACHE: v = proc.ccr; break;
	case MEM_HITMISS:
		v = 0;
		break;
	case MEM_ERROR:
		v = 0;
		break;

	default:
		if(a >= MMG_KIPDR0 && a < MMG_KIPDR0 + 0100)
			amode = 0;
		else if(a >= MMG_SIPDR0 && a < MMG_SIPDR0 + 0100)
			amode = 1;
		else if(a >= MMG_UIPDR0 && a < MMG_UIPDR0 + 0100)
			amode = 3;
		else {
			warn("mmg_fetch(%o)", a);
			Trap4(020);
		}
		aspace = ((a & 020) != 0);
		if(a & 040)
			v = proc.par[amode][aspace][(a >> 1) & 7];
		else
			v = proc.pdr[amode][aspace][(a >> 1) & 7];
		break;
	}
	return v;
}


void
memsys_store(IODev *dev UNUSED, unsigned a, int mode, ushort v)
{
	int	amode, aspace, areg;

	switch(a) {
	case MMG_0:
		v &= 0160001;
		if(mode & M_Low)
			SLB(proc.mmr0, v);
		else if(mode & M_High)
			SHB(proc.mmr0, v);
		else
			proc.mmr0 = v;
		if(proc.mmr0 & 0160000)
			freeze_mmr();
		else
			proc.mmfrozen = 0;
		set_mapfunc();
		proc.memops->mmr0(proc.mmg);
		break;

	case MMG_1:	break;
	case MMG_2:	break;

	case MMG_3:	
		if(mode & M_High)
			break;
		proc.mmr3 = v & 077;
		set_mapfunc();
		proc.memops->mmr3(proc.mmg);
		break;

	case MEM_CACHE:
		if(mode & M_Low)
			SLB(proc.ccr, v);
		else if(mode & M_High)
			SHB(proc.ccr, v);
		else
			proc.ccr = v;
		proc.memops->ccr(proc.mmg);
		proc.ccr &= 03377;
		break;

	case MEM_HITMISS:
		break;

	case MEM_ERROR:
		break;
	
	default:
		if(a >= MMG_KIPDR0 && a < MMG_KIPDR0 + 0100)
			amode = 0;
		else if(a >= MMG_SIPDR0 && a < MMG_SIPDR0 + 0100)
			amode = 1;
		else if(a >= MMG_UIPDR0 && a < MMG_UIPDR0 + 0100)
			amode = 3;
		else {
			warn("mmg_store(%o)", a);
			Trap4(020);
		}
		aspace = ((a & 020) != 0);
		areg = (a >> 1) & 7;
		if(a & 040) {
			/*	
			 * write to PAR
			 */
			if(mode & M_Low)
				SLB(proc.par[amode][aspace][areg], v);
			else if(mode & M_High)
				SHB(proc.par[amode][aspace][areg], v);
			else
				proc.par[amode][aspace][areg] = v;
			proc.pdr[amode][aspace][areg] &= ~0100;
			proc.memops->par(proc.mmg, amode, aspace, areg);
		} else {
			/*
			 * write to PDR
			 */
			v &= 0177416;

			if(mode & M_Low)
				SLB(proc.pdr[amode][aspace][areg], v);
			else if(mode & M_High)
				SHB(proc.pdr[amode][aspace][areg], v);
			else
				proc.pdr[amode][aspace][areg] = v;
			proc.pdr[amode][aspace][areg] &= ~0100;
			proc.memops->pdr(proc.mmg, amode, aspace, areg);
		}
		break;
	}
}


static void
set_mapfunc(void)
{
	if(proc.mmr0 & 1) {
		if(proc.mmr3 & 020) {
			proc.fetch = proc.memops->fetch22;
			proc.store = proc.memops->store22;
		} else {
			proc.fetch = proc.memops->fetch18;
			proc.store = proc.memops->store18;
		}
	} else {
		proc.fetch = proc.memops->fetch16;
		proc.store = proc.memops->store16;
	}
}

void
memsys_info(IODev *dev UNUSED)
{
	proc.memops->info(proc.mmg);
}
static int
memsysc_info(IODev *dev, int argc, char **argv UNUSED)
{
	if(argc != 0)
		return 1;
	memsys_info(dev);
	return 0;
}

/*
 * it's not clear, how it should work
 */
static ushort
get_mmr1(void)
{
	ushort	u;

	u = ((proc.mmr1reg[1] &   7) << 0)
	  | ((proc.mmr1chg[1] & 037) << 3);
	if((proc.mmr1chg[0] & 037) != 0)
		u = (u << 8)
		  | ((proc.mmr1reg[0] &   7) << 0)
	  	  | ((proc.mmr1chg[0] & 037) << 3);
	return u;
}

/*
 * if they are already frozen - no changes
 */
static void
freeze_mmr(void)
{
	if(!proc.mmfrozen) {
		proc.mmfrozen = 1;
		proc.fmmr0 = proc.mmr0;
		proc.fmmr1 = get_mmr1();
		proc.fmmr2 = proc.mmr2;
	}
}
