/*
**	This program maintains and updates the passwd file
**	under the AUSAM system. The command has the following format.
**
**		pwedit [-m] [-u] [-k] [-n] [-x] [-c] [-pstring] [-tstring] [filename]
**
**	where	-k	allows keyword specifications in the input
**		-n	causes encryption of passwords to be suspended
**		-x	causes extraction of all files onto standard output
**		-m	causes initial directory creation on adds
**		-u	allows uids not in the default range to be specified
**		-c	causes comments to appear in password entry listings
**			from -x option and editor temporarys...
**		-pstring the string is to be used as the alternate password
**			file name
**		-tstring specifies a string to be used as a template filename
**			that is "/usr/lib/pwetplts/string"
**		filename specifies a file from which entries are to be taken
**			for operation in non-interactive mode
**
**		Commands which may be given are; add, delete, update,
**			change, look and default.
*/

#include	<local-system>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<passwd.h>
#include	<class.h>
#include	<stdio.h>
#include	<signal.h>
#include	<ctype.h>


/*	some nice little defines to make life easier	*/ 
#define	PWETMPS		"/tmp/"			/* where tmp files are kept */
#define	PWETPLTS	"/usr/lib/pwe/tplts/"	/* where the templates are */
#define	PWEDFLTS	"/usr/lib/pwe/dflts/"	/* where the defaults are */
#define	PWETDFLT	"default"		/* the default name */

#define	SSIZE		100	/* name string lengths */


/*	now the global data	*/ 

/*	flags	*/ 
int	tflag, nflag, xflag, cflag, fflag, mflag, uflag;
int	sigstop;
int	status;	/* status used for waits */ 
int	owuid;	/* the caller's uid */
int	infield;	/* input item field being processed */

/*	signals */
extern
int	catsig();

struct csig
{
	char	s_signo;
	int	(*s_action)();
}
	csigs[] =
{
	{	SIGHUP,		SIG_IGN	}
	,{	SIGINT,		SIG_IGN	}
	,{	SIGQUIT,	SIG_IGN	}
	,{	SIGALRM,	catsig	}
	,{	SIGTERM,	catsig	}
	,{	SIGCPUTL,	catsig	}
};
#define	NCSIG	((sizeof csigs)/(sizeof csigs[0]))

/*	filename arrays	*/ 
char tname[SSIZE], dname[SSIZE], ename[SSIZE], fname[SSIZE];

/*	temp file templates for editor	*/ 
char *edtemp = "pweaXXXXX";

/*	the global last character	*/ 
int c;

/*	global passwd entry	*/ 
struct pwent pe;

/*	global default entry	*/ 
struct pwent de;

/*	putc and getc buffers	*/ 
FILE	*ibuf, *obuf;

/*	some definitions	*/ 
#define	ITEMKEY	1	/* a keyword item */
#define	ITEMNUM	2	/* a number item, may be byte,int or long */
#define	ITEMSTR	3	/* string item */
#define	ITEMBIT	4	/* a bit string */
#define	ITEMNL	5	/* an empty line */
#define	ITEMTERM 6	/* a terminal group spec in {}'s */

/*	definition of ITEMKEY command codes for internal use	*/ 
#define	ADDCOM	0	/* add command */
#define	DELCOM	1	/* delete command */
#define	CHNGCOM	2	/* change command */
#define	UPDTCOM	3	/* update command */
#define	LOOKCOM	4	/* lookat command */

/*	string to code matching	*/ 
char *commands[] =
{
	"add",		/* ADDCOM */ 
	"delete",	/* DELCOM */ 
	"change",	/* CHNGCOM */ 
	"update",	/* UPDTCOM */ 
	"look",		/* LOOKCOM */
	0
};

/*	some handles to hang on to things by	*/ 
long	pnum;		/* the global read number	*/ 
int	pcom;		/* type of command, +ve if known, -ve if not */ 
int	scom;		/* a place to keep it safe */ 
char pclass[CLASSMASKSIZE];	/* the present read class mask bits string */ 
char	*pstr;		/* the global read string */ 


main(argc, argv)
	int	argc;
	char	*argv[];
{
	/*	initialise io	*/ 


	ibuf = stdin;
	obuf = stdout;

#ifdef	DEBUG
	fprintf(obuf, "main\n");
	fflush(obuf);
#endif	DEBUG
	owuid = getuid();

	/*	get args and stuff	*/ 
	getargs(argc, argv);

	/*	initialise defaults etc	*/ 
	initialise();

	/*
	**	now have serveral possibilities.
	**	extraction mode, file input mode, or interaction.
	*/ 
	if(xflag)
		xtract();
	else
	{
		register struct csig	*sp;

		/*	initialise nasty signals	*/ 
		for ( sp = csigs ; sp < &csigs[NCSIG] ; sp++ )
			if ( signal( sp->s_signo, SIG_IGN ) != SIG_IGN )
				signal( sp->s_signo, sp->s_action );

		if(fflag)
			readfile();
		else
			interact();
	}

}


getargs(ac, av)
	register int	ac;
	register char	**av;
{
	/*	things with a '-' are flags, without is a filename	*/ 

#ifdef	DEBUG
	fprintf(obuf, "getargs\n");
	fflush(obuf);
#endif	DEBUG
	while(--ac)
	{
		if(**++av == '-')
		{
			switch(av[0][1])
			{
		    case 't':
				settname(*av+2);
				tflag++;
				break;
		    case 'p':
				setpname(*av+2);
				break;
		    case 'm':
				mflag++;
				break;
		    case 'u':
				uflag++;
				break;
		    case 'n':
				nflag++;
				break;
		    case 'x':
				xflag++;
				break;
		    case 'c':
				cflag++;
				break;
		    default:
				error(1, *av);
				break;
			}
		}
		else
		{
			setfname(*av);
			fflag++;
		}
	}

	if ( (!xflag) && (!fflag) )
	{
		struct stat	statb[3];

		if ( fstat(0, &statb[0]) != SYSERROR
		   && fstat(2, &statb[2]) != SYSERROR
		   &&  ( statb[0].st_dev != statb[2].st_dev
			|| statb[0].st_ino != statb[2].st_ino
		   )   )
			fflag++;
	}
#ifdef	DEBUG
	fprintf(obuf, "tflag=%d,nflag=%d,xflag=%d\n", tflag, nflag, xflag);
	fflush(obuf);
	fprintf(obuf, "tname->%s<-,dname->%s<-,fname->%s<-\n", tname, dname, fname);
	fflush(obuf);
#endif	DEBUG
}


initialise()
{
	/*
	**	set default file names if necessary, open template and default files
	**	and initialise default pwent.
	*/ 

#ifdef	DEBUG
	fprintf(obuf, "initialise\n");
	fflush(obuf);
#endif	DEBUG
	if(!tflag)
	{
		strcpy(tname, PWETPLTS);
		strcat(tname, PWETDFLT);
		strcpy(dname, PWEDFLTS);
		strcat(dname, PWETDFLT);
	}
	getdefault();
}


settname(s)
	register char	*s;
{
	/*	make up the filenames for templates and defaults	*/ 

#ifdef	DEBUG
	fprintf(obuf, "settname\n");
	fflush(obuf);
#endif	DEBUG
	strcpy(tname, PWETPLTS);
	strcat(tname, s);
	strcpy(dname, PWEDFLTS);
	strcat(dname, s);
}


setpname(s)
	char	*s;
{
	/*	set the default password file name	*/ 
#ifdef	DEBUG
	fprintf(obuf, "setpname\n");
	fflush(obuf);
#endif	DEBUG
	pwfile(s);
}


setfname(s)
	char	*s;
{
	/*	set input file name	*/ 

#ifdef	DEBUG
	fprintf(obuf, "setfname\n");
	fflush(obuf);
#endif	DEBUG
	strcpy(fname, s);
}


xtract()
{
	/*	explode whole passwd file onto SO with adds for each entry	*/ 
	register FILE *rsbuf;
	register int i;

#ifdef	DEBUG
	fprintf(obuf, "xtract\n");
	fflush(obuf);
#endif	DEBUG
	/*	save old output	*/ 
	rsbuf = obuf;
	fflush(obuf);
	obuf = stdout;

	/*	output	*/ 
	for(i = 0; i < PWTABSIZE; i++)
	{
		pe.pw_limits.l_uid = i;
		if(getpwlog(&pe, (char *)0, 0) == PWERROR)
			continue;
		/*	real entry, output it	*/ 
		if(getpwe(0, &pe) == PWERROR)
			continue;
		putpe("add", &pe, 1);
	}

	/*	restore old SO	*/ 
	fflush(obuf);
	obuf = rsbuf;
}


interact()
{
	/*
	**	it interacts with the cretins
	*/ 
	register int ri, rf;
	FILE *ribuf, *robuf;
	register cc;

#ifdef	DEBUG
	fprintf(obuf, "interact\n");
	fflush(obuf);
#endif	DEBUG
	while(1)
	{
		/* major loop */ 
		if(sigstop)
			break;
		rf = 0;
		if( owuid == 0)
		{
			do
			{
				putc('>', obuf);
				fflush(obuf);
			}
			while(((ri = getitem()) != ITEMKEY) && (ri != 0));
			if(ri == 0)
				break;
			scom = pcom;
		}
		else
			scom = LOOKCOM;

		if(scom != ADDCOM)
		{
			/* only need uid and lname if not an add command */ 
			fprintf(obuf, "lname: ");
			fflush(obuf);
			if((ri = getitem()) == 0)
				break;
			if(ri == ITEMSTR)
			{
				pe.pw_strings[LNAME] = pstr;
				rf |= 02;
			}
			if( rf == 0 || scom == DELCOM)
			{
				fprintf(obuf, "uid: ");
				fflush(obuf);
				if((ri = getitem()) == 0)
					break;
				if(ri == ITEMNUM)
				{
					/* got a number, not just nothing */ 
					if( pnum >= NUSERS )
					{
						error(15, (char *)0);
						continue;
					}
					pe.pw_limits.l_uid = (ushort)pnum;
					rf |= 01;
				}
			}
		}

		/* now have command and other data, just use it */ 
		if(scom == DELCOM)
		{
			/* delete command, must have both uid and lname */ 
			if((rf&03) != 03)
			{
				error(17, (char *)0);
			}
			else
			{
				if(delpwent(&pe) != 1)
					error(9, (char *)0);
			}
		}
		else
		{
			/* for add command or flags 01|02 must go on */ 
			if((scom == ADDCOM) || (rf&03))
			{
				fflush(obuf);
				robuf = obuf;
				strcpy(ename, PWETMPS);
				strcat(ename, edtemp);
				mktemp(ename);
				if ( (obuf = fopen(ename, "w")) == NULL )
					error(0, ename);

				if(scom == ADDCOM)
				{
					/* for an add must copy over template */ 
					ribuf = ibuf;
					if ( (ibuf = fopen(tname, "r")) == NULL )
						error(0, fname);
					while((cc = getc(ibuf)) != EOF)
						putc(cc, obuf);
					fclose(ibuf);
					ibuf = ribuf;
				}
				else
				{
					/* must be change or update */ 
					if(rf&01)
					{
						if(getpwe(0, &pe) == PWERROR)
							rf |= 04;
					}
					else
					{
						if(getpwe(1, &pe) == PWERROR)
							rf |= 04;
					}

					/* free up the interactive login name */ 

					if(!(rf&04))
						putpe(commands[scom], &pe, 1);
				}

				/* finished with obuf */ 
				fflush(obuf);
				fclose(obuf);
				obuf = robuf;

				if(!(rf&04))
				{
					/* return input absorbed into buffer */
					fseek(ibuf, (long)-(ibuf->_cnt), 1);
					ibuf->_cnt = 0;
					/* now spawn an elfic ed */ 
					switch ( fork() )
					{
					 default:
						/* parent */ 
						wait(&status);
					 case SYSERROR:
						break;

					 case 0:
						/* child */ 
						if(execl("/bin/em", "pwede", "-e", ename, 0) == SYSERROR)
						{
							error(16, (char *)0);
						}
					}

					/* now have returned so validate results */ 
					if(scom != LOOKCOM)
					{
						ribuf = ibuf;
						if ( (ibuf = fopen(ename, "r")) == NULL )
							error(0, ename);
						if(getpe(&pe) < 0)
						{
							error(5, (char *)1);
							rf |= 04;
						}
						else
							if(putpw(&pe) == PWERROR)
								error(6, (char *)1);
						fclose(ibuf);
						ibuf = ribuf;
					}
				}
				unlink(ename);
			}
		}
	}
}


putpe(rs, rpe, ff)
	register char		*rs;
	register struct pwent	*rpe;
	int			ff;
{
	/*
	**	Output the pwent struct at *rpe with the string at *rs as the
	**	first line of output.
	**	An ff of 1 causes freeing of string area after output
	*/ 
	register int	i;
	static char	decstr[]	= "%d%s\n";
	static char	longstr[]	= "%ld%s\n";
	static char	nullstr[]	= "";

#ifdef	DEBUG
	fprintf(obuf, "putpe\n");
	fflush(obuf);
#endif	DEBUG
	/*	command out	*/ 
	fprintf(obuf, "%s\n", rs);

	/*	now all other entries	*/ 
	fprintf(obuf, decstr, rpe->pw_limits.l_uid, cflag?"\t/* uid */":nullstr);
	fprintf(obuf, decstr, rpe->pw_gid, cflag?"\t/* gid */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_shares, cflag?"\t/* shares */":nullstr);
	fprintf(obuf, longstr, rpe->pw_limits.l_usage, cflag?"\t/* usage */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_nice, cflag?"\t/* nice */":nullstr);
	printclass(rpe);
	printterms(rpe);
	fprintf(obuf, "0%o%s\n", rpe->pw_limits.l_flags, cflag?"\t/* flags */":nullstr);
	fprintf(obuf, "0%o%s\n", rpe->pw_xflags, cflag?"\t/* xflags */":nullstr);
	fprintf(obuf, decstr, rpe->pw_reason, cflag?"\t/* reason */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_dlimit, cflag?"\t/* dlimit */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_doverflw, cflag?"\t/* doverflw */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_duse, cflag?"\t/* duse */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_plimit, cflag?"\t/* plimit */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_mlimit, cflag?"\t/* mlimit */":nullstr);
	fprintf(obuf, decstr, rpe->pw_limits.l_mplimit, cflag?"\t/* mplimit */":nullstr);
	if(rpe->pw_pword[0] == 0)
		fprintf(obuf, "\"\"");
	else
	{
		fprintf(obuf, "\"%.*s\"", CRYPTLEN, rpe->pw_pword);
	}
	if (cflag) fprintf(obuf, "\t/* password */");
	fprintf(obuf, "\n");
	fprintf(obuf, longstr, rpe->pw_age, cflag?"\t/* password age */":nullstr);
	fprintf(obuf, longstr, rpe->pw_contime, cflag?"\t/* contime */":nullstr);
	fprintf(obuf, longstr, rpe->pw_cputime, cflag?"\t/* cputime */":nullstr);
	fprintf(obuf, longstr, rpe->pw_extime, cflag?"\t/* extime */":nullstr);
	fprintf(obuf, decstr, rpe->pw_warn, cflag?"\t/* warnings */":nullstr);
	fprintf(obuf, decstr, rpe->pw_pages, cflag?"\t/* page limit*/":nullstr);
	fprintf(obuf, decstr, rpe->pw_pgused, cflag?"\t/* pages used */":nullstr);
#ifdef	TERMBOOK
	fprintf(obuf, decstr, rpe->pw_tblim & 0377, cflag?"\t/* tblim */":nullstr);
	fprintf(obuf, decstr, rpe->pw_tbrate, cflag?"\t/* tbrate */":nullstr);
#endif	TERMBOOK

	for(i = 0; i < PWSLENCNT; i++)
		if(rpe->pw_strings[i] != 0)
		{
			fprintf(obuf, "\"%s\"\n", rpe->pw_strings[i]);
		}
		else
		{
			error(2, (char *)i);
		}
	if(ff)
	{
		free(rpe->pw_strings[0]);
		for(i = 0; i < PWSLENCNT; i++)
			rpe->pw_strings[i] = 0;
	}
}


 int
getpwe(ct, rpe)
	int			ct;
	register struct pwent	*rpe;
{
	/*
	**	using call type, ct, gets an entry into *rpe. cts are:
	**		0	by uid
	**		1	by login name
	*/ 
	register int		rsum, ri;
	char			*as, *ss;
	extern char *		malloc();

#ifdef	DEBUG
	fprintf(obuf, "getpw\n");
	fflush(obuf);
#endif	DEBUG
	/*	first determine the string sizes	*/ 

	switch(ct)
	{
    case 0:	/* by uid */ 
		if(getpwlog(rpe, (char *)0, 0) == PWERROR)
		{
			error(12, (char *)0);
			return(PWERROR);
		}
		break;
    case 1:	/* by login name */ 
		ss = rpe->pw_strings[0];
		if(getpwuid(rpe, (char *)0, 0) == PWERROR)
		{
			error(13, (char *)0);
			return(PWERROR);
		}
		break;
	}

	for(ri = 0, rsum = 0; ri < PWSLENCNT; ri++)
	{
		rsum += (int)rpe->pw_strings[ri];
	}

	if((as = malloc(rsum)) == 0)
	{
		error(14, (char *)0);
		return(PWERROR);
	}

	switch(ct)
	{
    case 0:	/* by uid we get it */ 
		if(getpwlog(rpe, as, rsum) != rsum)
		{
			error(12, (char *)0);
			return(PWERROR);
		}
		break;
    case 1:	/* by login name */ 
		rpe->pw_strings[0] = ss;
		if(getpwuid(rpe, as, rsum) != rsum)
		{
			error(13, (char *)0);
			return(PWERROR);
		}
		break;
	}

	return 0;
}


 int
putpw(rpe)
	register struct pwent	*rpe;
{
	/*
	**	add change or update a passwd file entry using the stuff in *rep
	*/ 
	register int		ri;
	register		ret = PWERROR;

#ifdef	DEBUG
	fprintf(obuf, "putpw\n");
	fflush(obuf);
#endif	DEBUG
	switch(scom)
	{
    case ADDCOM:	/* add */ 
		/* enforce the bounds limits for this site */ 
		if(!uflag)
		{
			rpe->pw_limits.l_uid = UIDLB;
			rpe->pw_gid = GIDLB;
		}
		if(addpwent(rpe) == PWERROR)
			error(8, rpe->pw_strings[LNAME]);
		else
		/* make directories etc if needed */ 
		if(mflag)
			mkit();
		ret = 0;
		break;
    case CHNGCOM:	/* change */ 
		if(chngpwent(rpe) != 1)
			error(10, (char *)0);
		else
			ret = 0;
		break;
    case UPDTCOM:	/* update */ 
		if(updtpwent(rpe) != 1)
			error(11, (char *)0);
		else
			ret = 0;
		break;
    default:
		ret = 0;
		break;
	}

	for(ri = 0; ri < PWSLENCNT; ri++)
	{
		free(rpe->pw_strings[ri]);
		rpe->pw_strings[ri] = 0;
	}

	return ret;
}




readfile()
{
	/*
	**	opens file called "*fname" and decodes input to form
	**	new and modified passwd entries. Sets up input buffers also.
	*/ 
	register FILE *rsbuf;
	register int ri, rn;

	/*	save old input buffer	*/ 
#ifdef	DEBUG
	fprintf(obuf, "readfile\n");
	fflush(obuf);
#endif	DEBUG
	rsbuf = ibuf;

	/*	open and use the new stuff	*/ 

	if ( *fname == '\0' )
		ibuf = stdin;
	else
		ibuf = fopen(fname, "r");
	if ( ibuf == NULL )
		error(0, (*fname=='\0'?"stdin":fname));

	for(rn=0 ; c!=EOF ;)
	{
		if(sigstop)
			break;
		if ( (ri=getpe(&pe)) < 0 )
		{
			if ( c != EOF )
				error(5, (char *)rn);
			break;
		}

		rn++;

		if(ri >= 0 && putpw(&pe) == PWERROR)
		{
			error(6, (char *)rn);
			break;
		}
	}
	fprintf(stdout, "Pwedit: %d items processed\n", rn);

	/*	flush and restore to exit	*/ 
	fflush(obuf);
	fclose(ibuf);
	ibuf = rsbuf;
}


getdefault()
{
	/*
	**	open files etc and read defaults into "&de".
	**	Note that defaults of zero force entries to be requested for them
	*/ 
	register FILE *rsbuf;

#ifdef	DEBUG
	fprintf(obuf, "getdefault\n");
	fflush(obuf);
#endif	DEBUG
	rsbuf = ibuf;
	if ( (ibuf = fopen(dname, "r")) == NULL )
		error(0, dname);
	if(getpe(&de) < 0)
		error(5, (char *)0);
	fclose(ibuf);
	ibuf = rsbuf;
}


getpe(rpe)
	register struct pwent	*rpe;
{
	/*
	**	reads and validates the entry from "ibuf" into "*rpe".
	**	returns: 1 for success, 0 for EOF, -1 for error
	*/ 
	register char		*rpw;
	register int		rn;

#ifdef	DEBUG
	fprintf(obuf, "getpe\n");
	fflush(obuf);
#endif	DEBUG
	infield = 0;

	/*	get command	*/ 
	if(getitem() != ITEMKEY)
		return(-1);
	if((scom = pcom) < 0)
		return(-1);

	/*	get uid	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_uid))
		return(-1);
	rpe->pw_limits.l_uid = (ushort)pnum;

	/*	get gid	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_gid))
		return(-1);
	rpe->pw_gid = (ushort)pnum;

	/*	get shares	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_shares))
		return(-1);
	rpe->pw_limits.l_shares = (ushort)pnum;

	/*	get usage	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_usage))
		return(-1);
	rpe->pw_limits.l_usage = pnum;

	/*	get nice	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_nice))
		return(-1);
	rpe->pw_limits.l_nice = (ushort)pnum;

	/*	read class mask bit string	*/ 
	if(getitem() != ITEMBIT)
		return(-1);
	for(rn = 0; rn < CLASSMASKSIZE; rn++)
		rpe->pw_limits.l_cmask[rn] = pclass[rn];

	/*	read terminal groups string	*/ 
	if(getitem() != ITEMTERM)
		return(-1);
	rpe->pw_tmask = (tmask_t)pnum;

	/*	get flags	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_flags))
		return(-1);
	rpe->pw_limits.l_flags = (ushort)pnum;

	/*	get xflags	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_xflags))
		return(-1);
	rpe->pw_xflags = (ushort)pnum;

	/*	get reason	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_reason))
		return(-1);
	rpe->pw_reason = (ushort)pnum;

	/*	get dlimit	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_dlimit))
		return(-1);
	rpe->pw_limits.l_dlimit = pnum;

	/*	get doverflw	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_doverflw))
		return(-1);
	rpe->pw_limits.l_doverflw = pnum;

	/*	get duse	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_duse))
		return(-1);
	rpe->pw_limits.l_duse = pnum;

	/*	get plimit	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_plimit))
		return(-1);
	rpe->pw_limits.l_plimit = pnum;

	/*	get mlimit	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_mlimit))
		return(-1);
	rpe->pw_limits.l_mlimit = (musage_t)pnum;

	/*	get mplimit	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_limits.l_mplimit))
		return(-1);
	rpe->pw_limits.l_mplimit = (musage_t)pnum;

	/*	get passwd	*/ 
	if(getitem() != ITEMSTR)
		return(-1);

	/*	and also get age which may be reset	*/
	if ( getitem() != ITEMNUM )		return -1;
	if ( !bcheck(sizeof rpe->pw_age) )	return -1;
	rpe->pw_age = pnum;

	if(*pstr == 0)
	{
		/* null passwd */ 
		for(rn = 0; rn < CRYPTLEN; rn++)
			rpe->pw_pword[rn] = 0;
	}
	else if((scom != ADDCOM) || (nflag))
	{
		/* no encryption */ 
		for(rn = 0; rn < CRYPTLEN; rn++)
			rpe->pw_pword[rn] = pstr[rn];
	}
	else
	{
		register	i;
		register long	salt;
		char		saltc[2];
		extern char *	crypt();
		extern long	time();

		/* encrypt */ 

		salt = time( &rpe->pw_age );

		for ( i = 0 ; i < sizeof saltc ; i++ )
		{
			register char	cc = (char)((salt>>(i*6))&077) + '.';

			if ( cc > '9' )
				cc += 7;
			if ( cc > 'Z' )
				cc += 6;
			saltc[i] = cc;
		}

		rpw = crypt(pstr,saltc);
		for(rn = 0; rn < CRYPTLEN; rn++)
			rpe->pw_pword[rn] = rpw[rn];
	}

	/*	get contime	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_contime))
		return(-1);
	rpe->pw_contime = pnum;

	/*	get cputime	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_cputime))
		return(-1);
	rpe->pw_cputime = pnum;

	/*	get extime	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_extime))
		return(-1);
	rpe->pw_extime = pnum;

	/*	get warn	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_warn))
		return(-1);
	rpe->pw_warn = pnum;

	/*	get pages	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_pages))
		return(-1);
	rpe->pw_pages = (pages_t)pnum;

	/*	get pgused	*/ 
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_pgused))
		return(-1);
	rpe->pw_pgused = (pages_t)pnum;

#ifdef TERMBOOK
	/*	get tblim	*/
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_tblim))
		return(-1);
	rpe->pw_tblim = (char)pnum;

	/*	get tbrate	*/
	if(getitem() != ITEMNUM)
		return(-1);
	if(!bcheck(sizeof rpe->pw_tbrate))
		return(-1);
	rpe->pw_tbrate = (char)pnum;

#endif	TERMBOOK

	/*	finally get strings	*/ 
	for(rn = 0; rn < PWSLENCNT; rn++)
	{
		if(getitem() != ITEMSTR)
			return(-1);
		if(rpe->pw_strings[rn] != 0)
		{
			free(rpe->pw_strings[rn]);
		}
		rpe->pw_strings[rn] = pstr;
		pstr = 0;
	}

	return c != EOF;
}


getitem()
{
	/*
	**	returns a number corresponding to the type of item read.
	**	return values are define previously. Returns the negative of the
	**	normal return for an error item. Returns 0 for eof.
	*/ 
	register int finished;
	register int rv;
	register int ri;
	extern char *	getstring();

#ifdef	DEBUG
	fprintf(obuf, "getitem\n");
	fflush(obuf);
#endif	DEBUG
	infield++;
	finished = 0;

	c = getc(ibuf);

	while(!finished)
	{
		if(c == EOF)
			return(0);
		if ( islower(c) )
		{
			/*	a keyword	*/ 
			rv = ITEMKEY;
			ungetc(c, ibuf);
			pstr = getstring(pstr, 0);
			for(ri = 0; (strcmp(commands[ri], pstr) != 0); ri++)
			{
				if(commands[ri] == '\0')
				{
					ri = -1;
					break;
				}
			}
			pcom = ri;
			finished = 1;
		}
		else
		if ( isdigit(c) )
		{
			/*	a number of some form	*/ 
			rv = ITEMNUM;
			if(c == '0')
			{
				if(getnum(8) < 0)
					rv = -rv;
			}
			else
			{
				if(getnum(10) < 0)
					rv = -rv;
			}
			finished = 1;

		}
		else
			switch(c)
			{
		    case '"':	/*	a string	*/ 
				rv = ITEMSTR;
				pstr = getstring(pstr, 1);
				finished = 1;
				break;
		    case '/':	/*	a comment	*/ 
				flushline();
				break;
		    case '`':	/*	a bit string	*/ 
				rv = ITEMBIT;
				if(getclass() < 0)
					rv = -rv;
				finished = 1;
				break;
		    case '{':	/* a terminal group spec */ 
				rv = ITEMTERM;
				if(getterms() < 0)
					rv = -rv;
				finished = 1;
				break;
		    case '\n':	/* a null line */ 
				rv = ITEMNL;
				finished = 1;
				break;
		    default:	/*	read another char and do again	*/ 
				c = getc(ibuf);
				break;
			}
	}

	flushline();
	return(rv);
}


getnum(rbase)
	register int	rbase;
{
	/*
	**	get a number terminated by a blank in base rbase
	*/ 
	register char	cc;

#ifdef	DEBUG
	fprintf(obuf, "getnum\n");
	fflush(obuf);
#endif	DEBUG
	pnum = 0;

	while(((cc = c-'0') >= 0) && (cc < rbase))
	{
		pnum = (pnum*rbase)+cc;
		c = getc(ibuf);
	}

	return(0);
}


printclass(rpe)
	register struct pwent	*rpe;
{
	/*
	**	prints the bitwise representation of the pw_limits.l_cmask
	*/ 
	register int		rn, ri;

#ifdef	DEBUG
	fprintf(obuf, "printclass\n");
	fflush(obuf);
#endif	DEBUG
	putc('`', obuf);
	for(rn = 0; rn < CLASSMASKSIZE; rn++)
	{
		/* byte by byte */ 
		for ( ri = 1<<(BITSINMASK-1) ; ri ; ri >>= 1 )
		{
			/* bit by bit */ 
			if ((rpe->pw_limits.l_cmask[rn]&ri) == 0)
				putc('0', obuf);
			else
				putc('1', obuf);
		}
		if ( rn < (CLASSMASKSIZE-1) )
			putc(' ', obuf);
	}
	putc('`', obuf);
	putc('\n', obuf);
}

printterms(rpe)
	register struct pwent	*rpe;
{
	/*
	**	prints the groupwise representation of pw_tmask
	*/ 
	register		rn, ri;

#ifdef	DEBUG
	fprintf(obuf, "printterms\n");
	fflush(obuf);
#endif	DEBUG

	rn = rpe->pw_tmask;
	putc('{', obuf);
	for(ri = 0; ri < ((sizeof rpe->pw_tmask)*BITSINBYTE); ri++)
	{
		if(rn&01)
		{
			putc('a'+ri, obuf);
		}
		rn >>= 1;
	}
	putc('}', obuf);
	putc('\n', obuf);
}


getclass()
{
	/*	read class mask into pclass	*/ 
	register int	ri, rn;

#ifdef	DEBUG
	fprintf(obuf, "getclass\n");
	fflush(obuf);
#endif	DEBUG

	for(rn = 0; rn < CLASSMASKSIZE; rn++)
	{
		pclass[rn] = 0;

		for ( ri = 1<<(BITSINMASK-1); ri; ri >>= 1 )
		{
			while ( (c = getc(ibuf)) == ' ' );
			if((c == '0') || (c == '1'))
			{
				if(c == '1')
					pclass[rn] |= ri;
			}
			else
				return(-1);
		}
	}

	return(0);
}

getterms()
{
	/*
	**	return terminal mask in pnum
	*/ 
	register	rc, ri;

#ifdef	DEBUG
	fprintf(obuf, "getterms\n");
	fflush(obuf);
#endif	DEBUG

	ri = 0;
	while((rc = getc(ibuf)) != EOF && rc != '\n')
	{
		if(rc == '}')
		{
			pnum = ri;
			return 0;
		}
		if(rc >= 'a' && rc <= 'p')	/* only valid groups so far */ 
		{
			ri ^= 1 << (rc-'a');
		}
		else if(rc == '!')
			ri = ~ri;
		else
			return -1;
	}
	return -1;
}


flushline()
{
	/* flushes ibuf until a '\n' or eof are reached */ 

#ifdef	DEBUG
	fprintf(obuf, "flushline\n");
	fflush(obuf);
#endif	DEBUG
	while((c != '\n') && (c != EOF))
	{
		c = getc(ibuf);
	}

}


bcheck(ri)
	register int	ri;
{
	/*
	**	checks to see if the global long pnum has a number small enough
	**	to fit in "ri" bytes.
	*/ 
#ifdef	DEBUG
	fprintf(obuf, "bcheck\n");
	fflush(obuf);
#endif	DEBUG
	switch(ri)
	{
	 case 1:	return (!(pnum&~0xff));
	 case 2:	return (!(pnum&~0xffff));
	 case 3:	return (!(pnum&~0xffffff));
	 case 4:	return 1;
	}

	return 0;
}



 char
readc(tm)
	int	tm;
{
	/*
	**	reads characters from the ibuf file one by one.
	**	returns characters, '\0' for EOF or end of string
	**	specified by value of tm.
	*/ 

#ifdef	DEBUG
	fprintf(obuf, "readc\n");
	fflush(obuf);
#endif	DEBUG
	if ( (c = getc(ibuf)) == EOF )
		return '\0';

	if(c == '\\')
	{
		c = getc(ibuf);
		return(c);
	}

	if(tm)
	{
		/* terminate with '"' */ 
		if(c == '"')
			return('\0');
	}
	else
	{
		/* terminate with ' ' or '\n' */ 
		if((c == ' ') || (c == '\n'))
			return('\0');
	}

	return(c);
}


 char *
getstring(ptr, tm)
	char		*ptr;
	int		tm;
{
	/*
	**	read a string from the input source specified by present ibuf.
	**	free the area pointed to by ptr (if non-zero). return a pointer
	**	to a newly alloced string, or 0 on error.
	**	tm specifies the termination method, as in 1 for a """ terminator
	**	and 0 for normal ones like " ", "\n" etc
	*/ 
	register char	*string;
	extern char *	realloc();
	extern char *	malloc();
	register	s, slen, finished;

#ifdef	DEBUG
	fprintf(obuf, "getstring\n");
	fflush(obuf);
#endif	DEBUG

	if(ptr)
	{
		free(ptr);
		ptr = 0;
	}

	if ( (string = malloc(slen = SSIZE)) == NULL )
		error(14, (char *)0);

	finished = string == NULL;
	s = 0;

	while(!finished)
	{
		while(s < slen)
		{
			if((string[s++] = readc(tm)) <= 0)
			{
				finished = 1;
				break;
			}
		}
		if(finished)
			string = realloc(string, slen);
		else
			string = realloc(string, slen += SSIZE);
	}

	return(string);
}


error(ri, rp)
	register int		ri;
	register char *		rp;
{
	/*	error output and actions	*/ 
	register FILE		*rsbuf;

	rsbuf = obuf;
	obuf = stdout;

	fprintf(obuf, "pwedit: ");

	switch(ri)
	{
    case 0:
		fprintf(obuf, "cannot open file %s\n", rp);
		fflush(obuf);
		exit(1);
		break;
    case 1:
		fprintf(obuf, "unknown argument %s\n", rp);
		break;
    case 2:
		fprintf(obuf, "string %d not used at present\n", rp);
		fflush(obuf);
		exit(1);
		break;
    case 3:
		fprintf(obuf, "cannot creat file %s\n", rp);
		fflush(obuf);
		exit(1);
		break;
    case 4:
		fprintf(obuf, "cannot alloc buffer area for file %s\n", rp);
		fflush(obuf);
		exit(1);
		break;
    case 5:
		fprintf(obuf, "error in input file in item %d at field %d\n", rp, infield);
		break;
    case 6:
		fprintf(obuf, "error writing passwd file at item %d\n", rp);
		fflush(obuf);
		exit(1);
		break;
    case 7:
		fprintf(obuf, "invalid character is %c\n", rp);
		fflush(obuf);
		exit(1);
		break;
    case 8:
		fprintf(obuf, "passwd add error login name %s\n", rp);
		break;
    case 9:
		fprintf(obuf, "passwd delete error\n");
		break;
    case 10:
		fprintf(obuf, "passwd change error\n");
		break;
    case 11:
		fprintf(obuf, "passwd update error\n");
		break;
    case 12:
		fprintf(obuf, "passwd getpwlog error\n");
		break;
    case 13:
		fprintf(obuf, "passwd getpwuid error\n");
		break;
    case 14:
		fprintf(obuf, "cannot alloc string area\n");
		fflush(obuf);
		exit(1);
		break;
    case 15:
		fprintf(obuf, "uid too large\n");
		break;
    case 16:
		fprintf(obuf, "editor execl failed\n");
		fflush(obuf);
		exit(1);
		break;
    case 17:
		fprintf(obuf, "not enough data for interactive action requested\n");
		break;
    case 18:
		fprintf(obuf, "cannot fork mkdir\n");
		break;
    case 19:
		fprintf(obuf, "cannot exec mkdir for %s\n", rp);
		fflush(obuf);
		exit(1);
		break;
    case 20:
		fprintf(obuf, "mkdir failed for %s\n", rp);
		break;
    case 21:
		fprintf(obuf, "chown failed for %s\n", rp);
		break;
    default:
		fprintf(obuf, "spurious error call\n");
		fflush(obuf);
		exit(1);
		break;
	}

	fflush(obuf);
	obuf = rsbuf;
}


catsig()
{
	register struct csig	*sp;

	for ( sp = csigs ; sp < &csigs[NCSIG] ; sp++ )
		if ( sp->s_action != SIG_IGN )
			signal( sp->s_signo, SIG_IGN );

	sigstop++;
}


mkit()
{
	register int cpid;


	if(cpid = fork())
	{
		/* parent */ 
		if(cpid == SYSERROR)
		{
			error(18, (char *)0);
		}
		else
		{
			wait(&status);
			if(status)
				error(20, pe.pw_strings[DIRPATH]);
			else
			{
				if(chown(pe.pw_strings[DIRPATH], pe.pw_limits.l_uid, pe.pw_gid) == SYSERROR)
					error(21, pe.pw_strings[DIRPATH]);
			}
		}
	}
	else
	{
		/* child */ 
		execl("/bin/mkdir", "mkdir", pe.pw_strings[DIRPATH], 0);
		error(19, pe.pw_strings[LNAME]);
	}
}


