/*
 * TJU16 tape driver
 *
 *	minor device number is ndcccttt (8 bits) where
 *
 *		n is the no-rewind bit
 *		d selects 800 bpi (instead of 1600)
 *		c is TM02 number (up to 8)
 *		t is transport number (up to 8)
 */

#include <param.h>
#include <systm.h>
#include <buf.h>
#include <conf.h>
#include <dir.h>
#include <file.h>
#include <signal.h>
#include <user.h>
#include <errno.h>

extern struct user u;

struct	device
{
	int	htcs1;	/* shared */
	int	htwc;	/* rh70 */
	caddr_t	htba;	/* rh70 */
	int	htfc;	/* tm02 */
	int	htcs2;	/* rh70 */
	int	htds;	/* tm02 */
	int	hter;	/* tm02 */
	int	htas;	/* tm02 */
	int	htck;	/* tm02 */
	int	htdb;	/* rh70 */
	int	htmr;	/* tm02 */
	int	htdt;	/* tm02 */
	int	htsn;	/* tm02 */
	int	httc;	/* tm02 */
	int	htbae;	/* rh70 */
	int	htcs3;	/* rh70 */
};

struct	buf	httab;		/* device queue header */
struct	buf	rhtbuf;		/* raw i/o buffer header */
struct	buf	chtbuf;		/* command buffer header */

#define	NUNIT	2
#define	HTADDR	((struct device *)0172440)

char	h_openf[NUNIT];	/* 0 closed; 1 unit open; -1 unit has gone off-line */
daddr_t	h_blkno[NUNIT];
daddr_t	h_nxrec[NUNIT];
#define	INF	1000000		/* a large number */


/*
 * htcs1
 */
#define	GO	001
#define	NOP	000	/* no-op */
#define	REW	006	/* rewind */
#define	DCLR	010	/* drive clear */
#define	ERASE	024	/* erase */
#define	WTM	026	/* write tape mark */
#define	SFORW	030	/* space forward */
#define	SREV	032	/* space reverse */
#define	WCOM	060	/* write forward */
#define	RCOM	070	/* read forward */

#define	IENABLE	0100	/* interrupt enable */
#define	RDY	0200	/* rh70 ready */
#define	TRE	040000	/* transfer error */
#define	SC	0100000	/* special condition */

/*
 * httc
 */
#define P800	01300	/* 800 + normal mode */
#define	P1600	02300	/* 1600 + normal mode */

/*
 * htcs2
 */
#define	CLR	040	/* clear rh70, all tm02s, and all transports */
#define	NED	010000	/* non-existent tm02 */

/*
 * htds
 */
#define	TM	04	/* tape mark detected */
#define PES	040	/* selected transport is in 1600bpi mode */
#define	SSC	0100	/* slave status change */
#define	DRY	0200	/* tm02 ready */
#define EOT	02000	/* eot tape marker detected */
#define WRL	04000	/* no write ring in tape */
#define MOL	010000	/* transport on line */
#define ERR	040000	/* tm02 error - set if any in hter are set */

/*
 * hter
 */
#define FCE	01000	/* frame count */
#define CS	02000	/* 800: illegal TM (TM set in htds); 1600: correctable skew */
#define COR	0100000	/* 800: CRC error; 1600: correctable data error */
#define HARD	064023	/* UNS|OPI|NEF|FMT|RMR|ILR|ILF */


/*
 * internal states (in httab.b_active)
 */
#define	SIO	1	/* data transfer command */
#define	SSFOR	2	/* spacing forward */
#define	SSREV	3	/* spacing backwards */
#define SRETRY	4	/* error retry */
#define SCOM	5	/* non-data transfer command */
#define SOK	6


htopen(dev, flag)
{
	register unit, ds;

	unit = minor(dev) & 077;
	if (unit >= NUNIT || h_openf[unit])
	{
		u.u_error = ENXIO;
		return;
	}
	h_blkno[unit] = 0;
	h_nxrec[unit] = INF;
	ds = hcommand(dev, NOP);
	if ((ds & MOL) == 0 || ((flag & FWRITE) && (ds & WRL)))
		u.u_error = ENXIO;
	if (u.u_error == 0)
		h_openf[unit]++;
}

htclose(dev, flag)
{
	register int unit;
	register cmnd;

	unit = minor(dev) & 077;
	if (flag & FWRITE)	/* if opened for writing, append an EOT */
	{
		hcommand(dev, WTM);
		hcommand(dev, WTM);
	}
	if ((dev & 0200) == 0)
		cmnd = REW;
	else if (flag & FWRITE)
		cmnd = SREV;
	else if (h_blkno[unit] <= h_nxrec[unit])
		cmnd = SFORW;
	else
		cmnd = NOP;	/* as have just read a TM */
	hcommand(dev, cmnd);
	h_openf[unit] = 0;
}

hcommand(dev, com)
{
	register struct buf *bp;

	bp = &chtbuf;
	spl5();
	while (bp->b_flags & B_BUSY)
	{
		bp->b_flags |= B_WANTED;
		sleep((caddr_t)bp, PRIBIO);
	}
	spl0();
	bp->b_dev = dev;
	bp->b_resid = com;
	bp->b_blkno = 0;
	bp->b_flags = B_BUSY|B_READ;
	htstrategy(bp);
	iowait(bp);
	if (bp->b_flags & B_WANTED)
		wakeup((caddr_t)bp);
	bp->b_flags = 0;
	return(bp->b_resid);
}

htstrategy(bp)
register struct buf *bp;
{
	register daddr_t *p;

	if (bp != &chtbuf)
	{
		p = &h_nxrec[minor(bp->b_dev) & 077];
		if (bp->b_blkno > *p)
		{
			bp->b_flags |= B_ERROR;
			bp->b_error = ENXIO;
			iodone(bp);
			return;
		}
		if (bp->b_blkno == *p && bp->b_flags & B_READ)
		{
			clrbuf(bp);
			bp->b_resid = bp->b_bcount;
			iodone(bp);
			return;
		}
		if ((bp->b_flags & B_READ) == 0)
			*p = bp->b_blkno + 1;
	}
	bp->av_forw = NULL;
	spl5();
	if (httab.b_actf == NULL)
		httab.b_actf = bp;
	else
		httab.b_actl->av_forw = bp;
	httab.b_actl = bp;
	if (httab.b_active == 0)
		htstart();
	spl0();
}

htstart()
{
	register struct buf *bp;
	register unit, den;
	daddr_t blkno;

    loop:
	if ((bp = httab.b_actf) == NULL)
		return;
	unit = minor(bp->b_dev) & 0177;
	HTADDR->htcs2 = ((unit >> 3) & 07);	/* tm02 number */
	if (unit & 0100)
		den = P800;
	else
		den = P1600;
	den |= (unit & 07);
	if ((HTADDR->httc & 03777) != den)
		HTADDR->httc = den;
	if (HTADDR->htcs2 & NED || (HTADDR->htds & (MOL|DRY)) != (MOL|DRY))
		goto abort;
	unit &= 077;
	blkno = h_blkno[unit];
	if (bp == &chtbuf)
	{
		if (bp->b_resid == NOP)
		{
			bp->b_resid = HTADDR->htds;
			goto next;
		}
		httab.b_active = SCOM;
		HTADDR->htfc = 0;	/* space to next TM */
		HTADDR->htcs1 = bp->b_resid|IENABLE|GO;
		return;
	}
	if (h_openf[unit] < 0 || bp->b_blkno > h_nxrec[unit])
		goto abort;
	if (blkno == bp->b_blkno)
	{
		if (HTADDR->htds & EOT)
			goto abort;
		httab.b_active = SIO;
		den = HTADDR->htbae = bp->b_xmem;
		HTADDR->htba = bp->b_addr;
		HTADDR->htfc = -bp->b_bcount;
		HTADDR->htwc = -(bp->b_bcount>>1);
		den = IENABLE | GO | ((den & 03) << 8);
		if (bp->b_flags & B_READ)
			den |= RCOM;
		else
			den |= WCOM;
		HTADDR->htcs1 = den;
	}
	else
	{
		if (blkno < bp->b_blkno)
		{
			httab.b_active = SSFOR;
			HTADDR->htfc = blkno - bp->b_blkno;
			HTADDR->htcs1 = SFORW|IENABLE|GO;
		}
		else
		{
			httab.b_active = SSREV;
			HTADDR->htfc = bp->b_blkno - blkno;
			HTADDR->htcs1 = SREV|IENABLE|GO;
		}
	}
	return;

    abort:
	bp->b_flags |= B_ERROR;

    next:
	httab.b_actf = bp->av_forw;
	iodone(bp);
	goto loop;
}

htintr()
{
	register struct buf *bp;
	register int unit, state;
	int	err;

	if ((bp = httab.b_actf) == NULL)
		return;
	unit = minor(bp->b_dev) & 077;
	state = httab.b_active;
	httab.b_active = 0;
	if (HTADDR->htcs1 & TRE)
	{
		err = HTADDR->hter;
		if (HTADDR->htcs2 & 077400 || (err & HARD))
			state = 0;
		if (bp->b_flags & B_PHYS)
			err &= ~FCE;
		if ((bp->b_flags & B_READ) && (HTADDR->htds & PES))
			err &= ~(CS|COR);
		if ((HTADDR->htds & MOL) == 0)
			h_openf[unit] = -1;
		else if (HTADDR->htds & TM)
		{
			HTADDR->htwc = -(bp->b_bcount >> 1);
			h_nxrec[unit] = bp->b_blkno;
			state = SOK;
		}
		else if (state && err == 0)
			state = SOK;
		if (httab.b_errcnt > 2)
			printf("HTerr: cs1 0%o cs2 0%o ds 0%o er 0%o\n",
				HTADDR->htcs1, HTADDR->htcs2,
				HTADDR->htds, HTADDR->hter);
		htinit();
		if (state == SIO && ++httab.b_errcnt < 10)
		{
			httab.b_active = SRETRY;
			h_blkno[unit]++;
			HTADDR->htfc = -1;
			HTADDR->htcs1 = SREV|IENABLE|GO;
			return;
		}
		if (state != SOK)
		{
			bp->b_flags |= B_ERROR;
			state = SIO;
		}
	}
	else if (HTADDR->htcs1 & SC)
	{
		if (HTADDR->htds & ERR)
			htinit();
	}
	switch(state)
	{
	case SIO:
	case SOK:
		h_blkno[unit]++;

	case SCOM:
		httab.b_errcnt = 0;
		httab.b_actf = bp->av_forw;
		iodone(bp);
		bp->b_resid = (-HTADDR->htwc) << 1;
		break;

	case SRETRY:
		if ((bp->b_flags & B_READ) == 0)
		{
			httab.b_active = SSFOR;
			HTADDR->htcs1 = ERASE|IENABLE|GO;
			return;
		}

	case SSFOR:
	case SSREV:
		if (HTADDR->htds & TM)
		{
			if (state == SSREV)
			{
				h_nxrec[unit] = bp->b_blkno - HTADDR->htfc;
				h_blkno[unit] = h_nxrec[unit];
			}
			else
			{
				h_nxrec[unit] = bp->b_blkno + HTADDR->htfc - 1;
				h_blkno[unit] = h_nxrec[unit] + 1;
			}
		}
		else
			h_blkno[unit] = bp->b_blkno;
		break;

	default:
		return;
	}
	htstart();
}

htinit()
{
	register ocs2;
	register omttc;
	
	omttc = HTADDR->httc & 03777;	/* preserve old slave select, dens, format */
	ocs2 = HTADDR->htcs2 & 07;	/* preserve old unit */

	HTADDR->htcs2 = CLR;
	HTADDR->htcs2 = ocs2;
	HTADDR->httc = omttc;
	HTADDR->htcs1 = DCLR|GO;
}

htread(dev)
{
	htphys(dev);
	physio(htstrategy, &rhtbuf, dev, B_READ);
}

htwrite(dev)
{
	htphys(dev);
	physio(htstrategy, &rhtbuf, dev, B_WRITE);
}

htphys(dev)
{
	register unit;
	daddr_t a;

	unit = minor(dev) & 077;
	if (unit < NUNIT)
	{
		a = u.u_offset >> 9;
		h_blkno[unit] = a;
		h_nxrec[unit] = a + 1;
	}
}
