/*
**	Netstate -- manage state file
**
**	Bugs and comments to:	Piers Lauder
**				Dept of Comp Sci
**				Sydney University
**	Sep '80.
*/

#include	<local-system>
#ifdef	Aus2
#include	<types.h>
#include	<dir.h>
#endif	Aus2
#ifdef	Aus1
#include	<dir.h>
#include	"lvl7.h"
#endif	Aus1
#include	<stdio.h>
#include	"neth.h"
#include	"netstate.h"


short		silent;		/* TRUE if no map */
short		update;		/* TRUE for update */
char *		addhost;	/* a host to be added */
short		linkdown;	/* TRUE for link down */
short		breakhosts;	/* TRUE for break connection */
char *		brk1host;	/* first host name */
char *		brk2host;	/* second host name */
Nsp		brk1phost;	/* first host */
Nsp		brk2phost;	/* second host */
short		remove;		/* TRUE for host remove */
char *		rhost;		/* host to be removed */
time_t		cleartime;	/* hosts older than this get stats cleared */
Nsp		Up_stacs;	/* new state map */
char *		name;		/* invoked name */
time_t		ltime;		/* time now + HOUR */
char		statefile[]	= STATEFILE;

#define		max(A,B)	(A>B?A:B)

char	outbuf[BUFSIZ];
extern time_t	time();
extern char *	ctime();





main(argc, argv)
	int		argc;
	char *		argv[];
{
	register	pass;
	FILE *		fd;

	name = *argv;

	while ( --argc )
		if ( (*++argv)[0] == '-' )
		{
			switch ( (++*argv)[0] )
			{
			 case 'a':	addhost = ++*argv;
					break;
			 case 'b':	breakhosts++;
			 gethosts:
					brk1host = ++*argv;
					if ( (brk2host = strchr(brk1host, ':')) == NULLSTR )
						neterror("Usage: \"-%chost:host\"", argv[0][-1]);
					*brk2host++ = '\0';
					break;
			 case 'c':	if ( (cleartime = (time_t)atol(++*argv) * HOUR) == 0 )
						cleartime = 1;
					break;
			 case 'd':	linkdown++;
					goto gethosts;
			 case 'r':	remove++;
					rhost = ++*argv;
					break;
			 case 'u':	update++;
					break;
			 default:	neterror("unrecognised flag '%c'", (*argv)[0]);
			}
			silent++;
		}
		else
			neterror("unrecognised argument \"%s\"", *argv);

	if ( cleartime )
	{
		ltime = time((long *)0);
#		ifdef	GMT
		ltime += (long)GMT*60l*60l;
#		endif	GMT
		cleartime = ltime - cleartime;
	}

	if ( update )
	{
		rstate(stdin, &Up_stacs);
		for ( pass = 0 ; check(&Up_stacs, pass) ; pass++ );
	}

	if ( (fd = fopen(statefile, silent?"r+":"r")) != NULL )
	{
		rstate(fd, &State_stacs);
		if ( silent )
		{
			nice(-40);
			rewind(fd);
		}
		else
			fclose(fd);
	}
	else
		if ( silent )
			neterror("cannot open %s for update", statefile);
		else
			neterror("cannot read %s", statefile);

	if ( addhost != NULLSTR )
	{
		time_t	lt;
		char	temp[HOSTSIZE+3];

		sprintf(temp, ":%s@0", addhost);
		getstats(temp, (lt = time((long *)0)));
		setrates((long)0);
		modstats(lt);
	}

	for ( pass = 0 ; check(&State_stacs, pass) ; pass++ );
		
	if ( update )
	{
		merge();
		for ( pass = 0 ; check(&State_stacs, pass) ; pass++ );
	}

	if ( silent )
	{
		/* if ( !breakhosts || altroutes() ) */
			writestate(fd);
		fclose(fd);
	}
	else
	{
		setbuf(stdout, outbuf);
		printmap(stdout);
	}

	return 0;
}



/*
**	Print map of State_stacs
*/

printmap(fd)
	FILE *		fd;
{
	register Nsp	sp;

	fprintf(fd, "Network Map:-\n");

	for ( sp = State_stacs ; sp != NULLNSP ; sp = sp->ns_next )
		if ( sp->ns_type == host_t )
		{
			fprintf(fd, "%-*.*s", HOSTSIZE/2, HOSTSIZE, sp->ns_host);
			if ( sp->ns_bytes && sp->ns_date )
				fprintf(fd, "%9ld bytes, last active %.15s\n"
						,sp->ns_bytes
						,ctime(sp->ns_date)+4
					);
			else
				putc('\n', fd);
		}
		else
		{
			register	bytes_done = 0;

			fprintf(fd, " -> %*.*s", HOSTSIZE/2, HOSTSIZE, sp->ns_host);
			if ( sp->ns_bytes && sp->ns_date && sp->ns_rate > 1 && sp->ns_rate < 2000 )
			{
				fprintf(fd, "%6ld bytes/second  ", sp->ns_rate);
				bytes_done++;
			}
			if ( sp->ns_state == NS_DN )
			{
				if ( !bytes_done )
					fprintf(fd, "%21s", "");
				fprintf(fd, " (DOWN)\n");
			}
			else
				putc('\n', fd);
		}
}



/*VARARGS1*/
neterror(s, a, b)
	char *	s;
	char *	a;
	char *	b;
{
	extern	errno;

	fprintf(stderr, "%s: ", name);
	fprintf(stderr, s, a, b);
	if ( errno )
	{
		fflush(stderr);
		perror("\07");
	}
	else
		putc('\n', stderr);
	exit(1);
}



/*
**	Check consistency of network graph
*/

int
check(Stacs, pass)
	Nsp *		Stacs;
	int		pass;
{
	register Nsp	hp, np, lhp, lnp;
	register	hremoved = 0;

	if ( linkdown || breakhosts )
	{
		brk1phost = NULLNSP;
		brk2phost = NULLNSP;
	}

	for ( hp = Hosts_stacs, lhp = hp ; hp != NULLNSP ; lhp = hp, hp = hp->ns_prev )
	{
		if ( remove && pass == 0 && strcmp(rhost, hp->ns_host) == STREQUAL )
		{
			rmhost(hp, lhp, Stacs);
			hp = lhp;
			continue;
		}

		if ( (linkdown || breakhosts) && pass == 0 )
		{
			if ( brk1phost == NULLNSP && strcmp(hp->ns_host, brk1host) == STREQUAL )
				brk1phost = hp;
			else
			if ( brk2phost == NULLNSP && strcmp(hp->ns_host, brk2host) == STREQUAL )
				brk2phost = hp;
		}

		hp->ns_links = 0;	/* links for connected hosts */
		hp->ns_chosts = 0;	/* connected hosts adjacent in list */
	}

	if ( (linkdown || breakhosts) && pass == 0 && ( brk1phost == NULLNSP || brk2phost == NULLNSP ) )
		neterror("%s: \"%s\" doesn't exist"
			,linkdown?"link down":"break connection"
			,brk1phost==NULLNSP?brk1host:brk2host
			);

	/*
	**	Make connections
	*/

	for ( hp = Hosts_stacs ; hp != NULLNSP ; hp = hp->ns_prev )
	{
		char *	bhost;

		hp->ns_search = hp;	/* tag this host */

		if ( (breakhosts || linkdown) && pass == 0 )
		{
			if ( hp == brk1phost )
				bhost = brk2host;
			else
			if ( hp == brk2phost )
				bhost = brk1host;
			else
				bhost = NULLSTR;
		}
		else
			bhost = NULLSTR;

		if ( cleartime && (hp->ns_date < cleartime || hp->ns_date > ltime) )
			hp->ns_bytes = 0;

		for ( np = hp->ns_next, lnp = hp ; np != NULLNSP && np->ns_type != host_t ; lnp = np, np = np->ns_next )
		{
			if ( bhost != NULLSTR && strcmp(np->ns_host, bhost) == STREQUAL )
			{
				bhost = NULLSTR;
				if ( breakhosts )
				{
					lnp->ns_next = np->ns_next;
					free((char *)np);
					np = lnp;
					continue;
				}
				if ( linkdown )
					np->ns_state = NS_DN;
			}

			for ( lhp = Hosts_stacs ; lhp != NULLNSP ; lhp = lhp->ns_prev )
				if ( strncmp(np->ns_host, lhp->ns_host, HOSTSIZE) == STREQUAL )
				{
					if ( lhp->ns_search == hp )
					{
						warn("duplicate connected host \"%.*s\" for \"%.*s\" removed"
							,HOSTSIZE, np->ns_host
							,HOSTSIZE, hp->ns_host
						    );
						lnp->ns_next = np->ns_next;
						free((char *)np);
						np = lnp;
					}
					else
					{
						np->ns_prev = lhp;	/* connect graph */
						np->ns_date = lhp->ns_date;
						lhp->ns_links++;
						lhp->ns_search = hp;
					}
					break;
				}

			if ( lhp == NULLNSP )
			{
				if ( !remove || strcmp(rhost, np->ns_host) != STREQUAL )
					warn("unknown connected host \"%.*s\" for \"%.*s\" removed"
						,HOSTSIZE, np->ns_host
						,HOSTSIZE, hp->ns_host
					    );
				lnp->ns_next = np->ns_next;
				free((char *)np);
				np = lnp;
			}
			else
			for ( lhp = lhp->ns_next ; lhp != NULLNSP && lhp->ns_type != host_t ; lhp = lhp->ns_next )
				if ( strncmp(hp->ns_host, lhp->ns_host, HOSTSIZE) == STREQUAL )
				{
					if ( np->ns_rate != lhp->ns_rate )
						np->ns_rate = lhp->ns_rate = max(np->ns_rate, lhp->ns_rate);
					if ( np->ns_state != lhp->ns_state )
						np->ns_state = lhp->ns_state = NS_DN;
				}

			if ( np != lnp )
				hp->ns_chosts++;
		}

		if ( bhost )
			warn("break connection: \"%s\" is not connected to \"%.*s\""
				,bhost
				,HOSTSIZE, hp->ns_host
			    );
	}

	/*
	**	Check connections
	*/

	for ( hp = Hosts_stacs, lhp = hp ; hp != NULLNSP ; lhp = hp, hp = hp->ns_prev )
	{
		if ( hp->ns_links < hp->ns_chosts )
			for ( np = hp->ns_next, lnp = hp ; np != NULLNSP && np->ns_type != host_t ; lnp = np, np = np->ns_next )
				if ( np->ns_prev->ns_links > np->ns_prev->ns_chosts )
				{
					warn("dubious connection from \"%.*s\" to \"%.*s\" removed"
						,HOSTSIZE, hp->ns_host
						,HOSTSIZE, np->ns_host
					    );
					np->ns_prev->ns_links--;
					hp->ns_chosts--;
					lnp->ns_next = np->ns_next;
					free((char *)np);
					np = lnp;
				}

		if ( hp->ns_links < hp->ns_chosts || hp->ns_links == 0 )
		{
			if ( hp->ns_links == 0 )
			{
				if ( !breakhosts )
					warn("unconnected host \"%.*s\" removed"
						,HOSTSIZE, hp->ns_host
					    );
			}
			else
				warn("host %.*s and connections removed: links (%d) not equal to connected hosts (%d)"
					,HOSTSIZE, hp->ns_host
					,hp->ns_links
					,hp->ns_chosts
				    );

			rmhost(hp, lhp, Stacs);
			hremoved++;
		}
	}

	return hremoved;
}



/*
**	Remove a host and connections from Hosts_stacs and Stacs
*/

rmhost(hp, lhp, Stacs)
	register Nsp	hp;
	register Nsp	lhp;
	Nsp *		Stacs;
{
	register Nsp	np;

	/*
	**	Unlink from Hosts list
	*/

	if ( hp != Hosts_stacs )
		lhp->ns_prev = hp->ns_prev;
	else
		Hosts_stacs = hp->ns_prev;

	/*
	**	Unlink from "next" list
	*/

	if ( lhp == hp )
		lhp = NULLNSP;

	if ( (np = hp->ns_prev) != NULLNSP )
	{
		/** Find previous in "next" list **/
		for ( ; np->ns_next != hp ; np = np->ns_next );
		np->ns_next = lhp;
	}
	else
		*Stacs = lhp;

	/*
	**	Remove connections
	*/

	for ( np = hp->ns_next ; np != NULLNSP && np != lhp ; np = hp->ns_next )
	{
		hp->ns_next = np->ns_next;
		free((char *)np);
	}

	free((char *)hp);
}



/*
**	Merge Up_stacs with State_stacs
*/

merge()
{
	register Nsp	up, hp, np, lup;
	int		hostmoved = 0;
	int		newstats = 0;

	for ( up = Up_stacs, lup = up ; up != NULLNSP ; lup = up, up = up->ns_next )
	{
		if ( up->ns_type == host_t )
		{
			if ( hostmoved )
			{
				lup->ns_next = NULLNSP;
				hostmoved = 0;
			}

			for ( hp = Hosts_stacs ; hp != NULLNSP ; hp = hp->ns_prev )
				if ( strncmp(up->ns_host, hp->ns_host, HOSTSIZE) == STREQUAL )
					break;

			if ( hp == NULLNSP )
			{
				/** New host **/

				if ( (np = Hosts_stacs) != NULLNSP )
				{
					for ( ; np->ns_next != NULLNSP ; np = np->ns_next );
					np->ns_next = up;
				}
				else
					State_stacs = up;

				up->ns_prev = Hosts_stacs;
				Hosts_stacs = up;
				up->ns_bytes = 0;
				hostmoved++;
				hp = up;
			}
			else
			if ( up->ns_date > hp->ns_date )
			{
				newstats++;
				hp->ns_date = up->ns_date;
			}
			else
				newstats = 0;
		}
		else
		if ( !hostmoved )
		{
			for ( np = hp->ns_next ; np != NULLNSP && np->ns_type != host_t ; np = np->ns_next )
				if ( strncmp(up->ns_host, np->ns_host, HOSTSIZE) == STREQUAL )
				{
					if ( newstats )
					{
						np->ns_state = up->ns_state;
						np->ns_rate = up->ns_rate;
					}
					break;
				}

			if ( (np == NULLNSP || np->ns_type == host_t)
				&& strncmp(up->ns_host, NETID, HOSTSIZE) != STREQUAL
				&& strncmp(hp->ns_host, NETID, HOSTSIZE) != STREQUAL
			   )
			{
				lup->ns_next = up->ns_next;
				up->ns_next = hp->ns_next;
				hp->ns_next = up;
				up = lup;
			}
		}
	}
}



/*
**	Check that we can still reach both hosts disconnected
*/
/*
altroutes()
{
	register Nsp	hp;

	for ( hp = Hosts_stats ; hp != NULLNSP ; hp = hp->ns_prev )
		hp->ns_search = NULLNSP;

	if ( brk1phost != Home_stac )
		if ( searchg(brk1phost, 0) == 0 )
			return 0;

	if ( brk2phost != Home_stac )
		return searchg(brk2phost, 0);

	return 1;
}
*/




/*VARARGS1*/
warn(s, a, b, c, d)
	char *	s;
	char *	a;
	char *	b;
	char *	c;
	char *	d;
{
	fprintf(stderr, "%s: ", name);
	fprintf(stderr, s, a, b, c, d);
	putc('\n', stderr);
}
