/*
**	netq [-a] [-v] [-qhostname ...] [lname]
**
**	Print out contents of net queues
**
**		default:	show user's jobs on all queues
**		-a		show all jobs on all queues
**		-qhostname	show user's jobs on queue hostname
**		-a -qhostname	show all jobs on queue hostname
**		-v		tell origin and data name
**		lname		set user under consideration
**
**	Bugs and comments to:	Piers Lauder
**				Dept of Comp Sci
**				Sydney University
**	July '80.
*/


#include	<local-system>
#ifdef	Aus2
#include	<types.h>
#include	<dir.h>
#include	<stat.h>
#endif	Aus2
#ifdef	Aus1
#include	<stat16.h>
#include	<dir.h>
#endif	Aus1
#include	<passwd.h>
#include	<signal.h>
#include	<stdio.h>
#include	<sgtty.h>
#ifdef	Aus1
#include	"lvl7.h"
#endif	Aus1
#include	"net.h"
#include	"neth.h"


#define		NETS		20		/* maximum hosts expected */
#define		MAXJOBS		99		/* maximum jobs that will be printed from queues */

char		spooldir[]	= SPOOLDIR;
char *		netdir;
char		lockfile[HOSTSIZE+(DIRSIZ+1)];

struct
{
	ino_t	nd_ino;
	char	nd_name[HOSTSIZE];
}
		netdirs[NETS];
short		nets;

uid_t		uid;
char		all;
char		verbose;
char *		name;
struct pwent	p;

extern char *	strncat();
extern char *	strcat();
extern char *	strncpy();
extern char *	strcpy();
extern long	atol();

#define		dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))



main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	register	i;
	char	outbuf[BUFSIZ];

	name = *argv;

	if ( chdir(spooldir) == SYSERROR )
		neterror("\"chdir\" error - %s", spooldir);

	uid = getuid();

	while ( --argc )
		if ( **++argv == '-' )
		{
			switch ( (*argv)[1] )
			{
			 case 'a':
				all++;
				break;

			 case 'h':	/* alt. for 'q' */
			 case 'q':
				if ( nets >= NETS )
					neterror("too many hosts!");
				strncpy(netdirs[nets].nd_name, &(*argv)[2], DIRSIZ);
				if ( access(netdirs[nets].nd_name, 0) == SYSERROR )
					neterror("unrecognised host \"%s\"", netdirs[nets].nd_name);
				else
					nets++;
				break;

			 case 'v':
				verbose++;
				break;

			 default:
				neterror("unrecognised flag '%c'", (*argv)[1]);
			}
		}
		else
		{
			p.pw_strings[LNAME] = *argv;

			if ( getpwuid(&p, (char *)0, 0) == PWERROR )
				neterror("who?");
			else
				uid = p.pw_uid;
		}

	if ( nets == 0 && findnets("", 1) == 0 )
		neterror("no hosts found");

	setbuf(stdout, outbuf);

	for ( i = 0 ; i < nets ; i++ )
	{
		netdir = netdirs[i].nd_name;

		strncat(strncpy(lockfile, netdir, DIRSIZ), LOCKFILE, DIRSIZ);

		if ( all || access(lockfile, 0) != SYSERROR )
		{
			printlockfile();
		
			if ( chdir(netdir) == SYSERROR )
				perror(netdir);
			else
			{
				printqueue();
				chdir(spooldir);
			}
		}
		else
			printf("Host %s:\n     daemon inactive\n", netdir);

		fflush(stdout);
	}

	return 0;
}




/*
**	Fatal error exit
*/

/*VARARGS1*/

neterror(s, a)
	char *		s;
	char *		a;
{
	extern		errno;

	fprintf(stderr, "%s: ", name);
	fprintf(stderr, s, a);
	if ( errno )
		perror(" ");
	fprintf(stderr, "\nUsage: \"%s [-a] [-v] [-qhostname] [lname]\"\n", name);
	exit(1);
}



/*
**	Search spooldir for network directories
*/

findnets(path, level)
	char *		path;
	int		level;
{
	register FILE *	fd;
	register	found = 0;

	if ( (fd = fopen(".", "r")) == NULL )
		perror(spooldir);
	else
	{
		struct direct	dir;

		while ( fread((char *)&dir, sizeof dir, 1, fd) == 1 )
			if ( dir.d_ino && dir.d_name[0] != '~' && !dots(dir.d_name) )
			{
				register	i;

				for ( i = 0 ; i < nets ; i++ )
					if ( netdirs[i].nd_ino == dir.d_ino )
						break;
				if ( i == nets )
				{
#					ifdef	Aus2
					struct stat	dstat;
#					endif	Aus2
#					ifdef	Aus1
					struct statbuf	dstat;
#					endif	Aus1
					char		d[DIRSIZ+1];

					strncpy(d, dir.d_name, DIRSIZ);
					d[DIRSIZ] = '\0';

					if ( stat(d, &dstat) != SYSERROR
						&& (dstat.st_mode & S_IFMT) == S_IFDIR
						&& nets < NETS
					   )
					{
						if ( level < HLEVELS )
							if ( chdir(d) != SYSERROR )
							{
								char	p[HOSTSIZE];

								if ( access(LOCKFILE, 0) == SYSERROR )
									i = findnets(strcat(strcat(strcpy(p, path), d), "/"), level+1);
								else
									i = 0;
								chdir("..");
								if ( i )
								{
									found += i;
									continue;
								}
							}
						strcat(strcpy(netdirs[nets].nd_name, path), d);
						netdirs[nets].nd_ino = dir.d_ino;
						nets++;
						found++;
					}
				}
			}

		fclose(fd);
	}

	return found;
}



/*
**	Print daemon status from contents of lockfile
*/

printlockfile()
{
	register FILE *	fd;
	register char *	cp;
	struct lock	lock;

	printf("Host %s:\n     daemon", netdir);

	if ( (fd = fopen(lockfile, "r")) == NULL
		|| fread((char *)&lock, sizeof lock, 1, fd) != 1
		|| lock.lk_pid == 0
	   )
	{
		if ( fd == NULL )
		{
			lock.lk_size = 0;
			lock.lk_rcvd = 0;
			lock.lk_sent = 0;
			lock.lk_rate = 0;
			lock.lk_state = net_inactive;
		}
		else
		{
			fclose(fd);
		}
		printf(" inactive,");
	}
	else
	{
		fclose(fd);

		if ( uid )
			printf(" active,");
		else
			printf(" active (pid=%d),", lock.lk_pid);
	}

	switch ( lock.lk_state )
	{
	 case net_inactive:	cp = "inactive";	break;
	 case net_closed:	cp = "closed";		break;
	 case net_opening:	cp = "opening";		break;
	 case net_idle:		cp = "idle";		break;
	 case net_send:		cp = "sending";		break;
	 case net_recv:		cp = "receiving";	break;
	 case net_down:		cp = "down (host dead)";	break;
	 case net_error:	cp = "transmission fault";	break;
	 default:		cp = "state unknown";
	}

	printf(" line %s\n", cp);

	printf("     %ld bytes received, %ld bytes transmitted\n     %ld bytes/second\n"
			,lock.lk_rcvd
			,lock.lk_sent
			,lock.lk_rate
		);

	if ( lock.lk_size && !all )
		printf("     %ld bytes in queue\n", lock.lk_size);
}




/*
**	Print contents of queue directory
*/

printqueue()
{
	register FILE *	fd;

	if ( (fd = fopen(".", "r")) == NULL )
		perror(netdir);
	else
	{
		struct direct	dir;
		char *		files[MAXJOBS];
		register	ents = 0;
		register	i;
		register long	bytes = 0;
		extern char *	malloc();
		extern int	compare();

		while ( fread((char *)&dir, sizeof dir, 1, fd) == 1 )
			if ( dir.d_ino && dir.d_name[0] != '.' )
			{
				if ( ents < MAXJOBS )
				{
					register char *	s;

					files[ents++] = s = malloc(DIRSIZ+1);
					strncpy(s, dir.d_name, DIRSIZ);
				}
				else
					break;
			}

		fclose(fd);

		if ( ents )
		{
			qsort((char *)files, (unsigned)ents, sizeof files[0], compare);
			printf("     %s%d job%s in queue\n"
				,ents==MAXJOBS?">":""
				,ents
				,ents==1?"":"s"
			);
		}
		else
			printf("     queue empty\n");

		for ( i = 0 ; i < ents ; i++ )
		{
			register FILE *	efd;

			if ( (efd = fopen(files[i], "r")) != NULL )
			{
				struct netr	netrec;
				char		netfname[NETDATASIZE+1];
				char		netlname[NETDATASIZE+1];
				char		netname[NETDATASIZE+1];
				char		nethost[NETDATASIZE+1];
				char		message[NETDATASIZE+1];
				long		netsize;
				uid_t		netuid;
				char *		type;

				netlname[0] = '\0';
				netname[0] = '\0';
				nethost[0] = '\0';
				netsize = 0;
				netuid = (uid_t)-1;

				while ( fread((char *)&netrec, sizeof netrec, 1, efd) == 1 )
					switch ( netrec.nr_flag )
					{
					 case NC_NOGO:	type = "STOPPED";	goto fname;
					 case NC_MAIL:	type = "mail";		goto fname;
					 case NC_FILE:	type = "file";		goto fname;
					 case NC_PRINT:	type = "print";		goto fname;
					 case NC_HSTUP:	type = "host up";	goto fname;
					 case NC_HSTDN:	type = "host down";	goto fname;
					 case NC_REMOVE:type = "host remove";	goto fname;
					 case NC_STATE:	type = "net state";
					fname:
						strncpy(netfname, netrec.nr_data, NETDATASIZE);
						break;

					 case NC_LNAME:
						strncpy(netlname, netrec.nr_data, NETDATASIZE);
						break;

					 case NC_NAME:
						strncpy(netname, netrec.nr_data, NETDATASIZE);
						break;

					 case NC_HOST:
						strncpy(nethost, netrec.nr_data, NETDATASIZE);
						break;

					 case NC_UID:
						netuid = (uid_t)atoi(netrec.nr_data);
						break;

					 case NC_SIZE:
						netsize = atol(netrec.nr_data);
						break;

					 case NC_MESSAGE:
						strncpy(message, netrec.nr_data, NETDATASIZE);
						type = message;
						break;
					}

				if ( all || uid == netuid )
				{
					printf("%8d: %9.9s from %-8.8s for %-8.8s at %-10.*s %6ld bytes\n"
						,i+1
						,type
						,netlname
						,netname
						,HOSTSIZE
						,nethost
						,netsize
					);
					if ( verbose )
					{
						register FILE *	ffd;

						if ((ffd = fopen(netfname, "r")) != NULL )
						{
							netheader(ffd);
							fclose(ffd);
							printf("\t\t\"%s\" from %s at %s\n"
								,datname
								,orgname
								,orghost
							);
						}
					}
				}

				bytes += netsize;
				fclose(efd);
			}
		}

		while ( ents-- )
			free(files[ents]);

		if ( bytes && all )
			printf("     Total%57ld bytes\n", bytes);
	}
}




/*
**	Return order of two directory entries
*/

compare(d1, d2)
	char **		d1;
	char **		d2;
{
	return strncmp(*d1, *d2, DIRSIZ);
}
