/*
**	Net -- spool network jobs for transmission.
**
**	Bugs and comments to:	Piers Lauder
**				Dept of Comp Sci
**				Sydney University
**	July '80.
**
**	SUID => ROOT
*/


#include	<local-system>
#ifdef	Aus2
#include	<types.h>
#include	<stat.h>
#include	<dir.h>
#endif	Aus2
#include	<passwd.h>
#ifdef	Aus1
#include	<lnode.h>
#include	<stat16.h>
#include	<dir.h>
#endif	Aus1
#include	<signal.h>
#include	<stdio.h>
#include	<sgtty.h>
#include	<errno.h>
#ifdef	Aus2
#include	<fcntl.h>
#endif	Aus2
#ifdef	Aus1
#include	"lvl7.h"
#endif	Aus1
#include	"net.h"
#include	"neth.h"


#define		MAXBYTES	1000000L	/* more bytes cause abort */
#define		JOBSLOW		10000		/* more bytes cause low priority */

char *		name;				/* invoked name */
char		daemon_name[]	= "daemon";

char		tfnmh[]		= "~tXXXXXX";
char		cfnmh[]		= "~cXXXXXX";
char		dfnmh[]		= "/bXXXXXXXXX";
#define		PRIPOSN		1
#define		QUICK		'a'
#define		SLOW		'c'

char		spooldir[]	= SPOOLDIR;
#define		NAMESIZE	(sizeof spooldir + DIRSIZ)
char		lockfile[NAMESIZE+HOSTSIZE];
char		tfn[NAMESIZE+DIRSIZ];
char		cfn[NAMESIZE+DIRSIZ];
char		dfn[NAMESIZE+HOSTSIZE];

char		empty[]		= "";
char		bin_net[]	= BIN_NET;
char		netchange[]	= NETCHANGE;
char		bin_sh[]	= BIN_SH;
char		bin_sh_param[]	= "-c";
char		exec_err[]	= "can't exec %s for %s";
char		netid[]		= NETID;

struct pwent	pe;
short		xflag;
short		cfnused;
FILE *		tfd;
FILE *		cfd;
long		bytes;
time_t		ltime;
short		rflg;
short		quick;
char *		ttymesg;

extern char *	findhost();
extern char *	prevhost();
extern char *	nextup();
extern char *	strcat();
extern char *	strcpy();
extern char *	strchr();
extern char *	strrchr();
extern char *	itoa();
extern char *	ltoa();
extern char *	mktemp();
extern char *	mytty();
extern char *	ctime();
extern long	lseek();
extern time_t	time();


/*
**	Signals
*/

int		rubout();

struct csig
{
	int	s_signo;
	int	(*s_action)();
	int	(*s_oldact)();
}
		csigs[] =
{
	 {	SIGHUP,		rubout	}
	,{	SIGINT,		rubout	}
	,{	SIGQUIT,	rubout	}
	,{	SIGPIPE,	rubout	}
	,{	SIGTERM,	rubout	}
};
#define		NCSIGS	(sizeof csigs/sizeof csigs[0])

#ifdef	Aus2
struct stat	statbuf;
#define		stsize(A)	(A)->st_size
#endif	Aus2
#ifdef	Aus1
struct statbuf	statbuf;
#define		stsize(A)	(((long)((A)->sb_size0)<<16)|(A)->sb_size1)
#endif	Aus1

extern int	errno;





main(argc, argv)
	register int	argc;
	register char	*argv[];
{
	register	f;
	FILE *		F;
	register char	*ap;
	char		namebuf[100];
	register struct csig	*sp;

	for ( sp = csigs ; sp < &csigs[NCSIGS] ; sp++ )
		if ( (sp->s_oldact = signal(sp->s_signo, SIG_IGN)) != SIG_IGN )
			signal(sp->s_signo, sp->s_action);

	name = *argv++;
	argc--;

	dataction = " ";	/* 1 space, will be overwritten */
	dataction[0] = NC_MAIL;
	actparam = empty;

	while ( argc > 0 && (ap = *argv)[0] == '-' )
	{
		switch ( ap[1] )
		{
		 case 'D':
			dataction[0] = NC_HSTDN;
			destname = daemon_name;
			datname = &ap[2];
			break;

		 case 'N':
			datname = &ap[2];
			break;

		 case 'O':
			orghost = &ap[2];
			break;

		 case 'P':
			actparam = &ap[2];
			break;

		 case 'R':
			dataction[0] = NC_REMOVE;
			destname = daemon_name;
			datname = &ap[2];
			break;

		 case 'S':
			netstats = &ap[2];
			break;

		 case 'U':
			dataction[0] = NC_HSTUP;
			destname = daemon_name;
			datname = &ap[2];
			break;

		 case 'e':
			quick++;
			break;

		 case 'f':
			dataction[0] = NC_FILE;
			break;

		 case 'h':
			desthost = &ap[2];
			break;

		 case 'm':
			ttymesg = &ap[2];
			break;

		 case 'u':	/* alt. for 'n' */
		 case 'n':
			destname = &ap[2];
			break;

		 case 'p':
			dataction[0] = NC_PRINT;
			break;

		 case 'r':
			rflg++;
			break;

		 case 's':
			dfnmh[PRIPOSN] = SLOW;
			break;

		 case 'x':
			xflag++;
			break;

		 default:
			warn("bad flag %c ignored", ap[1]);
		}
		argc--;
		argv++;
	}

	if ( argc && (dataction[0] == NC_HSTDN || dataction[0] == NC_HSTUP || dataction[0] == NC_REMOVE) )
	{
		warn("bad usage");
		return 1;
	}

	if ( xflag == 0 && (destname == (char *)0 || desthost == (char *)0) )
	{
		warn("name and/or desthost not specified");
		return 1;
	}

	pe.pw_uid = getuid();

	while ( getpwlog(&pe, namebuf, sizeof namebuf) == PWERROR )
	{
		if ( errno == ENFILE )
			sleep(60);
		else
		{
			warn("Ghost who walks, can never die.");
			return 1;
		}
	}
	pwclose();

	/*
	**	Check network access
	*/

	if ( pe.pw_uid > DAEMONUID
		&& ((pe.pw_xflags&USENET) == 0
			|| dataction[0] == NC_HSTDN
			|| dataction[0] == NC_HSTUP
			|| dataction[0] == NC_REMOVE
			|| (dataction[0] == NC_PRINT
				&& strcmp(destname, pe.pw_strings[LNAME]))
			|| orghost != NULLSTR
			|| netstats != NULLSTR
		   )
	   )
	{
		warn("No permission.");
		return 1;
	}

	orgname = pe.pw_strings[LNAME];
	if ( orghost == NULLSTR )
		orghost = netid;
	if ( netstats == NULLSTR )
		netstats = empty;

	strcat(strcat(strcpy(tfn, spooldir), WORKDIR), tfnmh);
	strcat(strcat(strcpy(cfn, spooldir), WORKDIR), cfnmh);

	while ( (cfd = fopen(mktemp(cfn), "wf")) == NULL )
		neterror(cfn);

	chown(cfn, DAEMONUID, pe.pw_gid);
	chmod(cfn, 0600);

	ltime = time((long *)0);
#	ifdef	GMT
	ltime += (long)GMT*60l*60l;	/* for those systems that DON'T run on GMT!! */
#	endif	GMT

	if ( argc == 0 )
		switch ( dataction[0] )
		{
		 case NC_HSTDN:
		 case NC_HSTUP:
		 case NC_REMOVE:
			quick++;
			bytes += makeheader(cfd, ltime);
			cfnused++;
			break;
		 default:
			if ( datname == NULLSTR )
				datname = "stdin";
			copy(fdopen(0, "r"));
		}
	else
	if ( argc == 1 )
	{
		if ( datname == NULLSTR )
			datname = *argv;

		while ( *datname == '/' )
			datname++;	/* skip '/' */

		while ( access(*argv, 4) == SYSERROR )
			neterror(*argv);
		while ( (F = fopen(*argv, "r")) == NULL )
			neterror(*argv);

		if ( dataction[0] == NC_FILE )
			setmode(F);

		copy(F);

		fclose(F);

		if ( rflg )
			if ( remvp(*argv) != SYSERROR )	/* check for remove permission */
				unlink(*argv);
			else
				warn("%s: cannot remove", *argv);
	}
	else
		neterror("only one file arg allowed");

	if ( cfnused )
	{
		struct lock	lock;
		char *		nextname();

		fclose(cfd);

		ap = desthost;
		f = 0;

		while ( access(strcat(strcpy(dfn, spooldir), ap), 0) == SYSERROR )
			if ( f++ || (ap = findhost(ap, bytes)) == NULLSTR )
				if ( strcmp(destname, orgname) != STREQUAL && strcmp(desthost, netid) == STREQUAL )
					local();
				else
				if ( ap == NULLSTR )
				{
					neterror("destination unknown: %s", desthost);
					f = 0;	/* try again */
				}
				else
				{
					if ( dataction[0] == NC_HSTDN || dataction[0] == NC_HSTUP || dataction[0] == NC_REMOVE )
					{
						unlink(cfn);
						return 0;
					}
					neterror("link \"%s\" to %s unknown", ap, desthost);
				}

		while ( (tfd = fopen(mktemp(tfn), "wf")) == NULL )
			neterror(tfn);

		chown(tfn, DAEMONUID, pe.pw_gid);
#		ifdef	Aus2
		chmod(tfn, 0640);
#		endif	Aus2
#		ifdef	Aus1
		chmod(tfn, 0604);
#		endif	Aus1

		netmesg(NC_UID, itoa(pe.pw_uid));
		netmesg(NC_LNAME, pe.pw_strings[LNAME]);

		netmesg(dataction[0], cfn);
		netmesg(NC_UNLINK, cfn);
		netmesg(NC_SIZE, ltoa(bytes));

		if ( ttymesg )
		{
			netmesg(NC_MESSAGE, ttymesg);
			netmesg(NC_TTYNAME, mytty());
		}

		netmesg(NC_NAME, destname);
		netmesg(NC_HOST, desthost);

		fclose(tfd);

		if ( (pe.pw_uid <= DAEMONUID) && (quick > 0) )
			dfnmh[PRIPOSN] = QUICK;	/* ensure us super user types dont wait */
		else
		{
			if ( bytes > JOBSLOW )
				dfnmh[PRIPOSN] = SLOW;
			if ( quick > 0 && (pe.pw_xflags & (ADMIN|TUTOR)) == 0 )
				quick = 0;
		}

		strcat(dfn, dfnmh);
		strcat(strcat(strcpy(lockfile, spooldir), ap), LOCKFILE);

		nice(-40);

		for ( sp = csigs ; sp < &csigs[NCSIGS] ; sp++ )
			if ( sp->s_oldact != SIG_IGN )
				signal(sp->s_signo, SIG_IGN);

		if ( (f = open(lockfile, O_RDWR)) != SYSERROR )
		{
			if ( read(f, (char *)&lock, sizeof lock) == sizeof lock )
			{
				lock.lk_size += bytes;
				lseek(f, (long)0, 0);
				write(f, (char *)&lock, sizeof lock);
				close(f);
			}
			else
			{
				close(f);
				perror(lockfile);
				lock.lk_state = net_inactive;
			}
		}
		else
			lock.lk_state = net_inactive;

		if ( lock.lk_pid == 0
		    || lock.lk_state == net_inactive
		    || lock.lk_state == net_opening
		    || lock.lk_state == net_down )
			if ( dataction[0] == NC_HSTDN || dataction[0] == NC_HSTUP )
			{
				unlink(tfn);
				unlink(cfn);
				return 0;
			}

		while ( link(tfn, nextname(dfn)) == SYSERROR )
			neterror("cannot link %s", dfn);
		unlink(tfn);

		ap = NULLSTR;

		switch ( lock.lk_state )
		{
		 case net_inactive:	ap = "is inactive";	break;
		 case net_closed:	ap = "is closed";	break;
		 case net_down:		ap = "is down";		break;
		 case net_error:	ap = "has a high error rate";
					break;
		 case net_idle:		if ( lock.lk_pid )
						kill(lock.lk_pid, NS_GETGOING);
		}

		if ( ap != NULLSTR && pe.pw_uid != DAEMONUID )
			warn("Warning! link to %s %s", desthost, ap);
	}
	else
		unlink(cfn);

	return 0;
}



/*
**	Deliver local files
*/

local()
{
	chown(cfn, pe.pw_uid, pe.pw_gid);

	switch ( fork() )
	{
	 case 0:	execl(bin_net, bin_net+5, "-x", cfn, 0);
			neterror(exec_err, bin_net);

	 case SYSERROR:	neterror("fork error - try again");

	 default:	netwait();
	}

	unlink(cfn);
	exit(0);
}



/*
**	Spool a file
*/

copy(f)
	register FILE *	f;
{
	register	c;
	register long	n;

	if ( xflag )	/* file from network */
	{

		fstat(fileno(f), &statbuf);

		c = netheader(f);				/* read in header */

		while ( !upstate(netstats, stsize(&statbuf)-c, ltime) )	/* update the state file */
			neterror("can't update state file");

		c = 0;

		if ( strcmp(desthost, netid) == STREQUAL || (strcmp(orghost, netid) == STREQUAL && ++c) )	/* destination reached */
		{
			terminate(f, c);
			return;
		}
	}

	bytes += makeheader(cfd, ltime);

	n = 0;

	while ( (c = getc(f)) != EOF )
	{
		putc(c, cfd);
		if ( ++n > MAXBYTES && !xflag )
			neterror("%s too large!", datname);
	}

	bytes += n;

	cfnused++;
}



/*
**	Destination processing for file from network
*/

terminate(f, flag)
	register FILE *	f;
	int		flag;
{
	register	c;
	FILE *		pfd;
	extern FILE *	popen();
	char		buf[SSIZ];
	char		usfn[NAMESIZE+DIRSIZ+100];
	struct pwent	netpwe;
	uid_t		uid = DAEMONUID;
	uid_t		gid = DAEMONGID;
	short		inhome;
	short		lostfile = 0;

	if ( flag )
		reverse();

	buf[0] = '\0';

	if ( dataction[0] == NC_PRINT || dataction[0] == NC_FILE )
	{
		netpwe.pw_strings[LNAME] = destname;

		while ( getpwuid(&netpwe, buf, SSIZ) == PWERROR )
		{
			char *	savestats;

			if ( errno == ENFILE )
			{
				sleep(60);
				continue;
			}

			if ( flag )
				neterror("user %s has disappeared!", destname);
	
			dataction[0] = NC_MAIL;
			savestats = netstats;
			netstats = empty;
	
			reverse();
			bytes += makeheader(cfd, ltime);
			reverse();
	
			bytes += fprintf(cfd, "From daemon %.24s netmail from %s\nTo: %s\nSubject: unknown user\n\n%s is unknown at %s!\n\n"
						,ctime(ltime)
						,netid
						,orgname
						,destname
						,netid
					);
	
			cfnused++;
			dataction[0] = NC_FILE;
			netstats = savestats;
			netpwe.pw_gid = DAEMONGID;
			lostfile++;
			break;
		}
		pwclose();
	}

	switch ( dataction[0] )
	{
	 case NC_STATE:
		sprintf(buf, "%s -u", BIN_NETSTATE);
	 case NC_HSTUP:
		{
			register char *	cp1;

			if ( strcmp(netid, datname) == STREQUAL )
			{
				char *	savestats = netstats;

				dataction[0] = NC_STATE;
				netstats = empty;
				reverse();
				bytes += makeheader(cfd, ltime);
				reverse();
				bytes += writestate(cfd);
				netstats = savestats;
				quick++;
				cfnused++;
			}

			if ( (cp1 = nextup()) != NULLSTR )
			{
				register char *	cp2;

				while ( (cp2 = nextup()) != NULLSTR )
					dohostup(cp2);

				if ( cfnused )
				{
					dohostup(cp1);
					reverse();
				}
				else
				{
					free(desthost);
					desthost = cp1;
					bytes += makeheader(cfd, ltime);
					quick++;
					cfnused++;
				}
			}
			else
				if ( cfnused )
					reverse();
		}

		if ( buf[0] != '\0' )
			break;
		return;

	 case NC_HSTDN:
	 case NC_REMOVE:
		if ( hconnected(datname, orghost, dataction[0] == NC_REMOVE) )
		{
			register	n;
			register char *	bp;
			static char	format[] = "%s -S%s -%c%s -C%s -F%s -T%s";

			n = strlen(netchange);
			n += strlen(spooldir);
			n += strlen(datname);
			n += strlen(orghost);
			n += strlen(prevhost());
			n += strlen(netstats);
			n += sizeof format;

			if ( n > SSIZ )
				bp = malloc(n);
			else
				bp = buf;

			sprintf(bp, format
				,netchange
				,spooldir
				,dataction[0]==NC_REMOVE?'R':'D'
				,datname
				,orghost
				,prevhost()
				,netstats
			);
			close(1); dup(2);
			close(0); open("/dev/null", 0);
			unlink(cfn);
			execl(bin_sh, bin_sh+5, bin_sh_param, bp, 0);
			neterror(exec_err, bin_sh, buf);
		}
		return;

	 case NC_PRINT:
		uid = netpwe.pw_uid;
		gid = netpwe.pw_gid;
		sprintf(buf, "lpr -i'%s:%s' -s %s", orgname, orghost, actparam);
		break;

	 case NC_FILE:
		if ( lostfile || netpwe.pw_dlimit
		     || stat(strcat(strcat(strcpy(usfn, netpwe.pw_strings[DIRPATH]), UNFDIR), datname), &statbuf) != SYSERROR
		     || (pfd = fopen(usfn, "w")) == NULL
		   )
		{
			register char *	cp;
			register	i;
			char		ufn[DIRSIZ+1];

			cp = lostfile ? "lost" : destname;
			for ( i = 0 ; i < MAXUNAME && *cp != '\0' ; i++ )
				ufn[i] = *cp++;
			while ( i < MAXUNAME )
				ufn[i++] = '.';
			while ( i < DIRSIZ )
				ufn[i++] = 'X';
			ufn[i] = '\0';

			strcat(strcat(strcpy(usfn, SPOOLDIR), FILESDIR), ufn);

			while ( (pfd = fopen(mktemp(usfn), "w")) == NULL )
				neterror("cannot open %s, %s abandoned", usfn, datname);
			chown(usfn, DAEMONUID, netpwe.pw_gid);
			chmod(usfn, 0600);

			makeheader(pfd, ltime);
			inhome = 0;
		}
		else
		{
			chown(usfn, netpwe.pw_uid, netpwe.pw_gid);
			chmod(usfn, getmode());
			inhome = 1;
		}

		if ( !feof(f) )
			while ( (c = getc(f)) != EOF )
				putc(c, pfd);
		fclose(pfd);

	 case NC_MAIL:
		sprintf(buf, "%s %s"
#				ifdef	RMAIL
				,(pe.pw_uid<=DAEMONUID && dataction[0]==NC_MAIL)?"rmail":"mail"
#				else	RMAIL
				,"mail"
#				endif	RMAIL
				,lostfile?"root":destname
			);
		break;

	 default:
		neterror("unrecognised network file type: '%s'", dataction);
	}

	if ( (pfd = popen(buf, "w", uid, gid)) != NULL )
	{
		switch ( dataction[0] )
		{
		 case NC_FILE:
			fprintf(pfd, "%s \"%s\" %s %s %s %s\n\n\"%s\" %s%s\n%s\n%s\n"
				,lostfile?"Subject: LOST FILE!":"Subject: Net file"
				,datname
				,"from"
				,orgname
				,"at"
				,orghost
				,datname
				,inhome?"is spooled as ":lostfile?"sent to unknown user ":"awaits your collection"
				,inhome?usfn:lostfile?destname:"."
				,inhome?"":"Use \"netget\" to save it in 7 days"
				,inhome?"":"otherwise it will be removed."
				);
			break;

		 case NC_MAIL:
#			ifdef	RMAIL
			if ( pe.pw_uid > DAEMONUID )
#			endif	RMAIL
				fprintf(pfd, "%s %s at %s\n\n"
					,"Subject: net mail from"
					,orgname
					,orghost
					);
			goto copyit;

		 case NC_PRINT:
#			ifdef	PRINTLPSTATS
			fprintf(pfd, "%s \"%s\" %s %s %s %s\n"
				,"Print file"
				,datname
				,"via network from"
				,orgname
				,"at"
				,orghost
				);
			printstats(pfd);
			putc('\f', pfd);
#			endif	PRINTLPSTATS

		 case NC_STATE:
		 copyit:
			if ( !feof(f) )
				while ( (c = getc(f)) != EOF )
					putc(c, pfd);
		}
		pclose(pfd);
	}
	else
		neterror("cannot pipe to \"%s\", %s abandoned", buf, datname);
	if ( lostfile )
		reverse();
}



/*
**	Reverse message addresses
*/

reverse()
{
	register char *	cp;

	cp = orghost; orghost = desthost; desthost = cp;
	cp = orgname; orgname = destname; destname = cp;
}



/*
**	Fork off new "net" to do extra HOST UP message
*/

dohostup(cp)
	char *	cp;
{
	switch ( fork() )
	{
	 case 0:
		{
			register	i;
			char		temp[HOSTSIZE+2];
			char		temp1[HOSTSIZE+2];
			char		temp2[MAXSTATS+2];
			char		temp3[HOSTSIZE+2];
		
			for ( i = 0 ; ; i++ )
				if ( i != 2 )
					if ( close(i) == SYSERROR )
						break;

			execl(bin_net, bin_net+5
				,strcat(strcpy(temp, "-U"), datname)
				,strcat(strcpy(temp1, "-O"), orghost)
				,strcat(strcpy(temp2, "-S"), netstats)
				,strcat(strcpy(temp3, "-h"), cp)
				,0
				);

			neterror(exec_err, bin_net, temp);
		}
		break;
	 case SYSERROR:
		warn("fork error");
		break;
	 default:
		netwait();
	}
}



/*
**	Netwait - for children
*/

netwait()
{
	int	state;

	errno = 0;

#	ifdef	Aus2
	while ( wait(&state) != SYSERROR )
#	endif	Aus2
#	ifdef	Aus1
	while ( waitx(&state) != SYSERROR )
#	endif	Aus1
	{
#		ifdef	Aus1
		fseek(stderr, (long)0, 2);
#		endif	Aus1
		if ( state & 0xff )
			warn("child died signal %d", state & 0xff);
		if ( (state>>8) & 0xff )
			warn("child error exit %d", (state>>8) & 0xff );
	}

	errno = 0;
}


/*VARARGS1*/
neterror(s, a, b)
	char *	s;
	char *	a;
	char *	b;
{
	warn(s, a, b);
	if ( errno )
		perror("");
	if ( pe.pw_uid == DAEMONUID )
	{
		if ( errno == ENFILE )
		{
			sleep(60);
			return;
		}
		fprintf(stderr, "\t[tfn=%s, cfn=%s]\n", tfn, cfn);
		exit(2);
	}
	else
		rubout(SIGINT);
}



/*VARARGS1*/
warn(s, a, b)
	char *	s;
	char *	a;
	char *	b;
{
	fprintf(stderr, "%s: ", name);
	fprintf(stderr, s, a, b);
	fputs("\07\n", stderr);
	fflush(stderr);
}


netmesg(c, s)
	char		c;
	register char	*s;
{
	register char	*cp;
	struct netr	netm;

	netm.nr_flag = c;

	if ( s == NULL )
		s = empty;

	for ( cp = netm.nr_data ; cp < &netm.nr_data[NETDATASIZE] ; )
		if ( *cp++ = *s )
			s++;

	fwrite((char *)&netm, sizeof netm, 1, tfd);
}



/*
**	Stuff ascii mode into "actparam"
*/

#define	MODESIZE	26	/* " MODE=0777 DATE=NnnnNNNnnn" */

setmode(fd)
	FILE *		fd;
{
	register char *	s;

	fstat(fileno(fd), &statbuf);
#	ifdef	GMT
	statbuf.st_mtime += (long)GMT*60l*60l;
#	endif

	s = malloc((unsigned)(MODESIZE+strlen(actparam)+1));

	sprintf(s, "%s MODE=0%o DATE=%ld", actparam, statbuf.st_mode&0777, statbuf.st_mtime);

	actparam = s;
}



/*
**	Extract ascii mode from actparam
*/

getmode()
{
	register char *	cp;

	for ( cp = actparam ; (cp = strchr(cp, 'M')) != NULL ; cp++ )
		if ( strncmp(cp, "MODE=0", 6) == STREQUAL )
		{
			register char	c;
			register	o;

			for ( o = 0, cp += 6 ; (c = *cp++) >= '0' && c <= '7' ; o = (o<<3)+(c-'0') );
			return o & 0777;
		}

	return 0600;	/* default */
}


remvp(file)
	char		*file;
{
	register char	*p,*q,*r;
	int		wflag;

	wflag = 0;
	if ( stat(file, &statbuf) == SYSERROR || (statbuf.st_mode & S_IFMT) != S_IFREG )
		return SYSERROR;

	p = file;
	r = p+1;
	if ( *p == 0 )
		return SYSERROR;
	if ( *p == '/' )
		q = "/";
	else
		q = ".";
	while(*r)
	{
		if ( *r++ == '/' )
		{
			while(*r == '/' )
				r++;
			if ( *r )
				p = r;
			q = file;
		}
	}
	if ( q == file )
	{
		*--p = 0;
		wflag = access(q, 2);
		*p = '/';
	}
	else
		wflag = access(q, 2);
	return wflag;
}



rubout(i)
	int	i;
{
	signal(i, SIG_IGN);
	unlink(tfn);
	unlink(cfn);
	exit(1);
}


char	tnchars[]	= "+-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

char *
nextname(s)
	char		*s;
{
	register char	*p,*q;
	register	pid;
	register long	systime;

	pid = getpid();

	for ( p = s ; *p ; p++ );
	for ( q = p-3 ; p != q ; )
	{
		*--p = tnchars[pid & 077];
		pid >>= 6;
	}

	systime = ltime;
	systime += bytes * 60 / 1000;	/* 1 minute degradation per 1000 bytes */
	systime -= quick * 60 * 60;	/* 1 hour improvement per quick */

	for ( q = p-6 ; p != q ; )
	{
		*--p = tnchars[systime & 077];
		systime >>= 6;
	}

	return s;
}



/*
**	Own version of stdio lib "popen.c"
*/

#define	tst(a,b)	(*mode == 'r'? (b) : (a))
#define	RDR	0
#define	WTR	1
static	int	popen_pid;
static char *	pcmd;

FILE *
popen(cmd,mode,uid,gid)
	char	*cmd;
	char	*mode;
	uid_t	uid;
	uid_t	gid;
{
	int	p[2];
	register myside, yourside, pid;
	extern	piperr();

	while ( pipe(p) == SYSERROR )
	{
		if ( errno == ENFILE )
			sleep(60);
		else
			return NULL;
	}
	myside = tst(p[WTR], p[RDR]);
	yourside = tst(p[RDR], p[WTR]);
	if ( (pid = fork()) == 0 )
	{
		/* myside and yourside reverse roles in child */
		int	stdio;

		stdio = tst(0, 1);
		close(myside);
		close(stdio);
		fcntl(yourside, 0, stdio);
		close(yourside);
		close(tst(1, 0));
#		ifdef	Aus2
		setgid(gid);
#		endif	Aus2
		setuid(uid);
		execl(bin_sh, bin_sh+5, bin_sh_param, cmd, 0);
		neterror(exec_err, bin_sh, cmd);
	}
	if ( pid == -1 )
		return NULL;
	popen_pid = pid;
	close(yourside);
	signal(SIGPIPE, piperr);
	pcmd = cmd;
	return(fdopen(myside, mode));
}

pclose(ptr)
	FILE *	ptr;
{
	register r;
	int	status;

	fclose(ptr);

#	ifdef	Aus2
	while((r = wait(&status)) != popen_pid && r != -1);
#	endif	Aus2
#	ifdef	Aus1
	while((r = waitx(&status)) != popen_pid && r != -1);
#	endif	Aus1

	if ( r == -1 )
		status = -1;

	signal(SIGPIPE, rubout);

	return(status);
}

piperr()
{
	neterror("SIGPIPE on %s", pcmd);
}


char *
ltoa(l)
	long		l;
{
	static char	buf[11];

	sprintf(buf, "%ld", l);
	return buf;
}
