#
/*
 *	Optimised RHP04 and RJP04 driver
 *
 *	When instructed correctly (via the ifdefs) this driver will attempt
 *	to queue io requests by drive, cylinder and sector, and endeavour
 *	to overlap seeks with data transfers for more than one drive.
 *	Method is very similar to Childrens Museum RK driver.
 *
 *	Some simple error correction and recovery is attempted, but nothing
 *	fancy. You have been WARNED.
 *
 *	Bugs and comments to	Peter Ivanov
 *				Dept. Computer Science,
 *				University of New South Wales.
 */

#include	"../param.h"
#include	"../buf.h"
#include	"../conf.h"
#include	"../user.h"

#define	SWEEP	/* enable checking to force max head travel */
/*#define	NO_SEEK		/* to stop pre-seeking */
#define	RHP04		/* define for an RH70 type RP04 */
/*#define	RJP04		/* define for a UNIBUS type RP04 or DIVA */
/*#define	HPECC	/* enable data check correction */
/*#define	HPSTATS		/* define to enable stats collection */
#define	NO_OFFLINEQ	/* stop qing of requests for offlined drives */
#define	DROP_OFFLINEQ	/* delete drive request q when offlined */

struct	{
	int	hpcs1;	/* Control and Status register 1 */
	int	hpwc;	/* Word Count register */
	int	hpba;	/* Bus address register */
	int	hpda;	/* Desired disc address - sector/track */
	int	hpcs2;	/* Control and Status register 2 */
	int	hpds;	/* Drive status register */
	int	hper1;	/* Error register 1 */
	int	hpas;	/* Attention summary register */
	int	hpla;	/* Disc position look-ahead register */
	int	hpdb;	/* Data buffer */
	int	hpmr;	/* Maintenance register */
	int	hpdt;	/* Drive type encoding */
	int	hpsn;	/* Drive serial number */
	int	hpof;	/* Offset register - contains fm22 bit */
	int	hpdc;	/* Desired cylinder address register */
	int	hpcc;	/* Current cylinder address register */
	int	hper2;	/* Error register 2 */
	int	hper3;	/* Error register 3 */
	int	hppos;	/* Burst error bit position */
	int	hppat;	/* Burst error bit pattern */
#ifdef	RHP04
	int	hpbae;	/* 11/70 bus extension register */
	int	hpcs3;	/* Control and Status register 3 */
#endif
	};

#define	HPADDR	0176700
#define	NDRV	1	/* the number of real rp04 drives on controller */
#define	NHPCYLS	411	/* 411 cylinders per pack */
#define	TVOL	(sizeof hp_sizes/sizeof hp_sizes[0])	/* total number of volumes */
#define	HPAGE	10	/* number of times this block may be pre-empted for io
				   before io is forced	*/
#define	INTLV	9	/* interleaving factor for rp04 type drives */
#define	HPSPL	spl5	/* priority of hp disc */

/*
 *	The following maps logical volumes of various sizes to locations
 *	on various drives. This must be altered to add extra drives.
 *	PLEASE GET THEM CORRECT FIRST TIME.
 */

struct	{
	unsigned	nblocks;
	int	cyloff;
	int	drive;
} hp_sizes[] {
/*	 0:	  0->155	(156cyls)	drive 0	*/	65208l,	  0,	0,	/* source disk */
/*	 1:	156->187	( 32cyls)	drive 0	*/	13376l,	156,	0,	/* user disk 1 */
/*	 2:	188->219	( 32cyls)	drive 0	*/	13376l,	188,	0,	/* user disk 2 */
/*	 3:	220->229	( 10cyls)	drive 0	*/	 4180l,	220,	0,	/* swap area */
/*	 4:	230->254	( 25cyls)	drive 0	*/	10450l,	230,	0,	/* system disk */
/*	 5:	255->410	(156cyls)	drive 0	*/	65208l,	255,	0,	/* user disk 3 */
/*	 6:	dummy					*/	    0l,	411,	0,	/* dummy */
/*	 7:	dummy					*/	    0l,	411,	0,	/* dummy */
};

/*
 *	structure of an hp disc queue
 */

struct hpq
	{
	struct	buf	*hp_bufp;	/* pointer to first buffer in queue for this drive */
#ifdef	SWEEP
	int	hp_lcyl;		/* last cylinder accessed on this drive */
	char	hp_dirf;		/* current direction of head movement */
#endif
#ifdef	HPSTATS
	char	hp_nreq;		/* number of requests in queue - stats */
	char	hp_rmax;			/* high water mark for q */
	unsigned	hp_cyls[NHPCYLS];	/* accesses per cylinder */
#endif
	} hp_q[NDRV];

struct hpq	*hp_ap;		/* pointer to queue of drive currently doing io */

/*
 *	We use a few buffer variables for other purposes here, ie:
 *		b_scratch	to hold unit/cylinder address of this request
 *		b_resid		to hold track/sector address of this request
 *		av_back		to hold request age count
 */

#define	hp_cyl		b_scratch	/* at least an int */
#define	hp_trksec	b_resid		/* at least an int */
#define	hp_age		av_back

struct	devtab	hptab;

#ifndef	RAW_BUFFER_POOL
struct	buf	hpbuf;
#endif

/*	some handy masks	*/
#define	BYTE	0377
#define	LSD	07	/* least significant octal digit */
#define	UNIT	016000	/* the position of unit no. in hp_cyl */
#define	CYL	0777	/* the desired/present cylinder bits */

/*	some equally handy numbers	*/
#define	SBYTE	8	/* amount by which to shift for a byte */
#define	SUNIT	10	/* amount by which to shift unit nos. */

/*	a flag			*/
#define	B_SEEK	0200	/* the seeking flag for buffer header */

/*	drive commands (hpcs1)	*/
#define	GO	01
#define	PRESET	020
#define	RECAL	06
#define	RCLR	010
#define	SEEK	04

/*	hpds bits	*/
#define	DRY	0200	/* drive ready used for DIVAs */
#define	VV	0100	/* volume valid */
#define	MOL	010000	/* medium on line */
#define	ATA	0100000	/* attention bit */

/*	hpcs1 bits	*/
#define	IE	0100	/* interrupt enable bit */
#define	CRDY	0200	/* controller ready */
#define	ERR	040000	/* the OR of error bits */
#define	SC	0100000	/* special condition flag */

/*	hper1 bits	*/
#define	DTE	010000	/* drive timing error */
#define	OPI	020000	/* operation incomplete */
#define	UNS	040000	/* drive unsafe */
#define	DCK	0100000	/* data check error for RHPs and RJPs (DEC) */

/*	hpcs2 bits	*/
#define	CLR	040	/* controller clear */

/*	hpof bits	*/
#define	FMT22	010000	/* 16bit format for RHPs and RJPs (DEC) */

char	hp_onlinef;


hpopen()
{
	register int	n;

	/*
	 *	if there are no active drives,
	 *		then clear controller and wait for ready
	 *	for all possible non-initialised drives
	 *		initialise
	 */

	HPSPL();

	if( !hp_onlinef)
		{
		HPADDR->hpcs2 = CLR;
		while( !(HPADDR->hpcs1 & CRDY));
		}
	for(n=0; n<NDRV; n++)
		if( !(hp_onlinef & (1<<n)))
			hpdinit(n);

	spl0();
}

hpstrategy(bp)
register struct buf	*bp;
{
	register unsigned	p1, p2;
	register int	f;

	p1 = bp->b_dev.d_minor & BYTE;

/*
 *	Validate the request
 */

	if(
		(p1 >= TVOL)
	    || ((p2 = &hp_sizes[p1])->nblocks <= bp->b_blkno)
#ifdef	NO_OFFLINEQ | DROP_OFFLINEQ
	    || ( !(hp_onlinef & (1<< (p2->drive))))
#endif
	  )
		{
		bp->b_flags =| B_ERROR;
		iodone(bp);
		return;
		}

#ifdef	_1170 & RJP04
	if(bp->b_flags & B_PHYS)
		mapalloc(bp);
#endif

	bp->av_forw = 0;
	bp->hp_age = HPAGE;
	bp->hp_cyl = p2->cyloff | ((f = p2->drive & LSD) << SUNIT);

	p1 = bp->b_blkno;
	p2 = lrem(p1,22); /* sector */
	p1 = ldiv(p1,22); /* trk and cyl */
	bp->hp_trksec = ((p1%19) << SBYTE) | p2;
	bp->hp_cyl =+ p1/19;
	p2 = &hp_q[f];

/*
 *	Now "hp_cyl" contains "drive and cylinder"
 *	and "hp_trksec" contains the "track and sector"
 */

#ifdef	HPSTATS
	p2->hp_cyls[bp->hp_cyl & CYL]++;
	p2->hp_nreq++;
#endif

	HPSPL();
	if ((p1 = p2->hp_bufp) == NULL)
		{
		/* this queue is empty */
		p2->hp_bufp = bp;
		if(HPADDR->hpcs1 & CRDY)
			hpstart();
		}
	  else
		{
		/*
		 * the queue is not empty, so place in queue so as to
		 * minimise head movement.
		 */

#ifdef	SWEEP
		f = p2->hp_dirf;
#endif

		p2 = p1->av_forw;
		while(p2)
			{
			/* skip any overtaken blocks */
			if( !(p2->hp_age))
				p1 = p2;
			p2 = p2->av_forw;
			}

		for(; p2 = p1->av_forw; p1 = p2)

			if (	p1->hp_cyl <= bp->hp_cyl
			     && bp->hp_cyl <= p2->hp_cyl
			     || p1->hp_cyl >= bp->hp_cyl
			     && bp->hp_cyl >= p2->hp_cyl
			 )	{
				while (bp->hp_cyl == p2->hp_cyl)
					{
					/*
					 * for a cylinder match, do the
					 * rotational optimisation.
					 * INTLV should be set to an optimal
					 * value, thought to be around 9
					 * and should be altered if the discs
					 * are reformatted...... good luck.
					 */
					if(p2->hp_trksec > p1->hp_trksec)
						{
						if(bp->hp_trksec > p1->hp_trksec + INTLV
						&& bp->hp_trksec < p2->hp_trksec - INTLV)
							goto out;
						}
					  else
						if(bp->hp_trksec > p1->hp_trksec + INTLV
						|| bp->hp_trksec < p2->hp_trksec - INTLV)
							goto out;
					p1 = p2;
					if( !(p2 = p1->av_forw)) goto out;
					}
				break;
				}

#ifdef SWEEP
			  else
				if (f)
					{
					if(p2->hp_cyl < p1->hp_cyl)
						if(bp->hp_cyl > p1->hp_cyl)
							goto out;
						else
							f = 0;
					}
				  else
					{
					if(p2->hp_cyl > p1->hp_cyl)
						if(bp->hp_cyl < p1->hp_cyl)
							goto out;
						else
							f++;
					}
#endif

	out:
		bp->av_forw = p2;
		p1->av_forw = bp;

		while(p2)
			{
			/* count down overtaken blocks */
			if( p2->hp_age)
				p2->hp_age--;
			p2 = p2->av_forw;
			}
		}

	spl0();

}

/*
 *	hpstart() goes through all the queues and sets everybody
 *	seeking that should be.
 */
hpstart()
{
	/* called at HPSPL or greater */

	register struct buf *bp;
	register struct hpq *qp;

	if(hp_ap != NULL) return;

	for (qp = hp_q; qp < &hp_q[ NDRV ]; qp++)
#ifndef	NO_SEEK
		if ((bp = qp->hp_bufp) && (bp->b_flags&B_SEEK)==0)
			{
#endif
#ifdef	NO_SEEK
		if ( bp = qp->hp_bufp )
			{
#endif
#ifdef	SWEEP
			if(bp->hp_cyl > qp->hp_lcyl) qp->hp_dirf = 1;
			else if(bp->hp_cyl < qp->hp_lcyl) qp->hp_dirf = 0;
#endif
			HPADDR->hpcs2 = (bp->hp_cyl >> SUNIT) & LSD;
			HPADDR->hpdc = bp->hp_cyl & CYL;
#ifndef	NO_SEEK
			HPADDR->hpcs1 = IE | SEEK | GO;
			while( !(HPADDR->hpcs1 & CRDY));
			bp->b_flags =| B_SEEK;
#endif
#ifdef	NO_SEEK
#ifdef	SWEEP
			qp->hp_lcyl = bp->hp_cyl;
#endif
			hp_ap = qp;
#ifdef	RHP04
			rhstart(bp, &HPADDR->hpda, bp->hp_trksec, &HPADDR->hpbae);
#endif
#ifdef	RJP04
			rjpstart(bp, &HPADDR->hpda, bp->hp_trksec);
#endif
			return;
#endif
			}
}


hpintr()
{
	/* called at HPSPL or greater */

	register int	n;
	register struct hpq *qp;
	register struct buf *bp;

	/*
	 *	Here we have either a proper data transfer completion or
	 *	some special condition has occured
	 */

	if(HPADDR->hpcs1 & SC)
		{
		/*
		 *	Here have a special condition, which may be TRE,
		 *	a drive error or a nice little seek completion
		 */

		/* MOL, VV, DRY handling */

		if(HPADDR->hpas)
			/* attention is up somewhere */
			for(n=0; n<NDRV; n++)
				{
				HPADDR->hpcs2=n;
				if(HPADDR->hpds & MOL)
					{
					/* MOL, if !VV initialise */
					if( !(HPADDR->hpds & VV)) hpdinit(n);
					}
				  else
					{
					/* !MOL, flag as such */
					hp_onlinef =& ~(1<<n);
					qp= &hp_q[n];
					if(bp=qp->hp_bufp)
						{
						hperr(bp);
						bp->b_flags =& ~B_SEEK;
						if(qp == hp_ap)
							{
							HPADDR->hpcs1 = ERR;
							hp_ap=NULL;
							}
						}
#ifdef	DROP_OFFLINEQ
					while(bp=qp->hp_bufp)
						{
						bp->b_flags =| B_ERROR;
						qp->hp_bufp = bp->av_forw;
						iodone(bp);
						}
#endif
					HPADDR->hpas= (1<<n);
					HPADDR->hpcs1 = IE;
					return;
					}
				}

		/*
		 *	now have on/offline handled,
		 *	must handle other possibilities properly
		 */

#ifndef	NO_SEEK
		if( !(HPADDR->hpcs1 & ERR))
			/*
			 *	here have either a drive error or seek complete
			 */

			for(n=0; n<NDRV; n++)
				{
				HPADDR->hpcs2 = n;
				bp = (qp = &hp_q[n])->hp_bufp;
				if(HPADDR->hpds & ERR)
					{
					/*
					 *	here we have a real drive error on drive n
					 *	during a seek
					 */

					bp->b_flags =& ~B_SEEK;
					hperr(bp);
					n=0;
					if(HPADDR->hper1 & (UNS | DTE | OPI)) n++;
					HPADDR->hpcs1 = RCLR | GO;
					if(n)
						{
						while( !(HPADDR->hpds & DRY));
						HPADDR->hpcs1 = RECAL | GO;
						while( !(HPADDR->hpds & DRY));
						}
					if(errcnt()) goto ok;
					HPADDR->hpcs1 = IE | SEEK | GO;
					while( !(HPADDR->hpcs1 & CRDY));
					bp->b_flags =| B_SEEK;
					/* break below should be deleted if controller does
					   not queue interrupt requests */
					break;
					}
				  else
					if(HPADDR->hpds & ATA) 
						/*
						 *	here have a seek complete
						 *	or the odd
						 *	possibility of a MOL interrupt
						 */
						if( !(bp->b_flags & B_SEEK))
							{
							/* MOL */
							hpstart();
							}
						  else
							{
							bp->b_flags =& ~B_SEEK;
#ifdef	SWEEP
							qp->hp_lcyl = bp->hp_cyl;
#endif
							hp_ap = qp;
							/* the following line is redundant */
							/* the cyliner should be ok, BUT!! */
							HPADDR->hpdc = bp->hp_cyl & CYL;
#ifdef	RHP04
							rhstart(bp, &HPADDR->hpda, bp->hp_trksec, &HPADDR->hpbae);
#endif
#ifdef	RJP04
							rjpstart(bp, &HPADDR->hpda, bp->hp_trksec);
#endif
							break;
							}
				}
		  else
			{
#endif
			/*
			 *	here have a data transfer error
			 */
			if((qp = hp_ap) && (bp = qp->hp_bufp))
				{
				n = (bp->hp_cyl >> SUNIT) &LSD;
#ifdef	RHP04 & HPECC
				HPADDR->hpcs2 = n;
				if(HPADDR->hper1 == DCK)
					{
					if(hpecc()) return;
					}
				  else
					{
#endif
					hperr(bp);
#ifndef	NO_SEEK
					for(qp = hp_q; qp < &hp_q[NDRV]; qp++)
						if(qp->hp_bufp)
							qp->hp_bufp->b_flags =& ~B_SEEK;
#endif
					HPADDR->hpcs2 = CLR;
					while( !(HPADDR->hpcs1 & CRDY));
					HPADDR->hpcs2 = n;
					HPADDR->hpcs1 = RECAL | GO;
					while( !(HPADDR->hpds & DRY));
					if(errcnt()) goto ok;
					HPADDR->hpdc = bp->hp_cyl & CYL;
#ifdef	RHP04
					rhstart(bp, &HPADDR->hpda, bp->hp_trksec, &HPADDR->hpbae);
#ifdef	HPECC
					}
#endif
#endif
#ifdef	RJP04
				rjpstart(bp, &HPADDR->hpda, bp->hp_trksec);
#endif
				}
			  else
				hpstart();
#ifndef	NO_SEEK
			}
#endif
		}
	  else
		{
		/*
		 *	here have a normal data transfer completion
		 */
transfer:
		if((qp = hp_ap) && (HPADDR->hpcs1 & CRDY))
			{
			hp_ap = NULL;
			bp = qp->hp_bufp;

ok:
			qp->hp_bufp = bp->av_forw;
#ifdef	HPSTATS
			if( qp->hp_nreq-- > qp->hp_rmax)
				qp->hp_rmax = qp->hp_nreq+1;
#endif
			hpstart();
			bp->b_resid = HPADDR->hpwc;
			iodone(bp);
			}
		}

}

errcnt(bp)
register struct buf	*bp;
{
	if( ++(bp->b_error) > 10)
		{
		bp->b_flags =| B_ERROR;
		return(1);
		}
	return(0);
}

hperr(bp)
register struct buf	*bp;
{
	register int	*p;

	p = HPADDR;

	printf("HPERR  CS1=%o,WC=%o,BA=%o,DA=%o,CS2=%o,DS=%o,ER1=%o,AS=%o,DC=%o\n",
		HPADDR->hpcs1, HPADDR->hpwc, HPADDR->hpba, HPADDR->hpda,
		HPADDR->hpcs2, HPADDR->hpds, HPADDR->hper1, HPADDR->hpas, HPADDR->hpdc);
		printf("       BUFFER FLAGS=%o\n", bp->b_flags);
}

/*
 *	Physical IO:
 *	truncate transfers at the ends of logical volumes
 */

unsigned	hpresidue;

hpread(dev)
int	dev;
{
	if(hpphys(dev))
		{
#ifndef	RAW_BUFFER_POOL
		physio(hpstrategy, &hpbuf, dev, B_READ);
#endif
#ifdef	RAW_BUFFER_POOL
		physio(hpstrategy,      0, dev, B_READ);
#endif
		u.u_count =+ hpresidue;
		}
}

hpwrite(dev)
int	dev;
{
	if(hpphys(dev))
		{
#ifndef	RAW_BUFFER_POOL
		physio(hpstrategy, &hpbuf, dev, B_WRITE);
#endif
#ifdef	RAW_BUFFER_POOL
		physio(hpstrategy,      0, dev, B_WRITE);
#endif
		u.u_count =+ hpresidue;
		}
}

hpphys(dev)
int	dev;
{
	register unsigned a, b, c;

	a = hp_sizes[dev.d_minor & 07].nblocks;
	b = u.u_offset >> 9;
	if(b >= a)
		{
		u.u_error = ENXIO;
		return(0);
		}
	c = u.u_count;
	if(ldiv(c+511, 512) + b > a)
		c = (a - b) << 9;
	hpresidue = u.u_count - c;
	u.u_count = c;
	return(1);
}

hppowerf()
{
	/*
	 *	hp disc power fail recovery routine
	 *
	 *	It is ASSUMED that
	 *
	 *	. a power fail has occured and the cpu did a reset
	 *	. mem management registers have been restored
	 *	. as have unibus map regs as have
	 *	. the other vital ones.
	 */

	register struct buf	*bp;
	register struct hpq	*qp;
	register	d;

	long	n;

	HPADDR->hpcs2=CLR;
	hp_ap=NULL;
	for(d=0; d< NDRV; d++)
		{
		qp= &hp_q[d];
		if(bp=qp->hp_bufp)
			bp->b_flags =& ~B_SEEK;
		if(hp_onlinef & (1<<d))
			{
			/*
			 *	this drive was on when it died
			 */

			for(;;)
				{
				for(n=2000000; n>0; n--);
				if( (HPADDR->hpcs1 & CRDY) && (HPADDR->hpds & MOL) && (HPADDR->hpds & DRY) )
					{
					hpdinit(d);
					break;
					}
				printf("HPERR PWR FAIL RECVRY WAITING DRIVE %o\n", d);
				}
			}
		}

	hpstart();

	printf("HPERR PWR FAIL RECVRD\n");
}

hpdinit(d)
register int	d;
{
	/* called at HPSPL or greater */

	HPADDR->hpcs2 = d & LSD;
	if(HPADDR->hpds & MOL)
		{
		HPADDR->hpcs1 = RCLR | GO;
		/*
		 *	possibly some risk in these while loops here and
		 *	in previous code in that if the drive is maybe
		 *	MOL & !(DRY) ever....... not very likely and you
		 *	would be buggered if it was.....
		 */
		while( !(HPADDR->hpds & DRY));
		HPADDR->hpcs1 = PRESET | GO;
		while( !(HPADDR->hpds & DRY));
		HPADDR->hpof = FMT22;
		hp_onlinef =| (1<<d);
		}
}

/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/
/*****************************************************************************************/

#define	IENABLE	0100

/* code below to reside in m70/40/45?.s and defines to be checked as defined */

#ifdef	RJP04

/*
 *	device start routine for RJP04 type devices not connected via an
 *	RH70. Will work for things like TJU16s.
 *	Works for DIVA V Computrollers connected to Ampex DM9100s also.
 */

#define	RJPWCOM	060
#define	RJPRCOM	070

rjpstart(rbp,devloc,devblk)
register struct buf	*rbp;
int	*devloc;
{
	register int	*dp;
	register int	com;

	dp=devloc;
	*dp=devblk;		/* block address */
	*--dp=rbp->b_addr;	/* buffer address - map may have been called */
	*--dp=rbp->b_wcount;	/* word count - should be 2s complement */
	com= ((rbp->b_xmem & 03) << 8) |
		IENABLE | GO;
	if(rbp->b_flags & B_READ)
		com =| RJPRCOM;
	else
		com =| RJPWCOM;
	*--dp=com;
}
#endif
