#ifndef lint
static char rcsid[]="$Header: draw.c,v 5.0 87/03/24 14:18:21 solomon Stab $";
static char cr[]="Copyright (c) 1986, Marvin Solomon";
#endif lint
/*	draw.c	1.8	84/05/09
 *
 *	This file contains the functions for producing the graphics
 *   images in the canon/imagen driver for ditroff (skeleton version).
 *
 * 
 */

/* we really should include <math.h> here, but it contains all sorts of
 * stuff that gets lint upset, so we just declare what we need below
 */
#include <stdio.h>
#include <ctype.h>
#include <X/Xlib.h>
#include "proof.h"
#define MAXPOINTS 500
#define  pi		3.14159265358979324
extern int PPI, RES; /* points per inch and resolution (units per inch) */
extern int Foreground;
extern int Background;
extern int Overstrike;

double atan2(), sqrt(), pow();

/* Despite their names, the following procedures merely save relevant data in
 * the page buffer, leaving actual drawing for later (see dumpxxx() below).
 * When these procedures are called, the "current postion" in (hpos,vpos)
 * is expressed in device units.  Distances and postions saved, however,
 * are measured in screen unitts (pixels).
 */
drawline(dh, dv)
{
	hmot(dh);
	vmot(dv);
	SAVE(OPLINE)
	SAVE(hpos*PPI/RES)
	SAVE(vpos*PPI/RES)
}

drawcirc(d)
{
	hmot(d);
	SAVE(OPCIRC | (hpos*PPI/RES))
}

drawellip(hd, vd)
int hd;
int vd;
{
	hmot(hd);
	SAVE(OPELLIP)
	SAVE(hpos*PPI/RES)
	SAVE(vd*PPI/RES)
}

drawarc (cdh, cdv, pdh, pdv)
register int cdh;
register int cdv;
register int pdh;
register int pdv;
{
	hmot(cdh+pdh);
	vmot(cdv+pdv);
	SAVE(OPARC)
	SAVE(cdh*PPI/RES)
	SAVE(cdv*PPI/RES)
	SAVE(hpos*PPI/RES)
	SAVE(vpos*PPI/RES)
}

drawwig (buf, fp, pic)
char *buf;
FILE *fp;
int pic;
{
	register int len = strlen(buf);	/* length of the string in "buf" */
	register char *ptr = buf;		/* "walking" pointer into buf */

	SAVE(OPWIG | pic)
	SAVE(hpos*PPI/RES)
	SAVE(vpos*PPI/RES)
	while (*ptr == ' ') ptr++;		/* skip any leading spaces */

	while (*ptr != '\n') {			/* curve commands end with a '\n' */
		hmot(atoi(ptr));			/* convert motion to curve points */
		SAVE(hpos*PPI/RES)			/* and remember them */
		while (isdigit(*++ptr));	/* skip number*/
		while (*++ptr == ' ');		/* skip spaces 'tween numbers */
		vmot(atoi(ptr));
		SAVE(vpos*PPI/RES)
		while (isdigit(*++ptr));
		while (*ptr == ' ') ptr++;
					/* if the amount we read wasn't the */
					/*    whole thing, read some more in */
		if (len - (ptr - buf) < 15 && *(buf + len - 1) != '\n') {
			char *cop = buf;

			while (*cop++ = *ptr++);	/* copy what's left to the beginning */
			if (fgets ((cop - 1), len - (cop - buf), fp) == NULL)
				error (1, "unexpected end of input");
			ptr = buf;
		}
	}
	SAVE(-1)
}

drawthick(s)
int s;
{
	/* The thickness number is too big if interpreted as pixels, so
	 * we scale it down.  Since the current version of Gremlin uses
	 * 5 for thick, 3 for medium, and 1 for thin, the following function
	 * gives the thinest lines that still show some difference.
	 */
	s = (s + 1)/2;
	SAVE(OPTHICK | s)
}

drawstyle(s)
int s;
{
	SAVE(OPSTYLE | styleindex(s))
}

static struct { int tmask; Pattern xmask; } linestyle[] =
{
	{ -1,  SolidLine }, /* default; should be first */
	{ 04,  DottedLine },
	{ 020, DashedLine },
	{ 024, DotDashLine },
	{ 0,   0}
};

/* translate a funny troff line-style mask into an integer index */
int
styleindex(s)
{
	int i;
	for (i=0;linestyle[i].tmask;i++)
		if (linestyle[i].tmask == s) {
			return i;
		}
	fprintf(stderr,"line style %d unknown\n",s);
	return 0;
}

/*
 * Routines that actually do the drawing (some of the simpler ones
 * are done in "dumppage" in proof.c).
 */

/* Dump a broken (polygonal) line. */
Vertex *
dumppoly(v,n)
Vertex *v;
{
	v[n-1].flags = VertexDrawLastPoint;
	XDrawDashed(window,
		v,n,Thickness,Thickness,Foreground,linestyle[Style].xmask,
		Overstrike,AllPlanes);
	return v;
}

dumpline(newx, newy)
{
	Vertex v[2];
	v[0].x = hpos;
	v[0].y = vpos;
	v[0].flags = 0;
	v[1].x = newx;
	v[1].y = newy;
	(void) dumppoly(v,2);
	hpos = newx;
	vpos = newy;
}

/* Draw a circle with left side at (hpos,vpos) and right side at (newh, vpos) */
dumpcirc(newh)
{
	Arc((hpos + newh)/2,vpos,newh,vpos,0);
	hpos = newh;
}

/* Draw an ellipse of height height with left side at (hpos,vpos) and right
 * side at (newh, vpos) */
dumpellip(newh,height)
{
	Vertex v[5];
	int radius = (newh - hpos)/2;

	v[0].x = newh;
	v[1].x = hpos+radius;
	v[2].x = hpos;
	v[3].x = hpos+radius;
	v[4].x = newh;
	v[0].y = vpos;
	v[1].y = vpos+height/2;
	v[2].y = vpos;
	v[3].y = vpos-height/2;
	v[4].y = vpos;
	v[0].flags = VertexCurved|VertexStartClosed;
	v[1].flags = VertexCurved;
	v[2].flags = VertexCurved;
	v[3].flags = VertexCurved;
	v[4].flags = VertexCurved|VertexEndClosed;

	XDraw(window, v, 5, Thickness, Thickness, Foreground, Overstrike,
		AllPlanes);
	hpos = newh;
}

/* Draw an arc with center offset cdh,cdv (relative to current hpos,vpos)
 * and endpoint newh, newv
 */
dumparc (cdh, cdv, newh, newv)
{
	/* offset of end point relative to center */
	int pdh = newh - hpos - cdh;
	int pdv = newv - vpos - cdv;
    register double angle;

	/* figure angle from the three points...*/
	/* and convert (and round) to degrees */
    angle = (atan2((double) pdh, (double) pdv)
		- atan2 ((double) -cdh, (double) -cdv)) * 180.0 / pi;
	/* "normalize" and round */
    angle += (angle < 0.0)  ?  360.5 : 0.5;

    Arc(hpos + cdh, vpos + cdv, hpos, vpos, (int) angle);
	hpos = newh;
	vpos = newv;
}

/* Draw a curve or polygon.  P is a pointer to an array of shorts,
 * starting with the curve-type in (*p & OPERAND), followed by (x,y)
 * pairs, and ending with -1.
 * The result is a pointer to the -1 entry.  Unlike the above routines,
 * this one also updates hpos and vpos.
 *
 * I used to use to use X spline facilities, but they looked so awful,
 * I decided to use my own spline function (borrowed from another
 * ditroff driver).
 */
short *
dumpwig(p)
register short *p;
{
	int x[MAXPOINTS], y[MAXPOINTS], type, nextpoint=1;

	type = *p++ & OPERAND;
	while (*p >= 0) {
		x[nextpoint] = *p++;
		y[nextpoint] = *p++;
		if (nextpoint < MAXPOINTS-1) nextpoint++;
	}
	nextpoint--;
	hpos = x[nextpoint];
	vpos = y[nextpoint];
	switch (type) {
	case 0:
		GCurve(x, y, nextpoint);
		break;
	case 1:
		picurve(x, y, nextpoint);
		break;
	case 2:
		fprintf(stderr,"polygons not supported yet\n");
		break;
	default:
		fprintf(stderr,"internal error: unknown curve type %d\n",type);
		break;
	}
	return p;
}

/*----------------------------------------------------------------------------
 | Routine:	picurve (xpoints, ypoints, num_of_points)
 |
 | Results:	Draws a curve delimited by (not through) the line segments
 |		traced by (xpoints, ypoints) point list.  This is the "Pic"
 |		style curve.
 *---------------------------------------------------------------------------*/

picurve (x, y, npts)
int x[MAXPOINTS];
int y[MAXPOINTS];
int npts;
{
	register int i;			/* line segment traverser */
	register float w;		/* position factor */
	register int xp;		/* current point (and intermediary) */
	register int yp;
	register int j;			/* inner curve segment traverser */
	register int nseg;		/* effective resolution for each curve */
	float t1, t2, t3;		/* calculation temps */


	if (x[1] == x[npts] && y[1] == y[npts]) {
		x[0] = x[npts - 1];	/* if the lines' ends meet, make */
		y[0] = y[npts - 1];	/* sure the curve meets */
		x[npts + 1] = x[2];
		y[npts + 1] = y[2];
	} else {				/* otherwise, make the ends of the */
		x[0] = x[1];		/* curve touch the ending points of */
		y[0] = y[1];		/* the line segments */
		x[npts + 1] = x[npts];
		y[npts + 1] = y[npts];
	}

	for (i = 0; i < npts; i++) {	/* traverse the line segments */
		xp = x[i] - x[i+1];
		yp = y[i] - y[i+1];
		nseg = (int) sqrt((double)(xp * xp + yp * yp));
		xp = x[i+1] - x[i+2];
		yp = y[i+1] - y[i+2];/* "nseg" is the number of line */
							/* segments that will be drawn for */
							/* each curve segment.  ">> 3" is */
							/* dropping the resolution ( == / 8) */
		nseg = (nseg + (int) sqrt((double)(xp * xp + yp * yp))) >> 3;

		hpos = (x[i]+x[i+1]+1) >> 1;	/* the start of the first line seg */
		vpos = (y[i]+y[i+1]+1) >> 1;
		for (j = 1; j <= nseg; j++) {
			w = (float) j / (float) nseg;
			t1 = 0.5 * w * w;
			w -= 0.5;
			t2 = 0.75 - w * w ;
			w -= 0.5;
			t3 = 0.5 * w * w;
			xp = t1 * x[i+2] + t2 * x[i+1] + t3 * x[i] + 0.5;
			yp = t1 * y[i+2] + t2 * y[i+1] + t3 * y[i] + 0.5;
			dumpline (xp, yp);
		}
	}
}


/*----------------------------------------------------------------------------
 | Routine:	Arc (xcenter, ycenter, xstart, ystart, angle)
 |
 | Results:	This routine plots an arc centered about (cx, cy) counter
 |		clockwise starting from the point (px, py) through 'angle'
 |		degrees.  If angle is 0, a full circle is drawn. It does so
 |		by creating a draw-path around the arc whose density of
 |		points depends on the size of the arc.
 *---------------------------------------------------------------------------*/

Arc(cx,cy,px,py,angle)
register int cx;
register int cy;
int px, py, angle;
{
	double xs, ys, resolution, fullcircle;
	register int extent;
	register int nx;
	register int ny;
	register double epsilon;
	Vertex v[MAXPOINTS], *vp;

	xs = px - cx;
	ys = py - cy;

	/* calculate how fine to make the lines that build
	   the circle: r steps per radian of arc, where r is the radius */

	resolution = sqrt(xs * xs + ys * ys);

	epsilon = 1.0 / resolution;
	fullcircle = (2.0 * pi) * resolution;
	if (angle == 0)
		extent = fullcircle;
	else 
		extent = angle * fullcircle / 360.0;

	vp = v;
	if (extent > 1) {
		vp->x = px;
		vp->y = py;
		vp->flags = 0;
		vp++;
		while (--extent >= 0) {
			xs += epsilon * ys;
			nx = cx + (int) (xs + 0.5);
			ys -= epsilon * xs;
			ny = cy + (int) (ys + 0.5);
			vp->x = nx;
			vp->y = ny;
			vp->flags = 0;
			if (++vp - v == MAXPOINTS)
				vp = dumppoly(v,MAXPOINTS);
		}   /* end while */
	} else {			/* arc is too small: put out point */
		vp->x = px;
		vp->y = py;
		vp->flags = 0;
		vp++;
		vp->x = px;
		vp->y = py;
		vp->flags = 0;
		vp++;
	}
	(void) dumppoly(v,vp-v);
}  /* end Arc */



/*----------------------------------------------------------------------------
 | Routine:	Paramaterize (xpoints, ypoints, hparams, num_points)
 |
 | Results:	This routine calculates parameteric values for use in
 |		calculating curves.  The parametric values are returned
 |		in the array h.  The values are an approximation of
 |		cumulative arc lengths of the curve (uses cord length).
 |		For additional information, see paper cited below.
 *---------------------------------------------------------------------------*/

static Paramaterize(x, y, h, n)
int x[MAXPOINTS];
int y[MAXPOINTS];
float h[MAXPOINTS];
int n;
{
	register int dx;
	register int dy;
	register int i;
	register int j;
	float u[MAXPOINTS];


	for (i=1; i<=n; ++i) {
		u[i] = 0;
		for (j=1; j<i; j++) {
			dx = x[j+1] - x[j];
			dy = y[j+1] - y[j];
			u[i] += sqrt ((double) (dx * dx + dy * dy));
		}
	}
	for (i=1; i<n; ++i)  h[i] = u[i+1] - u[i];
}  /* end Paramaterize */


/*----------------------------------------------------------------------------
 | Routine:	PeriodicSpline (h, z, dz, d2z, d3z, npoints)
 |
 | Results:	This routine solves for the cubic polynomial to fit a
 |		spline curve to the the points  specified by the list
 |		of values.  The Curve generated is periodic.  The algorithms
 |		for this curve are from the "Spline Curve Techniques" paper
 |		cited below.
 *---------------------------------------------------------------------------*/

static PeriodicSpline(h, z, dz, d2z, d3z, npoints)
float h[MAXPOINTS];		/* paramaterization  */
int z[MAXPOINTS];		/* point list */
float dz[MAXPOINTS];			/* to return the 1st derivative */
float d2z[MAXPOINTS], d3z[MAXPOINTS];	/* 2nd and 3rd derivatives */
int npoints;				/* number of valid points */
{
	float d[MAXPOINTS]; 
	float deltaz[MAXPOINTS], a[MAXPOINTS], b[MAXPOINTS];
	float c[MAXPOINTS], r[MAXPOINTS], s[MAXPOINTS];
	int i;

						/* step 1 */
	for (i=1; i<npoints; ++i) {
		deltaz[i] = h[i] ? ((double) (z[i+1] - z[i])) / h[i] : 0;
	}
	h[0] = h[npoints-1];
	deltaz[0] = deltaz[npoints-1];

						/* step 2 */
	for (i=1; i<npoints-1; ++i) {
		d[i] = deltaz[i+1] - deltaz[i];
	}
	d[0] = deltaz[1] - deltaz[0];

						/* step 3a */
	a[1] = 2 * (h[0] + h[1]);
	b[1] = d[0];
	c[1] = h[0];
	for (i=2; i<npoints-1; ++i) {
		a[i] = 2*(h[i-1]+h[i]) - pow ((double) h[i-1],(double)2.0) / a[i-1];
		b[i] = d[i-1] - h[i-1] * b[i-1]/a[i-1];
		c[i] = -h[i-1] * c[i-1]/a[i-1];
	}

						/* step 3b */
	r[npoints-1] = 1;
	s[npoints-1] = 0;
	for (i=npoints-2; i>0; --i) {
		r[i] = -(h[i] * r[i+1] + c[i])/a[i];
		s[i] = (6 * b[i] - h[i] * s[i+1])/a[i];
	}

						/* step 4 */
	d2z[npoints-1] = (6 * d[npoints-2] - h[0] * s[1] 
					   - h[npoints-1] * s[npoints-2]) 
					 / (h[0] * r[1] + h[npoints-1] * r[npoints-2] 
						+ 2 * (h[npoints-2] + h[0]));
	for (i=1; i<npoints-1; ++i) {
		d2z[i] = r[i] * d2z[npoints-1] + s[i];
	}
	d2z[npoints] = d2z[1];

						/* step 5 */
	for (i=1; i<npoints; ++i) {
		dz[i] = deltaz[i] - h[i] * (2 * d2z[i] + d2z[i+1])/6;
		d3z[i] = h[i] ? (d2z[i+1] - d2z[i])/h[i] : 0;
	}
}  /* end PeriodicSpline */


/*----------------------------------------------------------------------------
 | Routine:	NaturalEndSpline (h, z, dz, d2z, d3z, npoints)
 |
 | Results:	This routine solves for the cubic polynomial to fit a
 |		spline curve the the points  specified by the list of
 |		values.  The alogrithms for this curve are from the
 |		"Spline Curve Techniques" paper cited below.
 *---------------------------------------------------------------------------*/

static NaturalEndSpline(h, z, dz, d2z, d3z, npoints)
float h[MAXPOINTS];		/* parameterization */
int z[MAXPOINTS];		/* Point list */
float dz[MAXPOINTS];	/* to return the 1st derivative */
float d2z[MAXPOINTS], d3z[MAXPOINTS];	/* 2nd and 3rd derivatives */
int npoints;			/* number of valid points */
{
	float d[MAXPOINTS];
	float deltaz[MAXPOINTS], a[MAXPOINTS], b[MAXPOINTS];
	int i;

						/* step 1 */
	for (i=1; i<npoints; ++i) {
		deltaz[i] = h[i] ? ((double) (z[i+1] - z[i])) / h[i] : 0;
	}
	deltaz[0] = deltaz[npoints-1];

						/* step 2 */
	for (i=1; i<npoints-1; ++i) {
		d[i] = deltaz[i+1] - deltaz[i];
	}
	d[0] = deltaz[1] - deltaz[0];

						/* step 3 */
	a[0] = 2 * (h[2] + h[1]);
	b[0] = d[1];
	for (i=1; i<npoints-2; ++i) {
		a[i] = 2*(h[i+1]+h[i+2]) - pow((double) h[i+1],(double) 2.0)/a[i-1];
		b[i] = d[i+1] - h[i+1] * b[i-1]/a[i-1];
	}

						/* step 4 */
	d2z[npoints] = d2z[1] = 0;
	for (i=npoints-1; i>1; --i) {
		d2z[i] = (6 * b[i-2] - h[i] *d2z[i+1])/a[i-2];
	}

						/* step 5 */
	for (i=1; i<npoints; ++i) {
		dz[i] = deltaz[i] - h[i] * (2 * d2z[i] + d2z[i+1])/6;
		d3z[i] = h[i] ? (d2z[i+1] - d2z[i])/h[i] : 0;
	}
}  /* end NaturalEndSpline */


/*----------------------------------------------------------------------------
 | Routine:	GCurve(xpoints, ypoints, num_points)
 |
 | Results:	This routine generates a smooth curve through a set of points.
 |		The method used is the parametric spline curve on unit knot
 |		mesh described in "Spline Curve Techniques" by Patrick
 |		Baudelaire, Robert Flegal, and Robert Sproull -- Xerox Parc.
 *---------------------------------------------------------------------------*/

#define PointsPerInterval 32

GCurve(x, y, numpoints)
int x[MAXPOINTS];
int y[MAXPOINTS];
int numpoints;
{
	float h[MAXPOINTS], dx[MAXPOINTS], dy[MAXPOINTS];
	float d2x[MAXPOINTS], d2y[MAXPOINTS], d3x[MAXPOINTS], d3y[MAXPOINTS];
	float t, t2, t3;
	register int j;
	register int k;
	register int nx;
	register int ny;
	Vertex v[MAXPOINTS], *vp;

	/* Solve for derivatives of the curve at each point 
	 * separately for x and y (parametric).
	 */
	Paramaterize(x, y, h, numpoints);
	if ((x[1] == x[numpoints]) && (y[1] == y[numpoints])) {
		/* closed curve */
		PeriodicSpline(h, x, dx, d2x, d3x, numpoints);
		PeriodicSpline(h, y, dy, d2y, d3y, numpoints);
	} else {
		NaturalEndSpline(h, x, dx, d2x, d3x, numpoints);
		NaturalEndSpline(h, y, dy, d2y, d3y, numpoints);
	}

	/* generate the curve using the above information and 
	 * PointsPerInterval vectors between each specified knot.
	 */

	vp = v;
	vp->x = x[1];
	vp->y = y[1];
	vp->flags = 0;
	vp++;
	for (j=1; j<numpoints; ++j) {
		if ((x[j] == x[j+1]) && (y[j] == y[j+1])) continue;
		for (k=1; k<=PointsPerInterval; ++k) {
			t = (float) k * h[j] / (float) PointsPerInterval;
			t2 = t * t;
			t3 = t * t * t;
			nx = x[j] + (int) (t * dx[j] + t2 * d2x[j]/2 + t3 * d3x[j]/6);
			ny = y[j] + (int) (t * dy[j] + t2 * d2y[j]/2 + t3 * d3y[j]/6);
			vp->x = nx;
			vp->y = ny;
			vp->flags = 0;
			if (++vp - v == MAXPOINTS)
				vp = dumppoly(v,MAXPOINTS);
		}  /* end for k */
	}  /* end for j */
	(void) dumppoly(v,vp-v);
}  /* end GCurve */
