/* pim.c: Panel Interface Module driver */
#include "param.h"
#include "dir.h"
#include "signal.h"
#include "seg.h"
#include "ipm.h"
#include "user.h"
#include "sid.h"
#include "errno.h"
#include "conf.h"
#include "proc.h"
#include "buf.h"
#include "stream.h"

/*
 * This is two drivers in one.
 * pm.cs is the control/status driver.
 * pm.event is the interrupt driven pim events driver.
 */

#define	ioprobe(x)	fubyte(x)


struct	pim	{
	u_char	bpmr;	/* board presence mask register */
	u_char	tailr;	/* try active interlock lines register */
	u_char	limcr;	/* LED and interupt 6 mask register */
	u_char	riimr;	/* reserved inputs interrupt mask register */
	u_char	ivnr;	/* interrupt vector number register */
	u_char	ror;	/* reserved output register */
	u_char	res[2];	/* reserved */
	u_char	issr;	/* interlock signals status register */
	u_char	vsr;	/* various status register */
	u_char	bpsr;	/* board presence sense register */
	u_char	res2[5];/* reserved */
};

#define	PMA	((struct pim *) 0xFD000000)
#define	VECTOR	76	/* 130 */

static	char rdstr[30];	/* formated information from pim status */


pmread(dev)		/* read bits from interface */
	dev_t dev;
{
	register char *cp;
	register unsigned char c;
	register unsigned short w;
	register i, n, iam_b;

	if (ioprobe(PMA) == -1)
		return;
	cp = rdstr;
	*cp++ = pim_active();
	w = (PMA->vsr << 8)|PMA->bpsr;
	for (i = 1; i < 17; i++) {
		*cp++ = (w & 0x8000) ? '1' : '0';
		w <<= 1;
	}
	*cp++ = '\n';
	i++;
	n = min(u.u_count, i - u.u_offset);
	if (n <= 0)
		return;
	iomove(&rdstr[u.u_offset], n, B_READ);
}


pmwrite(dev)		/* set things based on commands */
	dev_t dev;
{
	register c;

	if (ioprobe(PMA) == -1)
		return;
	while ((c = cpass()) != -1)
		switch (c) {
		case 'B':		/* Bid on interlock */
			PMA->tailr = 7;
			break;
		case 'b':		/* release interlock */
			PMA->tailr = 0;
			break;
		case 'I':		/* set in_service led */
			PMA->limcr &= ~0040;
			break;
		case 'i':		/* reset in_service led */
			PMA->limcr |= 0040;
			break;
		case 'F':		/* set fault led */
			PMA->limcr &= ~0100;
			break;
		case 'f':		/* reset fault led */
			PMA->limcr |= 0100;
			break;
		case 'S':		/* I don't feel so good! */
			PMA->limcr &= ~0200;
			break;
		case 's':		/* I'm getting better! */
			PMA->limcr |= 0200;
			break;
		}
	c = pim_active();		/* set side affects */
}

pmclose(dev, flag)	/* check to see if we need to release interlock */
	dev_t dev;
{
	if (minor(dev) == 1)
		PMA->tailr = 0;
}
	


/*
 *	Stream based event driver.  Readonly.
 */

static	struct 	queue *pimq;		/* read end of the event stream */

int	putq(), srv(), pimqopen(), nullsys(), pimdns();

struct	qinit pimqinit[] = {
/*up*/	{ putq, srv, pimqopen, nullsys , 10, 2 },
/*dn*/	{ putq, pimdns, nullsys, nullsys, 30, 2}
};

pimqopen(dev, q)
	dev_t dev;
	struct queue *q;
{
	pimq = q;
	return;
}

pimopen(dev, flag)	/* check for pim card */
	dev_t dev;
{
	int i = 0;
	extern lbolt;

	if (ioprobe(PMA) == -1) {
		u.u_error = ENXIO;
		return;
	}
	splhigh();
	PMA->ivnr = VECTOR;
	PMA->limcr |= 007; /* tell me everything */
	spl0();
}

pimintr()		/* something changed */
{
	register struct block *bp;
	register x;

	x = PMA->issr;	/* clear interrupts */
	/* should mask things other things */
	x = pim_active();	/* read to set side affects */
	if (pimq == NULL)
		return;
	if (pimq->flag & QFULL)
		return;
	bp = allocb(1);
	*bp->wptr++ = pim_active();
	bp->class |= S_DELIM;
	putq(pimq, bp);
}

pimdns(q)		/* just ditch it */
	register struct queue *q;
{
	register struct block *bp;

	while (bp = getq(q)) {
		if (bp->type == M_IOCTL) {
			bp->type = M_IOCNAK;
			bp->rptr = bp->wptr;
			qreply(q, bp);
			continue;
		}
		freeb(bp);
	}
}


pimclose()
{
	pimq = NULL;
}

/*
 * pim_active also sets values the cm uses to filter ethernet frames.
 */

static
pim_active()	/* return who owns interlock */
{
	register c, iam_b;
	extern int activenc, complex;
	int r;

	iam_b = PMA->vsr & 4;
	complex = (iam_b) ? 2 : 1;
	c = PMA->issr;
	 r =  c & 1 ? (iam_b ? 'b' : 'a')
			: c & 0100 ? (iam_b ? 'a' : 'b')
			: '-';
	if (iam_b && r == 'b' || !iam_b && r == 'a')
		activenc = 1;
	else
		activenc = 0;
	return r;
}
