/* sid.c: Routines to do adaptive ipc */
#include "param.h"
#include "systm.h"
#include "map.h"
#include "types.h"
#include "ipm.h"
#include "sid.h"
#include "file.h"
#include "seg.h"
#include "signal.h"
#include "dir.h"
#include "proc.h"
#include "user.h"
#include "errno.h"

#define	SIDPRI		26
#define	MESGBASE	(0x03000000)
#define	MESGLIMIT	(0x03000000 + 128 * 1024)
#define	MAXMESGSIZE	(64*1024)

static	struct	map	msgmap[NMESG];	/* could be smaller */
static	struct	mesg	*mesglist;
static	errsid, scopesid, nodesid, netsid;
static	my_scope, my_node, my_net;
static	struct	mesg	*mesgalloc();
static	long		*getpte();
static	struct mesg	*ulkmsg();
static	struct sid	*lookupsid();
static	struct	sid	*mon_sp;	/* filter sid */
extern	int selwait;

/*
 * put each message header on the free list.
 */

sidinit()
{
	register struct mesg *p;
	register a;

	for (p = mesg; p < &mesg[NMSGS]; p++) {
		p->next = mesglist;
		mesglist = p;
	}
	a = malloc(coremap, NMESG);
	mfree(msgmap, NMESG, a);
}

setsid()		/* assign dflt or identifer */
{
	register struct  a {
		int	code;
		int	value;
	} *uap;

	uap = (struct a *)u.u_ap;
	switch (uap->code) {
	case 0:	my_scope = uap->value;	break;
	case 1:	my_node = uap->value;	break;
	case 2:	my_net = uap->value;	break;
	case 3:	scopesid = uap->value;	break;
	case 4:	nodesid = uap->value;	break;
	case 5:	netsid = uap->value;	break;
	case 6:	errsid = uap->value;	break;
	default: u.u_error = EINVAL;	break;
	}
}

getsid()		/* get value of globals */
{
	register struct a {
		int	code;
	} *uap;

	uap = (struct a *)u.u_ap;
	switch (uap->code) {
	case 0: u.u_rval1 = my_scope;	break;
	case 1: u.u_rval1 = my_node;	break;
	case 2: u.u_rval1 = my_net;	break;
	case 3: u.u_rval1 = scopesid;	break;
	case 4: u.u_rval1 = nodesid;	break;
	case 5:	u.u_rval1 = netsid;	break;
	case 6:	u.u_rval1 = errsid;	break;
	default: u.u_error = EINVAL;	break;
	}
}

msgbind()
{
	register struct a {
		int sid;
	} *uap;
	register struct sid *p;
	register struct file *fp;

	uap = (struct a *)u.u_ap;
	if (uap->sid == 0)
		uap->sid = free_sid();
	for (p = sid; p < &sid[NSIDS]; p++) {
		if (p->serv_id != 0) {
			if (p->serv_id == uap->sid) {
				u.u_error = EBUSY;
				return;
			}
			continue;
		}
		if ((fp = falloc()) == NULL)
			return;
		p->serv_id = uap->sid;
		p->rsel = p->wsel = NULL;
		fp->f_flag |= FSID;
		fp->f_sid = p;
		init_queue(p);
		if (uap->sid == MON_SID)
			mon_sp = p;
		return;
	}
	u.u_error = ENOMEM;
}

sidioctl(fp, cmd, cmarg)
	register struct file *fp;
	register cmd, cmarg;
{
	switch (cmd) {
	case SIDGETSID:	/* return service id for this file */
		if (suword(cmarg, fp->f_sid->serv_id) == -1)
			u.u_error = EFAULT;
		break;
	default:
		u.u_error = EINVAL;
		break;
	}
}


closesid(fp)		/* dispose of this service port */
	register struct file *fp;
{
	register struct sid *sp;
	register struct mesg *mp;
	struct mesg *mesgget();
	register wkupflag = 0;		/* so we call wakup(msgmap) once */

	if ((fp->f_flag & FSID) == 0)
		return;
	if ((unsigned)fp->f_count > 1) {
		fp->f_count--;
		return;
	}
	sp = fp->f_sid;
	if (!emptyq(sp))
		wakeup(&mesglist);
	while ((mp = mesgget(sp)) != NULL) {
		if (mp->addr) {
			mfree(msgmap, mp->size, mp->addr);
			if (wkupflag++ == 0)
				wakeup(msgmap);
		}
		mesgfree(mp);
	}
	if (sp->serv_id == MON_SID)
		mon_sp = NULL;
	sp->serv_id = 0;
	fp->f_count = 0;
}

msgclean()		/* clean up user segment on exit */
			/* called from exit/sys1.c */
{
	register struct mesg *mp, *next;
	register long *p;

	if (u.u_msglist != NULL) {
		wakeup(msgmap);
		wakeup(&mesglist);
	}
	for (mp = u.u_msglist; mp; ) {
		mfree(msgmap, mp->size, mp->addr);
		next = mp->next;
		mesgfree(mp);
		mp = next;
	}
	u.u_msglist = NULL;
	bzero((char *)u.u_pte, sizeof u.u_pte);
}

smsgalloc()		/* get some message space */
{
	register struct	a {
		int	size;		/* how big a block */
	} *uap;
	register n, a;
	register long *p;
	register struct mesg *mp;

	uap = (struct a *)u.u_ap;
	if (uap->size > MAXMESGSIZE) {
		u.u_error = EINVAL;
		return;
	}
	n = btoc(uap->size);
	if ((p = getpte(n)) == NULL) {
		u.u_error = ENOMEM;
		return;
	}
	while ((mp = mesgalloc()) == NULL)
		sleep(&mesglist, SIDPRI);
	while ((a = malloc(msgmap, n)) == 0)
		sleep(msgmap, SIDPRI);
	clearmsg(a, n);
	setpte(a, p, n);
	mp->addr = (caddr_t) a;
	mp->size = n;
	mp->next = u.u_msglist;
	u.u_msglist = mp;
	u.u_rval1 = ctob(p - u.u_pte) | MESGBASE;
}


msgfree()		/* return the space to the system */
{
	register long *p;
	register struct mesg *mp;
	register struct a {
		caddr_t	place;
	} *uap;

	uap = (struct a *) u.u_ap;
	if ((mp = ulkmsg(uap->place)) == NULL)
		return;
	mfree(msgmap, mp->size, mp->addr);
	mesgfree(mp);
	wakeup(&mesglist);
	wakeup(msgmap);
}

#define	CHECK_NET(x)	((x) && my_net && netsid && (x) != my_net)
#define	CHECK_NODE(x)	((x) && my_node && nodesid && (x) != my_node)
#define	CHECK_SCOPE(x)	((x) && my_scope && scopesid && (x) != my_scope)

msgsend()		/* ship a message to someone else */
{
	struct	a {
		struct	mesghdr	*mhp;
		char		*msgp;
	} *uap;
	register struct mesg *mp;
	register struct sid *sp;
	struct mesghdr mh;
	register sid;
	extern activenc;

	uap = (struct a *)u.u_ap;
	if (copyin(uap->mhp, (char *)&mh, sizeof mh) < 0)  {
		u.u_error = EFAULT;
		return;
	}
	if ((mp = ulkmsg(uap->msgp)) == NULL)
		return;
	bcopy((char *)&mh, (char *)&mp->header, sizeof mh);
	if (mon_sp && (mp->header.flags & MESG_RELAY) == 0) {
		mp->header.orig.net = u.u_procp->p_pid;
		mesgput(mon_sp, mp);
		wakeup(mon_sp);
		return;
	}
	if (CHECK_NET(mh.dest.net))
		sid = netsid;
	else if (CHECK_NODE(mh.dest.node))
		sid = nodesid;
	else if (CHECK_SCOPE(mh.dest.scope)) {
		if (mh.dest.scope == 1 && activenc)
			sid = mh.dest.sid;
		else if (mh.dest.scope == 2 && !activenc)
			sid = mh.dest.sid;
		else
			sid = scopesid;
	} else
		sid = mh.dest.sid;
	if ((sp = lookupsid(sid)) == NULL
	&& (errsid == 0 || (sp = lookupsid(errsid)) == NULL)) {
		u.u_error = ENXIO;
		/* since we can't send it, we must put it back */
		mp->next = u.u_msglist;
		u.u_msglist = mp;
		setpte(mp->addr, getpte(mp->size), mp->size);
		return;
	}
	mp->header.flags &= ~MESG_RELAY;
	mesgput(sp, mp);
	wakeup(sp);
}


smsgrecv()		/* get a message from my queue */
{
	register struct file *fp;
	register struct mesg *mp;
	register struct sid *sp;
	long *p;
	register struct a {
		int	fd;
		struct	mesg	*mhp;
	} *uap;

	uap = (struct a *) u.u_ap;
	if ((fp = getf(uap->fd)) == NULL)
		return;
	if ((fp->f_flag & FSID) == 0) {
		u.u_error = EBADF;
		return;
	}
	while ((mp = mesgget(fp->f_sid)) == NULL)
		sleep(fp->f_sid, SIDPRI);
	if (uap->mhp)
	if (copyout((caddr_t) &mp->header,
		(caddr_t)uap->mhp,
		sizeof (struct mesghdr)) < 0)
	{
		u.u_error = EFAULT;
		mesgputb(fp->f_sid, mp);
		return;
	}
	if ((p = getpte(mp->size)) == NULL) {
		u.u_error = ENOMEM;
		mesgputb(fp->f_sid, mp);
		return;
	}
	setpte(mp->addr, p, mp->size);
	mp->next = u.u_msglist;
	u.u_msglist = mp;
	u.u_rval1 = ctob(p - u.u_pte) | MESGBASE;
}

sidselect(fp, mode)		/* will I return from IO? */
	register struct file *fp;
	register mode;
{
	register struct sid *sp;

	if (mode == FWRITE)
		return 1;		/* you can always write */
	/* must be a read */
	sp = fp->f_sid;
	if (emptyq(sp)) {
		if (sp->rsel && sp->rsel != u.u_procp)
			sp->flag |= RSELSID;
		sp->rsel = u.u_procp;
		return 0;
	}
	return 1;
}


static
sidselwake(sp)
	register struct sid *sp;
{
	if (sp->flag & RSELSID)
		wakeup(&selwait);
	else if (sp->rsel)
		wake1up(&selwait, sp->rsel);
	sp->flag &= ~RSELSID;
	sp->rsel = NULL;
}




static
init_queue(sp)		/* reset the two queues */
	struct sid *sp;
{
	sp->norm.first = sp->norm.last =
	sp->import.first = sp->import.last = NULL;
	sp->count = 0;
}

static
emptyq(sp)		/* true if sp has no messages */
	register struct sid *sp;
{
	return (sp->norm.first == NULL && sp->import.first == NULL);
}
static struct mesg *
mesgget(sp)		/* dequeue a message from the sid */
	register struct sid *sp;
{
	register struct mesg *mp;
	register struct sidq *q;

	for (q = &sp->import; q; q = (q == &sp->import) ? &sp->norm : NULL){
		if ((mp = q->first) == NULL)
			continue;
		q->first = mp->next;
		if (q->first == NULL)		/* got last one */
			q->last = NULL;
		mp->next = NULL;
		return mp;
	}
	return NULL;
}


static
mesgput(sp, mp)	/* insert message on service queue */
	register struct sid *sp;
	register struct mesg *mp;
{
	register struct sidq *q;

	sp->count++;
	q = (mp->header.flags & MESG_IMPORTANT) ? &sp->import : &sp->norm;
	mp->next = NULL;
	if (q->last == NULL)
		q->first = mp;
	else
		q->last->next = mp;
	q->last = mp;
	if (sp->rsel)
		sidselwake(sp);
}

static
mesgputb(sp, mp)	/* put message back on queue */
	register struct sid *sp;
	register struct mesg *mp;
{
	register struct sidq *q;

	q = (mp->header.flags & MESG_IMPORTANT) ? &sp->import : &sp->norm;
	mp->next = q->first;
	q->first = mp;
	if (q->last == NULL)
		q->last = mp;
}

static
mesgfree(mp)		/* free a message */
	register struct mesg *mp;
{
	mp->next = mesglist;
	mesglist = mp;
	mp->addr = 0;
	mp->size = 0;
	bzero((char *)&mp->header, sizeof mp->header);
}

static
struct	mesg	*
mesgalloc()		/* allocate a message  */
{
	register struct mesg *mp;

	if ((mp = mesglist) == NULL)
		return NULL;
	mesglist = mp->next;
	mp->next = NULL;
	return mp;
}


static long *
getpte(an)		/* return pointer to 'n' empty pte's */
	int an;
{
	register n;
	register long *p, *q;

	for (p = u.u_pte; p < &u.u_pte[NPTE-an]; p++) {
		for (n = an, q = p; n; n--, q++)
			if (*q)
				break;
		if (n == 0)
			return p;
	}
	return NULL;
}

static
setpte(addr, p, n)		/* set user pte */
	register long *p;
	register n;
{
	register a;

	a = ctob(addr);
	while (n--) {
		*p++ = a | 1;
		a += NBPC;
	}
	asm("	.word	0xf000");	/* pflusha */
	asm("	.word	0x2400");
	asm("	movl	#0x3919,d0");		/* clear cache */
	asm("	.long	0x4e7b0002");		/* movec d0,  */
}

static
resetpte(p, n)		/* clear user pte for message space */
	register long *p;
	register n;
{
	while (n--)
		*p++ = 0;
}


static
clearmsg(a, n)		/* zero out message space */
	register a, n;
{
	while (n--)
		clearseg(a++);
}
static struct mesg *
ulkmsg(place)		/* unlink a message from the user */
	register place;
{
	register a;
	register long *p;
	register struct mesg *mp, **q;
	register times = 0;	/* no. times around a linked list */

	if (place < MESGBASE 
	||  place >= MESGBASE + ctob(NPTE)) {
		u.u_error = EINVAL;
		return NULL;
	}
	place -= MESGBASE;
	p = &u.u_pte[btoc(place)];
	a = *p & ~0377;		/* get just the physical address */
	a = btoc(a);		/* turn into click */
	q = &u.u_msglist;
	for (mp = u.u_msglist; mp; q = &mp->next, mp = mp->next) {
		if (mp->addr == (caddr_t) a)
			break;
		if (mp == u.u_msglist && ++times > 1) {
			mp = NULL;
			printf ("ulkmsg:endless loop in u.u_msglist\n");
			break;
		}
		if (mp->next == mp) {
			printf ("ulkmsg: mp 0x%x points to itself\n", mp);
			mp = NULL;
			break;
		}
	}
	if (mp == NULL) {
		u.u_error = EINVAL;
		return NULL;
	}
	resetpte(p, mp->size);
	*q = mp->next;		/* unlink from upage */
	mp->next = NULL;
	return mp;
}

static struct sid *
lookupsid(serv_id)		/* search for sid `sid' */
	register serv_id;
{
	register struct sid *sp;

	if (serv_id == 0)
		return NULL;
	for (sp = sid; sp < &sid[NSIDS]; sp++)
		if (sp->serv_id == serv_id)
			return sp;
	return NULL;
}

static
free_sid()		/* find a free sid to use */
{
	static this_sid = 10000;
	register struct sid *sp;

again:
	for (sp = sid; sp < &sid[NSIDS]; sp++)
		if (sp->serv_id == this_sid) {
			if (++this_sid > 32000)
				this_sid = 10000;
			goto again;
		}
	this_sid++;
	return this_sid - 1;
}

