/* RWDOCS */
/* Upper half of IP, consisting of send/receive primitives, including
 * fragment reassembly, for higher level protocols.
 * Not needed when running as a standalone gateway.
 */
#define	TLB	30 * (1000/MSPTICK)	/* Reassembly limit time */
#include <stdio.h>
#include "global.h"
#include "mbuf.h"
#include "timer.h"
#include "internet.h"
#include "iface.h"
#include "ip.h"
#include "icmp.h"
#include "tcp.h"
#include "udp.h"

int32 ip_ttl = MAXTTL;	/* Default time-to-live for IP datagrams */
struct reasm *reasmq;
void free();

#define	INSERT	0
#define	APPEND	1
#define	PREPEND	2

/* Send an IP datagram to the router.
 */
int
ip_send(Source,Destination,Protocol,TypeOfService,TimeToLive,bp,length,id,DontFrag)
int32 Source;			/* Source address */
int32 Destination;		/* Destination address */
int32 Protocol;			/* Protocol */
int32 TypeOfService;		/* Type of service */
int32 TimeToLive;		/* Time to live */
struct mbuf *bp;		/* Data */
int32 length;			/* data length  */
int32 id;			/* identification */
int32 DontFrag;			/* Don't fragment flag */
{

/* RWDOCE */
	struct mbuf *tbp;
	struct ip ip;		/* Pointer to IP header */
	static int32 id_cntr;	/* Datagram serial number */

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif

		/* if length isn't specifed, get mbuf length */
	if(length == 0 && bp != NULLBUF)
		length = len_mbuf(bp);

		/* if sequence number isn't specifed, get one */
	if(id == 0)
		id = id_cntr++;		

		/* if time to live isn't specified, add one */
	if(TimeToLive == 0)
		TimeToLive = ip_ttl;

	/* Fill in IP header */
	ip.tos = TypeOfService;
	ip.length = IPLEN + length;
	ip.id = id;
	if(DontFrag)
		ip.fl_offs = DF;
	else
		ip.fl_offs = 0;
	ip.ttl = TimeToLive;
	ip.protocol = Protocol;
	ip.source = Source;
	ip.dest = Destination;
	ip.optlen = 0;

		/* router expects network byte order */
	if((tbp = htonip(&ip,bp)) == NULLBUF){
		free_p(bp);
		return -1;
	}
	return ip_route(tbp,0);		/* Toss it to the router */
}


/* RWDOCS */
/* Reassemble incoming IP fragments and dispatch completed datagrams
 * to the proper transport module
 */
void
ip_recv(ip,bp,rxbroadcast)
struct ip *ip;		/* IP header */
struct mbuf *bp;	/* Data */
int rxbroadcast;	/* True if broadcast */
{

/* RWDOCE */
	void (*recv)();	/* Function to call with completed datagram */

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	/* Initial check for protocols we can't handle */
	switch((ip->protocol & 0xff)) {
	case TCP_PTCL:
		recv = tcp_input;
		break;
	case UDP_PTCL:
		recv = udp_input;
		break;
	case ICMP_PTCL:
		recv = icmp_input;
		break;
	default:
		/* Send an ICMP Protocol Unknown response... */
		ip_stats.badproto++;
		/* ...unless it's a broadcast */
		if(!rxbroadcast){
			icmp_output(ip,bp,DEST_UNREACH,PROT_UNREACH,(union icmp_args *)0);
		}
		free_p(bp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return;
	}

	/* If we have a complete packet, call the next layer
	 * to handle the result. Note that fraghandle passes back
	 * a length field that does NOT include the IP header
	 */
	if((bp = fraghandle(ip,bp)) != NULLBUF)
		(*recv)(bp,ip->protocol,ip->source,ip->dest,ip->tos,
			ip->length - (IPLEN + ip->optlen),rxbroadcast);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
}

/* RWDOCS */
/* Process IP datagram fragments
 * If datagram is complete, return it with ip->length containing the data
 * length (MINUS header); otherwise return NULLBUF
 */
struct mbuf *
fraghandle(ip,bp)
struct ip *ip;		/* IP header, host byte order */
struct mbuf *bp;	/* The fragment itself */
{

/* RWDOCE */

	register struct reasm *rp; /* Pointer to reassembly descriptor */
	struct frag *lastfrag,*nextfrag,*tfp;
	struct mbuf *tbp;
	int32 i;
	int32 offset;		/* Index of first byte in fragment */
	int32 last;		/* Index of first byte beyond fragment */
	int32 mf;		/* 1 if there are more frags */

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	offset = (ip->fl_offs & F_OFFSET) << 3;	/* Convert to bytes */
	last = offset + ip->length - (IPLEN + ip->optlen);
	mf = (ip->fl_offs & MF) ? 1 : 0;

	rp = lookup_reasm(ip);
	if(offset == 0 && !mf) {

		/* Complete datagram received. free up resources
		 * and return pointer to completed datagram
		 */
		if(rp != NULLREASM)
			free_reasm(rp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return bp;
	}

	if(rp == NULLREASM){
		/* First fragment; create new reassembly descriptor */
		if((rp = creat_reasm(ip)) == NULLREASM){
			/* No space for descriptor, drop fragment */
			free_p(bp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
			return NULLBUF;
		}
	}

	/* Keep restarting timer as long as we keep getting fragments */
	stop_timer(&rp->timer);
	start_timer(&rp->timer);

	/* If this is the last fragment, we now know how long the
	 * entire datagram is; record it
	 */
	if(!mf)
		rp->length = last;

	/* Set nextfrag to the first fragment which begins after us,
	 * and lastfrag to the last fragment which begins before us
	 */
	lastfrag = NULLFRAG;
	for(nextfrag = rp->fraglist;nextfrag != NULLFRAG;nextfrag = nextfrag->next){
		if(nextfrag->offset > offset)
			break;
		lastfrag = nextfrag;
	}

	/* Check for overlap with preceeding fragment */
	if(lastfrag != NULLFRAG  && offset < lastfrag->last){
		/* Strip overlap from new fragment */
		i = lastfrag->last - offset;
		pullup(&bp,NULLCHAR,i);
		if(bp == NULLBUF)
			return NULLBUF;	/* Nothing left */
		offset += i;
	}

	/* Look for overlap with succeeding segments */
	for(; nextfrag != NULLFRAG; nextfrag = tfp){
		tfp = nextfrag->next;	/* save in case we delete fp */

		if(nextfrag->offset >= last)
			break;	/* Past our end */
		/* Trim the front of this entry; if nothing is
		 * left, remove it.
		 */
		i = last - nextfrag->offset;
		pullup(&nextfrag->buf,NULLCHAR,i);
		if(nextfrag->buf == NULLBUF){
			/* superseded; delete from list */
			if(nextfrag->prev != NULLFRAG)
				nextfrag->prev->next = nextfrag->next;
			else
				rp->fraglist = nextfrag->next;
			if(tfp->next != NULLFRAG)
				nextfrag->next->prev = nextfrag->prev;
			freefrag(nextfrag);
		} else
			nextfrag->offset = last;
	}

	/* Lastfrag now points, as before, to the fragment before us;
	 * nextfrag points at the next fragment. Check to see if we can
	 * join to either or both fragments.
	 */
	i = INSERT;
	if(lastfrag != NULLFRAG && lastfrag->last == offset)
		i |= APPEND;
	if(nextfrag != NULLFRAG && nextfrag->offset == last)
		i |= PREPEND;

	switch(i){
	case INSERT:	/* Insert new desc between lastfrag and nextfrag */
		tfp = newfrag(offset,last,bp);
		tfp->prev = lastfrag;
		tfp->next = nextfrag;
		if(lastfrag != NULLFRAG)
			lastfrag->next = tfp;	/* Middle of list */
		else
			rp->fraglist = tfp;	/* First on list */
		if(nextfrag != NULLFRAG)
			nextfrag->prev = tfp;
		break;
	case APPEND:	/* Append to lastfrag */
		append(&lastfrag->buf,bp);
		lastfrag->last = last;	/* Extend forward */
		break;
	case PREPEND:	/* Prepend to nextfrag */
		tbp = nextfrag->buf;
		nextfrag->buf = bp;
		append(&nextfrag->buf,tbp);
		nextfrag->offset = offset;	/* Extend backward */
		break;
	case (APPEND|PREPEND):
		/* Consolidate by appending this fragment and nextfrag
		 * to lastfrag and removing the nextfrag descriptor
		 */
		append(&lastfrag->buf,bp);
		append(&lastfrag->buf,nextfrag->buf);
		nextfrag->buf = NULLBUF;
		lastfrag->last = nextfrag->last;

		/* Finally unlink and delete the now unneeded nextfrag */
		lastfrag->next = nextfrag->next;
		if(nextfrag->next != NULLFRAG)
			nextfrag->next->prev = lastfrag;
		freefrag(nextfrag);
		break;
	}

	if(rp->fraglist->offset == 0 && rp->fraglist->next == NULLFRAG
		&& rp->length != 0) {
		/* We've gotten a complete datagram, so extract it from the
		 * reassembly buffer and pass it on.
		 */
		bp = rp->fraglist->buf;
		rp->fraglist->buf = NULLBUF;
		/* Tell IP the entire length */
		ip->length = rp->length + (IPLEN + ip->optlen);
		free_reasm(rp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return bp;
	} else
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return NULLBUF;
}

/* RWDOCS */
struct reasm *
lookup_reasm(ip)
struct ip *ip;
{

/* RWDOCE */
	register struct reasm *rp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	for(rp = reasmq;rp != NULLREASM;rp = rp->next){
		if(ip->source == rp->source && ip->dest == rp->dest
		 && ip->protocol == rp->protocol && ip->id == rp->id)
			return rp;
	}
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return NULLREASM;
}


/* RWDOCS */
/* Create a reassembly descriptor,
 * put at head of reassembly list
 */
struct reasm *
creat_reasm(ip)
register struct ip *ip;
{

/* RWDOCE */
	register struct reasm *rp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if((rp = (struct reasm *)calloc(1,sizeof(struct reasm))) == NULLREASM){
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return rp;	/* No space for descriptor */
		}
	rp->source = ip->source;
	rp->dest = ip->dest;
	rp->id = ip->id;
	rp->protocol = ip->protocol;
	rp->timer.start = TLB;
	rp->timer.func = ip_timeout;
	rp->timer.arg = (char *)rp;

	rp->next = reasmq;
	if(rp->next != NULLREASM)
		rp->next->prev = rp;
	reasmq = rp;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return rp;
}

/* RWDOCS */
/* Free all resources associated with a reassembly descriptor */
void
free_reasm(rp)
register struct reasm *rp;
{

/* RWDOCE */
	register struct frag *fp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	stop_timer(&rp->timer);

	/* Remove from list of reassembly descriptors */
	if(rp->prev != NULLREASM)
		rp->prev->next = rp->next;
	else
		reasmq = rp->next;

	if(rp->next != NULLREASM)
		rp->next->prev = rp->prev;

	/* Free any fragments on list, starting at beginning */
	while((fp = rp->fraglist) != NULLFRAG) {
		rp->fraglist = fp->next;
		free_p(fp->buf);
		free((char *)fp);
	}

	free((char *)rp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
}

/* RWDOCS */
/* Handle reassembly timeouts by deleting all reassembly resources */
void
ip_timeout(arg)
int *arg;
{

/* RWDOCE */
	register struct reasm *rp;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	rp = (struct reasm *)arg;
	free_reasm(rp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
}

/* RWDOCS */
/* Create a fragment */
struct frag *
newfrag(offset,last,bp)
int offset,last;
struct mbuf *bp;
{

/* RWDOCE */
	struct frag *fp;

#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	if((fp = (struct frag *)calloc(1,sizeof(struct frag))) == NULLFRAG){
		/* Drop fragment */
		free_p(bp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
		return NULLFRAG;
	}
	fp->buf = bp;
	fp->offset = offset;
	fp->last = last;
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
	return fp;
}

/* RWDOCS */
/* Delete a fragment, return next one on queue */
void
freefrag(fp)
struct frag *fp;
{

/* RWDOCE */
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif

	free_p(fp->buf);
	free((char *)fp);
#ifdef COHPROF
profile(__LINE__,__FILE__);
#endif
}

/* end of ip.c */
