#!/bin/sh
# Install.sh: installation script for "cohulip", version 1.0
# By fwb of Mark Williams Company, 09/13/94
# Based on the program "inst.c" by Randy Wright

. /usr/lib/shell_lib.sh

########################################
#
# Some local functions used throughout this script:
#
########################################

# ask the user a question; get Y/N reply
# $1 is prompt; $2 is default reply (y or n)
get_reply () {
	if [ $# -eq 2 ]; then
		read_input "$1" reply $2 require_yes_or_no
	else
		read_input "$1" reply "y" require_yes_or_no
	fi

	case $reply in
		y|Y|yes|Yes|YES)	reply=y ;;
		n|N|no|No|NO)		reply=n ;;
	esac
}

# wait for user to press key
wait_a_bit () {
	/bin/echo "\nPress any key to continue ... \c" ; read JUNK
	clear
}

########################################
#
# and awaaaaaaaaay we go ... !
#
########################################

/bin/echo "
                         WELCOME TO COHULIP

This script installs the COHERENT user-level IP package (cohulip) onto your
system.

We strongly suggest that you read the document ./doc/install.p *before* you
attempt to install cohulip onto your system.  If you wish to stop installation
and review this chapter, answer 'n' to the following question.
"

get_reply "Do you wish to continue installation?" y
if [ $reply = n ] ; then
	/bin/echo "
\nInstallation of cohulip has been aborted.  To try again, log in as the
superuser and execute script Install.sh.
"
	exit 1
fi

echo "
\nWe assume that you have read the chapter on cohulip installation, and are
continuing installation.  Please note that you can interrupt this program
at any time without damaging your system in any way.  To interrupt
installation, just type <ctrl-C>.  You may have to type it more than once.
To resume installation, su to root and execute script Install.sh again.
"

wait_a_bit

#
# create home directory for cohulip
#
/bin/echo "
The first step in installation is to select a home directory for cohulip.
By default, cohulip is installed into directory $NETHOME.  To accept this
default, simply press <return>.
"

NETHOME=/usr/net
while : ; do
	read_input "Home directory for cohulip:" NETHOME $NETHOME

	get_reply "Install into directory $NETHOME?" y
	if [ $reply = "n" ]; then
		continue;
	fi

	if [ -f $NETHOME ]; then
		/bin/echo "Cannot use $NETHOME:  it is an ordinary file."
		continue
	fi

	if [ ! -e $NETHOME ]; then
		mkdir -p $NETHOME
	fi

	break
done

STARTUP=$NETHOME/startup.new

#
# de-archive the help, configuration, and source files
#
/bin/echo "Copying some start-up files and help files ..."
cp *.tar $NETHOME
HERE=`pwd`
cd $NETHOME
/usr/bin/gtar --same-owner -xpf help.tar
/usr/bin/gtar --same-owner -xpf net.tar
rm *.tar
mkdir finger
chmog 666 bin bin finger
cd $HERE

#
# copy the executable into /usr/bin
#
/bin/echo "Copying the executable 'net' into /usr/bin ..."
/bin/cp bin/net /usr/bin
/bin/chmog 511 bin bin /usr/bin/net

wait_a_bit

#
# set the name of the local host
#
/bin/echo "
The next step in installation is set your system's name.  Normally, this is
the name set in the file /etc/uuname, although you can use another name if
you wish.  Again, to accept the default, simply press <return>.
"

SYSNAME=`/bin/uname -s`
while : ; do
	get_reply "Set the system name to $SYSNAME?" y
	if [ $reply = "y" ]; then
		break
	fi

	read_input "Enter the name of your system:" SYSNAME $SYSNAME
done

/bin/echo "hostname $SYSNAME" > $STARTUP

wait_a_bit

#
# set the local host's IP address
#
echo "
The next step is to enter your system's IP address.  As the manual states,
some users are assigned an IP address when they log into their Internet
provider's machine, and so may not know it at this time.
"

get_reply "Enter your IP address now" y
if [ $reply = "y" ]; then
	while : ; do
		read_input "Enter your IP address (e.g., 123.456.789.012):" IP_ADDR ""

		is_empty "$IP_ADDR"
		if [ $? -eq 0 ]; then
			continue
		fi

		get_reply "Is your IP address $IP_ADDR?" y
		if [ $reply = "y" ]; then
			break;
		fi
	done
	echo "ip address $IP_ADDR" >> $STARTUP
fi

wait_a_bit

#
# handle SLIP attachment, if the user wants one
#
get_reply "Use a serial port or modem for SLIP?" y
if [ $reply = "n" ]; then
	HAVE_SLIP=0
else
	# get the port's name
	while : ; do
		read_input "Enter the serial port (e.g., com3fl):" SLIP_PORT ""

		is_empty "$SLIP_PORT"
		if [ $? -eq 0 ]; then
			continue
		fi

		get_reply "Serial port is $SLIP_PORT?" y
		if [ $reply = "y" ]; then
			if [ -c /dev/$SLIP_PORT ]; then
				break
			else
				/bin/echo "/dev/$SLIP_PORT is not a character-special file.  Try again"
			fi
		fi
	done

	# get the port's speed
	while : ; do
		read_input "Enter the port's speed (e.g., 19200):" SLIP_SPEED ""

		is_empty "$SLIP_SPEED"
		if [ $? -eq 0 ]; then
			continue
		fi

		get_reply "The serial port's speed is $SLIP_SPEED?" y
		if [ $reply = "y" ]; then
			break
		fi
	done

	/bin/echo "attach asy 0 $SLIP_PORT slip sl0 0 1500 $SLIP_SPEED" >> $STARTUP
	/bin/echo "Your SLIP interface is named sl0"
	HAVE_SLIP=1
fi

wait_a_bit

#
# handle Ethernet attachment, if the user wants one
#
get_reply "Use an NE1000 or NE2000 ethernet card?" y
if [ $reply = "n" ]; then
	HAVE_ETHERNET=0
else
	# get card's name
	while : ; do
		read_input "Enter the Ethernet card name (e.g., ne0):" ETHER_CARD ""

		is_empty "$ETHER_CARD"
		if [ $? -eq 0 ]; then
			continue
		fi

		get_reply "Your Ethernet card is named $ETHER_CARD?" y
		if [ $reply = "y" ]; then
			break
		fi
	done

	/bin/echo "attach nx 0 0 arpa $ETHER_CARD 5 1500" >> $STARTUP
	/bin/echo "Your Ethernet interface is named 'ne0'"
	HAVE_ETHERNET=1
fi

wait_a_bit

#
# set the default route, if necessary
#
if [ $HAVE_SLIP -eq 1 -a $HAVE_ETHERNET -eq 1 ]; then
	# user has both SLIP and Ethernet
	/bin/echo "
Because you wish to use cohulip with both an Ethernet network and a SLIP
network, you must select a default route.  This is the interface over which
packets will be sent when a route cannot be located locally.  This is normally
the interface that connects to the nearest gateway or router.
"

	while : ; do
		read_input "Enter 'sl0' for SLIP or 'ne0' for Ethernet:" ROUTE ""
		case $ROUTE in
			"sl0")	;;
			"ne0")	;;
			*)	/bin/echo "Please enter sl0 or ne0"
				continue;;
		esac

		if [ $ROUTE = "sl0" ]; then
			get_reply "Use SLIP (sl0) as default route?" y
		else
			get_reply "Use Ethernet (ne0) as default route?" y
		fi

		if [ $reply = "y" ]; then
			break
		fi

	done
	
	/bin/echo "route add default $ROUTE" >> $STARTUP
	/bin/echo "Your default route is $ROUTE"

	wait_a_bit
else
	# user has one or the other (or neither), but not both
	if [ $HAVE_SLIP -eq 1 ]; then
		/bin/echo "route add default sl0" >> $STARTUP
	fi

	if [ $HAVE_ETHERNET -eq 1 ]; then
		/bin/echo "route add default ne0" >> $STARTUP
	fi
fi

#
# see about the domain-name server
#
/bin/echo "
As noted in the installation manual, you can enter the IP address or name
of the domain-name server (DNS) that is available to your system.  Note that
you should NOT enter this information if (1) you do not have it, or
(2) your system does not have a permanently assigned IP address.

"
get_reply "Do you wish to enter the name or address of your domain-name server?" y
if [ $reply = "y" ]; then
	while : ; do
		/bin/echo "
Please enter the correct name or dotted-decimal address of your DNS:
e.g., '192.48.234.1' or 'myserver.com'.
"
		read_input "Enter name or address of your DNS:" DNS_ID ""

		is_empty "$DNS_ID"
		if [ $? -eq 0 ]; then
			continue
		fi

		get_reply "Your DNS has name or address $DNS_ID?" y
		if [ $reply = "y" ]; then
			break
		fi
	done

	/bin/echo "domain addserver $DNS_ID" >> $STARTUP
	/bin/echo "DNS $DNS_ID has been added to $STARTUP"
fi

wait_a_bit

#
# output some boilerplate
#
/bin/echo "ip ttl 32" >> $STARTUP
/bin/echo "tcp mss 1500" >> $STARTUP
/bin/echo "tcp window 1500" >> $STARTUP
/bin/echo "tcp irtt 2000" >> $STARTUP
/bin/echo "eol unix" >> $STARTUP

#
# Add the servers that the user requests
#
/bin/echo "
The next step will be to ask you about the servers that you wish to turn on.
Note that if you turn on a server, cohulip provides that service to people who
are accessing into your system.  If you do not turn on a given server, you will
still be able to use that service on other machines.

"

# the telnet/telunix server requires special handling
/bin/echo ""
get_reply "Turn on the server for telnet?" y
if [ $reply = "y" ]; then
	/bin/echo "start telunix 23" >> $STARTUP
	/bin/echo "Turning on server telnet"
fi

# all the other servers can be handled in the same way
for SERVER in echo ping finger ftp
do
	/bin/echo ""
	get_reply "Turn on the server for $SERVER?" y
	if [ $reply = "y" ]; then
		/bin/echo "start $SERVER" >> $STARTUP
		/bin/echo "Turning on server $SERVER"
	fi
done

wait_a_bit

#
# update /etc/profile
#
/bin/echo "export NETHOME=$NETHOME" >> /etc/profile
/bin/echo "export NETSPOOL=/usr/spool/uucppublic" >> /etc/profile

#
# some chatter, and wrap up
#
/bin/echo "
We have created a file called $STARTUP based upon
the information you have enter.  For the application 'net' to use this
information, you must invoke it with the following syntax:

	net startup.new

We have also set the environmental variables NETHOME and NETSPOOL within the
system-wide profile file '/etc/profile'.  However, this will not take effect
until you next log in.  The network software needs these variables to locate
its files.
"

wait_a_bit

/bin/echo "
The file $NETHOME/hosts.net allows you assign aliases to IP addresses.
Some examples are included in $NETHOME/hosts.net already.  The file
$NETHOME/domain.txt is maintained by the domain-name client application
if you have specified a name server.

File $NETHOME/ftpusers controls access to ftp services.  This file contains
examples.

If you find that $NETHOME/startup.new is suitable for your system as
we have created it, you can change its name to startup.net; it will then
be automatically executed each time you start net.

This concludes the installation of cohulip.
"

#
# end of Install.sh
