#
#include "../370.h"
#include "../param.h"
#include "../buf.h"
#include "../conf.h"
#include "../manifest.h"
#include "../proc.h"
#include "../systm.h"
#include "../tty.h"
#include "../user.h"

/* States that the terminal driver is in when expecting an interrupt */
#define ENABLING	1
#define WAITING		2
#define READING		3
#define HALTING		4
#define WRITING		5
#define SENSING		6
#define PREPARING	7
#define DISABLING	8

struct tty trms[NTRMS];
extern char revbit[];

/* Inititialize terminals */
trminit()
{
	struct tty *tp;
	for(tp = trms;tp < &trms[NTRMS]; tp++){
		tp->t_state = 0;
		if(tp->t_devaddr) hdv(tp->t_devaddr);
	}
}

/*
 * Upper level routines: trmopen, trmclose, trmread, trmwrite,
 * trmsgtty, and trmstart.
 */


/*
 * Open a terminal.  If already open, do nothing, otherwise set
 * default values and issue enable for line.
 */
trmopen(dev,flag)
{
	struct tty *tp;
	int sps,trmstart();
	if(dev.d_minor >= NTRMS){
		u->u_error = ENXIO;
		return;
	}
	tp = &trms[dev.d_minor];
	if(tp->t_state) return;		/* already open */
	tp->t_dev = dev;
	tp->t_devaddr = cdevsw[dev.d_major].d_baseaddr + dev.d_minor;
	tp->t_addr = &trmstart;
	tp->t_flags = XTABS|CRMOD;
	tp->t_erase = CERASE;
	tp->t_kill = CKILL;
	/* Get input and output buffers */
	tp->t_ibp = getblk(NODEV);
	tp->t_obp = getblk(NODEV);
	/* Enable line */
	sps = stnsm(~IOINT);
enb:	switch(trmenbl(tp->t_devaddr)){
	case 0:	/* okay */
		break;
	case 1:	/* CSW stored */
		if(CSW->unitstat == STATMOD+CUEND+BUSY)goto enb; /* busy */
		break;	/* shouldn't happen */
	case 2:	/* channel busy */
		goto enb;
	case 3:	/* not operational */
		setsm(sps);
		u->u_error = EIO;
		brelse(tp->t_ibp);
		brelse(tp->t_obp);
		return;
	}
	tp->t_state = ENABLING;
	setsm(sps);
	if(u->u_procp->p_ttyp == 0) u->u_procp->p_ttyp = tp;
}

/*
 * Close a terminal. Flush queues, halt activity, and disable line.
 */
trmclose(dev)
{
	struct tty *tp;
	int sps;
	tp = &trms[dev.d_minor];
	wflushtty(tp);	/* flush */
	sps = stnsm(~IOINT);
	if(tp->t_state != WRITING){
		tp->t_state = DISABLING;
		/* There should now be a read active, which we halt before disabling */
hlt:		switch(hdv(tp->t_devaddr)){
		case 0:	/* interrupt outstanding */
			stosm(IOINT);	/* let 'er rip */
			stnsm(~IOINT);
			goto hlt;
		case 1:	/* CSW stored */
			break;	/* as expected */
		case 2:	/* channel busy */
			goto hlt;
		case 3:	/* not operational */
			trmerr(tp,"halting for disable");
		}
	} else tp->t_state = DISABLING;
	/*
	 * During sleep, we get an interrupt indicating 
	 * I/O was halted, or write completed. We are
	 * still disabled when we return.
	 */
	while(tp->t_state) sleep(&tp->t_state, TTOPRI);
	tp->t_state = DISABLING;
dsbl:	switch(trmdsbl(tp->t_devaddr)){
	case 0:	/* okay */
		break;
	case 1:	/* CSW stored */
		if(CSW->unitstat == STATMOD+CUEND+BUSY) goto dsbl; /* busy */
		break;	/* shouldn't happen */
	case 2:	/* channel busy */
		goto dsbl;
	case 3:	/* not operational */
		trmerr(tp,"disabling");
	}
	setsm(sps);
	brelse(tp->t_ibp);
	brelse(tp->t_obp);
}

trmread(dev)
{
	ttread(&trms[dev.d_minor]);
}

trmwrite(dev)
{
	ttwrite(&trms[dev.d_minor]);
}

trmsgtty(dev,v)
{
	ttystty(&trms[dev.d_minor],v);
}

/*
 * Start output to a terminal.
 * If we waiting for something to be typed,
 * halt read and start writing.
 */
trmstart(tp)
struct tty *tp;
{
	int sps;
	sps = stnsm(~IOINT);
	if(tp->t_state == WAITING){
hrd:		switch(hdv(tp->t_devaddr)){
		case 0:	/* interrupt outstanding */
			stosm(IOINT);	/* let 'er rip */
			stnsm(~IOINT);
			if(tp->t_state == WAITING) goto hrd;
			break;
		case 1:	/* CSW stored */
			break;	/* as expected */
		case 2:	/* channel busy */
			goto hrd;
		case 3:	/* not operational */
			trmdead(tp);
		}
		tp->t_state = HALTING;
	}
	setsm(sps);
}


/*
 * Interrupt level routines: trmintr, trmnorm, trmrd, and trmsens.
 */


/*
 * Terminal interrupt routine. Entire routine is run disabled.
 */
trmintr(dev)
{
	struct tty *tp;
	int sps,len;
	char *cp;
	tp = &trms[dev.d_minor];

	/* Huge switch by current state */
	switch(tp->t_state){

	case 0:	/* nothing happening */
		return;

	case ENABLING:
		/* someone has dialed up */
		if(CSW->unitstat == CHEND+DEVEND) trmnorm(tp);
		else trmdead(tp);	/* shouldn't happen */
		return;

	case WAITING:
	case READING:
		/* terminal input */
		if(CSW->unitstat == 0 && CSW->chanstat&PCI){
			/* typing begun, but not finished */
			tp->t_state = READING;
			return;
		}
		len = tp->t_il - ((CSW->resid1<<8) | CSW->resid2);
		cp = tp->t_ibp->b_addr;
		while(len--) ttyinput(revbit[*cp++],tp); /* process chars */
		if(CSW->unitstat == CHEND+DEVEND){
			/* normally terminated read */
			trmnorm(tp);
			return;
		}
		if(CSW->unitstat == CHEND+DEVEND+UNITCHEK){
			/* read terminated by break */
			trmsens(tp);
			return;
		}
		trmerr(tp,"reading");
		trmnorm(tp);
		return;

	case HALTING:
		/* read was halted to begin write */
		trmnorm(tp);
		return;

	case WRITING:
		/* data written */
		tp->t_wresid = (CSW->resid1 << 8) | CSW->resid2;
		if(CSW->unitstat == CHEND+DEVEND){
			/* normal write */
			trmnorm(tp);
			return;
		}
		if(CSW->unitstat == CHEND+DEVEND+UNITCHEK){
			/* write interrupted by break */
			trmsens(tp);
			return;
		}
		trmerr(tp,"writing");
		trmnorm(tp);
		return;

	case SENSING:
		/* getting sense info */
		if(tp->t_sense & INTREQ){
			/*
			 * Either break was hit, or the line went dead.
			 * In order to tell the difference, we issue a 
			 * prepare command and attempt to halt it. If the
			 * halt works, the line is allright, if not, we'll
			 * get a later interrupt telling us the line is okay,
			 * or the line timed out.
			 */
			tp->t_state = PREPARING;
	prep:		switch(trmprep(tp->t_devaddr)){
			case 0:	/* okay */
				break;
			case 1:	/* CSW stored */
				if(CSW->unitstat==STATMOD+CUEND+BUSY)goto prep;
				break;	/* shouldn't happen */
			case 2:	/* channel busy */
				goto prep;
			case 3:	/* not operational */
				trmdead(tp);
				return;
			}
			/* now attempt halt */
	halt:		switch(hdv(tp->t_devaddr)){
			case 0:	/* interrupt outstanding */
				return;
			case 1:	/* CSW stored */
				break;	/* as expected */
			case 2:	/* channel busy */
				goto halt;
			case 3:	/* not operational */
				trmdead(tp);
				return;
			}
			return;
		}
		if(tp->t_sense & TIMEOUT){
			/*
			 * Line has been dead for 28 seconds, so assume hangup.
			 */
			trmhup(tp);
			return;
		}
		printf("dev=%x,sense=%x\n",tp->t_devaddr,tp->t_sense);
		trmnorm(tp);
		return;

	case PREPARING:
		/* Waiting for line to come to life. */
		switch(CSW->unitstat){
		case CHEND+DEVEND:	
		case CHEND+DEVEND+UNITEXCP:
			trmrd(tp);
			/*
			 * Make state READING rather than WAITING, so
			 * that read will not be halted in trmstart.
			 */
			tp->t_state = READING;
			return;
		case CHEND+DEVEND+UNITCHEK:
			trmsens(tp);
			return;
		default:
			trmerr(tp,"preparing");
			trmnorm(tp);
			return;
		}

	case DISABLING:
		tp->t_state = 0;
		wakeup(&tp->t_state);
		return;

	}	/* end of state switch */
}

/*
 * Perform normal action for the terminal.
 * If we need to restart an interrupted write, do so,
 * if there is anything to write, write it,
 * else read.
 */
trmnorm(tp)
struct tty *tp;
{
	int i,len;
	char *cp;
	if(tp->t_wresid > 0) {	/* restart write */
wr:		switch(wrterm(tp->t_devaddr,tp->t_obp->b_addr+tp->t_wresid,
			      tp->t_wresid)){
		case 0:	/* okay */
			break;
		case 1:	/* CSW stored */
			if(CSW->unitstat == STATMOD+CUEND+BUSY) goto wr;
			break;	/* shouldn't happen */
		case 2:	/* channel busy */
			goto wr;
		case 3:	/* not operational */
			trmdead(tp);
			return;
		}
		tp->t_state = WRITING;
		return;
	}
	if(tp->t_outq.c_cc == 0){	/* read */
		trmrd(tp);
		return;
	}
	/* write */
	cp = tp->t_obp->b_addr;
	len = min(BLKSIZE, tp->t_outq.c_cc);
	for(i=0;i<len;i++)
		*cp++ = revbit[getc(&tp->t_outq)];
	tp->t_state = WRITING;
w:	switch(wrterm(tp->t_devaddr,tp->t_obp->b_addr,len)){
	case 0:	/* okay */
		break;
	case 1:	/* CSW stored */
		if(CSW->unitstat == STATMOD+CUEND+BUSY) goto w;
		break;	/* shouldn't happen */
	case 2:	/* channel busy */
		goto w;
	case 3:	/* not operational */
		trmdead(tp);
		return;
	}
	wakeup(&tp->t_outq);
}

/*
 * Read!
 */
trmrd(tp)
struct tty *tp;
{
	tp->t_state = WAITING;
	tp->t_il = BLKSIZE;
r:	switch(rdterm(tp->t_devaddr,tp->t_ibp->b_addr,tp->t_il)){
	case 0:	/* okay */
		break;
	case 1:	/* CSW stored */
		if(CSW->unitstat == STATMOD+CUEND+BUSY) goto r;
		break;	/* shouldn't happen */
	case 2:	/* channel busy */
		goto r;
	case 3:	/* not operational */
		trmdead(tp);
		return;
	}
}

/*
 * Get the sense information for a terminal.
 */
trmsens(tp)
struct tty *tp;
{
	tp->t_state = SENSING;
s:	switch(trmsns(tp->t_devaddr, &tp->t_sense)){
	case 0:	/* okay */
		break;
	case 1:	/* CSW stored */
		if(CSW->unitstat == STATMOD+CUEND+BUSY) goto s;
		break;
	case 2:	/* channel busy */
		goto s;
	case 3:	/* not operational */
		trmdead(tp);
		return;
	}
}

/*
 * Error routines: trmdead, trmhup, and trmerr.
 */

trmdead(tp)
struct tty *tp;
{
	printf("line %x not operational\n",tp->t_devaddr);
	trmhup(tp);
}

trmhup(tp)
struct tty *tp;
{
	tp->t_state = 0;
	signal(tp, SIGHUP);
}

trmerr(tp,s)
struct tty *tp;
char *s;
{
	printf("term err: dev=%x,csw=%x,sense=%x,id=%s\n",
		tp->t_devaddr,(CSW+4)->integ,tp->t_sense,s);
}
