#
/*
 *	Copyright 1973 Bell Telephone Laboratories Inc
 */

/*
 *	DH-11 driver
 *	This driver calls on the DHDM driver.
 *	If the DH has no DM11-BB, then the latter will
 *	be fake. To insure loading of the correct DM code,
 *	lib2 should have dhdm.o, dh.o and dhfdm.o in that order.
 *
 *			BSB 10/21/76 for Harvard.
 */

#include "../param.h"
#include "../conf.h"
#include "../user.h"
#include "../tty.h"
#include "../proc.h"

#define	LNKLIN	31	/* line # of aiken link */
/*  *********************************************************************/

#define NMUX ((NDH11+15)/16)
#define	DHNCH	24		/* max number of DMA chars */

/*
 * Place from which to do DMA on output
 */
char	dh_clist[NDH11][DHNCH];

/*
 * External page addresses for the DHs on the system
 */
char *dhaddr[NMUX] { 0160040, 0160060 };

/*
 * TTY structs for the DH lines
 */
struct	tty dh11[NDH11];

/*
 * Hardware control bits
 */
#define	BITS6	01
#define	BITS7	02
#define	BITS8	03
#define	TWOSB	04
#define	PENABLE	020
/* DEC manuals incorrectly say this bit causes generation of even parity. */
#define	OPAR	040
#define	HDUPLX	040000

#define	IENABLE	030100
#define	PERROR	010000
#define	FRERROR	020000
#define	XINT	0100000
#define	LSPEED	03	/* low speed: 110 baud */
#define HSPEED	011	/* high speed: 1200 */

/*
 * Software copy of last dhbar (one per DH)
 */
int	dhsar[NMUX];

/*
 * External page layout
 */
struct dhregs {
	int dhcsr;
	int dhnxch;
	int dhlpr;
	int dhcar;
	int dhbcr;
	int dhbar;
	int dhbreak;
	int dhsilo;
};

/*
 * Open a DH11 line.
 */
dhopen(dev, flag)
{
	register struct tty *tp;
	register int d, *dhp;
	extern dhstart();

	if ((d = dev.d_minor) >= NDH11) {
		u.u_error = ENXIO;
		return;
	}
	dhp = dhaddr[d/16];
	tp = &dh11[d];
	tp->t_addr = dhstart;
	tp->t_dev = dev;
	dhp->dhcsr =| IENABLE;
	tp->t_state =| WOPEN|SSTART;
	if ((tp->t_state&ISOPEN) == 0
	    && (tp->t_speeds == 0 || d < NDHDM)) {
		/* high speed lines not reset normally */
		tp->t_speeds = (d < NDHDM) ? LSPEED | (LSPEED<<8):
					     HSPEED | (HSPEED<<8);
		tp->t_flags = ODDP|EVENP|ECHO|NODELAY|XTABS|CRMOD|LCASE;
		dhparam(tp);
		dmopen(dev);
	}
	tp->t_state =& ~WOPEN;
	tp->t_state =| ISOPEN;
	if (u.u_procp->p_ttyp == 0)
		u.u_procp->p_ttyp = tp;
}

/*
 * Close a DH11 line.
 */
dhclose(dev)
{
	register struct tty *tp;

	tp = &dh11[dev.d_minor];
	dmclose(dev);
	tp->t_state =& (CARR_ON|SSTART);
	wflushtty(tp);
}

/*
 * Read from a DH11 line.
 */
dhread(dev)
{
	ttread(&dh11[dev.d_minor]);
}

/*
 * write on a DH11 line
 */
dhwrite(dev)
{
	ttwrite(&dh11[dev.d_minor]);
}

/*
 * DH11 receiver interrupt.
 *		Harvard version using PIRQ.
 */
dhrint()
{
	register struct tty *tp;
	register int c, *dhp;
	int dev;

	PIRQ->integ =& ~020000;	/* Turn off programmed interrupt */
	dev = 0;
	do {
		dhp = dhaddr[dev/16];
		while ((c = dhp->dhnxch) < 0) {	/* char. present */
			tp = &dh11[((c>>8)&017) | dev];
			if (tp >= &dh11[NDH11])
				continue;
			if((tp->t_state&ISOPEN)==0 || (c&PERROR)) {
				wakeup(tp);
				continue;
			}
			if (c&FRERROR)		/* break */
				c = 0;		/* null (break signal) */
			if (tp == &dh11[LNKLIN])
				lnkinput(c, tp);
			else
				ttyinput(c, tp);
		}
	} while ((dev =+ 16) < 16 * NMUX);
}

/*
 * stty/gtty for DH11
 */
dhsgtty(dev, av)
{
	register struct tty *tp;

	tp = &dh11[dev.d_minor];
	if (!ttystty(tp, av))
		dhparam(tp);
}

/*
 * Set parameters from open or stty into the DH hardware
 * registers.
 */
dhparam(atp)
{
	register struct tty *tp;
	register int lpr, *dhp;

	tp = atp;
	dhp = dhaddr[tp->t_dev.d_minor/16];
	spl5();
	dhp->dhcsr.lobyte = (tp->t_dev.d_minor&017) | IENABLE;
	/*
	 * Hang up line?
	 */
	if (tp->t_speeds.lobyte==0) {
		tp->t_state =| HUPCL;
		dmclose(tp->t_dev);
		goto out;
	}
	lpr = (tp->t_speeds.hibyte<<10) | (tp->t_speeds.lobyte<<6);
/*	if (tp->t_speeds.lobyte == 4)		/* 134.5 baud */
/*		lpr =| BITS6|PENABLE|HDUPLX; else */
		if (tp->t_flags&EVENP)
			if (tp->t_flags&ODDP)
				lpr =| BITS8; else
				lpr =| BITS7|PENABLE; else
			lpr =| BITS7|OPAR|PENABLE;
	if (tp->t_speeds.lobyte == 3)	/* 110 baud */
		lpr =| TWOSB;
	dhp->dhlpr = lpr;
out:	spl0();
}

/*
 * DH11 transmitter interrupt.
 * Restart each line which used to be active but has
 * terminated transmission since the last interrupt.
 */
dhxint(dev)
{
	register struct tty *tp;
	register ttybit, bar;
	int *dhp;

	dhp = dhaddr[dev];
	bar = dhsar[dev] & ~dhp->dhbar;
	dhp->dhcsr =& ~XINT;
	ttybit = 1;
	for (tp = &dh11[dev*16]; bar; tp++) {
		if(bar&ttybit) {
			dhsar[dev] =& ~ttybit;
			bar =& ~ttybit;
			tp->t_state =& ~BUSY;
			if (tp == &dh11[LNKLIN])
				lnkxfr();
			else
				dhstart(tp);
		}
		ttybit =<< 1;
	}
}

/*
 * Start (restart) transmission on the given DH11 line.
 */
dhstart(atp)
{
	extern ttrstrt();
	register c, nch;
	register struct tty *tp;
	int sps, *dhp;
	char *cp;

	sps = PS->integ;
	spl5();
	tp = atp;
	/*
	 * If it's currently active, or delaying, or ^Sed,
	 * no need to do anything.
	 */
	if (tp->t_state&(TIMEOUT|BUSY|HALTOP))
		goto out;
	/*
	 * If this is a restart... simply go to it.
	 */
	dhp = dhaddr[tp->t_dev.d_minor/16];
	if(tp->t_state&SUSPEND)
		goto startup;
	/*
	 * t_char is a delay indicator which may have been
	 * left over from the last start.
	 * Arrange for the delay.
	 */
	if (c = tp->t_char)
		goto delay;
again:	cp = dh_clist[tp->t_dev.d_minor];
	/*
	 * Copy DHNCH characters, or up to a delay indicator,
	 * to the DMA area.
	 */
	nch = 0;
	while (nch > -DHNCH && (c = getc(&tp->t_outq))>=0) {
		if (c >= 0200) {
			tp->t_char = c;
			break;
		}
		*cp++ = c;
		nch--;
	}
	/*
	 * If the writer was sleeping on output overflow,
	 * wake him when low tide is reached.
	 */
	if (tp->t_outq.c_cc<=TTLOWAT && tp->t_state&ASLEEP) {
		tp->t_state =& ~ASLEEP;
		wakeup(&tp->t_outq);
	}
	/*
	 * If any characters were set up, start transmission;
	 * otherwise, check for possible delay.
	 */
	if (nch) {
startup:	dhp->dhcsr.lobyte = (tp->t_dev.d_minor & 017) | IENABLE;
		if(tp->t_state&SUSPEND) {
			tp->t_state =& ~SUSPEND;
			tp->t_char = 0;	/* clear delay count */
			if(dhp->dhbcr >= 0)
				goto again;
		} else {
			dhp->dhcar = cp+nch;
			dhp->dhbcr = nch;
		}
		c = 1<<(tp->t_dev.d_minor & 017);
		dhp->dhbar =| c;
		dhsar[tp->t_dev.d_minor/16] =| c;
		tp->t_state =| BUSY;
	} else if (c = tp->t_char) {
delay:		tp->t_char = 0;
		timeout(ttrstrt, tp, (c&0177)+6);
		tp->t_state =| TIMEOUT;
	}
    out:
	PS->integ = sps;
}

dhstop(atp)
{
	register struct tty *tp;
	register int i, *dhp;
	int dev;

	tp = atp;
	dhp = dhaddr[(dev = tp->t_dev.d_minor)/16];
	tp->t_state =& ~BUSY;
	tp->t_state =| SUSPEND;
	i = 1 << (dev & 017);
	dhsar[dev/16] =& ~i;
	dhp->dhbar =& ~i;
}
