#
/*
 * These routines handle the data structures of the  GUCS disk
 * handlers.  They are called by the specific device handlers.
 */

#include	"../param.h"
#include	"../buf.h"
#include	"../user.h"
#include	"../conf.h"
#include	"../disk.h"




/*
 * The strategy routine validates the request, and then inserts it
 * into the doubly linked list of requests for the logical disk.
 */

diskstrategy( pdp, aldp, abp)
struct pdisk	*pdp;
struct ldisk	*aldp;
struct buf	*abp;
	{
	register struct buf	*bp, *p;
	register struct ldisk	*ldp;

	bp = abp;
	ldp = aldp;

	bp->b_resid = 0;
	if ( bp->b_blkno + (-bp->b_wcount+255)/256  > ldp->l_nblk )
		bp->b_flags =| B_ERROR;
	if ( bp->b_flags&B_ERROR )  {
		bp->b_error = ENXIO;
		bp->b_resid = -bp->b_wcount;
		iodone(bp);
		return;
		}

	spl5();
	/* Put request in I/O queue for ld, in blkno order */
	for ( p= ldp->av_forw;  p!= ldp && bp->b_blkno>p->b_blkno;
				p= p->av_forw );
	/* insert bp in front of p */
	p->av_back->av_forw = bp;
	bp->av_back = p->av_back;
	bp->av_forw = p;
	p->av_back = bp;

	/* Increment count of requests at this prio level */
	(pdp->p_schedule +ldp->l_prio) ->s_reqcount++;

	spl0();
	diskstart( pdp );
	}



/*
 * The start routine first checks if the physical device is busy.
 * If not it selects the next request for service and
 * initiates an overlapped seek.
 */

diskstart( apdp )
struct pdisk	*apdp;
	{
	register struct pdisk	*pdp;
	register struct prilev	*prio;
	register char		*p;
	struct buf		*bp;
	struct ldisk		*ldp;
	int			sps;

	pdp = apdp;
	sps = PS->integ;
	spl5();

	if ( pdp->p_flags&P_BUSY )
		goto out;

	/* Find highest prio level with request waiting */
	for ( prio= pdp->p_schedule; prio->s_reqcount ==0; prio++ );
	if ( prio->s_reqcount < 0 )
		/* list terminator => no requests */
		goto out;
	prio->s_reqcount--;
	pdp->p_flags =| P_BUSY;
	PS->integ = sps;

	/* Find next request */
	if ( (p=prio->s_nextbp) == 0 )  {
		/* Search order list for next disk waiting,
		 * starting from where we left off last time */
		for ( p= prio->s_order +prio->s_ordptr;; p++ )  {
			if ( *p == ',' )
				/* change head direction */
				prio->s_headdir = -prio->s_headdir;
			else  if ( *p == ';' )  {
				/* back to the list head */
				p = prio->s_order -1;  /* incr. to s_order */
				prio->s_headdir = 1;
				}
			else	{
				ldp = &(pdp->p_ldisks)[*p];
				if ( ldp->av_forw != ldp )
					/* Got one */
					break;
				}
			}
		/* Update the pointers to new position */
		prio->s_ordptr =  p - prio->s_order;
		prio->s_curldisk = ldp;
		/* from now on p points to bp - request chosen */
		p =  prio->s_headdir>0? ldp->av_forw: ldp->av_back;
		}
	else	ldp = prio->s_curldisk;
	pdp->p_buf = p;
	pdp->p_ldp = ldp;

	/* Remove bp from I/O queue */
	p->av_forw->av_back = p->av_back;
	p->av_back->av_forw = p->av_forw;

	/* Set the next bp, if none move to next ld */
	bp = prio->s_headdir>0? p->av_forw: p->av_back;
	if ( bp == ldp )  {
		/* no more requests */
		bp = 0;
		prio->s_ordptr++;
		}
	prio->s_nextbp = bp;

	(*pdp->p_start)(pdp);
	return;

      out:
	PS->integ = sps;
	}



/*
 * Put the physical cylinder number in p_addr1,
 * and the number of residual blocks in p_addr2.
 */

diskflip( apdp, blkspercyl )
struct pdisk	*apdp;
int		blkspercyl;
	{
	register struct pdisk	*pdp;
	register struct buf	*bp;
	register int		n;
	int			cyl;

	pdp = apdp;
	bp = pdp->p_buf;
	n = bp->b_blkno;
	/* Calculate the virtual cyl no, and remaining blocks */
	cyl = ldiv( n, blkspercyl);
	pdp->p_addr2 = lrem( n, blkspercyl );

	if ( pdp->p_ldp->l_flags &L_FLIPCYL )  {
		/* Map high cyl no to low cyl no and vice versa */
		cyl =  ldiv( pdp->p_ldp->l_nblk -1, blkspercyl) - cyl;

		/* A multiblock transfer may split over two cyls.
		 * Put the words remaining to transfer in b_resid */
		if ( pdp->p_addr2 + (-bp->b_wcount+255)/256
				> blkspercyl )  {
			n =  (pdp->p_addr2 - blkspercyl)*256;
			bp->b_resid = bp->b_wcount - n;
			bp->b_wcount = n;
			}
		}
	pdp->p_addr1 =  pdp->p_ldp->l_locyl + cyl;
	}



/*
 * Hand back request as completed.
 * Non-zero status implies request failed with error "status".
 */

diskfinish( adt, status, resid )
struct devtab	*adt;
int		status, resid;
	{
	register struct buf		*bp;
	register struct pdisk		*pdp;
	register struct devtab		*dt;

	dt = adt;
	pdp = dt->d_actf;
	bp = pdp->p_buf;

	/* Report errors */
	diskerror( pdp );
	pdp->p_errcnt = 0;
	pdp->p_errsprtd = 0;
	pdp->p_resid = 0;

	/* Remove from the r/w queue */
	dt->d_actf = dt->d_actf->d_actf;
	if ( dt->d_actf == 0 )
		dt->d_actl = dt;

	/* I/O error ? */
	if ( status )  {
		bp->b_flags =| B_ERROR;
		bp->b_error = status;
		bp->b_resid =+ resid;
		}
	/* If request was split by diskflip, initiate
	 * the rest of the transfer  */
	else  if ( bp->b_resid )  {
		dpadd( &bp->b_xmem, (-bp->b_wcount)<<1 );
		bp->b_blkno =- bp->b_wcount/256;
		bp->b_wcount = bp->b_resid;
		bp->b_resid = 0;
		(*pdp->p_start)(pdp);
		return;
		}

	pdp->p_flags =& ~P_BUSY;
	diskstart( pdp );
	iodone( bp);
	}



/*
 * Increment the error count for the transfer, and print
 * print the number of errors so far if required.
 */

diskfail( apdp, error, resid)
struct pdisk	*apdp;
int		error, resid;
	{
	register struct pdisk	*pdp;

	pdp = apdp;
	if ( pdp->p_error !=error  ||  pdp->p_resid !=resid )
		diskerror( pdp);
	pdp->p_error = error;
	pdp->p_resid = resid;
	return( ++pdp->p_errcnt );
	}



/*
 * Report any un-reported errors on the transfer.
 */

diskerror( apdp )
struct pdisk	*apdp;
	{
	register struct	pdisk	*pdp;
	register struct buf	*bp;
	register int		errs;
	int			dev;

	pdp = apdp;
	bp = pdp->p_buf;
	errs = pdp->p_errcnt - pdp->p_errsprtd;
	if ( errs > 0 )  {
		dev = bp->b_dev;
		printf("SYSERR: %d of %o dev %d/%d @blk %d\n", errs,
			pdp->p_error, dev.d_major, dev.d_minor,
			bp->b_blkno +(pdp->p_resid -bp->b_wcount)/256 );
		pdp->p_errsprtd = pdp->p_errcnt;
		}
	}
