/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)queue.c	1.11 85/05/31";

/*
**	Find messages awaiting transmission, and print out details.
**
**	SETUID ==> ACSNETUID.
*/

char *	Usage		= "[-[A][F][M][R][T<level>][V][W]] [link|alias|node ...]";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"Passwd.h"
#include	"command.h"
#include	"debug.h"
#include	"ftheader.h"
#include	"handlers.h"
#include	"header.h"
#include	"spool.h"
#include	"state.h"
#include	"sub_proto.h"
#include	"Stream.h"

#include	<ndir.h>
#include	<errno.h>
#if	KILL_0 != 1
#include	<signal.h>
#endif	KILL_0 != 1


/*
**	Parameters set from arguments
*/

bool	All;				/* Print out all messages found */
bool	AllLinks;			/* Print out all links found */
bool	Fast;				/* Don't search messages */
bool	MessageDetails;			/* Print header details from messages */
char *	Name;				/* Program invoked name */
bool	Reroute;			/* Look for re-routed messages */
int	Traceflag;			/* Trace level */
char *	User;				/* Different user to look for */
bool	Verbose;			/* Print all details from ftp header */
bool	Warnings;			/* Whinge */

/*
**	List for found links.
*/

typedef struct LLel *	LLl_p;

typedef struct LLel
{
	LLl_p	lql_next;
	char *	lql_link;
	ino_t	lql_ino;
}
	LLel;

LLl_p	LinkList;			/* Head of linked list */
LLl_p *	LinkVec;			/* Vector of sorted links */
int	NLinks;				/* No. of links found */

/*
**	Structure for messages in a queue.
*/

typedef struct mq_el	ML;
struct mq_el
{
	ML *	ml_next;
	char *	ml_name;
};

/*
**	Miscellaneous info.
*/

bool	Ack;				/* Message is an acknowledgement */
char *	Format		= "%4d: %-7s from %-8.8s to %-13.13s at %-13.13s %10ld bytes\n";
char *	FormatV		= "%4d: %s from %s to %s at %s %ld bytes\n";
char *	FFormatV	= "%4d: %s from %s at %s to %s at %s %ld bytes\n";
char *	HeaderFile;			/* Current message header */
char *	HomeNode;			/* Here */
Passwd	Me;				/* My name */
long	MesgDataSize;			/* Size of data part */
char *	MesgFile;			/* Data part of message (if any) */
long	MesgHdrSize;			/* Size of header */
long	MesgLength;			/* Total size of message */
char *	Spooldir	= SPOOLDIR();
Time_t	Time;				/* Time of invocation */
long	Time_to_die;			/* From timeout commands */

#define	Fprintf		(void)fprintf
#define	Fflush		(void)fflush
#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))
#define	MSGINDENT	(6+7+1)

/*
**	Routines
*/

void	MsgStats(), ListLink(), SortLinks(), usage();
bool	FindLinks();
int	compare(), msgcmp();
bool	read_com(), read_header(), read_ftp();



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	register int	i;
	char		buf[BUFSIZ];

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					All = true;
					AllLinks = true;
					continue;

				case 'F':
					Fast = true;
					continue;

				case 'M':
					MessageDetails = true;
					continue;

				case 'R':
					Reroute = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'V':
					Verbose = true;
					continue;

				case 'W':
					Warnings = true;
					continue;

				case 'u':
					User = ++*argv;
					goto break2;

				default:
					usage("unrecognised flag '%c'", c);
					return 1;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			char *	link = *argv;

			for ( i = 0 ; ; i++ )
			{
				char *		linkp = concat(Spooldir, link, NULLSTR);
				struct stat	statb;

				if
				(
					stat(linkp, &statb) != SYSERROR
					&&
					(statb.st_mode&S_IFMT) == S_IFDIR
					&&
					!dots(link)
				)
				{
					ListLink(Spooldir, link, (ino_t)0);
					break;
				}

				free(linkp);

				if ( i == 0 )
				{
					NodeLink nl;

					if ( (link = FindAlias(*argv)) == NULLSTR )
						link = *argv;

					if ( FindAddress(link, &nl) )
					{
						link = nl.nl_name;
						continue;
					}
				}

				Error("unknown link \"%s\"", *argv);
				return 1;
			}
		}
	}

	if ( NLinks == 0 && !FindLinks(Spooldir) )
	{
		Error("no network links found!");
		return 1;
	}

	if ( !GetUser(&Me, getuid()) )
	{
		Error("passwd error for uid %d: \"%s\"", Me.P_uid, Me.P_error);
		return 1;
	}

	if ( !(Me.P_flags & P_SU) && (All || User != NULLSTR) )
	{
		if ( User != NULLSTR )
		{
			Error("No permission.");
			return 1;
		}
		else
			All = false;
	}

	if ( User != NULLSTR && !GetUid(&Me, User) )
	{
		Error("passwd error for user \"%s\": \"%s\"", User, Me.P_error);
		return 1;
	}

	if ( Reroute )
		ListLink(REROUTEDIR(), ".", (ino_t)0);

	SortLinks();

	HomeNode = NodeName();

	setbuf(stdout, buf);

	for ( i = 0 ; i < NLinks ; i++ )
	{
		MsgStats(LinkVec[i]->lql_link);
		Fflush(stdout);
	}

	return 0;
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	(void)fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}



/*
**	Queue a network link directory.
*/

void
ListLink(path, link, ino)
	char *		path;
	char *		link;
	ino_t		ino;
{
	register LLl_p	hllp;

	if ( link == NULLSTR || link[0] == '\0' )
		return;

	hllp = Talloc(LLel);

	hllp->lql_link = concat(path, link, NULLSTR);

	if ( strcmp(link, ".") == STREQUAL )
		hllp->lql_link[strlen(path)-1] = '\0';

	hllp->lql_ino = ino;

	hllp->lql_next = LinkList;
	LinkList = hllp;

	NLinks++;

	Trace2(1, "list link \"%s\"", hllp->lql_link);
}




/*
**	Search spooldir for link directories.
*/

bool
FindLinks(path)
	char *				path;
{
	register DIR *			dirp;
	register struct direct *	direp;
	bool				found = false;

	Trace2(1, "find links in \"%s\"", path);

	if ( (dirp = opendir(path)) == NULL )
	{
		Syserror("cannot read \"%s\"", path);
		return found;
	}

	while ( (direp = readdir(dirp)) != NULL )
	if ( direp->d_name[0] != WORKFLAG && !dots(direp->d_name) )
	{
		register LLl_p	hllp;
		struct stat	dstat;
		char *		newpath;

		Trace2(2, "entry \"%s\"", direp->d_name);

		for ( hllp = LinkList ; hllp != (LLl_p)0 ; hllp = hllp->lql_next )
			if ( hllp->lql_ino == direp->d_ino )
				break;

		newpath = concat(path, direp->d_name, NULLSTR);

		if
		(
			hllp == (LLl_p)0
			&&
			stat(newpath, &dstat) != SYSERROR
			&&
			(dstat.st_mode & S_IFMT) == S_IFDIR
		)
		{
			ListLink(path, direp->d_name, direp->d_ino);
			found = true;
		}

		free(newpath);
	}

	closedir(dirp);

	return found;
}


/*
**	Sort the links into alphabetic order
*/

void
SortLinks()
{
	register LLl_p	lp;
	register LLl_p *lpp;

	LinkVec = lpp = (LLl_p *)Malloc(sizeof(LLl_p) * NLinks);

	for ( lp = LinkList ; lp != (LLl_p)0 ; lp = lp->lql_next )
		*lpp++ = lp;

	DODEBUG(if((lpp-LinkVec)!=NLinks)Fatal1("bad NLinks"));

	Trace2(1, "found %d links", NLinks);

	if ( NLinks > 1 )
		qsort((char *)LinkVec, NLinks, sizeof(LLl_p), compare);
}



/*
**	Alpha compare
*/

int
compare(lpp1, lpp2)
	char *	lpp1;
	char *	lpp2;
{
	return strcmp((*(LLl_p *)lpp1)->lql_link, (*(LLl_p *)lpp2)->lql_link);
}



/*
**	Search a directory for N-N command files and print details.
*/


void
MsgStats(dir)
	char *				dir;
{
	register DIR *			dirp;
	register struct direct *	direp;
	register int			msgcount;
	register long			total;
	int				maxcount;
	char *				link;
	char *				d_state = "inactive";
	char *				l_state;
	bool				force;
	int				procid = 0;
	ML *				msgs;
	ML **				msgs_vec;

	Trace2(1, "MsgStats \"%s\"", dir);

	if ( (link = strrchr(dir, '/')) != NULLSTR && link[1] != '\0' )
		link++;
	else
		link = dir;

	if ( (dirp = opendir(dir)) == NULL )
	{
		Syserror("Can't read \"%s\"", dir);
		return;
	}

	if ( AllLinks || NLinks == 1 )
		force = true;
	else
		force = false;

	msgcount = 0;
	total = 0;
	msgs = (ML *)0;

	while ( (direp = readdir(dirp)) != NULL )
	{
		/*
		**	Find valid command file.
		*/

		switch ( direp->d_name[0] )
		{
		default:
			if ( strcmp(direp->d_name, STATUSFILE) == STREQUAL )
			{
				register int	n;
				register int	fd;
				NN_state	nnstate;
				char *		status = concat(dir, "/", direp->d_name, NULLSTR);

				if ( (fd = open(status, O_READ)) == SYSERROR )
				{
					free(status);
					continue;
				}

				if ( read(fd, (char *)&nnstate, sizeof nnstate) == sizeof nnstate )
				{
					if
					(
						(procid = nnstate.procid)
						&&
						(
							kill(procid, SIG0) != SYSERROR
							||
							errno == EPERM
						)
					)
					{
						switch ( nnstate.procstate )
						{
						default:
						case PROC_IDLE:	
							d_state = "idle";
							break;

						case PROC_RUNNING:
							d_state = "active";
							break;

						case PROC_ERROR:
							d_state = "waiting for error to be fixed";
							force = true;
							nnstate.linkstate = LINK_DOWN;
							break;

						case PROC_OPENING:
							d_state = "waiting for open on device";
							force = true;
							nnstate.linkstate = LINK_DOWN;
							break;
						
						case PROC_CALLING:
							d_state = "waiting for connection";
							nnstate.linkstate = LINK_DOWN;
							break;

						}

						switch ( nnstate.linkstate )
						{
						default:
						case LINK_DOWN:
							l_state = "down";
							break;

						case LINK_UP:
							l_state = "up";
						}
					}
					else
						procid = 0;
				}

				(void)close(fd);

				free(status);
			}

			continue;

		case SMALL_ID:
		case MEDIUM_ID:
		case LARGE_ID:
			break;
		}

		Trace2(2, "found \"%s\"", direp->d_name);

		msgcount++;

		if ( !Fast )
		{
			register ML *	msgp;

			msgp = Talloc(ML);
			msgp->ml_next = msgs;
			msgs = msgp;
			msgp->ml_name = newstr(direp->d_name);
		}
	}

	closedir(dirp);

	if ( msgcount || force )
	{
		if ( link[0] == WORKFLAG )
			Fprintf(stdout, "%s", link);
		else
		{
			Fprintf(stdout, "%-*s daemon %s", NODE_NAME_SIZE, link, d_state);

			if ( procid )
			{
				if ( Me.P_flags & P_SU )
					Fprintf(stdout, " (%d)", procid);
				Fprintf(stdout, ", line %s", l_state);
			}
		}

		Fprintf(stdout, "\n%*s", NODE_NAME_SIZE+1, "");
	}

	if ( msgcount )
	{
		Fprintf
		(
			stdout,
			"%d message%s in queue\n",
			msgcount,
			msgcount>1?"s":""
		);
	}
	else
	{
		if ( force )
			Fprintf(stdout, "queue empty\n");
		return;
	}

	if ( Fast )
		return;

	{
		register ML *	msgp;
		register int	i;

		msgs_vec = (ML **)Malloc(msgcount * sizeof(ML *));

		msgp = msgs;
		i = 0;
		do
			msgs_vec[i++] = msgp;
		while
			( (msgp = msgp->ml_next) != (ML *)0 );

		if ( msgcount > 1 )
			qsort((char *)msgs_vec, msgcount, sizeof msgs_vec[0], msgcmp);
	}

	maxcount = msgcount;

	while ( msgcount-- > 0 )
	{
		register int	fd;
		register char *	fp;
		Time_t		mtime;

		fp = concat(dir, "/", msgs_vec[msgcount]->ml_name, NULLSTR);

		free(msgs_vec[msgcount]->ml_name);
		free((char *)msgs_vec[msgcount]);

		if ( (fd = open(fp, O_READ)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read commands file \"%s\"", fp);

			free(fp);
			continue;
		}

		MesgLength = 0;
		MesgDataSize = 0;
		MesgHdrSize = 0;

		if ( MesgFile != NULLSTR )
			free(MesgFile);
		MesgFile = NULLSTR;

		if ( HeaderFile != NULLSTR )
			free(HeaderFile);
		HeaderFile = NULLSTR;

		Time_to_die = 0;

		if
		(
			!ReadCom(fd, &mtime, read_com)
			||
			HeaderFile == NULLSTR
		)
		{
			if ( Warnings )
				Warn("bad commands file \"%s\"", fp);

			(void)close(fd);
		}
		else
		{
			bool	timed_out;
			Time_t	ttd = mtime + Time_to_die;

			(void)close(fd);

			timed_out = (bool)(Time_to_die > 0 && Time > ttd);

			if
			(
				(
					All
					||
					!timed_out
				)
				&&
				read_header(timed_out, maxcount-msgcount)
			)
			{
				total += MesgLength;

				if ( Time_to_die > 0 && MessageDetails )
					Fprintf
					(
						stdout,
						"%*sTime-to-die: %.15s\n",
						MSGINDENT, "",
						ctime(&ttd)+4
					);
			}
		}
		
		free(fp);
	}

	if ( maxcount > 1 && total > 0 )
		Fprintf(stdout, "%*ld Total.\n", 72, total);

	free((char *)msgs_vec);
}



/*
**	Compare two command names for message priority (reverse order).
*/

int
msgcmp(msgp1, msgp2)
	char *	msgp1;
	char *	msgp2;
{
	return strcmp((*(ML **)msgp2)->ml_name, (*(ML **)msgp1)->ml_name);
}



/*
**	Read HeaderFile, and print details.
*/

bool
read_header(timed_out, num)
	bool		timed_out;
	int		num;
{
	register int	fd;
	HdrReason	hr;
	bool		val = false;

	Trace2(1, "read_header \"%s\"", HeaderFile);

	if ( (fd = open(HeaderFile, O_READ)) == SYSERROR )
	{
		if ( Warnings )
			SysWarn("Can't read \"%s\"", HeaderFile);

		return false;
	}
	
	if ( (hr = ReadHeader(fd)) != hr_ok )
	{
		if ( Warnings )
			Warn("Header error \"%s\"", HeaderReason(hr));

		(void)close(fd);
		return false;
	}

	/*
	**	Print protcocol details if from HomeNode, or All.
	*/

	if ( All || strcmp(HdrSource, HomeNode) == STREQUAL )
	{
		char *		cp;
		Handler *	handler;
		Time_t		msg_date;

		if ( MessageDetails )
		{
			struct stat	statb;

			(void)fstat(fd, &statb);
			msg_date = statb.st_mtime - atol(HdrTt);
		}

		if ( (cp = GetEnv(ENV_ACK)) != NULLSTR )
		{
			Ack = true;
			free(cp);
			cp = "ACK";
		}
		else
		{
			Ack = false;

			if ( (handler = GetHandler(HdrHandler)) == (Handler *)0 )
				cp = "Message";
			else
				cp = handler->descrip;
		}

		if ( HdrSubpt[0] == FTP )
			val = read_ftp(fd, num, cp);
		else
		{
			(void)close(fd);
			val = false;
		}

		if ( val == false )
		{
			if ( !All )
				return false;

			Fprintf
			(
				stdout,
				Verbose?FormatV:Format,
				num,
				cp,
				HdrSource,
				HdrHandler,
				HdrDest,
				MesgLength
			);

			val = true;
		}

		/*
		**	Print message header info if requested.
		*/

		if ( MessageDetails )
		{
			if ( HdrEnv[0] != '\0' )
				Fprintf
				(
					stdout,
					"%*sEnv=\"%s\"\n",
					MSGINDENT, "",
					ExpandString(HdrEnv, strlen(HdrEnv))
				);

			Fprintf
			(
				stdout,
				"%*sRoute=\"%s\"\n",
				MSGINDENT, "",
				ExpandString(HdrRoute, strlen(HdrRoute))
			);

			Fprintf
			(
				stdout,
				"%*sDate: %.15s\n",
				MSGINDENT, "",
				ctime(&msg_date)+4
			);

			val = true;
		}

		if ( timed_out )
		{
			Fprintf(stdout, "%*s(TIMED OUT)\n", MSGINDENT, "");
			val = true;
		}
	}
	else
		(void)close(fd);

	return val;
}



/*
**	Function called from "ReadCom" to process a command.
**	Remember last file name (containing header).
*/

bool
read_com(name, base, range)
	char *	name;
	long	base;
	long	range;
{
	Trace4(2, "command \"%s %ld %ld\"", name, base, range);

	if ( range > 0 )
	{
		MesgLength += range;
		if ( MesgFile != NULLSTR )
			free(MesgFile);
		MesgFile = HeaderFile;
		HeaderFile = name;
		MesgDataSize = MesgHdrSize;
		MesgHdrSize = range;
	}
	else
	{
		Time_to_die = base;
		free(name);
	}
	
	return true;
}



/*
**	Read the data part of message for FTP header.
*/

bool
read_ftp(fd, num, type)
	int		fd;
	int		num;
	char *		type;
{
	FthReason	fthr;

	Trace5
	(
		2,
		"read_ftp type %s, datasize %ld, hdrsize %ld, datalength %ld",
		type,
		MesgDataSize,
		MesgHdrSize,
		DataLength
	);

	if ( DataLength == 0 )
	{
		(void)close(fd);

		if ( (fd = open(MesgFile, O_READ)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read message data from \"%s\"", MesgFile);

			return false;
		}
	}
	else
		MesgDataSize = DataLength;

	if ( (fthr = ReadFtHeader(fd, MesgDataSize, false)) != fth_ok )
	{
		if ( Warnings )
			Warn("FT Header \"%s\" error", FTHREASON(fthr));

		(void)close(fd);
		return false;
	}

	(void)close(fd);

	if ( !All && strcmp(FthFrom, Me.P_name) != STREQUAL )
		return false;

	/*
	**	Have valid FTP message to print
	*/

	if ( Ack )
	{
		char *	cp = FthTo;

		FthTo = FthFrom;
		FthFrom = cp;
	}

	if ( !Verbose )
	{
		Fprintf
		(
			stdout,
			Format,
			num,
			type,
			(!All || strcmp(HdrSource, HomeNode) == STREQUAL)
				? FthFrom : HdrSource,
			FthTo,
			HdrDest,
			MesgLength
		);
	}
	else
	{
		register FthFD_p	fp;

		Fprintf
		(
			stdout,
			FFormatV,
			num,
			type,
			FthFrom,
			HdrSource,
			FthTo,
			HdrDest,
			MesgLength
		);

		if ( (fthr = GetFthFiles()) != fth_ok )
		{
			Warn("FT Header \"%s\" error", FTHREASON(fthr));
			return true;
		}

		for ( fp = FthFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
			Fprintf
			(
				stdout,
				"      %10ld  %.12s  %s\n",
				fp->f_length,
				ctime(&fp->f_time)+4,
				fp->f_name
			);

		FreeFthFiles();
	}

	return true;
}
