/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Rstate.c	1.30 85/08/16
*/

/*
**	Read a state file and structure it in memory.
*/

#define	STDIO

#include	"global.h"
#include	"address.h"
#include	"debug.h"
#include	"state.h"

#include	"statefile.h"

#undef	Extern
#define	Extern
#include	"node.h"

#include	<setjmp.h>


#ifndef	DEFUNCT_TIME
#define	DEFUNCT_TIME	(60*60*24*7L)	/* Statefiles time out after 1 week */
#endif

static Crc_t	StCRC;
static Entry *	Source;
static Time_t	SourceDate;
static char *	SourceAddr;
static Address *AddressP;

extern jmp_buf	NameErrJmp;
extern char *	HomeNode;

static Token	Rcheck();
static void	RsetFlags();
static Token	Rtoken();



void
Rstate(fd, foreign, sourceaddr, sourcedate, ignoreCRC)
	FILE *		fd;
	bool		foreign;
	char *		sourceaddr;
	Time_t		sourcedate;
	bool		ignoreCRC;
{
	register Node *	np;
	register Data *	dp;
	register Node *	op;
	register Entry *nep;
	register Entry *oep;
	Token		accept;
	Token		expect;
	int		tokens;
	char *		errs1;
	char *		errs2;
	char		temp[TOKEN_SIZE+1];

	Trace4(1, "Rstate %d %s %s", fileno(fd), foreign?"true":"false", sourceaddr==NULLSTR?"":sourceaddr);

	if ( tokens = setjmp(NameErrJmp) )
	{
		Error("Bad home nodename \"%s\" at char %d", HomeNode, tokens-1);
		return;
	}

	if ( Home == (Entry *)0 )
	{
		Home = Enter(HomeNode, NodeHash);

		if ( !(Home->e_states & S_FOUND) )
		{
			Home->e_states |= S_FOUND;
			NodeCount++;
			ClearDomains(Home, true);
		}
	}

	if ( foreign )
	{
		SourceAddr = sourceaddr;
		SourceDate = sourcedate;

		AddressP = SplitAddress(sourceaddr);

		SetHierarchy(NULLSTR, &ForeignHier);
	}
	else
		SetHierarchy(NULLSTR, &DomHier);

	StCRC = 0;
	Source = (Entry *)0;

	accept = t_any;
	expect = t_any;

	np = (Node *)0;
	dp = (Data *)0;
	oep = (Entry *)0;

	if ( tokens = setjmp(NameErrJmp) )
	{
		errs1 = "Bad node name in %s statefile: \"%s\" (char %d)";
		errs2 = foreign?"foreign":"home";

		if ( ignoreCRC )
			Warn(errs1, errs2, temp, tokens-1);
		else
		{
			Error(errs1, errs2, temp, tokens-1);
			return;
		}
	}

	for ( tokens = 0 ; ; tokens++ )
	{
		switch ( Rtoken(fd, temp, accept, &expect) )
		{
		case t_node:
			nep = Enter(temp, NodeHash);
			np = nep->e_node;
			oep = (Entry *)0;
			dp = (Data *)0;

			Trace2(2, "Node \"%s\"", nep->e_name);

			if ( foreign )
			{
				if ( (accept = Rcheck(nep)) == t_eof )
				{
					AnteState = true;
					return;
				}
			}
			else
				accept = t_any;

			if ( !(nep->e_states & S_FOUND) )
			{
				NodeCount++;
				nep->e_states |= S_FOUND;
			}

			continue;

		case t_nodehier:
			if ( !foreign )
			{
				if ( np == (Node *)0 )
				{
					errs1 = "Unexpected node hierarchy";
					errs2 = temp;
					break;
				}
				if ( dp != (Data *)0 )
					continue;	/* Ignore link hierarchy */
				if ( np->n_hierarchy != NULLSTR )
					free(np->n_hierarchy);
				np->n_hierarchy = newstr(temp);
			}
			else
			{
				/*
				**	Link hierarchy
				*/

				if ( dp == (Data *)0 )
				{
					if ( oep == Home )
						continue;

					errs1 = "Unexpected link hierarchy";
					errs2 = temp;
					break;
				}
				if
				(
					op->n_hierarchy != NULLSTR
					&&
					strccmp(temp, op->n_hierarchy) != STREQUAL
				)
				{
					register Link **lpp;

					/*
					**	This node thinks that this link
					**	has a different hierarchy
					**	==> different link?
					*/

					if ( (lpp = IsLinked(nep, oep)) != (Link **)0 )
					{
						if ( Links(oep) == 1 )
						{
							/*
							**	But wait!
							**	Perhaps it's right!
							**	(The hierarchy has really changed,
							**	and we didn't believe it before.)
							*/

							Warn
							(
								"Changed hierarchy for node \"%s\" linked to \"%s\" from \"%s\" to \"%s\"",
								oep->e_name,
								nep->e_name,
								op->n_hierarchy,
								temp
							);
							free(op->n_hierarchy);
							op->n_hierarchy = newstr(temp);

						}
						else
						{
							dp = (Data *)0;
							Unlink(nep, lpp);
							Warn
							(
								"Broke putative link between \"%s.%s\" and \"%s.%s\" -- node of same name already exists called \"%s.%s\"",
								nep->e_name, np->n_hierarchy,
								oep->e_name, temp,
								oep->e_name, op->n_hierarchy
							);
						}
					}
				}
			}
			continue;

		case t_link:
			dp = (Data *)0;

			if ( np == (Node *)0 )
			{
				errs1 = "Unexpected link";
				errs2 = temp;
				break;
			}

			oep = Enter(temp, NodeHash);
			op = oep->e_node;

			Trace2(2, "Link \"%s\"", oep->e_name);

			if ( foreign && oep == Home )
				continue;

			if ( op == np )
			{
				errs1 = "Link to self for";
				errs2 = oep->e_name;
				break;
			}

			dp = MakeLink(nep, oep);

			if ( foreign && nep == Source )
			{
				if ( !(oep->e_states & S_FOUND) )
				{
					NodeCount++;
					oep->e_states |= S_FOUND;
				}

				(void)MakeLink(oep, nep);

				dp->d_states &= ~FOREIGN_FLAGS;
				dp->d_cost = 0;
				dp->d_speed = 0;
			}

			continue;

		case t_alias:
			if ( np != (Node *)0 )
			{
				errs1 = "Unexpected alias";
				errs2 = temp;
				break;
			}
			nep = Enter(temp, AliasHash);
			expect = t_aliasv;
			continue;

		case t_aliasv:
			if ( expect != t_aliasv )
			{
				errs1 = "Unexpected alias value";
				errs2 = temp;
				break;
			}
			expect = t_any;
			if ( foreign )
			{
				Warn
				(
					"Unexpected alias \"%s=>%s\" in foreign statefile from \"%s\"",
					nep->e_name,
					temp,
					sourceaddr
				);
				continue;
			}
			if ( nep->e_states & S_FOUND )
			{
				errs1 = "Duplicate alias value";
				errs2 = temp;
				break;
			}
			else
			{
				nep->e_value = newstr(temp);
				AliasCount++;
				nep->e_states |= S_FOUND;
			}
			continue;

		case t_xalias:
			if ( np != (Node *)0 )
			{
				errs1 = "Unexpected export alias";
				errs2 = temp;
				break;
			}
			nep = Enter(temp, AliasHash);
			if ( foreign )
			{
				if
				(
					(nep->e_states & S_FOUND) 
					||
					Lookup(temp, NodeHash) != (Entry *)0
					||
					Lookup(temp, DomainHash) != (Entry *)0
				)
				{
					if
					(
						nep->e_value == NULLSTR
						||
						strcmp(nep->e_value, sourceaddr) != STREQUAL
					)
						Warn
						(
							"Rejected export alias \"%s\" for \"%s\" -- already exists",
							temp, sourceaddr
						);

					continue;	/* Reject if already exists */
				}
				nep->e_value = newstr(sourceaddr);
				AliasCount++;
				nep->e_states |= S_FOUND;
			}
			else
			{
				if
				(
					!(nep->e_states & S_FOUND)
					||
					strcmp(nep->e_value, Home->e_name) != STREQUAL
				)
				{
					errs1 = "Bad export alias name";
					errs2 = temp;
					break;
				}
				nep->e_next = XAliases;
				XAliases = nep;
			}
			continue;
			
		case t_domain:
			if ( np == (Node *)0 )
			{
				errs1 = "Unexpected domain";
				errs2 = temp;
				break;
			}
			oep = AddDomain(temp, &np->n_domains, S_FOUND);
			if ( nep == Home )
				oep->e_states |= S_OURDOM;
			continue;

		case t_topdom:
			if ( np == (Node *)0 )
			{
				errs1 = "Unexpected domain";
				errs2 = temp;
				break;
			}
			oep = AddDomain(temp, &np->n_domains, S_FOUND|S_TOPDOM);
			if ( nep == Home )
				oep->e_states |= S_OURDOM;
			continue;

		case t_othdom:
			if ( np == (Node *)0 )
			{
				errs1 = "Unexpected other domain";
				errs2 = temp;
				break;
			}
			if ( foreign )
				np = Source->e_node;
			(void)AddDomain(temp, &np->n_domains, S_TOPDOM|S_OTHDOM|S_FOUND);
			continue;

		case t_domhier:
			if ( np != (Node *)0 )
			{
				errs1 = "Unexpected domain hierarchy";
				errs2 = temp;
				break;
			}
			if ( foreign )
				(void)AddDomain(temp, &ForeignHier, (States)0);
			else
				(void)AddDomain(temp, &DomHier, S_HIERDOM);
			continue;

		case t_state:
			if ( np != (Node *)0 )
			{
				Time_t	t;

				if ( (t = atol(temp)) > np->n_state )
					np->n_state = t;
			}
			continue;

		case t_date:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
			{
				Time_t	t;

				if ( (t = atol(temp)) > np->n_date )
					np->n_date = t;
			}
#			endif	NODE_STATS
			continue;

		case t_recvd:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_recvd += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_sent:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_sent += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_passto:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_passto += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_passfrom:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_passfrom += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_time:
#			if	LINK_STATS == 1
			if ( dp != (Data *)0 )
				dp->d_time += (ulong)atol(temp);
#			endif	LINK_STATS
			continue;

		case t_bytes:
#			if	LINK_STATS == 1
			if ( dp != (Data *)0 )
				dp->d_bytes += (ulong)atol(temp);
#			endif	LINK_STATS
			continue;

		case t_speed:
			if ( foreign && nep != Source )
				continue;
			if ( dp != (Data *)0 )
				dp->d_speed = (ulong)atol(temp);
			continue;

		case t_cost:
			if ( foreign && nep != Source )
				continue;
			if ( dp != (Data *)0 )
				dp->d_cost = (ulong)atol(temp);
			continue;

		case t_nflags:
			if ( np != (Node *)0 )
				RsetFlags(&nep->e_states, temp, foreign);
			continue;

		case t_lflags:
			if ( dp != (Data *)0 )
				RsetFlags(&dp->d_states, temp, foreign);
			continue;

		case t_spooler:
			if ( np == (Node *)0 )
				continue;
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_spooler != NULLSTR )
				free(np->n_handlers->h_spooler);
			np->n_handlers->h_spooler = newstr(temp);
			continue;

		case t_connector:
			if ( np == (Node *)0 )
				continue;
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_connector != NULLSTR )
				free(np->n_handlers->h_connector);
			np->n_handlers->h_connector = newstr(temp);
			continue;

		case t_caller:
			if ( np == (Node *)0 )
				continue;
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_caller != NULLSTR )
				free(np->n_handlers->h_caller);
			np->n_handlers->h_caller = newstr(temp);
			continue;

		case t_comment:
			if ( np == (Node *)0 )
				continue;
#			if	ALL_COMMENTS != 1
			if ( nep != Home )
				continue;
#			endif	ALL_COMMENTS != 1
			if ( np->n_comment != NULLSTR )
				free(np->n_comment);
			np->n_comment = newstr(temp);
			continue;

		case t_handler:
			if ( oep == (Entry *)0 || !(oep->e_states & S_OURDOM) )
			{
				errs1 = "Unexpected domain handler";
				errs2 = temp;
				break;
			}
			if ( oep->e_handler != NULLSTR )
				free(oep->e_handler);
			oep->e_handler = newstr(temp);
			continue;

		case t_eof:
			goto out;

		default:
			Fatal1("Rstats: unknown token");
		}

		if ( foreign )
		{
			Error("%s \"%s\"", errs1, errs2);
			AnteState = true;
		}
		else
			Warn("%s \"%s\" ignored", errs1, errs2);
	}

out:
	if
	(
		!ignoreCRC
		&&
		tokens > 0
		&&
		(
			getc(fd) != C_EOF
			||
			getc(fd) != LOCRC(StCRC)
			||
			getc(fd) != HICRC(StCRC)
			||
			getc(fd) != '\n'
		)
	)
	{
		Error("bad %sstatefile CRC", foreign?"foreign ":"");
		AnteState = true;
		return;
	}

	if ( foreign )
	{
		FreeAddress(AddressP);

		if ( Source == (Entry *)0 )
		{
			AnteState = true;
			return;
		}

		if ( sourcedate > 0 )
			Source->e_node->n_state = sourcedate;

		CheckHierarchy(Source, &ForeignHier);
		CheckSourceDomains(Source);
	}
	else
		Home->e_node->n_hierarchy = DomString(&DomHier, DOMAIN_SEP);
}



/*
**	Read a separator followed by a token (name or number)
**	and return token type.
*/

static Token
Rtoken(fd, string, accept, expect)
	FILE *		fd;
	char *		string;
	Token		accept;
	Token *		expect;
{
	register char *	s;		/* Known to be r11 for vaxen */
	register int	n;
	register int	c;
#	if	vax
	register int	last_crc;	/* Known to be r8 for vaxen */
#	else	vax
	register Crc_t	last_crc;
#	endif	vax
	register Token	token;
	register Token	new_token;
	register int	repeat_c;

	Trace2
	(
		3,
		"Rtoken accept=%s",
		accept==t_any?"any"
		  :accept==t_foreign?"foreign"
		  :accept==t_node?"node"
		  :"???"
	);

	for ( n = 0, s = string, token = t_null, repeat_c = 0 ; ; )
	{
		DODEBUG(char	tokenc);

		c = getc(fd);
		*s = c & 0177;

#		if	vax
		last_crc = StCRC;	/* Now in r8 */
		asm("	crc	_crc16t,r8,$1,(r11)	");
		asm("	movw	r0,_StCRC		");
#		else	vax
		StCRC = acrc((last_crc = StCRC), s, 1);
#		endif	vax

		switch ( c )
		{
		case C_ALIAS:		new_token = t_alias;	goto found;
		case C_ALIASV:		new_token = t_aliasv;	goto found;
		case C_BYTES:		new_token = t_bytes;	goto found;
		case C_CALLER:		new_token = t_caller;	goto found;
		case C_COMMENT:		new_token = t_comment;	goto found;
		case C_CONNECTOR:	new_token = t_connector;goto found;
		case C_COST:		new_token = t_cost;	goto found;
		case C_DATE:		new_token = t_date;	goto found;
		case C_DOMAIN:		new_token = t_domain;	goto found;
		case C_DOMHIER:		new_token = t_domhier;	goto found;
		case C_HANDLER:		new_token = t_handler;	goto found;
		case C_LFLAGS:		new_token = t_lflags;	goto found;
		case C_LINK:		new_token = t_link;	goto found;
		case C_NFLAGS:		new_token = t_nflags;	goto found;
		case C_NODE:		new_token = t_node;	goto found;
		case C_NODEHIER:	new_token = t_nodehier;	goto found;
		case C_OTHDOM:		new_token = t_othdom;	goto found;
		case C_PASSFROM:	new_token = t_passfrom;	goto found;
		case C_PASSTO:		new_token = t_passto;	goto found;
		case C_RECVD:		new_token = t_recvd;	goto found;
		case C_SENT:		new_token = t_sent;	goto found;
		case C_SPEED:		new_token = t_speed;	goto found;
		case C_SPOOLER:		new_token = t_spooler;	goto found;
		case C_STATE:		new_token = t_state;	goto found;
		case C_TIME:		new_token = t_time;	goto found;
		case C_TOPDOM:		new_token = t_topdom;	goto found;
		case C_XALIAS:		new_token = t_xalias;	goto found;
		case C_SPARE1:		new_token = t_ignore;	goto found;
		case C_SPARE2:		new_token = t_ignore;	goto found;

		case C_EOF:		(void)ungetc(c, fd);
					StCRC = last_crc;
		case EOF:		if ( n == 0 || token == t_null )
						return t_eof;
					Trace1(1, "Rtoken at EOF!");
					goto out;
found:
			if ( n == 0 || token == t_null )
			{
				if ( repeat_c )
				{
					if ( c != repeat_c )
						break;
					repeat_c = 0;
					token = t_null;
					continue;
				}

				if ( accept == t_foreign )
				{
					switch ( new_token )
					{
					case t_alias:
					case t_aliasv:
					case t_bytes:
					case t_caller:
					case t_connector:
					case t_date:
					case t_handler:
					case t_ignore:
					case t_passfrom:
					case t_passto:
					case t_recvd:
					case t_sent:
					case t_spooler:
					case t_state:
					case t_time:
					case t_topdom:
						token = t_null;
						continue;
					}
				}
				else
				if ( accept != t_any && new_token != accept )
				{
					token = t_null;
					continue;
				}

				switch ( token = new_token )
				{
				case t_ignore:
					token = t_null;
					continue;

				case t_caller:
				case t_comment:
				case t_connector:
				case t_handler:
				case t_spooler:
					repeat_c = c;
				}

				DODEBUG(tokenc = c);
				s = string;
				n = 0;
				continue;
			}
			else
			{
				if ( repeat_c && c != '\n' )
				{
					if ( c != repeat_c )
						break;
				}
				else
				{
					(void)ungetc(c, fd);
					StCRC = last_crc;
				}

out:
				*s = '\0';
				Trace4(3, "Rtoken '%c'(%d) \"%s\"", tokenc, (int)token, string);
				if ( (new_token = *expect) != t_any && token != new_token )
				{
					*expect = t_any;
					Warn("Read unexpected token %d value \"%s\" (expected %d)", token, string, new_token);
				}
				return token;
			}
		}

		if ( ++n <= TOKEN_SIZE )
		{
			if ( *s >= ' ' || *s == '\t' )
				s++;
			else
				n--;
		}
		else
			n = TOKEN_SIZE;
	}
}



/*
**	Decode state flags from string.
*/

static void
RsetFlags(sp, cp, foreign)
	register States *sp;
	register char *	cp;
	bool		foreign;
{
	register int	c;
	register States	accept;

	if ( foreign )
	{
		accept = FOREIGN_FLAGS;

		if ( *sp & S_LAN )
			accept &= ~S_DOWN;
	}
	else
		accept = EXTERNAL_FLAGS;

	*sp &= ~accept;
	accept |= S_DOWN;

	while ( c = *cp++ )
		*sp |= (1 << (c-'A')) & accept;
}



/*
**	Check foreign node integrity.
*/

static Token
Rcheck(ep)
	register Entry *ep;
{
	register Node *	np;

	np = ep->e_node;

	if ( Source == (Entry *)0 )
	{
		register int	i;
		char *		hierarchy;

		/*
		**	This is first node in statefile,
		**	but hierarchy should already have been read.
		*/

		if ( (Source = Enter(AddressP->ad_node, NodeHash)) == Home )
		{
			Error
			(
				"Foreign node name same as Home: \"%s\"",
				SourceAddr
			);
			return t_eof;
		}

		if ( Source != ep )
		{
			Error
			(
				"Foreign node name not same as statefile: \"%s\" != \"%s\"",
				SourceAddr,
				ep->e_name
			);
			return t_eof;
		}

		if
		(
			ForeignHier.d_head == (Domain *)0
			&&
			AddressP->ad_domains >= 2
		)
		{
			/*
			**	No stated hierarchy, but neighbours think
			**	there is, so use domains from source address.
			*/

			for ( i = AddressP->ad_domains ; i > 0 ; i++ )
			{
				(void)AddDomain
				(
					AddressP->ad_strings[i],
					&Source->e_node->n_domains,
					S_FOUND
				);
				(void)AddDomain
				(
					AddressP->ad_strings[i],
					&ForeignHier,
					(States)0
				);
			}

			Warn("Made up hierarchy for node from source address \"%s\"", SourceAddr);

			hierarchy = newstr(strchr(SourceAddr, DOMAIN_SEP)+1);
		}
		else
		{
			hierarchy = DomString(&ForeignHier, DOMAIN_SEP);

			for ( i = AddressP->ad_domains ; i > 0 ; i-- )
			{
				register Entry *	dp;
				register Domain *	hp;

				if ( (dp = Lookup(AddressP->ad_strings[i], DomainHash)) != (Entry *)0 )
				{
					for ( hp = ForeignHier.d_head ; hp != (Domain *)0 ; hp = hp->d_next )
						if ( hp->d_entry == dp )
							break;
				}
				else
					hp = (Domain *)0;

				if ( hp == (Domain *)0 )
				{
					Error("Source address \"%s\" inconsistent with hierarchy \"%s\"", SourceAddr, hierarchy);
					return t_eof;
				}
			}
		}

		if ( SourceDate > 0 )
		{
			extern Time_t	Time;

			if ( np->n_state >= SourceDate && np->n_state < Time )
			{
				Report2("foreign statefile for \"%s\" date earlier than last", Source->e_name);

				return t_eof;
			}

#			if	NODE_STATS == 1
			if ( SourceDate > np->n_date || np->n_date > Time )
				np->n_date = SourceDate;
#			endif	NODE_STATS == 1

			if
			(
				np->n_hierarchy != NULLSTR
				&&
				hierarchy != NULLSTR
				&&
				strcmp(np->n_hierarchy, hierarchy) != STREQUAL
				&&
				(SourceDate - np->n_state) < DEFUNCT_TIME
				&&
				(
					Home->e_node->n_hierarchy == NULLSTR
					||
					strcmp(Home->e_node->n_hierarchy, hierarchy) != STREQUAL
				)
			)
			{
				Warn
				(
					"state discarded for node \"%s\" because hierarchy changed from \"%s\" to \"%s\"",
					Source->e_name,
					np->n_hierarchy,
					hierarchy
				);
				return t_eof;
			}
		}

		/*
		**	Clear all variables which this statefile may legally set
		*/

		{
			register Link **lpp;

			for ( lpp = &np->n_l_first ; *lpp != (Link *)0 ; )
				if ( (*lpp)->l_entry != Home )
					Unlink(Source, lpp);
				else
					lpp = &(*lpp)->l_next;
		}

		if ( np->n_hierarchy != NULLSTR )
			free(np->n_hierarchy);
		np->n_hierarchy = hierarchy;

		if ( np->n_comment != NULLSTR )
		{
			free(np->n_comment);
			np->n_comment = NULLSTR;
		}
	}

	if
	(
		ep == Home
		||
		(
			ep != Source
			&&
			(
				np->n_state > 0						/* Had a statefile direct */
				||
				(
					Source->e_node->n_state > 0		/* Had a statefile from source before */
					&&
					IsLinked(ep, Source) == (Link **)0	/* This node not linked to source */
				)
			)
		)
	)
		return t_node;		/* Skip line */

	ClearDomains(ep, true);
	ep->e_states &= ~FOREIGN_FLAGS;

	return t_foreign;	/* Reject rude tokens */
}
