/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Rcommands.c	1.19 85/08/16
*/

/*
**	Read commands from file and update state info. in place.
*/

#define	STDIO

#include	"global.h"
#include	"address.h"
#include	"debug.h"
#include	"state.h"

#include	"node.h"
#include	"statefile.h"

#include	<setjmp.h>

/*
**	Commands
*/

#define	LINE_LENGTH	200		/* Maximum length of a command line */

#define	DELIM		" \t"		/* Command field delimiters */

#define	MAXSPEED	100000		/* Maximum speed for a link */
#define	MAXCOST		65535		/* Maximum cost for a link */

#define	ALIAS_C		','		/* Char to delimit an alias */
#define	COMMENT_C	'#'		/* Char to delimit start of comment */
#define	DOMAIN_C	','		/* Char to delimit domains */
#define	FLAG_C		','		/* Char to delimit flags */
#define	HIER_C		DOMAIN_SEP	/* Char to delimit domains in hierarchy */
#define	LINK_C		','		/* Char to delimit link from node */
#define	QUOTE_C		'!'		/* Char to start quoted string */

typedef enum
{
	ct_add,		/* <node> [flag[,...]]		(add a new node) */
	ct_alias,	/* <name> [node[.domain ...]]	(give value for an alias) */
	ct_break,	/* <node1>,<node2>		(remove link between node1 and node2) */
	ct_caller,	/* <node> [<caller>]		(set caller field in node description) */
	ct_clear,	/* <node>			(clear state date for this node) */
	ct_comment,	/* <node> [<comment ...>]	(set comment field in node description) */
	ct_connector,	/* <node> [<connector>]		(set connector field in node description) */
	ct_cost,	/* <node1>,<node2> <cost> 	(set cost of link between node1 and node2) */
	ct_domain,	/* <node> primary-domain[,secondary-domain ...] (assign node to domains) */
	ct_domhier,	/* [domain-in[. ...]]		(assign domain hierarchy for this node) */
	ct_flag,	/* <node1>[,<node2>] {+|-}<flag>[,...] (add|remove flags at node1 or in link to node2) */
	ct_handler,	/* <domain> [<handler>]		(set handler field in domain description) */
	ct_link,	/* <node1>,<node2> [flag[,...]]	(make link between node1 and node2) */
	ct_remdom,	/* <domain>			(remove domain) */
	ct_remove,	/* <node>			(remove node and links) */
	ct_speed,	/* <node1>,<node2> <speed> 	(set speed of link between node1 and node2) */
	ct_spooler,	/* <node> [<spooler>]		(set spooler field in node description) */
	ct_xalias	/* [<name>[,...]]		(give name for exported alias for this node) */
}
		CmdType;

typedef enum
{
	cf_command, cf_node, cf_link, cf_optlink, cf_params, cf_optparams
}
		CmdFlag;

#define	CF_COMMAND	(1<<(int)cf_command)
#define	CF_NODE		(1<<(int)cf_node)
#define	CF_LINK		(1<<(int)cf_link)
#define	CF_OPTLINK	(1<<(int)cf_optlink)
#define	CF_PARAMS	(1<<(int)cf_params)
#define	CF_OPTPARAMS	(1<<(int)cf_optparams)

typedef struct
{
	char *	c_name;
	CmdType	c_type;
	int	c_flags;
}
		Command;

#define	CMDZ	sizeof(Command)

typedef Command *Cmdp;

Command		Cmds[] =
{
	{"add",		ct_add,		CF_OPTPARAMS},
	{"alias",	ct_alias,	CF_OPTPARAMS},
	{"break",	ct_break,	CF_LINK},
	{"caller",	ct_caller,	CF_OPTPARAMS},
	{"clear",	ct_clear,	0},
	{"comment",	ct_comment,	CF_OPTPARAMS},
	{"connector",	ct_connector,	CF_OPTPARAMS},
	{"cost",	ct_cost,	CF_LINK|CF_PARAMS},
	{"domain",	ct_domain,	CF_OPTPARAMS},
	{"flag",	ct_flag,	CF_OPTLINK|CF_PARAMS},
	{"handler",	ct_handler,	CF_OPTPARAMS},
	{"hierarchy",	ct_domhier,	0},
	{"link",	ct_link,	CF_LINK|CF_OPTPARAMS},
	{"remdom",	ct_remdom,	0},
	{"remove",	ct_remove,	0},
	{"speed",	ct_speed,	CF_LINK|CF_PARAMS},
	{"spooler",	ct_spooler,	CF_OPTPARAMS},
	{"xalias",	ct_xalias,	0}
};

#define	NCMDS	((sizeof Cmds)/CMDZ)

/*
**	Flags (see state.h).
*/

typedef struct
{
	char *	f_name;
	short	f_min;
	States	f_state;
}
		Flag;

#define	FLGZ	sizeof(Flag)

typedef Flag *	Flgp;

Flag		Flags[] =
{
	{"call",	2,	S_CALL},
	{"con",		2,	S_CON},
	{"conterm",	4,	S_CONTERMINAL},
	{"dead",	2,	S_DEAD},
	{"down",	2,	S_DOWN},
	{"foreign",	1,	S_FOREIGN},
	{"intermittent",1,	S_INTERMITTENT},
	{"lan",		2,	S_LAN},
	{"local",	2,	S_LOCAL},
	{"msg",		1,	S_MSG},
	{"msgterm",	4,	S_MSGTERMINAL},
	{"terminal",	1,	S_MSGTERMINAL|S_CONTERMINAL},
	{"workstation",	1,	S_LOCAL}
};

#define	NFLGS	((sizeof Flags)/FLGZ)

/*
**	Miscellaneous
*/

static int	Line;

static char *	AliasString();
static char *	SetAlias();
int		cmpflg();
int		cmpcom();
static void	fixlan();
static int	getline();
static bool	set_states();

extern jmp_buf	NameErrJmp;



void
Rcommands(fd, warn)
	FILE *			fd;
	bool			warn;
{
	register Cmdp		cmdp;
	register char *		cp;
	register Link **	lpp;
	register Entry *	np;
	register Entry *	lp;
	register char *		errp;
	register Data *		dp;
	register bool		home;
	register ulong		n;
	char *			errfmt = "%s command \"%s\" at line %d";
	char *			nolink = "no link exists for";
	char			buf[LINE_LENGTH];

	Trace2(1, "Rcommand fd %d", fileno(fd));

	Line = 0;

	while ( getline(fd, buf) != EOF )
	{
		Trace2(2, "got line \"%s\"", buf);

		if ( (cp = strtok(buf, DELIM)) == NULLSTR )
			continue;

		if
		(
			(cmdp = (Cmdp)bsearch
				      (
					cp,
					(char *)Cmds,
					NCMDS,
					CMDZ,
					cmpcom
				      )
			) == (Cmdp)0
		)
		{
			errp = "unknown";
			Error(errfmt, errp, cp, Line);
			continue;
		}

		if ( (cp = strtok(NULLSTR, DELIM)) == NULLSTR )
		{
			if ( cmdp->c_type == ct_domhier )
			{
				SetHierarchy(cp, &DomHier);
				continue;
			}

			if ( cmdp->c_type == ct_xalias )
			{
				XAliases = (Entry *)0;
				continue;
			}

			errp = "no node for";
			Error(errfmt, errp, cmdp->c_name, Line);
			continue;
		}

		if ( setjmp(NameErrJmp) )
		{
			switch ( cmdp->c_type )
			{
			case ct_alias:
			case ct_xalias:
				errp = "bad alias for";	
				break;
			case ct_domain:
			case ct_domhier:
			case ct_remdom:
			case ct_handler:
				errp = "bad domain name for";
				break;
			default:
				errp = "bad node name for";
			}

			Error(errfmt, errp, cmdp->c_name, Line);
			continue;
		}

		home = false;

		if
		(
			(errp = strchr(cp, LINK_C)) == NULLSTR
			||
			cmdp->c_type == ct_xalias
		)
		{
			if ( cmdp->c_flags & CF_LINK )
			{
				errp = "missing link for";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}

			lp = (Entry *)0;

			switch ( cmdp->c_type )
			{
			case ct_alias:
			case ct_domhier:
			case ct_handler:
			case ct_remdom:
			case ct_xalias:
				np = (Entry *)cp;	/* Remember name */
				break;

			default:
				if ( (np = Enter(cp, NodeHash)) == Home )
					home = true;
				break;
			}
		}
		else
		{
			if ( !(cmdp->c_flags & (CF_LINK|CF_OPTLINK)) )
			{
				errp = "extraneous link for";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}

			*errp++ = '\0';

			np = Enter(cp, NodeHash);
			lp = Enter(errp, NodeHash);

			if ( np == lp )
			{
				errp = "link to self in";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}

			if ( np == Home || lp == Home )
				home = true;

			if ( (lpp = IsLinked(np, lp)) == (Link **)0 )
			{
				static Link *	empty_link;

				lpp = &empty_link;
			}
		}

		if
		(
			(cp = strtok(NULLSTR, "")) == NULLSTR
			||
			((cp += strspn(cp, DELIM)), *cp == '\0')
		)
		{
			if ( cmdp->c_flags & CF_PARAMS )
			{
				errp = "missing parameter list for";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}

			cp = NULLSTR;
		}
		else
		{
			if ( !(cmdp->c_flags & (CF_PARAMS|CF_OPTPARAMS)) )
			{
				errp = "unexpected parameter list for";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}

			if ( *cp == QUOTE_C )
				cp++;

			if ( *cp == '\0' )
				cp = NULLSTR;
		}

		switch ( cmdp->c_type )
		{
		case ct_add:
			if ( cp != NULLSTR )
				(void)set_states(&np->e_states, cp, NODE_FLAGS, home);

			if ( !(np->e_states & S_FOUND) )
			{
				NodeCount++;
				np->e_states |= S_FOUND;
			}
			continue;

		case ct_remove:
			if ( home )
			{
				errp = "cannot remove own node in";
				break;
			}

			if ( np->e_states & S_FOUND )
			{
				NodeCount--;
				np->e_states &= ~S_FOUND;
			}

			if ( IsLinked(Home, np) != (Link **)0 )
				ChangeState = true;
			continue;

		case ct_clear:
			np->e_node->n_state = 0;
			continue;

		case ct_link:
			if
			(
				!(np->e_states & S_FOUND)
				||
				!(lp->e_states & S_FOUND)
			)
			{
				errp = "cannot make link to unknown node for";
				break;
			}

			dp = MakeLink(np, lp);

			if ( home && !(dp->d_states & S_FOUND) )
				ChangeState = true;

			(void)MakeLink(lp, np);

			if ( cp == NULLSTR )
				dp->d_states |= S_MSG;
			else
			if ( set_states(&dp->d_states, cp, LINK_FLAGS, home) )
				fixlan(np, lp);

			continue;

		case ct_break:
			if ( *lpp != (Link *)0 )
			{
				Unlink(np, lpp);

				if ( (lpp = IsLinked(lp, np)) != (Link **)0 )
					Unlink(lp, lpp);

				if ( home )
					ChangeState = true;
			}

			continue;

		case ct_flag:
			if ( lp != (Entry *)0 )
			{
				if ( *lpp == (Link *)0 )
				{
					errp = nolink;
					break;
				}

				if ( set_states(&(*lpp)->l_data->d_states, cp, LINK_FLAGS, home) )
					fixlan(np, lp);
			}
			else
				(void)set_states(&np->e_states, cp, NODE_FLAGS, home);

			continue;

		case ct_speed:
			if ( *lpp == (Link *)0 )
			{
				errp = nolink;
				break;
			}

			if ( (n = atol(cp)) < 0 || n > MAXSPEED )
			{
				errp = "speed out of range";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}

			if ( home && (*lpp)->l_data->d_speed != n )
				ChangeState = true;

			(*lpp)->l_data->d_speed = n;
			continue;

		case ct_cost:
			if ( *lpp == (Link *)0 )
			{
				errp = nolink;
				break;
			}

			if ( (n = atol(cp)) < 0 || n > MAXCOST )
			{
				errp = "cost out of range";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}

			if ( home && (*lpp)->l_data->d_cost != n )
				ChangeState = true;

			(*lpp)->l_data->d_cost = n;
			continue;

		case ct_alias:
			if ( (errp = SetAlias((char *)np, cp)) != NULLSTR )
				Error(errfmt, errp, cmdp->c_name, Line);
			continue;

		case ct_xalias:
			lpp = (Link **)AliasString(XAliases);

			XAliases = (Entry *)0;
			do
			{
				if ( (cp = strchr((char *)np, ALIAS_C)) != NULLSTR )
					*cp++ = '\0';
				if ( (errp = SetAlias((char *)np, Home->e_name)) != NULLSTR )
				{
					Error(errfmt, errp, cmdp->c_name, Line);
					break;
				}
				lp = Lookup((char *)np, AliasHash);
				lp->e_next = XAliases;
				XAliases = lp;
			}
#					ifndef	perq
				while ( (char *)(np = (Entry *)cp) != NULLSTR );
#					else	perq
				/*
				** original gave compiler error (sic):-
				** "Rcommands.c", line 498: compiler roor: no hoptab for PCNOV
				*/
				while (np = (Entry *)cp);
#					endif	perq

			if ( (cp = AliasString(XAliases)) == NULLSTR )
			{
				if ( lpp != (Link **)0 )
				{
					free((char *)lpp);
					ChangeState = true;
				}
			}
			else
			if ( lpp == (Link **)0 )
			{
				free(cp);
				ChangeState = true;
			}
			else
			if ( strcmp(cp, (char *)lpp) != STREQUAL )
			{
				free(cp);
				free((char *)lpp);
				ChangeState = true;
			}
			continue;
			
		case ct_domain:
			SetDomains(np, cp);
			continue;

		case ct_domhier:
			SetHierarchy((char *)np, &DomHier);
			continue;

		case ct_remdom:
			if
			(
				(lp = Lookup((char *)np, DomainHash)) != (Entry *)0
				&&
				(lp->e_states & S_FOUND)
			)
			{
				DomainCount--;
				if ( lp->e_states & S_OURDOM )
					ChangeState = true;
				lp->e_states = 0;
			}
			continue;

		case ct_handler:
			if ( (lp = Lookup((char *)np, DomainHash)) == (Entry *)0 )
			{
				errp = "unknown domain in";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}
			if ( lp->e_handler != NULLSTR )
				free(lp->e_handler);
			lp->e_handler = cp==NULLSTR?cp:newstr(cp);
			continue;

		case ct_spooler:
			if ( cp != NULLSTR && strlen(cp) > TOKEN_SIZE )
			{
				errp = "spooler name too long in";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}
			if ( np->e_node->n_handlers == (Hndl *)0 )
				np->e_node->n_handlers = Talloc(Hndl);
			else
			if ( np->e_node->n_handlers->h_spooler != NULLSTR )
				free(np->e_node->n_handlers->h_spooler);

			np->e_node->n_handlers->h_spooler = cp==NULLSTR?cp:newstr(cp);
			continue;

		case ct_connector:
			if ( cp != NULLSTR && strlen(cp) > TOKEN_SIZE )
			{
				errp = "connector name too long in";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}
			if ( np->e_node->n_handlers == (Hndl *)0 )
				np->e_node->n_handlers = Talloc(Hndl);
			else
			if ( np->e_node->n_handlers->h_connector != NULLSTR )
				free(np->e_node->n_handlers->h_connector);

			np->e_node->n_handlers->h_connector = cp==NULLSTR?cp:newstr(cp);
			continue;

		case ct_caller:
			if ( cp != NULLSTR && strlen(cp) > TOKEN_SIZE )
			{
				errp = "caller name too long in";
				Error(errfmt, errp, cmdp->c_name, Line);
				continue;
			}
			if ( np->e_node->n_handlers == (Hndl *)0 )
				np->e_node->n_handlers = Talloc(Hndl);
			else
			if ( np->e_node->n_handlers->h_caller != NULLSTR )
				free(np->e_node->n_handlers->h_caller);

			np->e_node->n_handlers->h_caller = cp==NULLSTR?cp:newstr(cp);
			continue;

		case ct_comment:
			if ( cp != NULLSTR )
			{
				if ( strlen(cp) > TOKEN_SIZE )
				{
					errp = "comment too long in";
					Error(errfmt, errp, cmdp->c_name, Line);
					continue;
				}

				if ( strchr(cp, C_COMMENT) != NULLSTR )
				{
					errp = "illegal character in";
					Error(errfmt, errp, cmdp->c_name, Line);
					continue;
				}
			}

			if ( home )
			{
				if ( cp == NULLSTR )
				{
					if ( np->e_node->n_comment != NULLSTR )
						ChangeState = true;
				}
				else
				if
				(
					np->e_node->n_comment == NULLSTR
					||
					strcmp(cp, np->e_node->n_comment) != STREQUAL
				)
					ChangeState = true;
			}

			if ( np->e_node->n_comment != NULLSTR )
				free(np->e_node->n_comment);

			np->e_node->n_comment = cp==NULLSTR?cp:newstr(cp);
			continue;
		}

		if ( warn )
			Warn(errfmt, errp, cmdp->c_name, Line);
	}
}


/*
**	Read a command line, ignoring comments and empty lines.
*/

static int
getline(fd, buf)
	FILE *		fd;
	char *		buf;
{
	register char *	cp;
	register int	c;
	register char *	ep;
	register bool	ignore;
	register bool	empty;

	if ( feof(fd) )
		return EOF;

	ep = buf + LINE_LENGTH-1;

	for ( ;; )
	{
		cp = buf;
		empty = true;
		ignore = false;

		while ( (c = getc(fd)) != EOF )
		{
			if ( c == '\n' )
			{
				Line++;
				break;
			}

			if ( ignore )
				continue;

			if ( c == COMMENT_C )
			{
				ignore = true;
				continue;
			}

			if ( cp < ep )
			{
				*cp++ = c;
				empty = false;
			}
			else
			{
				Error("command too long at line %d", Line);
				ignore = true;
			}
		}

		if ( !empty )
		{
			*cp = '\0';
			return 0;
		}

		if ( c == EOF )
			return EOF;
	}
}


/*
**	Decode state flags from string.
*/

static int	FlagLen;

static bool
set_states(sp, cp, mask, home)
	States *	sp;
	register char *	cp;
	States		mask;
	bool		home;
{
	register Flgp	flgp;
	register char *	ep;
	States		old = *sp;
	bool		set;
	bool		down = false;
	char *		errfmt = "%s flag \"%s\" at line %d";

	do
	{
		set = true;

		switch ( *cp )
		{
		case '-':	set = false;
		case '+':	cp++;
		}

		if ( (ep = strchr(cp, FLAG_C)) != NULLSTR )
		{
			FlagLen = ep-cp;
			*ep++ = '\0';
		}
		else
			FlagLen = strlen(cp);

		if ( (flgp = (Flgp)bsearch(cp, (char *)Flags, NFLGS, FLGZ, cmpflg)) == (Flgp)0 )
		{
			Error(errfmt, "unknown", cp, Line);
			continue;
		}

		if ( !(flgp->f_state & mask) )
		{
			Error(errfmt, "illegal", cp, Line);
			continue;
		}

		if ( set )
		{
			*sp |= flgp->f_state;

			if ( flgp->f_state == S_DOWN )
				down = true;

			if ( flgp->f_state == S_CALL )
				*sp |= S_INTERMITTENT;
		}
		else
			*sp &= ~flgp->f_state;
	}
		while ( (cp = ep) != NULLSTR );

	if ( home && ((old ^ *sp) & FOREIGN_FLAGS ) )
		ChangeState = true;

	return (bool)(home && down && (*sp & S_LAN));
}



/*
**	If one of the nodes is Home,
**	flag all links marked S_LAN from other node as being down.
*/

static void
fixlan(np1, np2)
	Entry *			np1;
	Entry *			np2;
{
	register Entry *	op;
	register Link *		lp;

	Trace3(2, "fixlan between \"%s\" and \"%s\"", np1->e_name, np2->e_name);

	if ( np1 == Home )
		op = np2;
	else
	if ( np2 == Home )
		op = np1;
	else
		return;

	for ( lp = op->e_node->n_l_first ; lp != (Link *)0 ; lp = lp->l_next )
		if ( lp->l_data->d_states & S_LAN )
			lp->l_data->d_states |= S_DOWN;
}



int
cmpcom(cp, cmdp)
	char *	cp;
	char *	cmdp;
{
	Trace3(3, "cmpcom \"%s\" <> \"%s\"", cp, ((Cmdp)cmdp)->c_name);
	return strcmp(cp, ((Cmdp)cmdp)->c_name);
}



int
cmpflg(cp, flgp)
	char *	cp;
	char *	flgp;
{
	Trace3(3, "cmpflg \"%s\" <> \"%s\"", cp, ((Flgp)flgp)->f_name);
	return strncmp(cp, ((Flgp)flgp)->f_name, max(FlagLen, ((Flgp)flgp)->f_min));
}



/*
**	Turn alias list into a string.
*/

static char *
AliasString(ap)
	Entry *	ap;
{
	register Entry *	ep;
	register char *		cp;
	register int		len;
	char *			str;

	if ( (ep = ap) == (Entry *)0 )
		return NULLSTR;

	len = 0;
	do
	{
		len += strlen(ep->e_name) + 1;
	}
		while ( (ep = ep->e_next) != (Entry *)0 );

	str = cp = Malloc(len);

	ep = ap;
	do
	{
		cp = strcpyend(cp, ep->e_name);
		*cp++ = LINK_C;
	}
		while ( (ep = ep->e_next) != (Entry *)0 );

	*--cp = '\0';

	return str;
}



/*
**	Add alias from parameter list.
**	Avoid non-existent domain/node names;
**	a null list means remove.
*/

static char *
SetAlias(np, cp)
	char *		np;
	register char *	cp;
{
	register Entry *ap;
	register char *	dp;

	Trace3(2, "SetAlias \"%s\" for address \"%s\"", np, cp);

	if ( cp == NULLSTR )
	{
		if
		(
			(ap = Lookup(np, AliasHash)) != (Entry *)0
			&&
			(ap->e_states & S_FOUND)
		)
		{
			free(ap->e_value);
			ap->e_value = NULLSTR;
			ap->e_states &= ~S_FOUND;
			AliasCount--;
		}

		return NULLSTR;
	}

	if ( (dp = strrchr(cp, HIER_C)) != NULLSTR )
	{
		if ( Lookup(dp+1, DomainHash) == (Entry *)0 )
			return "need valid domain name for";
	}
	else
		if ( Lookup(cp, NodeHash) == (Entry *)0 )
			return "need valid node name for";

	ap = Enter(np, AliasHash);

	if ( !(ap->e_states & S_FOUND) )
	{
		AliasCount++;
		ap->e_states |= S_FOUND;
	}
	else
		free(ap->e_value);

	ap->e_value = newstr(cp);

	return NULLSTR;
}



/*
**	Set domains for node
*/

void
SetDomains(ep, cp)
	Entry *			ep;
	register char *		cp;
{
	register Entry *	dp;
	register char *		hp;
	register DomHead *	dhp;

	Trace3(2, "SetDomains \"%s\" for node \"%s\"", cp, ep->e_name);

	dhp = &ep->e_node->n_domains;

	if ( ep == Home )
		hp = DomString(dhp, DOMAIN_C);
	else
		hp = NULLSTR;

	ClearDomains(ep, true);

	if ( cp == NULLSTR )
	{
		if ( hp != NULLSTR )
		{
			free(hp);
			ChangeState = true;
		}
		return;
	}

	if ( ep == Home && (hp == NULLSTR || strcmp(cp, hp) != STREQUAL) )
		ChangeState = true;

	if ( hp != NULLSTR )
		free(hp);

	do
	{
		if ( (hp = strchr(cp, DOMAIN_C)) != NULLSTR )
			*hp++ = '\0';

		dp = AddDomain(cp, dhp, S_FOUND);

		if ( ep == Home )
			dp->e_states |= S_OURDOM;	/* Mark domain valid for hierarchy */

		dp->e_states &= ~(S_OTHDOM|S_TOPDOM);
	}
	while
		( (cp = hp) != NULLSTR );
}



/*
**	Set hierarchy for local domains
*/

void
SetHierarchy(cp, dhp)
	char *			cp;
	register DomHead *	dhp;
{
	register Domain *	dp;
	register char *		hp;
	States			states;

	Trace3(2, "SetHierarchy (%s) to \"%s\"", (dhp==&DomHier)?"home":"foreign", cp);

	if ( dhp == &DomHier )
	{
		states = S_HIERDOM;
		hp = DomString(dhp, HIER_C);
	}
	else
	{
		states = 0;
		hp = NULLSTR;
	}

	while ( (dp = dhp->d_head) != (Domain *)0 )
	{
		dp->d_entry->e_states &= ~(S_HIERDOM|S_TOPDOM);
		dhp->d_head = dp->d_next;
		free((char *)dp);
	}

	dhp->d_tail = &dhp->d_head;
	dhp->d_last = (Domain *)0;
	dhp->d_count = 0;

	if ( cp == NULLSTR )
	{
		if ( hp != NULLSTR )
		{
			free(hp);
			ChangeState = true;
		}
		return;
	}

	if ( dhp == &DomHier && (hp == NULLSTR || strcmp(cp, hp) != STREQUAL) )
		ChangeState = true;

	if ( hp != NULLSTR )
		free(hp);

	do
	{
		if ( (hp = strchr(cp, HIER_C)) != NULLSTR )
			*hp++ = '\0';

		(void)AddDomain(cp, dhp, states);
	}
	while
		( (cp = hp) != NULLSTR );
}
