/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)state.c	1.20 85/06/26
*/

/*
**	Update the status and statistics file.
*/

#define	FILE_CONTROL
#define	STAT_CALL

#include	"global.h"
#include	"command.h"
#include	"debug.h"

#include	"daemon.h"
#include	"driver.h"
#include	"Stream.h"
#include	"AQ.h"


static int	StateFd;		/* Initialised by "ReadState()" */



void
Update(flag)
	StCom		flag;
{
	register bool	all_done;
	register Str_p	strp;
	register int	etime;
	static short	idle_time;

	switch ( flag )
	{
	case up_date:
		if ( NNstate.lasttime == LastTime )
			return;
		break;

	case up_error:
		NNstate.procstate = PROC_ERROR;
		break;

	case up_finish:
		NNstate.procid = 0;
		break;

	case up_opening:
		NNstate.procstate = PROC_OPENING;
		break;

	case up_force:
		break;
	}
	
	Trace2(1, "Update(%d)", (int)flag);

	UpdateTime = 0;
	NNstate.thistime = LastTime;

	if
	(
		(
			StateFd == SYSERROR
			&&
			(StateFd = creat(Statusfile, 0644)) == SYSERROR
		)
		||
		lseek(StateFd, (long)0, 0) == SYSERROR
		||
		write(StateFd, (char *)&NNstate, sizeof NNstate) != sizeof NNstate
	)
	{
		Syserror(Statusfile);
		return;
	}

	if ( flag == up_finish || flag == up_opening || flag == up_error )
		return;

	if ( (etime = LastTime - NNstate.lasttime) > 0 )
	{
		register int	speed;

		Trace2(2, "Update etime %d", etime);

		if
		(
			etime > 9
			&&
			(speed = (outByteCount+inByteCount)/etime) > NNstate.maxspeed
		)
			NNstate.maxspeed = speed;

		NNstate.allbytes += inByteCount + outByteCount;
		inByteCount = 0;
		outByteCount = 0;
		NNstate.lasttime = LastTime;

		if ( (Receiving || Transmitting) && NNstate.linkstate != LINK_DOWN )
			NNstate.activetime += etime;
	}

	all_done = true;
	Receiving = 0;

	for ( strp = &inStreams[Fstream] ; strp < &inStreams[Nstreams] ; strp++ )
	{
		if ( strp->str_state == STR_ACTIVE )
			Receiving++;
		else
		if ( strp->str_state == STR_EMPTY )
			continue;

		all_done = false;
	}

	Transmitting = 0;

	for ( strp = &outStreams[Fstream] ; strp < &outStreams[Nstreams] ; strp++ )
	{
		if ( strp->str_state == STR_ACTIVE )
			Transmitting++;
		else
		if ( strp->str_state == STR_EMPTY )
			continue;

		all_done = false;
	}

	if ( (Receiving || Transmitting) && NNstate.linkstate != LINK_DOWN )
	{
		idle_time = 0;
		NNstate.procstate = PROC_RUNNING;
	}
	else
	if ( (idle_time += etime) > 0 )
	{
		static short	done_time;

		Trace3(1, "Update IDLE time %d, done_time %d", idle_time, done_time);

		NNstate.procstate = PROC_IDLE;

		if ( all_done )
		{
			done_time += etime;

			if ( BatchMode && done_time >= BatchTime )
				finish(0);
		}
		else
			done_time = 0;
	}

	SetSpeed();
}



/*
**	Read in the status file and queue function to make consistent.
**	Return old process id.
*/

int
ReadState()
{
	register int	count;
	struct stat	statb;

	qAction(FixStreams, (AQarg)0);

	for
	(
		count = 0
		;
		(StateFd = open(Statusfile, O_RDWR)) == SYSERROR
		;
	)
		if ( ++count > 3 )
			goto out;
		else
			(void)sleep(2);

	while ( (count = StateFd) <= 2 )
	{
		if ( (StateFd = dup(count)) == SYSERROR )
		{
			(void)close(count);
			goto out;
		}

		(void)close(count);
		(void)open("/dev/null", O_RDWR);
	}

	for
	(
		count = 0
		;
		fstat(StateFd, &statb) == SYSERROR
		||
		statb.st_size != sizeof NNstate
		||
		read(StateFd, (char *)&NNstate, sizeof NNstate) != sizeof NNstate
		;
	)
		if ( ++count > 3 )
			goto out;
		else
			(void)sleep(2);

	if ( strcmp(NNstate.version, StreamSCCSID) == STREQUAL )
		return NNstate.procid;

out:
	(void)strclr((char *)&NNstate, sizeof NNstate);

	(void)strcpy(NNstate.version, StreamSCCSID);

	if ( StateFd != SYSERROR )
	{
		(void)close(StateFd);
		StateFd = SYSERROR;
	}

	return 0;
}



/*
**	Make state self consistent
*/

void
FixStreams()
{
	register Str_p	strp;
	register int	i;

	Trace1(1, "FixStreams");

	for ( strp = inStreams, i = 0 ; strp < &inStreams[NSTREAMS] ; i++, strp++ )
	{
		strp->str_flags = 0;

		if ( strp->str_recv.rh_id[0] != '\0' )
		{
			strp->str_fname = concat
					  (
						  Workdir
						 ,strp->str_recv.rh_id
						 ,NULLSTR
					  );

			if ( (strp->str_fd = open(strp->str_fname, O_WRITE)) == SYSERROR )
			{
				if ( strp->str_state == STR_ENDING )
					strp->str_state = STR_AGAIN;
				else
				if ( strp->str_state != STR_ENDED )
					goto inbad;
				strp->str_fd = 0;
			}
			else
			{
				Trace4
				(
					2,
					"Found active message \"%s\" for stream %d, id \"%s\"",
					strp->str_fname,
					i,
					strp->str_id
				);

				strp->str_posn = lseek(strp->str_fd, (long)0, 2);

				strp->str_state = STR_ACTIVE;
			}
		}
		else
		{
			strp->str_fname = NULLSTR;
inbad:
			strp->str_fd = 0;

			if ( i < Fstream || i >= Nstreams )
				strp->str_state = STR_INACTIVE;
			else
				strp->str_state = STR_IDLE;
		}
	}

	for ( strp = outStreams, i = 0 ; strp < &outStreams[NSTREAMS] ; i++, strp++ )
	{
		strp->str_descr.dq_data = NULLSTR;
		strp->str_descr.dq_first = (DQl_p)0;
		strp->str_fd = 0;

		if
		(
			strp->str_id[0] != '\0'
			&&
			RdCommand(i)
		)
		{
			strp->str_state = STR_START;
		}
		else
		{
			strp->str_id[0] = '\0';

			if ( i < Fstream || i >= Nstreams )
				strp->str_state = STR_INACTIVE;
			else
				strp->str_state = STR_IDLE;
		}
	}

	Update(up_force);
}



/*
**	Execute change-state handler.
**
**	Attempt to dampen rapid changes.
*/

void
NewState(newstate, force)
	int		newstate;
	bool		force;
{
	static Time_t	change_time;
	static int	real_state	= -1;

	Trace4(1, "NewState(%d, %d) real_state=%d", newstate, force, real_state);

	if ( newstate == NNstate.linkstate )
	{
		if
		(
			newstate == real_state
			||
			(
				newstate == LINK_DOWN
				&&
				!force
				&&
				(
					BatchMode
					||
					(LastTime - change_time) < DOWNTIME
				)
			)
		)
		{
			Trace2(2, "NewState change_time=%lu", change_time);
			return;
		}
	}
	else
	{
		NNstate.linkstate = newstate;

		Update(up_force);

		if ( !force )
		{
			change_time = LastTime;
			return;
		}
	}

	if ( newstate == real_state )
		return;

	RunState((AQarg)newstate);
	real_state = newstate;
	change_time = LastTime;
}
