/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)children.c	1.15 85/07/20
*/

/*
**	Routines that handle child processes
*/

#define	STDIO

#include	"global.h"
#include	"command.h"
#include	"debug.h"
#include	"messagename.h"
#include	"spool.h"

#define	BM_DATA
#include	"bad.h"
#include	"daemon.h"
#include	"Stream.h"
#include	"driver.h"
#include	"AQ.h"

#include	<setjmp.h>
#include	<signal.h>


extern jmp_buf	QTimeouts;
extern bool	UseFd1;


#define	MINWAIT		3	/* Minimum seconds between "wait"s */
#define	MAXWAIT		60	/* Maximum seconds between "wait"s */
#define	MAXPROCWAIT	(30*60)	/* Maximum wait time for a process */
#define	STATEWAIT	30	/* No need to be interested in the state-changer */

static int	BadPid;
static Time_t	NSstarted;
static int	NSpid;
static bool	NewWait;
static int	CPid;
static int	RoutePid;
static char *	Router	= ROUTER;
static Time_t	Started[NSTREAMS];

void		WaitError(), StatusError();
static void	start_router();



/*
**	Terminate reception, pass message to handler.
*/

void
EndMessage(chan)
	AQarg		chan;
{
	register Str_p	strp = &inStreams[(int)chan];
	register int	i;
	char *		args[8];
	char		elapsedtime[16];

	Trace3(1, "EndMessage for channel %d state %d", (int)chan, strp->str_state);

	(void)fflush(stderr);

	if ( RouteDir != NULLSTR )
	{
		static char *	routename;
		static int	id;

		strp->str_fd = 0;

		if ( routename == NULLSTR )	
		{
			routename = concat(RouteDir, "/01234567890123", NULLSTR);
			for ( i = 0 ; i < NSTREAMS ; i++ )
			{
				MesgNumber += inStreams[i].str_messages;
				MesgNumber += outStreams[i].str_messages;
			}
		}

		while
		(
			access
			(
				EncodeName
				(
					routename,
					(strp->str_flags & STR_DUP) ? (char)MESG_DUP : (char)0,
					(char)id,	/* May be used to distinguish consecutive runs */
					(LastTime > strp->str_time) ? LastTime - strp->str_time : 0L
				),
				0
			)
			!= SYSERROR
		)
			if ( ++id > VC_MASK )
				Syserror("Can't creat \"%s\"", routename);

		while ( link(strp->str_fname, routename) == SYSERROR )
			Syserror("Can't link \"%s\" to \"%s\"", strp->str_fname, routename);

		while ( unlink(strp->str_fname) == SYSERROR )
			Syserror("Can't unlink \"%s\"", strp->str_fname);

		SndEOMA((int)chan);

		NNstate.allmessages++;
		strp->str_messages++;
		strp->str_bytes += strp->str_posn;
		strp->str_flags = 0;

		Update(up_force);

		if ( RoutePid == 0 )
			start_router();
		
		return;
	}

	switch ( strp->str_fd = fork() )
	{
	case SYSERROR:
		Report1("EndMessage cannot fork");
		qAction(EndMessage, chan);	/* Try again later */
		return;
	
	case 0:
		closeall();

		i = 0;
		args[i++] = MessageHandler;

		if ( LastTime > strp->str_time )
		{
			(void)sprintf(elapsedtime, "-t%lu", LastTime - strp->str_time);
			args[i++] = elapsedtime;
		}

		args[i++] = concat("-l", LinkDir, NULLSTR);
		args[i++] = concat("-h", HomeNode, NULLSTR);

		if ( strp->str_flags & STR_DUP )
			args[i++] = "-D";

		args[i++] = strp->str_fname;
		args[i++] = NULLSTR;

		for ( ;; )
		{
			execve(args[0], args, StripEnv());
			Syserror("cannot execve handler \"%s\"", args[0]);
		}
	}
	
	Started[(int)chan] = LastTime;

	if ( !Waiting )
	{
		Waiting = true;
		qAction(WaitHandler, (AQarg)(LastTime+MINWAIT));
	}
	else
		NewWait = true;

	Trace2(1, "EndMessage started process %d", strp->str_fd);
}



/*
**	Wait for handler to terminate;
**	if found, send an EOM_ACCEPT.
*/

void
WaitHandler(delay)
	AQarg		delay;
{
	register Str_p	strp;
	register int	chan;
	register int	procs;
	int		status;

	Trace2(1, "WaitHandler %ld", (Time_t)delay - LastTime);

	if ( (Time_t)delay >= LastTime )
	{
		if ( NewWait )
		{
			NewWait = false;

			delay = (AQarg)(LastTime+MINWAIT);
		}

		qAction(WaitHandler, delay);
		Waiting = true;
		return;
	}

	NewWait = false;

	for
	(
		strp = &inStreams[Fstream], chan = Fstream, procs = 0 ;
		chan < Nstreams ;
		chan++, strp++
	)
		if ( strp->str_state == STR_ENDING && strp->str_fd > 0 )
			procs++;
	
	if ( NSpid != 0 )
		procs++;
	else
	if ( procs == 0 )
	{
		Waiting = false;
		return;
	}

	do
	{
		CPid = 0;

		if ( !setjmp(QTimeouts) )
		{
			ALARM_ON(2);

			CPid = wait(&status);
			
			ALARM_OFF();

			if ( CPid == SYSERROR )
			{
				char *	fmt;

				if ( (CPid = NSpid) > 0 )
				{
					fmt = "newstatehandler (pid %d) disappeared!";
					NSpid = 0;
				}
				else
				{
					fmt = "handler (pid %d) for channel %d disappeared!";

					for
					(
						strp = &inStreams[Fstream], chan = Fstream ;
						chan < Nstreams ;
						chan++, strp++
					)
						if ( strp->str_state == STR_ENDING && (CPid = strp->str_fd) > 0 )
						{
							strp->str_fd = 0;
							if ( access(strp->str_fname, 0) == SYSERROR )
							{
								Warn(fmt, CPid, chan);
								goto ok;
							}

							qAction(EndMessage, (AQarg)chan);
							break;
						}

					if ( chan == Nstreams && BadPid != 0 )
					{
						BadPid = 0;
						Waiting = false;
						return;
					}
				}

				Warn(fmt, CPid, chan);
				procs--;
				continue;;
			}
		}
		else
		if ( CPid == 0 )
		{
			/*
			**	Timed out, try later
			*/

			for
			(
				strp = &inStreams[Fstream], chan = Fstream, procs = MAXWAIT ;
				chan < Nstreams ;
				chan++, strp++
			)
				if ( strp->str_state == STR_ENDING && strp->str_fd > 0 )
				{
					if ( (CPid = LastTime-Started[chan]) > MAXPROCWAIT )
					{
						Started[chan] = LastTime;
						WaitError(MessageHandler, CPid, strp->str_fd, chan);
					}

					if ( CPid < procs )
						procs = CPid;
				}
			
			if ( NSpid != 0 )
			{
				if ( (CPid = LastTime-NSstarted) > MAXPROCWAIT )
				{
					NSstarted = LastTime;
					WaitError(NewstateHandler, CPid, NSpid, 0);
				}

				if ( CPid < procs )
					procs = CPid;
			}

			if ( procs < MINWAIT || (!Transmitting && !Receiving) )
				procs = MINWAIT;

			qAction(WaitHandler, (AQarg)(LastTime+procs));
			Waiting = true;
			return;
		}

		Trace2(1, "WaitHandler found process %d", CPid);

		if ( CPid == NSpid )
		{
			NSpid = 0;

			if ( status )
				StatusError(NewstateHandler, "", status);

			procs--;
			continue;
		}

		if ( CPid == RoutePid )
		{
			RoutePid = 0;
			StatusError(Router, "", status);
			continue;
		}
		
		for
		(
			strp = &inStreams[Fstream], chan = Fstream ;
			chan < Nstreams ;
			chan++, strp++
		)
			if ( strp->str_state == STR_ENDING && CPid == strp->str_fd )
			{
				strp->str_fd = 0;

				if ( status )
				{
					qAction(EndMessage, (AQarg)chan);

					StatusError(MessageHandler, strp->str_fname, status);

					procs--;
					break;
				}

ok:
				SndEOMA(chan);

				strp->str_messages++;
				NNstate.allmessages++;
				strp->str_bytes += strp->str_posn;
				strp->str_flags = 0;

				Update(up_force);
				
				procs--;
				break;
			}

		if ( chan == Nstreams && BadPid == 0 )
			Report2("Collected orphan %d", CPid);
	}
		while ( procs > 0 || BadPid != 0 );

	Waiting = false;
}



void
StatusError(prog, file, status)
	char *	prog;
	char *	file;
	int	status;
{
	Error
	(
		 "\"%s %s\" bad exit status:%d signal:%d%s"
		,prog
		,file
		,(status>>8)&0xff
		,status&0x7f
		,status&0x80?" (core dumped)":""
	);
}



void
WaitError(prog, wait, pid, chan)
	char *	prog;
	int	wait;
	int	pid;
	int	chan;
{
	Error
	(
		"overlong wait (%d secs.) for process %d, \"%s\" on channel %d",
		wait,
		pid,
		prog,
		chan
	);
}


/*
**	Run state update program
*/

void
RunState(newstate)
	AQarg		newstate;
{
	register int	i;
	char *		args[7];

	while ( NSpid != 0 )
		WaitHandler((AQarg)0);
	
	(void)fflush(stderr);

	switch ( NSpid = fork() )
	{
	case SYSERROR:
		NSpid = 0;
		Report1("RunState cannot fork");
		qAction(RunState, newstate);
		return;

	case 0:
		closeall();

		i = 0;
		args[i++] = NewstateHandler;
		args[i++] = (int)newstate==LINK_DOWN?"-D":"-U";
		args[i++] = concat("-h", HomeNode, NULLSTR);
		if ( BatchMode )
			args[i++] = "-N";
#		ifdef	DEBUGPROGS
		if ( Traceflag )
			args[i++] = Traceflag==1?"-T1":"-T2";
#		endif	DEBUGPROGS
		args[i++] = LinkDir;
		args[i++] = NULLSTR;

		for ( ;; )
		{
			(void)execve(args[0], args, StripEnv());
			Syserror("can't execve \"%s\"", args[0]);
		}
	}

	NSstarted = LastTime;

	if ( !Waiting )
	{
		Waiting = true;
		qAction(WaitHandler, (AQarg)(LastTime+STATEWAIT));
	}

	Trace2(1, "RunState started process %d", NSpid);
}



/*
**	Move offending command file to bad directory,
**	and set bad message handler onto it.
**	Then reset the channel and queue NewMessage.
*/

void
BadMessage(chan, reason)
	AQarg		chan;
	BMReason	reason;
{
	register Str_p	strp = &outStreams[(int)chan];
	char *		mesg;

	Trace3(1, "BadMessage for channel %d state %d", (int)chan, strp->str_state);

	mesg = BMExplanations[(int)reason];

	Report3("bad message \"%s\" - %s", strp->str_id, mesg);

	if ( reason != bm_nodata && access(strp->str_id, 0) == 0 )
	{
		char *	badname = concat(BADDIR(), strp->str_id, NULLSTR);
		char *	args[7];

		while
		(
			access(badname, 0) != 1
			&&
			link(strp->str_id, badname) == SYSERROR
		)
			Syserror("cannot link \"%s\" to \"%s\"", strp->str_id, badname);

		Report3("bad message \"%s\" moved to \"%s\"", strp->str_id, badname);

		/*
		**	'NewMessage' will unlink bad command file.
		*/

		(void)fflush(stderr);

		switch ( BadPid = fork() )
		{
		case SYSERROR:
			BadPid = 0;
			Report1("BadMessage cannot fork");
			qAction(BadMessage, chan);		/* Try again */
			free(badname);
			return;

		case 0:
			closeall();

			args[0] = BadHandler;
			args[1] = concat("-d", badname, NULLSTR);
			args[2] = concat("-l", LinkDir, NULLSTR);
			args[3] = concat("-i", Name, NULLSTR);
			args[4] = concat("-h", HomeNode, NULLSTR);
			args[5] = concat("-e", mesg, NULLSTR);
			args[6] = NULLSTR;

			for ( ;; )
			{
				execve(BadHandler, args, StripEnv());
				Syserror("cannot execve handler \"%s\"", BadHandler);
			}
		}

		free(badname);

		if ( !Waiting )
		{
			Waiting = true;
			qAction(WaitHandler, (AQarg)(LastTime+STATEWAIT));
		}

		Trace2(1, "BadMessage started process %d", BadPid);
	}
	else
		Report2("bad message \"%s\" non-existent", strp->str_id);

	strp->str_state = STR_ERROR;
}


/*
**	Start routing daemon
*/

static void
start_router()
{
	char *	args[6];
	char	pid[12];

	for ( ;; )
	{
		switch ( RoutePid = fork() )
		{
		case SYSERROR:
			RoutePid = 0;
			Syserror("Can't fork");
			continue;
		
		case 0:
			closeall();

			if ( UseFd1 )
				(void)signal(SIGHUP, SIG_IGN);

			args[0] = Router;
			args[1] = concat("-d", RouteDir, NULLSTR);
			args[2] = concat("-h", HomeNode, NULLSTR);
			args[3] = concat("-l", LinkDir, NULLSTR);
			sprintf(pid, "-p%d", Pid);
			args[4] = pid;
			args[5] = NULLSTR;

			for ( ;; )
			{
				execve(Router, args, StripEnv());
				Syserror("cannot execve routing daemon \"%s\"", Router);
			}
		}

		break;
	}
}
