/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)changestate.c	1.24 85/07/04";

/*
**	Change of state processor, invoked by daemons.
**
**	Broadcast up/down/newhost message to statehandlers.
*/

#define	FILE_CONTROL
#define	LOCKING
#define	RECOVER
#define	STDIO

#include	"global.h"

#include	"address.h"
#include	"debug.h"
#include	"handlers.h"
#include	"header.h"
#include	"spool.h"
#include	"state.h"
#include	"sub_proto.h"


/*
**	Parameters set from arguments.
*/

char *	ChangeNode;		/* Name of node changing status */
bool	Down;			/* Link down */
char *	HomeNode;		/* Name of this node */
bool	Intermittent;		/* This is for an intermittent connection */
char *	LinkNode;		/* Link involved (default ChangeNode) */
char *	Name;			/* Program invoked name */
bool	NoTimeout;		/* Set to prevent timeout on message */
char *	OrgNode;		/* Origin Node (default HomeNode) */
int	Traceflag;		/* Global tracing control */
bool	Up;			/* Link up */

/*
**	Miscellaneous
*/

bool	AtHome;			/* Origin is us, (otherwise being used by gateway) */
NodeLink LinkD;			/* Route info. from FindNode() */
int	MesgFd;			/* File descriptor of Message */
int	Pid;			/* Used by UniqueName() in Exec...() */
char	ProtoType[2];		/* Header protocol type */
Time_t	Time;			/* Used by UniqueName() in Exec...() */
char	Ttd[TIME_SIZE+1];	/* "Time-to-die" for message */

char *	Message	= WORKDIR(state..message);

#define	WEEK	(1*7*24*60*60)
#define	free(A)

bool	FindLink();
void	finish(), newhost(), newstate(), send();
FILE *	start_state();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	if ( getuid() != ACSNETUID )
		Error("No permission");

	Pid = getpid();
	Time = time((long *)0);

	while ( (MesgFd = creat(UniqueName(Message, (long)0, Time), 0600)) == SYSERROR )
		Syserror("Can't creat \"%s\"", Message);

	DODEBUG(if(MesgFd==2)Fatal("MesgFd==2"));

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'D':
					Down = true;
					continue;

				case 'N':
					Intermittent = true;
					continue;

				case 'P':
					NoTimeout = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'U':
					Up = true;
					continue;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					goto break2;

				case 'o':
					OrgNode = ++*argv;
					goto break2;

				default:
					Warn("unrecognised flag '%c'", c);
					goto break2;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
			ChangeNode = *argv;
	}

	if ( !Up && !Down || Up && Down )
		Error("need either -D or -U");

	if ( ChangeNode == NULLSTR )
		Error("need name of node");

	if ( LinkNode == NULLSTR || LinkNode[0] == '\0' )
		LinkNode = ChangeNode;

	if ( HomeNode == NULLSTR )
		HomeNode = NodeName();

	if ( OrgNode == NULLSTR )
	{
		OrgNode = HomeNode;
		AtHome = true;
	}

	/*
	**	If node is unknown, add it to statefile and broadcast the new statefile;
	**	if state has changed, broadcast a state-change message.
	*/

	if
	(
		AtHome
		? !FindLink(ChangeNode)
		: !FindNode(ChangeNode, pt_msg, &LinkD)
	)
	{
		newhost();
	}
	else
	{
		if
		(
			Intermittent
			&&
			(LinkD.nl_flags & (S_INTERMITTENT|S_MSG)) == (S_INTERMITTENT|S_MSG)
		)
		{
			(void)unlink(Message);
			return 0;
		}

		newstate();
	}

	send();

	if ( !AtHome )
		return 0;

	Recover(ert_return);

	/*
	**	Try re-routing messages queued from node just down,
	**	or to node just up.
	*/

	ReadRoute();	/* Re-load changed routing tables */

	if ( Up )
		ReRoute(REROUTEDIR(.));
	else
		ReRoute(concat(SPOOLDIR(), LinkNode, NULLSTR));

	return 0;
}



/*
**	Called from the errors routines to cleanup
*/

void
finish(error)
	int		error;
{
	register int	i;

	if ( Traceflag == 0 )
		(void)unlink(Message);

	for ( i = 0 ; i < 20 ; i++ )
		UnLock(i);

	(void)exit(error);
}



/*
**	Run state program to add new node and write state info to message.
**
**	Node is configured as a normal node with MTP allowed on the link,
**	if necessary, this should be overridden in the commands file.
**
**	The new state info. is broadcast with a request for a reply.
*/

void
newhost()
{
	register FILE *	fd;
	register char *	errs;
	VarArgs		va;

	fd = start_state(&va);

	Report3("New host \"%s\" connected to \"%s\"", ChangeNode, OrgNode);

	(void)fprintf(fd, "add %s\n", ChangeNode);
	(void)fprintf(fd, "link %s,%s msg%s\n", ChangeNode, OrgNode, Intermittent?",int":"");

	if ( (errs = ExOPipeClose(fd)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}
	
	if ( ReadRoute() )
		(void)FindNode(ChangeNode, pt_msg, &LinkD);

	(void)sprintf(Ttd, "%lu", (ulong)WEEK);
	HdrTtd = Ttd;

	HdrEnv = MakeEnv
		 (
			ENV_NORET, NULLSTR,
			ENV_STATE, NULLSTR,
			ENV_NO_AUTOCALL, NULLSTR,
			AtHome?ENV_ENQ:NULLSTR, NULLSTR,
			NULLSTR
		 );
}



/*
**	Run state program to change link status and write state info to message.
**
**	Set up message header for sending state change to statehandlers.
*/

void
newstate()
{
	register FILE *	fd;
	char *		errs;
	VarArgs		va;

	fd = start_state(&va);

	(void)fprintf
	      (
		fd,
		"%s %s,%s msg,%cdown%s\n",
		AtHome?"link":"flag",
		ChangeNode,
		OrgNode,
		Up?'-':'+',
		Intermittent?",int":""
	      );

	if ( (errs = ExOPipeClose(fd)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}
	
	(void)sprintf(Ttd, "%lu", NoTimeout?(ulong)WEEK:(ulong)CHST_TTD);
	HdrTtd = Ttd;

	HdrEnv = MakeEnv
		 (
			ENV_NORET, NULLSTR,
			ENV_STATE, NULLSTR,
			Up?ENV_UP:ENV_DOWN, ChangeNode,
			Up?NULLSTR:ENV_NOTNODE, ChangeNode,
			NULLSTR
		 );
}



/*
**	Send Message.
**
**	Write Message, and call receiver to process it.
**
**	NB: this will invoke (via receiver) statehandler for this node.
*/

void
send()
{
#	ifndef	OLD_ADDR_FORMAT
	register int	i;
	char *		errs;
	VarArgs		va;
	static int	maxHdrLength;
#	else	OLD_ADDR_FORMAT
	extern void	send1();
#	endif	OLD_ADDR_FORMAT
	char *		primdom;
	char		addr[4];
	int		domind;

	HdrSource = OrgNode;
	HdrHandler = STATEHANDLER;
	DataLength = lseek(MesgFd, (long)0, 2);
	ProtoType[0] = STATE_PROTO;
	HdrSubpt = ProtoType;

	/*
	**	Always broadcast update to other's primary domain,
	**	and our sub-domains, unless other domain is inside ours.
	*/

	addr[0] = ATYP_MULTICAST;
	addr[1] = ATYP_BROADCAST;
	addr[2] = DOMAIN_SEP;
	addr[3] = '\0';

	if ( (primdom = DomInPrim(LinkD.nl_domind)) == NULLSTR )
	{
		domind = PrimDom();			/* Use our primary */

		if ( NodeinPrim(LinkD.nl_index) )
			LinkD.nl_domind = domind;	/* We are inside other */

		if ( (primdom = DomainName(LinkD.nl_domind)) == NULLSTR )
		{
			/*
			**	No domains at all - Just broadcast.
			*/

			addr[2] = '\0';	
			domind = LinkD.nl_domind;	/* Null */
		}
	}
	else
		domind = LinkD.nl_domind;		/* Other inside ours */

#	ifndef	OLD_ADDR_FORMAT

	HdrDest = concat(addr, primdom, NULLSTR);

	addr[2] = DOMAIN_SEP;

	for ( i = 0 ; (primdom = DomInHier(&domind)) != NULLSTR ; i++ )
		HdrDest = concat(HdrDest, addr, primdom, NULLSTR);
	
	if ( i == 0 )
		HdrDest++;

#	else	OLD_ADDR_FORMAT

	do
	{
		HdrDest = concat(&addr[1], primdom, NULLSTR);

		send1();

		addr[2] = DOMAIN_SEP;
	}
	while
		( (primdom = DomInHier(&domind)) != NULLSTR );

	(void)close(MesgFd);
	(void)unlink(Message);
}



void
send1()
{
	char *		errs;
	VarArgs		va;
	static int	maxHdrLength;

#	endif	OLD_ADDR_FORMAT

	while ( WriteHeader(MesgFd, DataLength, maxHdrLength) == SYSERROR )
		Syserror("Can't write \"%s\"", Message);

#	ifndef	OLD_ADDR_FORMAT
	(void)close(MesgFd);
#	else	OLD_ADDR_FORMAT

	/*
	**	Header may get trampled on by receiver ...
	*/

	HdrLength += NODE_NAME_SIZE+1+(TIME_SIZE+1)*2;

	if ( HdrLength > maxHdrLength )
		maxHdrLength = HdrLength;
#	endif	OLD_ADDR_FORMAT

	FIRSTARG(&va) = RECEIVER;
	NEXTARG(&va) = "-L";

#	ifdef	DEBUG
	if ( Traceflag )
	{
		NEXTARG(&va) = newstr("-T0");
		LASTARG(&va)[2] += Traceflag;
	}
#	endif	DEBUG

	NEXTARG(&va) = concat("-h", HomeNode, NULLSTR);
	if ( !AtHome )
		NEXTARG(&va) = concat("-l", LinkNode, NULLSTR);

#	ifndef	OLD_ADDR_FORMAT

	NEXTARG(&va) = Message;

#	else	OLD_ADDR_FORMAT

	NEXTARG(&va) = UniqueName(newstr(Message), (long)0, ++Time);

	while ( link(Message, LASTARG(&va)) == SYSERROR )
		Syserror("Can't link \"%s\" to \"%s\"", Message, LASTARG(&va));

#	endif	OLD_ADDR_FORMAT

	if ( (errs = Execute(&va)) != NULLSTR )
		Error(errs);
}



/*
**	Pre-amble to start a state program
*/

FILE *
start_state(vap)
	VarArgs *	vap;
{
	FILE *		fd;

	FIRSTARG(vap) = STATE;
	NEXTARG(vap) = "-CO";		/* Make changes and output local state */
	if ( AtHome )
		NEXTARG(vap) = "-RSXc";	/* Update the state, routing, and statistics files */
	else
		NEXTARG(vap) = "-N";	/* No check! */
	NEXTARG(vap) = concat("-h", OrgNode, NULLSTR);
#	ifdef	DEBUG
	if ( Traceflag )
		NEXTARG(vap) = Traceflag==1?"-T1":"-T2";
#	endif	DEBUG

	fd = ExPipeO(vap, MesgFd);

	if ( !AtHome )
		(void)fprintf(fd, "hierarchy\n");	/* Clear inappropriate hierarchy */

	return fd;
}


/*
**	Find link by name
*/

bool
FindLink(name)
	char *		name;
{
	register int	i;

	for ( i = 0 ; GetLink(i, &LinkD) ; i++ )
		if ( strccmp(name, LinkD.nl_name) == STREQUAL )
			return true;

	return false;
}
