#
/*
 */

/*
 * TM tape driver (DEC TM11; Kennedy 9100/9300)
 */

#include "../param.h"
#include "../buf.h"
#include "../conf.h"
#include "../user.h"

struct {
	int tmer;
	int tmcs;
	int tmbc;
	int tmba;
	int tmdb;
	int tmrd;
};

struct	devtab	tmtab;
struct	buf	rtmbuf;

char	t_openf[8];
char	*t_blkno[8];
char	*t_nxrec[8];
char	t_flags[8];
int	t_dens[8];

#define	TMADDR	0172520

#define	TMPRI	5

#define	GO	01
#define	RCOM	02
#define	WCOM	04
#define	WEOF	06
#define	SFORW	010
#define	SREV	012
#define	WIRG	014
#define	REW	016
#define	OFFL	0		/* go offline */
#define	LODENS	0		/* 9-channel  800 bpi */
#define	HIDENS	060000		/* 9-channel 1600 bpi */
#define	IENABLE	0100
#define	CRDY	0200
#define	TUR	1
#define	REWS	2
#define	WRL	4
#define	BOT	040
#define	SELR	0100
#define	EOT	02000
#define	BGL	04000
#define	EOF	040000
#define	HARD	0100200		/* ILC, NXM */

#define	FSEEK	1
#define	SSEEK	2
#define	SIO	4

#define	SREOF	010
#define	SREOI	020
#define	SRREV	040
#define	SRERR	0100

tmlopen(dev, flag)
{
	tmopen(dev, flag, LODENS);
}

tmhopen(dev, flag)
{
	tmopen(dev, flag, HIDENS);
}

tmopen(dev, flag, dens)
{
	register int unit;
	extern lbolt;

	unit = dev.d_minor;
	if (t_openf[unit])
		u.u_error = ENXIO;
	else {
		t_dens[unit] = dens;
		tcommand(unit, 0);		/* select unit */
		if (!(TMADDR->tmer & SELR))
			printf("Switch tape unit %d online\n", unit);
		t_blkno[unit] = 0;
		t_nxrec[unit] = 65535;
		t_flags[unit] = 0;
		while (!(TMADDR->tmer & SELR))
			sleep(&lbolt, 1);
		if (flag && TMADDR->tmer & WRL) {
			tcommand(unit, OFFL|GO);
			u.u_error = ENXIO;
		}
		else
			t_openf[unit]++;
	}
}

tmclose(dev, flag)
{
	register int unit;

	unit = dev.d_minor;
	if (flag) {
		if (t_openf[unit] != WEOF)
			tcommand(unit, WEOF|GO);
		if (t_flags[unit] < 2)
			tcommand(unit, WEOF|GO);
	}
	if (t_flags[unit] < 2) {
		tcommand(unit, REW|GO);
		if (!t_flags[unit])
			tcommand(unit, OFFL|GO);
	}
	t_openf[unit] = 0;
}

tcommand(unit, com)
{
	extern lbolt;

	while (tmtab.d_active || (TMADDR->tmcs & CRDY) == 0) {
		if (TMADDR->tmer & REWS)
			sleep(&lbolt, 1);
	}
	TMADDR->tmcs = t_dens[unit] | (unit<<8) | com;
}

tmstrategy(abp)
struct buf *abp;
{
	register struct buf *bp;
	register char **p;

	bp = abp;
	p = &t_nxrec[bp->b_dev.d_minor];
	if (*p <= bp->b_blkno) {
		if (*p < bp->b_blkno) {
			bp->b_flags =| B_ERROR;
			iodone(bp);
			return;
		}
		if (bp->b_flags & B_READ) {
			clrbuf(bp);
			iodone(bp);
			return;
		}
	}
	if ((bp->b_flags & B_READ) == 0)
		*p = bp->b_blkno + 1;
	bp->av_forw = 0;
	spl5();
	if (tmtab.d_actf == 0)
		tmtab.d_actf = bp;
	else
		tmtab.d_actl->av_forw = bp;
	tmtab.d_actl = bp;
	if (tmtab.d_active == 0)
		tmstart();
	spl0();
}

tmstart()
{
	register struct buf *bp;
	register int com;
	int unit;
	register char *blkno;

    loop:
	if ((bp = tmtab.d_actf) == 0)
		return;
	unit = bp->b_dev.d_minor;
	blkno = t_blkno[unit];
	if (t_openf[unit] < 0 || (TMADDR->tmcs & CRDY) == 0) {
		bp->b_flags =| B_ERROR;
		tmtab.d_actf = bp->av_forw;
		iodone(bp);
		goto loop;
	}
	com = t_dens[unit] | (unit<<8) | ((bp->b_xmem & 03) << 4) | IENABLE;
	if (blkno != bp->b_blkno) {
		tmtab.d_active = SSEEK;
		if (blkno < bp->b_blkno) {
			com =| SFORW|GO;
			TMADDR->tmbc = blkno - bp->b_blkno;
		} else {
			if (bp->b_blkno == 0)
				com =| REW|GO;
			else {
				com =| SREV|GO;
				TMADDR->tmbc = bp->b_blkno - blkno;
			}
		}
		TMADDR->tmcs = com;
		return;
	}
	tmtab.d_active = SIO;
	TMADDR->tmbc = bp->b_wcount << 1;
	TMADDR->tmba = bp->b_addr;		/* core address */
	TMADDR->tmcs = com | ((bp->b_flags&B_READ) ? RCOM|GO :
	    ((tmtab.d_errcnt) ? WIRG|GO : WCOM|GO));
}

tmintr()
{
	register struct buf *bp;
	register int com, unit;

	if (tmtab.d_active == FSEEK) {
		unit = (TMADDR->tmcs>>8) & 07;
		if (TMADDR->tmer & (EOF|EOT|BOT|HARD)) {
			com = TMADDR->tmcs & (SFORW|SREV);
			if (TMADDR->tmer & EOF)
				t_openf[unit] = (com == SREV) ? 1 :
					((t_openf[unit] != SREOF)
					 ? SREOF : SREOI);	/* EOF found */
			else
			if ((TMADDR->tmer & EOT) && com == SFORW)
				t_openf[unit] = SREOI;		/* EOT found */
			else
			if ((TMADDR->tmer & BOT) && com == SREV)
				t_openf[unit] = SREOI|SRREV;	/* BOT found */
			else
				t_openf[unit] = SREOI|SRERR;	/* error */
			tmtab.d_active = 0;
			wakeup(&tmtab.d_active);
		}
		else {
			t_openf[unit] = 1;
			TMADDR->tmbc = -1;
			TMADDR->tmcs =| GO;
		}
		return;

	}
	if ((bp = tmtab.d_actf) == 0)
		return;
	unit = bp->b_dev.d_minor;
	if (TMADDR->tmcs < 0) {				/* error bit */
		if ((TMADDR->tmer & (HARD|EOT|EOF)) == 0) {
			if (tmtab.d_active == SIO) {
				if (++tmtab.d_errcnt < 10) {
					t_blkno[unit]++;
					tmtab.d_active = 0;
					if (TMADDR->tmer & BGL)
						tmtab.d_errcnt = 0;
					while (!(TMADDR->tmer & TUR));
					tmstart();
					return;
				}
				bp->b_flags =| B_ERROR;
			}
		}
		else {
			if (TMADDR->tmer & HARD)
				deverror(bp, TMADDR->tmer, TMADDR->tmcs);
			bp->b_flags =| B_ERROR;
			if (bp != &rtmbuf) {
				if ((TMADDR->tmer & EOF) == 0)
					t_openf[unit] = -1;
			}
			else
				t_openf[unit] = (TMADDR->tmer & (HARD|EOT) ||
						t_openf[unit] == SREOF)
						? SREOI : SREOF;
		}
		tmtab.d_active = SIO;
	}
	else
		t_openf[unit] = 1;
	if (tmtab.d_active == SIO) {
		tmtab.d_errcnt = 0;
		t_blkno[unit]++;
		tmtab.d_actf = bp->av_forw;
		tmtab.d_active = 0;
		bp->b_resid = TMADDR->tmbc;
		iodone(bp);
	} else
		t_blkno[unit] = bp->b_blkno;
	tmstart();
}

tmread(dev)
{
	register char *open;
	register int count;

	open = &t_openf[dev.d_minor];
	if (*open & WEOF || *open == SREOI || *open == (SREOI|SRERR))
		u.u_error = EIO;
	else {
		count = u.u_count;
		tmphys(dev);
		physio(tmstrategy, &rtmbuf, dev, B_READ);
		if (*open == SREOF) {
			u.u_error = 0;
			u.u_count = count;
		}
		else
			u.u_count = -rtmbuf.b_resid;
	}
}

tmwrite(dev)
{
	register char *open;

	open = &t_openf[dev.d_minor];
	if (*open == SREOI || *open == (SREOI|SRERR))
		u.u_error = EIO;
	else {
		tmphys(dev);
		physio(tmstrategy, &rtmbuf, dev, B_WRITE);
		*open = SREOI|WEOF;
	}
	u.u_count = 0;
}

/*
 * gtty: arg[0]:=density; arg[1]:=flags;
 * stty: arg[0]=density; arg[1]=flags;  
 *	 arg[2]: 0=no operation;
 *	 	 1=write EOF;
 *		 2=search EOF (forward);
 *		-2=search EOF (backward);
*/
tmsgtty(dev, v)
int *v;
{
	register char *open;
	register int unit, *av;
	extern lbolt;

	unit = dev.d_minor;
	if (av = v) {
		*av++ = (t_dens[unit] == LODENS) ? 0 : 1;
		*av = t_flags[unit];
		*++av = 0;
		return;
	}
	open = &t_openf[unit];
	av = u.u_arg;
	t_dens[unit] = (*av++) ? HIDENS : LODENS;
	t_flags[unit] = *av;
	if (*++av == 0)
		return;
	else
	if (*av == 1) {
		tcommand(unit, WEOF|GO);
		while ((TMADDR->tmcs & CRDY) == 0);
		*open = WEOF;
	}
	else {
		if (((*open & WEOF || *open == SREOI) && *av > 0) ||
		    (*open == (SREOI|SRREV) && *av < 0) ||
		    (*open == (SREOI|SRERR))) {
			u.u_error = EIO;
			return;
		}
		tcommand(unit, 0);		/* wait for unit ready */
		tmtab.d_active = FSEEK;
		TMADDR->tmbc = -1;
		spl5();
		if (*av > 0)
			TMADDR->tmcs =| IENABLE|SFORW|GO;
		else
			TMADDR->tmcs =| IENABLE|SREV|GO;
		sleep(&tmtab.d_active, TMPRI);
		spl0();
		if (*open & SREOI)
			u.u_error = EIO;
	}
}

tmphys(dev)
{
	register int unit, a;

	unit = dev.d_minor;
	a = lshift(u.u_offset, -9);
	t_blkno[unit] = a;
	t_nxrec[unit] = ++a;
}
