/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Paths.c	1.12 86/08/17
*/

/*
**	Calculate shortest paths to all nodes from 'home'.
*/

#include	"global.h"
#include	"debug.h"
#include	"state.h"

#include	"route.h"
#include	"node.h"

/*
**	The 'Ant' algorithm for calculating all shortest paths from a node:-
**
**	Create one ant to arrive at the starting node,
**		and put it at the head of a priority list;
**
**	while there are ants in the priority list
**	{
**		if node for the first ant has not been visited
**		{
**			mark node as visited, and set its shortest path;
**
**			for each edge to an unvisited node
**			{
**				create an ant to travel to that	node
**					through a distance proportional
**					to the cost of traversing the edge;
**
**				sort the ant into the priority list
**					based on distance;
**			}
**		}
**
**		remove first ant from list;
**	}
**
**	The 'Ant' algorithm has the following properties:-
**
**	If 'd' is the maximum number of shortest paths from one node to all others,
**	and 'n' is the number of nodes,
**	and 'e' is the number of edges,
**
**	then:
**		the maximum number of ants that may exist simultaneously is 'd',
**	and	the maximum number of ants that may be created is 'e'.
**
**	Ant creation is the most expensive operation,
**		however, the priority list sort is O('d'),
**		and therefore the algorithm is O('d' * 'e'),
**		which is proportional to O('n' squared).
*/



/*
**	Stucture for active Ant list
*/

typedef struct Ant *	Ant_p;

typedef struct Ant
{
	ulong	ant_dist;		/* Link distance */
	ulong	ant_adist;		/* Accumulated distance */
	Ant_p	ant_next;		/* Next ant in priority list */
	Entry *	ant_entry;		/* Destination */
	Entry *	ant_route;		/* Came via */
}
			Ant;

#define	DEAD_FACTOR	100		/* Cost of a link being dead */
#define	DOWN_FACTOR	3		/* Cost of a link being down */
#define	INTT_FACTOR	5		/* Cost of a link being intermittent */
#define	SIZE		102400l		/* Maximum speed for benefit of integer arithmetic */
#define	DFLT_SPEED	1024		/* Bytes/sec. for normal links */



void
Paths(home, type, linkno)
	Entry *			home;
	Rflags			type;
	int			linkno;
{
	register Ant *		nap;
	register Ant *		alist;
	register Ant **		app;
	register Domain *	dp;
	register Link *		lp;
	register States		term;

	static Ant *	freelist;

	States		save_states;

	DODEBUG(int edges_traversed=0; int ants_in_list=1; int max_ants=0; int ant_sorts=0);
	DODEBUG(int nodes_reached=0; int edges_searched=0; int ants_created=1);
	Trace6
	(
		1,
		"Paths from \"%s\", type 0%lo, linkno %d:  %d nodes, %d edges",
		home->e_name,
		type,
		linkno,
		NodeCount,
		EdgeCount
	);

	if ( (alist = freelist) != (Ant *)0 )
	{
		freelist = alist->ant_next;
		alist->ant_next = (Ant *)0;
		alist->ant_route = (Entry *)0;
		alist->ant_adist = 0;
	}
	else
		alist = Talloc(Ant);

	alist->ant_entry = home;

	save_states = home->e_states;
	home->e_states &= ~(S_MSGTERMINAL|S_CONTERMINAL);
	term = 0;
	if ( type & S_MSG )
		term |= S_MSGTERMINAL;
	if ( type & S_CON )
		term |= S_CONTERMINAL;

	while ( alist != (Ant *)0 )
	{
		register Entry *	q;

		DODEBUG(edges_traversed++; if(ants_in_list>max_ants) max_ants = ants_in_list);
		Trace5
		(
			2,
			"ant from %s at %s (route %s), dist %lu",
			alist->ant_route==(Entry *)0
				? "(null)"
				: alist->ant_route->e_name,
			alist->ant_entry->e_name,
			alist->ant_entry->e_route==(Entry *)0
				? "(null)"
				: alist->ant_entry->e_route->e_name,
			alist->ant_adist
		);

		if
		(
			(q = alist->ant_entry->e_route) == (Entry *)0
			||
			(
				q != alist->ant_route
				&&
				alist->ant_adist == alist->ant_entry->e_node->n_dist
			)
		)
		{
			extern bool	Warnings;	/* In Control/state.c */

			/*
			**	Found a shortest path to this node
			*/

			DODEBUG(nodes_reached++);

			if ( q != (Entry *)0 && (type & S_PRINTROUTE) && Warnings )
				Warn
				(
					"Duplicate shortest paths from \"%s\" to \"%s\" via \"%s\" and \"%s\"",
					home->e_name, alist->ant_entry->e_name,
					q->e_name, alist->ant_route->e_name
				);

			alist->ant_entry->e_node->n_dist = alist->ant_adist;

			if ( (alist->ant_entry->e_route = alist->ant_route) == (Entry *)0 )
				alist->ant_entry->e_route = home;	/* To stop back-tracking */

			if ( (type & S_LINKFORW) && alist->ant_route == Home )
			{
				/*
				**	Set bit in ForwTable for each node whose
				**	shortest path from 'home' lies via 'Home'.
				*/

				register int	i = (LinkCount * alist->ant_entry->e_index) + linkno;

				ForwTable[i/8] |= 1 << (i%8);

				Trace2(3, "Set bit %d in ForwTable", i);
			}

			/*
			**	Also found shortest path to all domains to which this node belongs
			*/

			if ( type & (S_DOMAINS|S_DOMROUTE) )
			for
			(
				dp = alist->ant_entry->e_node->n_domains.d_head ;
				dp != (Domain *)0 ;
				dp = dp->d_next
			)
			{
				if ( dp->d_entry->e_states & S_OTHDOM )
				{
					if ( !(dp->d_states & S_OTHDOM) )
						dp->d_entry->e_states &= ~S_OTHDOM;
				}
				else
				if ( dp->d_states & S_OTHDOM )
					continue;

				if
				(
					(type & S_DOMAINS)
					&&
					dp->d_entry->e_route == (Entry *)0
				)
				{
					dp->d_entry->e_route = alist->ant_entry->e_route;
					Trace3
					(
						2,
						"shortest path to domain %s via %s",
						dp->d_entry->e_name,
						dp->d_entry->e_route->e_name
					);
				}

				if
				(
					(type & S_DOMROUTE)
					&&
					alist->ant_route
				)
				{
					register int	i;

					/*
					**	Set bit in Domain Forwarding Table for this link.
					**	(link index has been set in route index).
					*/

					i = dp->d_entry->e_index * LinkCount + alist->ant_route->e_index;
					DODEBUG
					(
						if
						(
							Traceflag >= 2
							&&
							!(DomForwTable[i/8] & (1 << (i%8)))
						)
							Trace
							(
								2,
								"domain %s reached via %s",
								dp->d_entry->e_name,
								alist->ant_route->e_name
							);
					);
					DomForwTable[i/8] |= 1 << (i%8);
				}
			}

			/*
			**	Search all edges to unfound nodes linked to this one
			*/

			if ( q == (Entry *)0 && (alist->ant_entry->e_states & term) != term )
			for
			(
				lp = alist->ant_entry->e_node->n_l_first ;
				lp != (Link *)0 ;
				lp = lp->l_next
			)
			{
				DODEBUG(edges_searched++);

				if
				(
					lp->l_entry->e_route != (Entry *)0
					||
					!(lp->l_data->d_states & type)
				)
					continue;

				/*
				**	New edge to travel, calculate distance to node
				*/

				DODEBUG(ants_created++; ants_in_list++);

				if ( (nap = freelist) != (Ant *)0 )
					freelist = nap->ant_next;
				else
					nap = Talloc(Ant);

				nap->ant_entry = lp->l_entry;

				if ( type & S_PRINTROUTE )
					nap->ant_route = alist->ant_entry;
				else
				if ( (nap->ant_route = alist->ant_route) == (Entry *)0 )
					nap->ant_route = lp->l_entry;	/* A link */

				if ( type & S_CHEAP )
					nap->ant_dist = lp->l_data->d_cost;
				else
					nap->ant_dist = 0;

				if ( type & S_FAST )
				{
					register ulong	dist;

					if ( (dist = lp->l_data->d_speed) == 0 )
						dist = SIZE/DFLT_SPEED;
					else
						dist = SIZE/dist;

					if ( lp->l_data->d_states & S_INTERMITTENT )
						dist *= INTT_FACTOR;
					else
					if ( lp->l_data->d_states & S_DOWN )
						dist *= DOWN_FACTOR;

					nap->ant_dist += dist;
				}

				if ( nap->ant_dist == 0 )
					nap->ant_dist = 1;

				if ( lp->l_data->d_states & S_DEAD )
					nap->ant_dist *= DEAD_FACTOR;

				nap->ant_adist = alist->ant_adist + nap->ant_dist;

				Trace5
				(
					3,
					"new ant from \"%s\" at \"%s\" to \"%s\" dist %lu",
					nap->ant_route->e_name,
					alist->ant_entry->e_name,
					nap->ant_entry->e_name,
					nap->ant_dist
				);

				/*
				**	Sort ant into priority list
				*/

				for
				(
					app = &alist->ant_next ;
					*app != (Ant *)0 ;
					app = &(*app)->ant_next
				)
				{
					DODEBUG(ant_sorts++);

					if ( nap->ant_dist <= (*app)->ant_dist )
					{
						(*app)->ant_dist -= nap->ant_dist;
						break;
					}
					nap->ant_dist -= (*app)->ant_dist;
				}

				nap->ant_next = *app;
				*app = nap;
			}
		}

		DODEBUG(ants_in_list--);

		nap = alist;
		alist = alist->ant_next;
		nap->ant_next = freelist;
		freelist = nap;
	}

	home->e_states = save_states;

	DODEBUG
	(
		Trace4
		(
			1,
			"\tedges traversed = %d, edges searched = %d, ant sorts = %d",
			edges_traversed,
			edges_searched,
			ant_sorts

		);
		Trace4
		(
			1,
			"\tnodes reached = %d, ants created = %d, max ants = %d",
			nodes_reached,
			ants_created,
			max_ants

		);
	);
}
