/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)receiver.c	1.73 89/11/15";

/*
**	Handle messages from node-node daemon or sender.
**
**	Messages are passed to a local handler if destination is reached,
**	otherwise they are spooled for further transmission.
*/

#define	FILE_CONTROL
#define	STAT_CALL
#define	RECOVER
#define	LOCKING

#include	"global.h"

#include	"address.h"
#include	"command.h"
#include	"debug.h"
#include	"handlers.h"
#include	"header.h"
#include	"messagename.h"
#include	"spool.h"
#include	"state.h"
#include	"sysexits.h"

#include	<ndir.h>
#include	<signal.h>


/*
**	Parameters set from arguments.
*/

char *	BounceMesg;		/* Error string for message to be returned */
long	ComLength;		/* Length of message parts described in CommandsFile */
char *	CommandsFile;		/* Name of file containing commands for message parts */
char *	DmnDir;			/* Directory for router to operate from */
int	DmnPid;			/* Pid of NNdaemon */
ulong	ElapsedTime;		/* Journey time */
char *	HomeNode;		/* Name of this node */
bool	IgnEcho;		/* Don't do echo test (for gateways) */
char *	LinkNode;		/* Message has arrived from this node */
bool	Local;			/* Local invokation, return errors */
bool	MesgDup;		/* This message may be a duplicate */
char *	Message;		/* Current message being processed by receive() */
bool	NoCall;			/* Don't auto-call on this message */
char *	Name;			/* Program invoked name */
bool	Router;			/* Invoked as ROUTER */
int	Traceflag;		/* Global tracing control */

/*
**	Miscellaneous
*/

VarArgs	Args;			/* Global argument list for Exec...() */
int	Count;			/* Count used by spool1() */
bool	Explicit;		/* True if current message is using explicit routing */
char *	FilterMesg;		/* Message returned by a filter */
bool	Finish;			/* Terminate flag */
NodeLink FLinkD;		/* Info. for LinkNode */
char *	Home_Address;		/* Address of this node */
NodeLink LinkD;			/* Info. from DoRoute() */
char *	LSpoolDir = LSPOOLDIR;
bool	LockSet;		/* Router has set lockfile successfully */
int	MesgFd;			/* File descriptor for message being delivered */
bool	NoDeliver;		/* Don't deliver message locally */
bool	NoOpt;			/* True if no message ordering required */
int	Pid;			/* Process id */
int	RetVal;			/* Value returned by receiver */
char *	RouterLock	= "lock";
char *	Slash		= "/";
bool	StateMessage;		/* This message should have high priority in queue */
NodeLink SourceD;		/* Info. from FindAddress() */
char *	SpoolDir = SPOOLDIR();
char	Template[NNCFNZ+1];	/* Last component of node-node command file name */
Time_t	Time;			/* Current time */
ulong	Ttd;			/* Time-to-die for current message */

#define	IN	true
#define	OUT	false


void	Return(), addresserr(), deliver(), finish(), freeArgs(),
	handlebad(), putback(), router(), spool1();

bool	receive();

int	dofilter(), sigcatch();

extern bool	SetLock();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	if ( geteuid() != ACSNETUID )
	{
		Error("No permission.");
		exit(1);
	}

	if ( strcmp(*argv, ROUTER) == STREQUAL )
		Router = true;

	DODEBUG(EchoArgs(argc, argv));

	(void)sprintf(Template, "%c%-*.*s", SMALL_ID, sizeof Template-2, sizeof Template-2, "router");

	Pid = getpid();

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				if ( Router )
				switch ( c )
				{
				case 'd':
					DmnDir = ++*argv;
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					if ( FindAddress(LinkNode, &FLinkD) )
						goto break2;
					Error("Bad link name \"%s\"", LinkNode);
					exit(1);

				case 'p':
					DmnPid = atol(++*argv);
					break;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				default:
					Error("unrecognised flag '%c'", c);
					exit(1);
				}
				else
				switch ( c )
				{
				case 'C':
					NoCall = true;
					continue;

				case 'D':
					MesgDup = true;
					continue;

				case 'E':
					IgnEcho = true;
					continue;

				case 'L':
					Local = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'c':
					CommandsFile = ++*argv;
					goto break2;

				case 'd':
					ComLength = atol(++*argv);
					break;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					if ( FindAddress(LinkNode, &FLinkD) )
						goto break2;
					Error("Bad link name \"%s\"", LinkNode);
					exit(1);

				case 'r':
					BounceMesg = ++*argv;
					goto break2;

				case 't':
					ElapsedTime = atol(++*argv);
					break;

				default:
					Error("unrecognised flag '%c'", c);
					exit(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' || c == '-' )	/* KLUDGE! */
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			if ( Router )
			{
				Error("Unexpected file name");
				exit(1);
			}

			Time = time((long *)0);

			if ( !receive(*argv) )
				Error("Message echoed on link \"%s\"", LinkNode);

			if ( CommandsFile != NULLSTR )
			{
				(void)unlink(CommandsFile);
				CommandsFile = NULLSTR;
			}
		}
	}

	if ( Router )
	{
		if ( DmnDir == NULLSTR || LinkNode == NULLSTR || DmnPid == 0 )
		{
			Error("Need directory, link, and daemon's pid");
			exit(1);
		}

		router();
	}
	else
	if ( Message == NULLSTR )
	{
		Error("no spool file name");
		exit(1);
	}

	exit(RetVal);
}




/*
**	If "error" is non-zero, there has been an unrecoverable
**	system error and the daemon will hang pending operator intervention.
*/

void
finish(error)
	int	error;
{
	if ( Message != NULLSTR )
	{
		register int i;

		for ( i = 0 ; i < 20 ; i++ )
			UnLock(i);

		if ( Local )
			(void)unlink(Message);
	}

	if ( CommandsFile != NULLSTR )
		(void)unlink(CommandsFile);

	if ( DmnPid )
		(void)kill(DmnPid, SIGERROR);

	if ( LockSet )
		(void)unlink(RouterLock);

	(void)exit(error);
}


/*
**	Scan receiving directory for messages, and route them.
*/

void
router()
{
	register DIR *		dirp;
	register struct direct *direp;
	register int		count;
	bool			flush;

	(void)signal(SIGTERM, sigcatch);

	while ( (dirp = opendir(DmnDir)) == NULL )
		Syserror("Can't read \"%s\"", DmnDir);

	if ( chdir(DmnDir) == SYSERROR )
		Syserror("Can't chdir \"%s\"", DmnDir);

	while ( !SetLock(RouterLock, Pid) )
	{
		if ( kill(DmnPid, SIG0) == SYSERROR )
		{
			Error("Could not set \"%s/%s\"", DmnDir, RouterLock);
			return;
		}

		(void)sleep(20);
	}

	LockSet = true;

	for ( flush = false, count = 1 ; ; count = 0 )
	{
		while ( !Finish && (direp = readdir(dirp)) != NULL )
		{
			Parts		parts;
			struct stat	statb;

			if ( strlen(direp->d_name) != PNAMELEN )
				continue;

			count++;

			Time = time((long *)0);

			(void)stat(direp->d_name, &statb);

			(void)DecodeName(direp->d_name, &parts);

			ElapsedTime = parts.p_time + Time - statb.st_mtime;
			MesgDup = (parts.p_flag1 & MESG_DUP) ? true : false;

			if ( !receive(direp->d_name) && !flush )
			{
				(void)kill(DmnPid, SIGERROR);
				flush = true;
			}

			CheckRoute();
		}

		if ( Finish )
			return;

		if ( count == 0 )
		{
			if ( flush )
				Error("Message echoed on link \"%s\"", LinkNode);

			if ( kill(DmnPid, SIG0) == SYSERROR )
				break;

			(void)sleep(20);
		}

		rewinddir(dirp);
	}

	closedir(dirp);

	(void)unlink(RouterLock);
}


/*
**	Process message.
*/

bool
receive(message)
	char *		message;
{
	register char *	cp;
	HdrReason	reason;
	bool		expired;
	char *		handler;
	bool		filter = true;

	Message = message;

	Trace3(1, "Message \"%s\" received via \"%s\"", Message, (LinkNode==NULLSTR)?"":LinkNode);

	if ( HomeNode == NULLSTR )
		HomeNode = NodeName();

	if ( Home_Address == NULLSTR )
		Home_Address = concat(HomeNode, Hierarchy(), NULLSTR);

	while ( (MesgFd = open(Message, O_RDWR)) == SYSERROR )
		Syserror("Can't open \"%s\"", Message);

again:
	FreeHeader();

	if ( (reason = ReadHeader(MesgFd)) != hr_ok )
	{
		/*
		**	Bad message header.
		**
		**	Almost no recovery possible, except to pass
		**	it to the badhandler for post-mortem.
		*/

		(void)close(MesgFd);

		if ( Local )
		{
			Error("Header error \"%s\"", HeaderReason(reason));
			RetVal = 1;
			return true;
		}

		handlebad
		(
			concat
			(
				"-eHeader ",
				HeaderReason(reason),
				" error",
				NULLSTR
			)
		);

		(void)unlink(Message);

		return true;
	}

	if
	(
		!IgnEcho
		&&
		LinkNode != NULLSTR
		&&
		BounceMesg == NULLSTR
		&&
		(cp = LastNode(HdrRoute)) != NULLSTR
	)
	{
		if ( strcmp(HomeNode, cp) == STREQUAL )
		{
			free(cp);
	
			if ( Router )
				putback();

			(void)unlink(Message);

			return false;
		}
	
		free(cp);
	}

	if ( UpdateHeader(ElapsedTime, &Ttd, LinkNode) )
		expired = false;
	else
		expired = true;

	if ( HdrDest[0] == ATYP_EXPLICIT )
		Explicit = true;
	else
		Explicit = false;

	if ( strcmp(HdrHandler, STATEHANDLER) == STREQUAL )
		StateMessage = true;
	else
		StateMessage = false;

	/*
	**	Ensure "ID" field exists in environment.
	*/

	if ( (cp = GetEnv("ID")) == NULLSTR )
	{
		char *	ep;

		if ( (cp = strrchr(Message, '/')) == NULLSTR )
			cp = Message;
		else
			cp++;

		if ( MesgDup )
			HdrEnv = concat(HdrEnv, ep = MakeEnv(ENV_DUP, Home_Address, "ID", cp, NULLSTR), NULLSTR);
		else
			HdrEnv = concat(HdrEnv, ep = MakeEnv("ID", cp, NULLSTR), NULLSTR);

		free(ep);
	}
	else
	{
		free(cp);

		if ( MesgDup )
		{
			HdrEnv = concat(HdrEnv, cp = MakeEnv(ENV_DUP, Home_Address, NULLSTR), NULLSTR);
			free(cp);
		}
	}

	if ( BounceMesg != NULLSTR )
	{
		Return(newstr(BounceMesg));
		goto out;
	}

	/*
	**	If a filter has been specified for link, pass it the message.
	*/

	NoDeliver = false;

	if ( filter && (cp = FLinkD.nl_filter) != NULLSTR )
	{
		switch ( dofilter(cp, HdrSource, LinkNode, IN) )
		{
		case EX_OK:
			break;

		case EX_DROPMESG:
			if ( StateMessage )
				break;	/* Naughty! */
			(void)close(MesgFd);
			free(FilterMesg);
			goto out;

		case EX_NEWMESG:
			if ( StateMessage )
			{
				Recover(ert_finish);
				Error("illegal return status %d from filter \"%s\"", ExStatus, cp);
			}
			filter = false;
			free(FilterMesg);
			goto again;

		case EX_RETMESG:
			if ( StateMessage )
				break;	/* Naughty! */
			Return(FilterMesg);
			(void)close(MesgFd);
			goto out;

		case EX_EXMESG:
			NoDeliver = true;
			break;

		default:
			Recover(ert_finish);
			Error("unexpected return status %d from filter \"%s\", error \"%s\"", ExStatus, cp, FilterMesg);
		}

		if ( FilterMesg != NULLSTR )
			free(FilterMesg);
	}

	/*
	**	If this node is in destination, pass message to handler.
	*/

	if ( !Local )
		Recover(ert_return);

	if ( HdrDest[0] == '\0' )
	{
		if ( Local )
			goto hdr_handler;
		else
			Return(newstr("Bad destination address format"));
	}
	else
	if ( DestReached(deliver) )
	{
hdr_handler:
		if ( (handler = strrchr(HdrHandler, '/')) == NULLSTR )
			handler = HdrHandler;
		else
			handler++;	/* Suspicious! */
		handler = concat(INSLIB(), handler, NULLSTR);
		deliver(handler, NULLSTR);
		free(handler);
	}

	(void)close(MesgFd);

	Trace2(1, "New destination \"%s\"", HdrDest);

	/*
	**	If the message has not expired,
	**	and there are more nodes in destination,
	**	spool message for each outbound link.
	*/

	if ( !expired && HdrDest[0] != '\0' )
	{
		Hdr_Fields	save_header;

		if ( (cp = GetEnv(ENV_NO_AUTOCALL)) != NULLSTR )
		{
			free(cp);
			NoCall = true;
		}
		else
			NoCall = false;

		if ( (cp = GetEnv(ENV_NOOPT)) != NULLSTR )
		{
			free(cp);
			NoOpt = true;
		}
		else
			NoOpt = false;

		if ( StateMessage )
		{
			CheckRoute();	/* Re-load routing table */
			if ( LinkNode != NULLSTR )
				(void)FindAddress(LinkNode, &FLinkD);
		}

		Time = time((long *)0);		/* Update Time */

		Count = 0;
		save_header = HdrFields;

		(void)DoRoute(HdrSource, LinkNode, &LinkD, spool1, addresserr);

		HdrFields = save_header;
	}

out:
	Recover(ert_finish);

#	ifdef	DEBUG
	if ( Traceflag )
	{
		char *	tracemesg = UniqueName
				    (
					concat(TRACEDIR(), Template, NULLSTR),
					DataLength+ComLength,
					Time
				    );

		(void)link(Message, tracemesg);
		free(tracemesg);
	}
#	endif	DEBUG

	(void)unlink(Message);

	return true;
}



void
deliver(handler, address)
	char *	handler;
	char *	address;
{
	char *	errs;

	Trace3(1, "deliver(handler \"%s\", address \"%s\")", (handler==NULLSTR)?"":handler, address);

	if ( NoDeliver || (StateMessage && Local) )
		return;	/* Don't bother */

	if ( access(handler, 1) == SYSERROR )
	{
		/*
		**	Requested handler doesn't exist here.
		**
		**	Return message to source.
		*/

		Return(concat("Handler \"", handler, "\" unavailable", NULLSTR));
		return;
	}
/*
**	while ( WriteHeader(MesgFd, DataLength, HdrLength) == SYSERROR )
**		Syserror("Can't write header in \"%s\"", Message);
*/
	freeArgs();

	NEXTARG(&Args) = newstr(handler);
	if ( strchr(HdrDest, ATYP_BROADCAST) != NULLSTR )
		NEXTARG(&Args) = newstr("-B");
	if ( Traceflag )
		NEXTARG(&Args) = NumericArg('T', Traceflag);
	if ( address != NULLSTR )
	{
		NEXTARG(&Args) = concat("-a", address, NULLSTR);
		NEXTARG(&Args) = concat("-b", HdrHandler, NULLSTR);
	}
	if ( CommandsFile != NULLSTR )
		NEXTARG(&Args) = concat("-c", CommandsFile, NULLSTR);
	NEXTARG(&Args) = NumericArg('d', DataLength);
	NEXTARG(&Args) = concat("-e", HdrEnv, NULLSTR);
	NEXTARG(&Args) = concat("-h", HomeNode, NULLSTR);
	NEXTARG(&Args) = concat("-l", LinkNode, NULLSTR);
	NEXTARG(&Args) = concat("-s", HdrSource, NULLSTR);
	NEXTARG(&Args) = concat("-t", HdrTt, NULLSTR);
	NEXTARG(&Args) = newstr(Message);

	if ( (errs = Execute(&Args)) != NULLSTR )
	{
		/*
		**	Handler failed.
		**
		**	Return message to source.
		*/

		Return(errs);
	}

	DODEBUG
	(
		if ( access(Message, 0) == SYSERROR )
			Syserror
			(
				"handler \"%s\" removed \"%s\"!",
				handler,
				Message
			)
	);
}



/*
**	Called once for each link with appropriate address set in 'HdrDest'.
**
**	'LinkD' contains the link details.
**
**	'Message' contains the message.
*/

void
spool1(source)
	char *		source;
{
	register int	fd;
	register char *	mesgfile;
	register Time_t	T;
	bool		spoolok;

	/*
	**	If this node is inderdicted, ignore it.
	*/

	if ( StateMessage )
	{
		if ( LinkD.nl_flags & S_FOREIGN )
			return;

		if
		(
			(LinkD.nl_flags & S_INTERMITTENT)
			&&
			(
				(mesgfile = GetEnv(ENV_UP)) != NULLSTR
				||
				(mesgfile = GetEnv(ENV_DOWN)) != NULLSTR
			)
		)
		{
			free(mesgfile);
			return;
		}
	}

	if ( (mesgfile = GetEnv(ENV_NOTNODE)) != NULLSTR )
	{
		if ( strccmp(mesgfile, LinkD.nl_name) == STREQUAL )
		{
			free(mesgfile);
			return;
		}

		free(mesgfile);
	}

	/*
	**	Messages from "local" nodes are restricted to
	**	the primary domain of the sender.
	*/

	if
	(
		!StateMessage
		&&
		(HomeFlags & S_LOCAL)
		&&
		HomeAddress(HdrSource, (char **)0, INCL_HIER)
		&&
		!NodeinPrim(LinkD.nl_index)
	)
	{
		Return(concat("Can't send messages from \"local\" node \"", HdrSource, "\" off site", NULLSTR));
		return;
	}

	/*
	**	Messages from foreign nodes to foreign nodes
	**	are disallowed if we are a terminal node.
	*/

	if
	(
		!StateMessage
		&&
		(HomeFlags & S_MSGTERMINAL)
		&&
		!HomeAddress(HdrSource, (char **)0, INCL_HIER)
		&&
		!NodeinPrim(LinkD.nl_index)
		&&
		(
			!FindAddress(HdrSource, &SourceD)
			||
			!NodeinPrim(SourceD.nl_index)
		)
	)
	{
		Return(concat("Can't route foreign messages through \"terminal\" node \"", HdrSource, "\"", NULLSTR));
		return;
	}

	/*
	**	If a filter has been specified for link, pass it the message.
	*/

	if ( (mesgfile = LinkD.nl_filter) != NULLSTR )
	{
		switch ( dofilter(mesgfile, source, LinkD.nl_name, OUT) )
		{
		case EX_EXMESG:
		case EX_OK:
			break;

		case EX_DROPMESG:
			if ( StateMessage )
				break;	/* Naughty! */
			free(FilterMesg);
			return;

		case EX_RETMESG:
			if ( StateMessage )
				break;	/* Naughty! */
			Return(FilterMesg);
			return;

		default:
			Recover(ert_finish);
			Error("unexpected return status %d from filter \"%s\", error \"%s\"", ExStatus, mesgfile, FilterMesg);
		}

		if ( FilterMesg != NULLSTR )
			free(FilterMesg);
	}

	/*
	**	Make uniqueness a little easier
	*/

	T = Time - atol(HdrTt);
	if ( StateMessage )
		T /= 2;
	T += (Count+=2);

	Trace2(1, "Message passed to link \"%s\"", LinkD.nl_name);

	/*
	**	Make a link to Message
	*/

	mesgfile = concat(SpoolDir, LinkD.nl_name, Slash, LSpoolDir, NULLSTR);

	if ( access(mesgfile, 2) == SYSERROR )
		spoolok = false;
	else
		spoolok = true;

	free(mesgfile);

	mesgfile = UniqueName
		   (
			spoolok
			?concat(SpoolDir, LinkD.nl_name, Slash, LSpoolDir, Slash, Template, NULLSTR)
			:concat(WORKDIR(), Template, NULLSTR),
			DataLength+ComLength,
			T
		   );

	while ( link(Message, mesgfile) == SYSERROR )
		Syserror("Can't link \"%s\" to \"%s\"", Message, mesgfile);

	freeArgs();

	if ( LinkD.nl_spooler != NULLSTR )
	{
		char *	errs;

		/*
		**	If (through routing changes) this message is about
		**	to return to a node that it has already visited,
		**	send it back to its source.
		*/

		if
		(
			strccmp(HdrDest, LinkD.nl_name) != STREQUAL
			&&
			(fd = InRoute(LinkD.nl_name)) > 3
			&&
			(!Explicit || fd > 4)
		)
		{
			(void)unlink(mesgfile);

			Return
			(
				concat
				(
					"Routing loop detected to node \"",
					LinkD.nl_name,
					"\"",
					NULLSTR
				)
			);

			return;
		}

		/*
		**	This node requires a non-standard spooler to be invoked.
		*/

		NEXTARG(&Args) = newstr(LinkD.nl_spooler);
		if ( Traceflag )
			NEXTARG(&Args) = NumericArg('T', Traceflag);
		NEXTARG(&Args) = concat("-a", HdrDest, NULLSTR);
		if ( CommandsFile != NULLSTR )
			NEXTARG(&Args) = concat("-c", CommandsFile, NULLSTR);

		if ( MatchString(LinkD.nl_spooler, "Sun3_4") != NULLSTR )
			NEXTARG(&Args) = concat("-e", HdrEnv, NULLSTR);

		NEXTARG(&Args) = concat("-h", HomeNode, NULLSTR);
		NEXTARG(&Args) = concat("-l", LinkD.nl_name, NULLSTR);
		NEXTARG(&Args) = concat("-s", source, NULLSTR);
		NEXTARG(&Args) = NumericArg('t', ElapsedTime);
		NEXTARG(&Args) = mesgfile;

		if ( (errs = Execute(&Args)) != NULLSTR )
		{
			/*
			**	Spooler failed.
			**
			**	Return message to source.
			*/

			(void)unlink(mesgfile);	/* Because Args will be freed in "Return()" */

			Return(concat("Gateway handler failed :-\n", errs, NULLSTR));

			free(errs);
		}
		else
		{
			Trace4
			(
				1,
				"message passed to \"%s\" for \"%s\" via \"%s\"",
				LinkD.nl_spooler,
				HdrDest,
				LinkD.nl_name
			);

			UpStats(LinkD.nl_name);
		}
	}
	else
	{
		char *	headerfile;
		ComHead	comhead;
		char *	commands;
		char *	commandtemp;
		char *	savehdrenv;
		long	messagebase;

#		if	MAXRETSIZE != 0
		/*
		**	If returned message is too large, truncate it.
		*/

		if ( DataLength > MAXRETSIZE && (savehdrenv = GetEnv(ENV_RETURNED)) != NULLSTR )
		{
			char *	ep;

			free(savehdrenv);
			HdrEnv = concat(savehdrenv = HdrEnv, ep = MakeEnv(ENV_TRUNC, Home_Address, NULLSTR), NULLSTR);
			messagebase = DataLength - MAXRETSIZE;

			free(ep);
		}
		else
#		endif	MAXRETSIZE != 0
			messagebase = 0;

		/*
		**	Make a new file for header
		*/

		headerfile = UniqueName
			     (
				spoolok
				?concat(SpoolDir, LinkD.nl_name, Slash, LSpoolDir, Slash, Template, NULLSTR)
				:concat(WORK2DIR(), Template, NULLSTR),
				DataLength+ComLength,
				T + 1
			     );

		while ( (fd = creat(headerfile, 0600)) == SYSERROR )
			Syserror("Can't create \"%s\"", headerfile);

		HdrSource = source;

		while ( WriteHeader(fd, (long)0, 0) == SYSERROR )
			Syserror("Can't write header in \"%s\"", headerfile);

		(void)close(fd);

		if ( messagebase != 0 )
		{
			free(HdrEnv);
			HdrEnv = savehdrenv;
		}

		/*
		**	Make a command file for this message in
		**	appropriate command directory for node.
		*/

		commandtemp = UniqueName
			      (
				concat(WORK2DIR(), Template, NULLSTR),
				DataLength+ComLength,
				T
			      );

		/*
		**	4[5] commands: (see "command.h")
		**		send message; send header;
		**		unlink message; unlink header;
		**		[set timeout for message].
		*/
		
		FreeCom(&comhead, init_ch);

		AddCom(&comhead, mesgfile, messagebase, DataLength-messagebase);
		AddCom(&comhead, headerfile, (long)0, (long)HdrLength);
		AddCom(&comhead, mesgfile, (long)0, (long)0);
		AddCom(&comhead, headerfile, (long)0, (long)0);

		if ( Ttd > 0 )
			AddCom(&comhead, NULLSTR, (long)Ttd, (long)0);

		while ( (fd = creat(commandtemp, 0600)) == SYSERROR )
			Syserror("Can't create \"%s\"", commandtemp);

		if ( CommandsFile != NULLSTR )
			CopyFile(CommandsFile, fd, commandtemp);

		(void)WriteCom(&comhead, fd, commandtemp);

		(void)close(fd);

		FreeCom(&comhead, free_ch);

		/*
		**	If (through routing changes) this message is about
		**	to return to a node that it has already visited,
		**	spool it in the holding directory.
		*/

		if
		(
			strccmp(HdrDest, LinkD.nl_name) != STREQUAL
			&&
			(fd = InRoute(LinkD.nl_name)) > 0
			&&
			(!Explicit || fd > 3)
		)
		{
			if ( fd == 1 && ReRoutable() )
			{
				commands = UniqueName
					   (
						concat(REROUTEDIR(), Template, NULLSTR),
						DataLength+ComLength,
						T
					   );

				while ( link(commandtemp, commands) == SYSERROR )
					Syserror
					(
						"Can't move \"%s\" to \"%s\"",
						commandtemp,
						commands
					);
				
				UpStats(LinkD.nl_name);
			}
			else
			{
				Return
				(
					concat
					(
						"Routing loop detected to node \"",
						LinkD.nl_name,
						"\"",
						NULLSTR
					)
				);

				commands = Malloc(1);
				(void)unlink(headerfile);
				(void)unlink(mesgfile);
			}

			(void)unlink(commandtemp);
		}
		else
		{
			char *	calldir = NULLSTR;

			/*
			**	Now move commands into daemon's de-spooling directory.
			*/

			commands = UniqueName
				   (
					concat(SpoolDir, LinkD.nl_name, Slash, Template, NULLSTR),
					NoOpt
					? (long)MAX_MESG_DATA
					: StateMessage
					  ? (long)0
					  : (DataLength+ComLength+HdrLength),
					T
				   );

			while ( link(commandtemp, commands) == SYSERROR )
				Syserror("Can't move \"%s\" to \"%s\"", commandtemp, commands);

			(void)unlink(commandtemp);

			/*
			**	If link is "call-on-demand", start daemon.
			*/

			if
			(
				!NoCall
				&&
				(LinkD.nl_flags & S_CALL)
				&&
				!DaemonActive(calldir = concat(SpoolDir, LinkD.nl_name, NULLSTR), true)
			)
			{
				char *	callfile;

				callfile = concat(calldir, Slash, CALLFILE, NULLSTR);

				if ( (fd = access(callfile, 04)) == SYSERROR && LinkD.nl_caller == NULLSTR )
				{
					Warn
					(
						"Can't access \"%s\" for call-on-demand link \"%s\"",
						callfile,
						LinkD.nl_name
					);

					free(callfile);
				}
				else
				{
					char *	errs;

					if ( LinkD.nl_caller != NULLSTR )
					{
						NEXTARG(&Args) = newstr(LinkD.nl_caller);
						if ( fd == SYSERROR )
						{
							free(callfile);
							callfile = NULLSTR;
						}
					}
					else
						NEXTARG(&Args) = newstr(NNCALL);
					NEXTARG(&Args) = newstr("-&");
					NEXTARG(&Args) = concat("-h", LinkD.nl_name, NULLSTR);
					NEXTARG(&Args) = concat("-l", calldir, Slash, LOGFILE, NULLSTR);
					NEXTARG(&Args) = callfile;

					if ( (errs = Execute(&Args)) != NULLSTR )
					{
						Warn(errs);
						free(errs);
					}
				}
			}
			
			if ( calldir != NULLSTR )
				free(calldir);

			UpStats(LinkD.nl_name);
		}

		Trace3
		(
			1,
			"message spooled in \"%s\" for \"%s\"",
			commands,
			HdrDest
		);

		free(commands);
		free(commandtemp);
		free(headerfile);
		free(mesgfile);
	}
}



/*
**	Pass message to BadHandler
*/

void
handlebad(arg)
	char *	arg;	/* free()'d */
{
	char *	badmesg;
	char *	errs;

	Trace1(1, "Message passed to badhandler");

	/*
	**	Move mesg to BADDIR
	*/

	badmesg = UniqueName
		  (
			concat(BADDIR(), Template, NULLSTR),
			DataLength+ComLength,
			Time
		  );

	while ( link(Message, badmesg) == SYSERROR )
		Syserror("Can't move \"%s\" to \"%s\", (error: \"%s\")", Message, badmesg, arg);

	freeArgs();
	NEXTARG(&Args) = newstr(BADHANDLER);
	if ( Traceflag )
		NEXTARG(&Args) = NumericArg('T', Traceflag);
	NEXTARG(&Args) = arg;
	NEXTARG(&Args) = concat("-l", LinkNode, NULLSTR);
	NEXTARG(&Args) = concat("-i", Name, NULLSTR);
	NEXTARG(&Args) = concat("-h", HomeNode, NULLSTR);

	if ( CommandsFile != NULLSTR )
	{
		char *	badcom;

		/*
		**	Move CommandsFile to BADDIR
		*/

		badcom = UniqueName
			 (
				concat(BADDIR(), Template, NULLSTR),
				DataLength+ComLength,
				Time
			 );

		while ( link(CommandsFile, badcom) == SYSERROR )
			Syserror("Can't move \"%s\" to \"%s\"", CommandsFile, badcom);

		NEXTARG(&Args) = concat("-d", CommandsFile, NULLSTR);
	}

	NEXTARG(&Args) = badmesg;

	if ( (errs = Execute(&Args)) != NULLSTR )
	{
		/*
		**	A total disaster!
		*/

		Error(errs);

		finish(1);
	}
}



/*
**	Unknown node encountered in destination/source address:
**	return message to source if possible.
**
**	"LinkD.nl_name" contains an error string set by DoRoute().
*/

void
addresserr(address, source)
	char *	address;
	bool	source;
{
	char *	reason;

	if ( source )
		reason = "Source address \"";
	else
		reason = "Destination \"";

	Return
	(
		concat
		(
			reason,
			address,
			"\" ",
			LinkD.nl_name,
			" at \"",
			Home_Address,
			"\"",
			NULLSTR
		)
	);
}



/*
**	Return the message to its source with explanation of error
*/

bool	returning;

void
Return(error)
	char *		error;	/* free()'d */
{
	char *		cp;
	char *		errors;
	Hdr_Fields	save_header;
	Time_t		ttd;
#	if	LOG_RETURNED == 1
	Handler *	handler;
#	endif

	Trace2(1, "Message returned to sender, reason \"%s\"", error);

	errors = StripErrString(error);

	if ( strlen(errors) < 4 )
	{
		free(errors);

		if ( strlen(error) < 4 )
		{
			free(error);

			if ( Local )
			{
				RetVal = 1;	/* Cos reason already on fd2 */
				return;
			}

			error = newstr("Handler failed, reason unknown");
		}

		errors = newstr(error);
	}

	/*
	**	If this is a local message, don't bother.
	*/

	if
	(
		Local
		||
		HomeAddress(HdrSource, (char **)0, INCL_HIER)
	)
	{
		if ( Local )
		{
			Warn("Could not deliver message: %s", errors);
			RetVal = 1;
			return;
		}
		else
			handlebad(concat("-eCould not deliver message: ", error, NULLSTR));

		free(errors);
		free(error);
		return;
	}

	/*
	**	Check no recursion, or returning unwanted.
	*/

	cp = NULLSTR;

	if
	(
		returning
		||
		(cp = GetEnv(ENV_RETURNED)) != NULLSTR
		||
		(cp = GetEnv(ENV_NORET)) != NULLSTR
	)
	{
		if ( cp != NULLSTR )
			free(cp);
#		if	LOG_RETURNED == 0
		if ( (cp = GetEnv(ENV_NORET)) != NULLSTR )
			free(cp);
		else
#		endif	LOG_RETURNED == 0
		handlebad(concat("-eReturned message discarded: ", error, NULLSTR));
		free(errors);
		free(error);
		return;
	}

#	if	LOG_RETURNED == 1
	/*
	**	Log the error.
	*/
	if
	(
		(handler = GetHandler(HdrHandler)) == (Handler *)0
		||
		handler->ign_err == '0'
	)
		handlebad(concat("-eReturned message log: ", error, NULLSTR));
#	endif	LOG_RETURNED == 1

	free(error);

	returning = true;

	/*
	**	Save old header
	*/

	save_header = HdrFields;

	/*
	**	Set new header
	*/

	HdrEnv = MakeEnv
		 (
			ENV_RETURNED, NULLSTR,
			ENV_ERR1, errors,
			ENV_DESTINATION, HdrDest,
			ENV_ROUTE, HdrRoute,
			NULLSTR
		 );

	free(errors);

	HdrDest = HdrSource;
	HdrSource = Home_Address;

	HdrTtd = HdrTt = HdrRoute = "";

	(void)UpdateHeader((Time_t)0, &ttd, LinkNode);

	/*
	**	Spool it
	*/

	if ( !FindAddress(HdrDest, &LinkD) )
	{
		if ( FindAddress(LinkNode, &LinkD) )
		{
			char	sep[2];

			handlebad
			(
				concat
				(
					"-eDomain configuration problem?\nUnknown source \"",
					HdrDest,
					"\"\nMessage for \"",
					HdrDest,
					"\" returned to source via \"",
					LinkNode,
					"\"",
					NULLSTR
				)
			);

			if ( HdrDest[0] == ATYP_EXPLICIT )
				HdrDest++;

			sep[0] = ATYP_EXPLICIT;
			sep[1] = '\0';

			HdrDest = concat(sep, LinkNode, sep, HdrDest, NULLSTR);

			spool1(HdrSource);

			free(HdrDest);
		}
		else
			handlebad(concat("-eUnknown source \"", HdrDest, "\"", NULLSTR));
	}
	else
		spool1(HdrSource);

	/*
	**	Restore header
	*/

	free(HdrEnv);

	HdrFields = save_header;

	returning = false;
}



/*
**	Clear out Args (all must have been malloc'ed).
*/

void
freeArgs()
{
	register char *	cp;

	if ( NARGS(&Args) > MAXVARARGS )
		Fatal1("Too many VarArgs");
	
	while ( NARGS(&Args)-- )
		if ( (cp = ARG(&Args, NARGS(&Args))) != NULLSTR )
			free(cp);
	
	NARGS(&Args) = 0;
}



/*
**	Run filter on Message.
*/

int
dofilter(prog, source, link, in)
	char *		prog;
	char *		source;
	char *		link;
	bool		in;
{
	Trace4(1, "Message %s \"%s\" passed to filter \"%s\"", in?"from":"to", link, prog);

	freeArgs();

	NEXTARG(&Args) = newstr(prog);
	NEXTARG(&Args) = newstr(in?"-I":"-O");
	NEXTARG(&Args) = concat("-a", HdrDest, NULLSTR);
	NEXTARG(&Args) = concat("-b", HdrHandler, NULLSTR);
	if ( CommandsFile != NULLSTR )
		NEXTARG(&Args) = concat("-c", CommandsFile, NULLSTR);
	NEXTARG(&Args) = NumericArg('d', DataLength);
	NEXTARG(&Args) = concat("-e", HdrEnv, NULLSTR);
	NEXTARG(&Args) = concat("-h", HomeNode, NULLSTR);
	NEXTARG(&Args) = concat("-l", link, NULLSTR);
	NEXTARG(&Args) = concat("-s", source, NULLSTR);
	NEXTARG(&Args) = newstr(Message);

	FilterMesg = Execute(&Args);

	return ExStatus;
}



/*
**	Put message back on queue to a link after being echoed to router.
*/

void
putback()
{
	int	fd;
	char *	mesgfile;
	char *	headerfile;
	ComHead	comhead;
	char *	commands;
	char *	commandtemp;

	Trace3(1, "putback \"%s\" on link \"%s\"", Message, LinkNode);

	if ( !UpdateHeader(ElapsedTime, &Ttd, LinkNode) )
		return;	/* Expired */

	/*
	**	Make a link to Message
	*/

	mesgfile = UniqueName
		   (
			concat(WORKDIR(), Template, NULLSTR),
			DataLength,
			Time
		   );

	while ( link(Message, mesgfile) == SYSERROR )
		Syserror("Can't link \"%s\" to \"%s\"", Message, mesgfile);

	/*
	**	Make a new file for header
	*/

	headerfile = UniqueName
		     (
			concat(WORK2DIR(), Template, NULLSTR),
			DataLength,
			Time
		     );

	while ( (fd = creat(headerfile, 0600)) == SYSERROR )
		Syserror("Can't create \"%s\"", headerfile);

	while ( WriteHeader(fd, (long)0, 0) == SYSERROR )
		Syserror("Can't write header in \"%s\"", headerfile);

	(void)close(fd);

	/*
	**	Make a command file for this message in
	**	appropriate command directory for node.
	*/

	commandtemp = UniqueName
		      (
			concat(WORK2DIR(), Template, NULLSTR),
			DataLength,
			Time
		      );

	/*
	**	4[5] commands: (see "command.h")
	**		send message; send header;
	**		unlink message; unlink header;
	**		[set timeout for message].
	*/
	
	FreeCom(&comhead, init_ch);

	AddCom(&comhead, mesgfile, (long)0, DataLength);
	AddCom(&comhead, headerfile, (long)0, (long)HdrLength);
	AddCom(&comhead, mesgfile, (long)0, (long)0);
	AddCom(&comhead, headerfile, (long)0, (long)0);

	if ( Ttd > 0 )
		AddCom(&comhead, NULLSTR, (long)Ttd, (long)0);

	while ( (fd = creat(commandtemp, 0600)) == SYSERROR )
		Syserror("Can't create \"%s\"", commandtemp);

	(void)WriteCom(&comhead, fd, commandtemp);

	(void)close(fd);

	FreeCom(&comhead, free_ch);

	/*
	**	Now move commands into daemon's de-spooling directory.
	*/

	commands = UniqueName
		   (
			concat(SpoolDir, LinkNode, Slash, Template, NULLSTR),
			DataLength+HdrLength,
			Time/2
		   );

	while ( link(commandtemp, commands) == SYSERROR )
		Syserror("Can't move \"%s\" to \"%s\"", commandtemp, commands);

	(void)unlink(commandtemp);

	UpStats(LinkNode);

	free(commands);
	free(commandtemp);
	free(headerfile);
	free(mesgfile);
}



/*
**	Catch system termination and wind up.
*/

int
sigcatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);

	Finish = true;
}
