/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)stop.c	1.6 84/11/07";

/*
**	Stop messages awaiting transmission.
**
**	SETUID ==> ACSNETUID.
*/

char *	Usage		= "[-[A][M][R][T<level>][V][W][Y][u<user]] [link|alias|node ...]";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"Passwd.h"
#include	"command.h"
#include	"debug.h"
#include	"ftheader.h"
#include	"handlers.h"
#include	"header.h"
#include	"spool.h"
#include	"state.h"
#include	"sub_proto.h"
#include	"Stream.h"

#include	<ndir.h>
#include	<errno.h>


/*
**	Parameters set from arguments
*/

bool	All;				/* Remove all found */
bool	MessageDetails;			/* Print header details from messages */
char *	Name;				/* Program invoked name */
bool	Reroute;			/* Look for re-routed messages */
int	Traceflag;			/* Trace level */
char *	User;				/* Different user to look for */
bool	Verbose;			/* Print all details from ftp header */
bool	Warnings;			/* Whinge */
bool	Yes;				/* Be obsequious */

/*
**	List for found links.
*/

typedef struct LLel *	LLl_p;

typedef struct LLel
{
	LLl_p	lql_next;
	char *	lql_link;
	ino_t	lql_ino;
}
	LLel;

LLl_p	LinkList;			/* Head of linked list */
LLl_p *	LinkVec;			/* Vector of sorted links */
int	NLinks;				/* No. of links found */

/*
**	Structure to remember "unlink" file names.
*/

typedef struct UnlinkN *UN_p;
typedef struct UnlinkN
{
	UN_p	un_next;
	char *	un_name;
}
	UnlinkN;

UN_p	UnFreelist;
UN_p	UnList;

/*
**	Miscellaneous info.
*/

char *	Format		= "%-7s from %-8.8s to %-13.13s at %-13.13s %10ld bytes\n";
char *	FormatV		= "%s from %s to %s at %s  %ld bytes\n";
char *	HeaderFile;			/* Current message header */
char *	HomeNode;			/* Here */
char	LinkName[132];			/* Link name for printing */
bool	LinkPrinted;			/* True after LinkName printed */
Passwd	Me;				/* My name */
long	MesgDataSize;			/* Size of data part */
char *	MesgFile;			/* Data part of message (if any) */
long	MesgHdrSize;			/* Size of header */
long	MesgLength;			/* Total size of message */
char *	Spooldir	= SPOOLDIR();
int	Stopped;			/* Messages stopped */
Time_t	Time;				/* Time of invocation */
long	Time_to_die;			/* From timeout command */

#define	Fprintf		(void)fprintf
#define	Fflush		(void)fflush
#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))

/*
**	Routines
*/

void	ListLink(), SortLinks(), Stopem(), print_header(), usage();
bool	FindLinks(), query(), searchstatus();
int	compare(), msgcmp();
bool	read_com(), read_header(), read_ftp();



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	register int	i;
	char		buf[BUFSIZ];

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					All = true;
					continue;

				case 'M':
					MessageDetails = true;
					continue;

				case 'R':
					Reroute = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'V':
					Verbose = true;
					continue;

				case 'W':
					Warnings = true;
					continue;

				case 'Y':
					Yes = true;
					continue;

				case 'u':
					User = ++*argv;
					goto break2;

				default:
					usage("unrecognised flag '%c'", c);
					return 1;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			char *	link = *argv;

			for ( i = 0 ; ; i++ )
			{
				char *		linkp = concat(Spooldir, link, NULLSTR);
				struct stat	statb;

				if
				(
					stat(linkp, &statb) != SYSERROR
					&&
					(statb.st_mode&S_IFMT) == S_IFDIR
					&&
					!dots(link)
				)
				{
					ListLink(Spooldir, link, (ino_t)0);
					break;
				}

				free(linkp);

				if ( i == 0 )
				{
					NodeLink nl;

					if ( (link = FindAlias(*argv)) == NULLSTR )
						link = *argv;

					if ( FindAddress(link, &nl) )
					{
						link = nl.nl_name;
						continue;
					}
				}

				Error("unknown link \"%s\"", *argv);
				return 1;
			}
		}
	}

	if ( NLinks == 0 && !FindLinks(Spooldir) )
	{
		Error("no network links found!");
		return 1;
	}

	if ( !GetUser(&Me, getuid()) )
	{
		Error("passwd error for uid %d: \"%s\"", Me.P_uid, Me.P_error);
		return 1;
	}

	if ( !(Me.P_flags & P_SU) && (All || User != NULLSTR) )
	{
		Error("No permission.");
		return 1;
	}

	if ( User != NULLSTR && !GetUid(&Me, User) )
	{
		Error("passwd error for user \"%s\": \"%s\"", User, Me.P_error);
		return 1;
	}

	if ( Reroute )
		ListLink(REROUTEDIR(), ".", (ino_t)0);

	SortLinks();

	HomeNode = NodeName();

	setbuf(stdout, buf);

	for ( i = 0 ; i < NLinks ; i++ )
		Stopem(LinkVec[i]->lql_link);

	Fprintf(stdout, "%d message%s stopped.\n", Stopped, Stopped==1?"":"s");

	return 0;
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	Fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}



/*
**	Queue a network link directory.
*/

void
ListLink(path, link, ino)
	char *		path;
	char *		link;
	ino_t		ino;
{
	register LLl_p	hllp;

	if ( link == NULLSTR || link[0] == '\0' )
		return;

	hllp = Talloc(LLel);

	hllp->lql_link = concat(path, link, NULLSTR);

	if ( strcmp(link, ".") == STREQUAL )
		hllp->lql_link[strlen(path)-1] = '\0';

	hllp->lql_ino = ino;

	hllp->lql_next = LinkList;
	LinkList = hllp;

	NLinks++;

	Trace2(1, "list link \"%s\"", hllp->lql_link);
}




/*
**	Search spooldir for link directories.
*/

bool
FindLinks(path)
	char *				path;
{
	register DIR *			dirp;
	register struct direct *	direp;
	bool				found = false;

	Trace2(1, "find links in \"%s\"", path);

	if ( (dirp = opendir(path)) == NULL )
	{
		Syserror("cannot read \"%s\"", path);
		return found;
	}

	while ( (direp = readdir(dirp)) != NULL )
	if ( direp->d_name[0] != WORKFLAG && !dots(direp->d_name) )
	{
		register LLl_p	hllp;
		struct stat	dstat;
		char *		newpath;

		Trace2(2, "entry \"%s\"", direp->d_name);

		for ( hllp = LinkList ; hllp != (LLl_p)0 ; hllp = hllp->lql_next )
			if ( hllp->lql_ino == direp->d_ino )
				break;

		newpath = concat(path, direp->d_name, NULLSTR);

		if
		(
			hllp == (LLl_p)0
			&&
			stat(newpath, &dstat) != SYSERROR
			&&
			(dstat.st_mode & S_IFMT) == S_IFDIR
		)
		{
			ListLink(path, direp->d_name, direp->d_ino);
			found = true;
		}

		free(newpath);
	}

	closedir(dirp);

	return found;
}


/*
**	Sort the links into alphabetic order
*/

void
SortLinks()
{
	register LLl_p	lp;
	register LLl_p *lpp;

	LinkVec = lpp = (LLl_p *)Malloc(sizeof(LLl_p) * NLinks);

	for ( lp = LinkList ; lp != (LLl_p)0 ; lp = lp->lql_next )
		*lpp++ = lp;

	DODEBUG(if((lpp-LinkVec)!=NLinks)Fatal1("bad NLinks"));

	Trace2(1, "found %d links", NLinks);

	if ( NLinks > 1 )
		qsort((char *)LinkVec, NLinks, sizeof(LLl_p), compare);
}



/*
**	Alpha compare
*/

int
compare(lpp1, lpp2)
	LLl_p *lpp1;
	LLl_p *lpp2;
{
	return strcmp((*lpp1)->lql_link, (*lpp2)->lql_link);
}



/*
**	Search a directory for appropriate N-N command files,
**	print details, and prompt for action.
*/

void
Stopem(dir)
	char *				dir;
{
	register DIR *			dirp;
	register struct direct *	direp;
	register int			fd;
	char *				link;

	Trace2(1, "Stopem \"%s\"", dir);

	if ( (link = strrchr(dir, '/')) != NULLSTR && link[1] != '\0' )
		link++;
	else
		link = dir;

	(void)sprintf(LinkName, "Link to %s:-\n", link);
	LinkPrinted = false;

	if ( (dirp = opendir(dir)) == NULL )
	{
		Syserror("Can't read \"%s\"", dir);
		return;
	}

	while ( (direp = readdir(dirp)) != NULL )
	{
		register char *	fp;
		register UN_p	unp;
		Time_t		mtime;
		bool		val;
		bool		timed_out;

		/*
		**	Find valid command file.
		*/

		switch ( direp->d_name[0] )
		{
		default:
			continue;

		case SMALL_ID:
		case MEDIUM_ID:
		case LARGE_ID:
			break;
		}

		Trace2(2, "found \"%s\"", direp->d_name);

		fp = concat(dir, "/", direp->d_name, NULLSTR);

		if ( (fd = open(fp, O_RDWR)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read commands file \"%s\"", fp);

			free(fp);
			continue;
		}

		MesgLength = 0;
		MesgDataSize = 0;
		MesgHdrSize = 0;

		if ( MesgFile != NULLSTR )
			free(MesgFile);
		MesgFile = NULLSTR;

		if ( HeaderFile != NULLSTR )
			free(HeaderFile);
		HeaderFile = NULLSTR;

		Time_to_die = 0;

		if
		(
			!ReadCom(fd, &mtime, read_com)
			||
			HeaderFile == NULLSTR
		)
		{
			if ( Warnings )
				Warn("bad commands file \"%s\"", fp);

			val = false;
		}
		else
		{
			val = read_header();
			timed_out = (bool)(Time_to_die > 0 && Time > (mtime+Time_to_die));
		}

		if ( val && query(timed_out) )
		{
			char *	statusfile = concat(dir, "/", STATUSFILE, NULLSTR);

			if ( !timed_out )
			{
				ComHead	comhead;

				FreeCom(&comhead, init_ch);
				AddCom(&comhead, NULLSTR, (long)1, (long)0);	/* Timeout in 1 second */
				(void)WriteCom(&comhead, fd, fp);		/* 'fd' is at EOF after "ReadCom()" above */
				FreeCom(&comhead, free_ch);
			}

			if ( searchstatus(statusfile, direp->d_name) )
			{
				if ( !timed_out )
					Warn("Might not be stopped: message is being transmitted.");
			}

			if ( !timed_out )
				Stopped++;

			(void)unlink(fp);	/* Unlink command file */

			for ( unp = UnList ; unp != (UN_p)0 ; unp = unp->un_next )
			{
				if ( !timed_out )
				{
					register int	f;

					/*
					**	Truncate message
					*/

					if ( (f = creat(unp->un_name, 0600)) != SYSERROR )
						(void)close(f);
				}

				(void)unlink(unp->un_name);
			}

			free(statusfile);
		}

		while ( (unp = UnList) != (UN_p)0 )
		{
			UnList = unp->un_next;
			unp->un_next = UnFreelist;
			UnFreelist = unp;
			free(unp->un_name);
		}

		(void)close(fd);
		free(fp);
	}

	closedir(dirp);
}



/*
**	Search NNdaemon status file for transmitting command file name.
*/

bool
searchstatus(statusfile, commandfile)
	char *		statusfile;
	char *		commandfile;
{
	register int	i;
	register int	fd;
	NN_state	NNstate;
	bool		exists;

	Trace2(1, "searchstatus \"%s\"", statusfile);

	for ( i = 0, exists = false ; i < 3 ; i++ )
	{
		if ( (fd = open(statusfile, O_READ)) != SYSERROR )
		{
			if
			(
				read(fd, (char *)&NNstate, sizeof NNstate) == sizeof NNstate
				&&
				strcmp(NNstate.version, StreamSCCSID) == STREQUAL
			)
			{
				register Str_p	strp;

				for ( strp = outStreams ; strp < &outStreams[NSTREAMS] ; strp++ )
					if
					(
						strp->str_id != '\0'
						&&
						strcmp(strp->str_id, commandfile) == STREQUAL
					)
					{
						(void)close(fd);
						return true;
					}

				(void)close(fd);
				return false;
			}

			exists = true;

			(void)close(fd);
		}

		(void)sleep(1);
	}

	if ( exists )
	{
		Error("status file version, re-compile");
		return true;
	}

	return false;
}



/*
**	Read HeaderFile, and extract details.
*/

bool
read_header()
{
	int		fd;
	HdrReason	hr;
	bool		val;
	char *		cp;
	Handler *	handler;
	Time_t		msg_date;

	Trace2(1, "read_header \"%s\"", HeaderFile);

	if ( (fd = open(HeaderFile, O_READ)) == SYSERROR )
	{
		if ( Warnings )
			SysWarn("Can't read \"%s\"", HeaderFile);

		return false;
	}

	if ( (hr = ReadHeader(fd)) != hr_ok )
	{
		if ( Warnings )
			Warn("Header error \"%s\"", HeaderReason(hr));

		(void)close(fd);
		return false;
	}

	/*
	**	Fetch protcocol details if from HomeNode, or All.
	*/

	if ( strcmp(HdrSource, HomeNode) != STREQUAL && !All )
	{
		(void)close(fd);
		return false;
	}

	if ( (handler = GetHandler(HdrHandler)) == (Handler *)0 )
		cp = "Message";
	else
		cp = handler->descrip;
	
	if ( MessageDetails )
	{
		struct stat	statb;

		(void)fstat(fd, &statb);
		msg_date = statb.st_mtime - atol(HdrTt);
	}

	if ( HdrSubpt[0] == FTP )
		val = read_ftp(fd, cp);
	else
	{
		val = false;
		(void)close(fd);
	}

	if ( val == false )
	{
		if ( !All )
			return false;

		print_header();

		Fprintf
		(
			stdout,
			Verbose?FormatV:Format,
			cp,
			HdrSource,
			HdrHandler,
			HdrDest,
			MesgLength
		);
	}

	/*
	**	Print message header info if requested.
	*/

	if ( MessageDetails )
	{
		if ( HdrEnv[0] != '\0' )
			Fprintf
			(
				stdout,
				"%*sEnv=\"%s\"\n",
				6+7+1, "",
				ExpandString(HdrEnv, strlen(HdrEnv))
			);

		Fprintf
		(
			stdout,
			"%*sRoute=\"%s\"\n",
			6+7+1, "",
			ExpandString(HdrRoute, strlen(HdrRoute))
		);

		Fprintf
		(
			stdout,
			"%*sDate: %.15s\n",
			6+7+1, "",
			ctime(&msg_date)+4
		);
	}

	return true;
}



/*
**	Function called from "ReadCom" to process a command.
**	Remember last file name (containing header),
**	and all "unlink" file names.
*/

bool
read_com(name, base, range)
	char *	name;
	long	base;
	long	range;
{
	Trace4(2, "command \"%s %ld %ld\"", name, base, range);

	if ( range > 0 )
	{
		MesgLength += range;
		if ( MesgFile != NULLSTR )
			free(MesgFile);
		MesgFile = HeaderFile;
		HeaderFile = name;
		MesgDataSize = MesgHdrSize;
		MesgHdrSize = range;
	}
	else
	if ( base == 0 )
	{
		register UN_p	unp;

		if ( (unp = UnFreelist) == (UN_p)0 )
			unp = Talloc(UnlinkN);
		else
			UnFreelist = unp->un_next;

		unp->un_next = UnList;
		UnList = unp;
		unp->un_name = name;
	}
	else
	{
		Time_to_die = base;
		free(name);
	}

	return true;
}



/*
**	Read the data part of message for FTP header.
*/

bool
read_ftp(fd, type)
	int		fd;
	char *		type;
{
	FthReason	fthr;

	Trace5
	(
		2,
		"read_ftp type %s, datasize %ld, hdrsize %ld, datalength %ld",
		type,
		MesgDataSize,
		MesgHdrSize,
		DataLength
	);

	if ( DataLength == 0 )
	{
		(void)close(fd);

		if ( (fd = open(MesgFile, O_READ)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read message data from \"%s\"", MesgFile);

			return false;
		}
	}
	else
		MesgDataSize = DataLength;

	if ( (fthr = ReadFtHeader(fd, MesgDataSize, false)) != fth_ok )
	{
		if ( Warnings )
			Warn("FT Header \"%s\" error", FTHREASON(fthr));

		(void)close(fd);
		return false;
	}

	(void)close(fd);

	if ( !All && strcmp(FthFrom, Me.P_name) != STREQUAL )
		return false;

	/*
	**	Have valid FTP message to print
	*/

	print_header();

	if ( !Verbose )
	{
		Fprintf
		(
			stdout,
			Format,
			type,
			FthFrom,
			FthTo,
			HdrDest,
			MesgLength
		);
	}
	else
	{
		register FthFD_p	fp;

		Fprintf
		(
			stdout,
			FormatV,
			type,
			FthFrom,
			FthTo,
			HdrDest,
			MesgLength
		);

		if ( (fthr = GetFthFiles()) != fth_ok )
		{
			Warn("FT Header \"%s\" error", FTHREASON(fthr));
			return true;
		}

		for ( fp = FthFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
			Fprintf
			(
				stdout,
				"%27s %10ld  %.12s\n",
				fp->f_name,
				fp->f_length,
				ctime(&fp->f_time)+4
			);

		FreeFthFiles();
	}

	return true;
}



/*
**	Print out header on demand
*/

void
print_header()
{
	if ( !LinkPrinted )
	{
		LinkPrinted = true;
		Fprintf(stdout, LinkName);
	}
}



/*
**	Interrogate for action
*/

bool
query(timed_out)
	bool		timed_out;
{
	register char *	cp;
	register int	n;
	char		buf[4];
	static char *	yes = "yes";

	if ( timed_out )
	{
		Fprintf(stdout, "(TIMED OUT)\n", yes);
		return true;
	}

	Fprintf(stdout, "Stop ? (y or n) ");

	if ( Yes )
	{
		Fprintf(stdout, "%s\n", yes);
		return true;
	}

	Fflush(stdout);

	for( cp = buf, n = 0 ; ; )
	{
		if ( read(0, cp, 1) <= 0 )
			finish(1);

		if ( *cp == '\n' )
		{
			*cp = '\0';
			break;
		}

		if ( ++n < sizeof buf )
			cp++;
	}

	if ( n > 0 && n <= strlen(yes) && strncmp(buf, yes, n) == STREQUAL )
		return true;

	return false;
}
