/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Domains.c	1.15 84/10/09
*/

/*
**	Domain handling routines.
*/

#include	"global.h"
#include	"debug.h"
#include	"state.h"

#include	"node.h"



/*
**	Add new domain to end of list, return Entry.
*/

Entry *
AddDomain(cp, hp, found)
	register char *		cp;
	register DomHead *	hp;
	bool			found;
{
	register Domain *	dp;
	register Entry *	ep;

	ep = Enter(cp, DomainHash);

	for ( dp = hp->d_head ; dp != (Domain *)0 ; dp = dp->d_next )
		if ( dp->d_entry == ep )
			return ep;	/* A duplicate! */

	dp = Talloc(Domain);

	dp->d_entry = ep;

	if ( found && !(ep->e_states & S_FOUND) )
	{
		DomainCount++;
		ep->e_states |= S_FOUND;
	}

	dp->d_previous = hp->d_last;
	hp->d_last = dp;

	*hp->d_tail = dp;
	hp->d_tail = &dp->d_next;
	hp->d_count++;

	return ep;
}



/*
**	Check that node is a member of each domain mentioned in hierarchy,
**	and that there are at least two domains in the hierarchy.
*/

void
CheckHierarchy(ep, hp)
	Entry *			ep;
	DomHead *		hp;
{
	register Domain **	dpp;
	register Domain *	dp;
	register Domain *	hdp;
	register Entry *	primdom;
	register bool		prim_in_hier;

	if ( hp->d_head != (Domain *)0 )
	{
		if ( (hdp = ep->e_node->n_domains.d_head) != (Domain *)0 )
			primdom = hdp->d_entry;
		else
		{
			Error
			(
				"Node \"%s\" not a member of any domain in hierarchy",
				ep->e_name
			);
			return;
		}

		prim_in_hier = false;
		hp->d_count = 0;

		for ( dpp = &hp->d_head ; (dp = *dpp) != (Domain *)0 ; )
		{
			if ( ep == Home )
			{
				if ( dp->d_entry->e_states & S_DOMAINS )
				{
ok:
					if ( dp->d_entry == primdom )
						prim_in_hier = true;

					hp->d_count++;
					dpp = &dp->d_next;
					continue;
				}
			}
			else
			if ( dp->d_entry->e_states & S_FOUND )
			{
				for ( dp = hdp ; dp != (Domain *)0 ; dp = dp->d_next )
					if ( (*dpp)->d_entry == dp->d_entry )
					{
						dp = *dpp;
						goto ok;
					}

				dp = *dpp;
			}

			Warn
			(
				"%somain \"%s\" removed from hierarchy for node \"%s\"",
				(dp->d_entry->e_states&S_FOUND)?"Foreign d":"D",
				dp->d_entry->e_name,
				ep->e_name
			);

			if ( (*dpp = dp->d_next) != (Domain *)0 )
				dp->d_next->d_previous = dp->d_previous;
			else
			{
				hp->d_tail = dpp;
				hp->d_last = dp->d_previous;
			}

			free((char *)dp);
		}

		if ( !prim_in_hier )
			Error
			(
				"Primary domain \"%s\" not in hierarchy for node \"%s\"",
				primdom->e_name,
				ep->e_name
			);
		else
		if ( hp->d_count < 2 )
			Error
			(
				"Less than 2 domains in hierarchy for node \"%s\"",
				ep->e_name
			);
	}
}



/*
**	Check that, either Home belongs to primary domain given,
**			or Home primary domain is in list.
*/

bool
CheckDomains(dp)
	register Domain *	dp;	/* Head of a node's domain list */
{
	register Entry *	hdp;

	if ( dp->d_entry->e_states & S_DOMAINS )
		return true;

	hdp = Home->e_node->n_domains.d_head->d_entry;

	while ( (dp = dp->d_next) != (Domain *)0 )
	{
		if ( dp->d_entry == hdp )
			return true;
	}

	return false;
}



/*
**	Check domains of each link.
*/

void
CheckLinkDomains()
{
	register Domain *	dp;
	register Link *		lp;

	if ( Home->e_node->n_domains.d_head == (Domain *)0 )
		return;

	for ( lp = Home->e_node->n_l_first ; lp != (Link *)0 ; lp = lp->l_next )
		if ( (dp = lp->l_entry->e_node->n_domains.d_head) != (Domain *)0 )
		{
			if ( !CheckDomains(dp) )
				Warn
				(
					"Link \"%s\" not a member of any of Home's domains",
					lp->l_entry->e_name
				);
		}
}




/*
**	Check source domains.
*/

void
CheckSourceDomains(source)
	Entry *			source;		/* Source node */
{
	register Domain *	dp;
	register Domain *	hdp;
	register Link **	lpp;
	register bool		first;
#	if	KEEP_FGN_SUB_D == 1
	register bool		remove;
#	endif	KEEP_FGN_SUB_D == 1

	if ( Home->e_node->n_domains.d_head == (Domain *)0 )
		return;

	/*
	**	Discard any foreign internal domains from node and links,
	**	using information from foreign domain hierarchy.
	**	(Know that there are 0 or >1 domains in each hierarchy.)
	*/

	if ( (dp = ForeignHier.d_head) == (Domain *)0 )
		return;

	if ( (hdp = DomHier.d_head) == (Domain *)0 )
		return;

	while ( hdp->d_entry == dp->d_entry )
	{
		if ( (dp = dp->d_next) == (Domain *)0 )
			return;

		if ( (hdp = hdp->d_next) == (Domain *)0 )
			return;
	}

	dp = ForeignHier.d_last;
	hdp = DomHier.d_last;

	/*
	**	Discard identical part
	*/

	while ( hdp->d_entry == dp->d_entry )
	{
		if ( (dp = dp->d_previous) == (Domain *)0 )
			return;

		if ( (hdp = hdp->d_previous) == (Domain *)0 )
			break;
	}

	/*
	**	Discard each higher level domain of which we are a member.
	*/

	while ( dp->d_entry->e_states & S_DOMAINS )
	{
		if ( (dp = dp->d_previous) == (Domain *)0 )
			return;
	}

	/*
	**	Remaining sub-domains should be removed
	*/
#	if	KEEP_FGN_SUB_D == 1
	/*
	**	where we belong to one of the same name.
	*/

	remove = false;
#	endif	KEEP_FGN_SUB_D == 1

	while ( (hdp = dp->d_previous) != (Domain *)0 )
	{
#		if	KEEP_FGN_SUB_D == 1
		if ( !remove )
		{
			if ( !(hdp->d_entry->e_states & S_DOMAINS) )
			{
				dp = hdp;
				continue;
			}

			remove = true;
		}
#		endif	KEEP_FGN_SUB_D == 1

		if ( (dp = hdp->d_previous) == (Domain *)0 )
			ForeignHier.d_head = hdp->d_next;
		else
			dp->d_next = hdp->d_next;

		hdp->d_next->d_previous = hdp->d_previous;

		/*
		**	Remove references from source.
		*/

		for
		(
			dp = source->e_node->n_domains.d_head, first = true ;
			dp != (Domain *)0 ;
			dp = dp->d_next, first = false
		)
			if ( dp->d_entry == hdp->d_entry )
			{
				if ( first )
				{
					Warn
					(
						"\"%s\" removed (has external sub-domain \"%s\" as its primary)",
						source->e_name,
						dp->d_entry->e_name
					);

					NodeCount--;
					source->e_states &= ~S_FOUND;
					break;
				}

				Trace3
				(
					1,
					"Discarding domain \"%s\" from node \"%s\"",
					dp->d_entry->e_name,
					source->e_name
				);

				if ( (dp->d_previous->d_next = dp->d_next) != (Domain *)0 )
					dp->d_next->d_previous = dp->d_previous;
				else
				{
					source->e_node->n_domains.d_last = dp->d_previous;
					source->e_node->n_domains.d_tail = &dp->d_previous->d_next;
				}

				source->e_node->n_domains.d_count--;

				free((char *)dp);
				break;
			}

		/*
		**	And from its links.
		**	(Remove any links whose primary domain is removed.)
		*/

		for ( lpp = &source->e_node->n_l_first ; (*lpp) != (Link *)0 ; )
		{
			/*
			**	This is unsatisfactory;
			**	need a way to ensure this doesn't remove
			**	a domain from one of "our" nodes.
			*/

			if
			(
				(*lpp)->l_entry == Home
				||
				(dp = (*lpp)->l_entry->e_node->n_domains.d_head) == (Domain *)0
			)
			{
				lpp = &(*lpp)->l_next;
				continue;
			}

			first = true;

			do
			{
				if ( dp->d_entry == hdp->d_entry )
				{
					if ( first )
						Unlink(source, lpp);
					else
					{
						if ( (dp->d_previous->d_next = dp->d_next) != (Domain *)0 )
							dp->d_next->d_previous = dp->d_previous;
						else
						{
							(*lpp)->l_entry->e_node->n_domains.d_last = dp->d_previous;
							(*lpp)->l_entry->e_node->n_domains.d_tail = &dp->d_previous->d_next;
						}

						(*lpp)->l_entry->e_node->n_domains.d_count--;
						
						free((char *)dp);
						lpp = &(*lpp)->l_next;
					}

					break;
				}

				first = false;
			}
			while
				( (dp = dp->d_next) != (Domain *)0 );

			if ( dp == 0 )
				lpp = &(*lpp)->l_next;
		}

		dp = hdp->d_next;

		free((char *)hdp);
	}

#	if	KEEP_FGN_SUB_D != 1
	if ( (dp = source->e_node->n_domains.d_head) == (Domain *)0 )
		return;

	if ( CheckDomains(dp) )
	{
		/*
		**	Discard any links not a member of one of Home's domains.
		*/

		for ( lpp = &source->e_node->n_l_first ; (*lpp) != (Link *)0 ; )
		{
			if ( (dp = (*lpp)->l_entry->e_node->n_domains.d_head) == (Domain *)0 )
			{
				lpp = &(*lpp)->l_next;
				continue;
			}

			if ( !CheckDomains(dp) )
				Unlink(source, lpp);
			else
				lpp = &(*lpp)->l_next;
		}
	}
#	endif	KEEP_FGN_SUB_D != 1
}



/*
**	Clear (all/unfound) domains from node.
*/

void
ClearDomains(ep, all)
	Entry *			ep;
	bool			all;
{
	register Domain **	dpp;
	register Domain *	dp;
	register Node *		np;

	np = ep->e_node;

	for ( dpp = &np->n_domains.d_head ; (dp = *dpp) != (Domain *)0 ; )
		if ( all || !(dp->d_entry->e_states & S_FOUND) )
		{
			if ( !all )
				Report3
				(
					"Node \"%s\" removed from domain \"%s\"",
					ep->e_name,
					dp->d_entry->e_name
				);

			if ( (*dpp = dp->d_next) == (Domain *)0 )
				np->n_domains.d_last = dp->d_previous;
			else
				dp->d_next->d_previous = dp->d_previous;

			if ( ep == Home )
				dp->d_entry->e_states &= ~S_DOMAINS;

			np->n_domains.d_count--;
			free((char *)dp);
		}
		else
			dpp = &dp->d_next;

	np->n_domains.d_tail = dpp;
}



/*
**	Extract a string representing domain list from a domain header
*/

char *
DomString(dhp, c)
	DomHead *		dhp;
	char			c;
{
	register Domain *	dp;
	register char *		cp;
	register int		len = 0;
	char *			str;

	if ( (dp = dhp->d_head) == (Domain *)0 )
		return NULLSTR;

	do
	{
		len += strlen(dp->d_entry->e_name) + 1;
	}
		while ( (dp = dp->d_next) != (Domain *)0 );

	str = cp = Malloc(len);

	dp = dhp->d_head;
	do
	{
		cp = strcpyend(cp, dp->d_entry->e_name);
		*cp++ = c;
	}
		while ( (dp = dp->d_next) != (Domain *)0 );

	*--cp = '\0';

	return str;
}
