#
/*
 *      just to turn M37 half-linefeed sequences (ESC-9,-8,-7) into
 *      Diablo half-linefeeds (ESC-U,-D,-<LF>)
 */

/* defined characters */
#define ESC     '\033'
#define CR      '\r'
#define LF      '\n'

/*
 * characters meant for the TTY37 or GSI,
 * to be translated; and their translations.
 */
#define M37HLF  '9'
#define DBOHLF  'U'
#define M37RHLF	'8'
#define DBORHLF	'D'
#define M37RLF	'7'
#define DBORLF  LF
#define GSIPLOT '\006'
#define DBOPLOT '3'
#define GSIUPLT '\007'
#define DBOUPLT '4'
#define GREEK   '\016'
#define ENGLISH '\017'

/* the following require one argument */
#define DBOAHT  '\t'
#define DBOAVT  '\013'
#define DBOVMI  '\036'
#define DBOHMI  '\037'

/* cursor motion, for the special-character plotting */
#define UP      ESC,LF
#define DOWN    LF
#define RIGHT   ' '
#define LEFT    '\b'

/* bits on the flags variable */
#define ESCFG   01
#define TEXTMD  02
#define SPECIAL 04
#define ONEMORE 010

int     flags,          /* stores the above states */
	svmode,         /* stores the prior mode of the diablo */
	mode[3];        /* for stty */

/*
 * The following are the translation tables for the
 * ascii-to-greek translation.
 * In some cases, where an apostrophe would be
 * appropriate, it has been substituted for, because
 * our apostrophe is too rigid.
 */
int	tab[]{
	'A',	/* alpha */
	'B',	/* beta */
	'D',	/* delta */
	'W',	/* DELTA */
	'S',	/* epsilon */
	'N',	/* eta */
	'\\',	/* gamma */
	'G',	/* GAMMA */
	'o',	/* infinity - not in M37 */
	'^',	/* integral */
	'L',	/* lambda */
	'E',	/* LAMBDA */
	'M',	/* mu */
	'[',	/* nabla (del) */
	'_',	/* not */
	'@',	/* nu */
	'C',	/* omega */
	'Z',	/* OMEGA */
	']',	/* partial */
	'U',	/* phi */
	'F',	/* PHI */
	'V',	/* psi */
	'H',	/* PSI */
	'J',	/* pi */
	'P',	/* PI */
	'K',	/* rho */
	'Y',	/* sigma */
	'R',	/* SIGMA */
	'I',	/* tau */
	'T',	/* theta */
	'O',	/* THETA */
	'X',	/* xi */
	'Q',	/* zeta */
	0
};

int *trans[]{
	alpha,
	beta,
	delta,
	DELTA,
	epsilon,
	eta,
	gamma,
	GAMMA,
	infinity,
	integral,
	lambda,
	LAMBDA,
	mu,
	nabla,
	not,
	nu,
	omega,
	OMEGA,
	partial,
	phi,
	PHI,
	psi,
	PSI,
	pi,
	PI,
	rho,
	sigma,
	SIGMA,
	tau,
	theta,
	THETA,
	xi,
	zeta,
	0
};

/*
 * The reason these are int arrays, is so that negative elements may
 * exist: a negative element -n means output the next char n times.
 */
int alpha[]	{LEFT,'c',RIGHT,RIGHT,'(',LEFT,0};
int beta[]      {'B',LEFT,LEFT,DOWN,DOWN,'|',RIGHT,RIGHT,UP,UP,0};
int delta[]     {'o',UP,UP,UP,'<',DOWN,DOWN,DOWN,0};
int DELTA[]     {LEFT,LEFT,'/',-3,DOWN,'-',-4,RIGHT,'-',UP,UP,UP,
		 '\\',LEFT,LEFT,0};
int epsilon[]	{'<','-',0};
int eta[]	{'n',RIGHT,RIGHT,DOWN,DOWN,'|',LEFT,LEFT,UP,UP,0};
int gamma[]	{')',RIGHT,'/',LEFT,0};
int GAMMA[]     {LEFT,LEFT,'|',RIGHT,RIGHT,UP,UP,UP,'-',-3,
		 DOWN,RIGHT,RIGHT,'`',LEFT,LEFT,0};
int infinity[]	{LEFT,LEFT,'c',-4,RIGHT,'o',LEFT,LEFT,0};
int integral[]  {'|',RIGHT,'^',LEFT,LEFT,-2,DOWN,'.',-4,DOWN,LEFT,'`',
		 -2,RIGHT,UP,UP,UP,
		 UP,UP,UP,0};
int lambda[]    {'\\',-4,DOWN,LEFT,'\'',DOWN,LEFT,'\'',UP,UP,
		 UP,UP,UP,RIGHT,RIGHT,0};
int LAMBDA[]	{LEFT,LEFT,'/',-4,RIGHT,'\\',LEFT,LEFT,0};
int mu[]	{'u',LEFT,LEFT,',',RIGHT,RIGHT,0};
int nabla[]     {LEFT,LEFT,'\\',UP,UP,UP,'-',-4,RIGHT,'-',-3,DOWN,
		 '/',LEFT,LEFT,0};
int not[]	{'-',-2,RIGHT,UP,',',DOWN,-2,LEFT,0};
int nu[]	{LEFT,'(',-3,RIGHT,'/',LEFT,LEFT,0};
int omega[]	{LEFT,'u',-3,RIGHT,'u',LEFT,LEFT,0};
int OMEGA[]	{'O',DOWN,DOWN,LEFT,'-',RIGHT,RIGHT,'-',LEFT,UP,UP,0};
int partial[]	{'o',RIGHT,DOWN,'`',LEFT,UP,'`',LEFT,UP,'`',RIGHT,DOWN,0};
int phi[]	{'o','/',0};
int PHI[]	{'o','[',']',0};
int psi[]       {'/','-',DOWN,DOWN,RIGHT,RIGHT,'\'',-4,LEFT,'\'',RIGHT,RIGHT,
		 UP,UP,0};
int PSI[]       {'[',']','-',DOWN,DOWN,RIGHT,RIGHT,'\'',-4,LEFT,'`',RIGHT,
		 RIGHT,UP,UP,0};
int pi[]        {UP,'-',-3,DOWN,'"',DOWN,'"',UP,
		 UP,UP,0};
int PI[]        {LEFT,LEFT,'[',']',-4,RIGHT,'[',']',LEFT,LEFT,UP,UP,
		 UP,'-',-3,DOWN,0};
int rho[]	{'o',LEFT,LEFT,DOWN,DOWN,'|',UP,UP,RIGHT,RIGHT,0};
int sigma[]	{'o',DOWN,RIGHT,RIGHT,'~',UP,LEFT,LEFT,0};
int SIGMA[]     {'>',-2,DOWN,'-',UP,UP,
		 UP,UP,UP,'-',-3,DOWN,0};
int tau[]	{'t',DOWN,RIGHT,RIGHT,'~',LEFT,LEFT,LEFT,'~',RIGHT,UP,0};
int theta[]	{'O','-',0};
int THETA[]	{'O','=',0};
int xi[]        {'c',RIGHT,DOWN,',',LEFT,UP,UP,UP,'c',LEFT,
		 DOWN,'`',RIGHT,DOWN,0};
int zeta[]      {'c',RIGHT,DOWN,',',UP,UP,UP,
		 LEFT,'<',DOWN,DOWN,0};

/*
 * interrupt-time and end-time routine
 */
restore(){
	if((flags & TEXTMD) == 0) {
		putesc(DBOUPLT);
	}
	flush();
	mode[2] = svmode;
	stty(1, mode);
	exit();
}

/*
 * The basic plan for this is your regular everyday Turing machine.
 * Each time the machine picks up a non-null character, it processes it
 * according to its current state. If it's seen an escape, it's got
 * ESCFG set; if it's seen the Shift-Out character, it'll have SPECIAL
 * set until it's seen Shift-In; if it's put out an escape sequence
 * which for the Diablo 1600 requires an argument, it'll have ONEMORE
 * set.
 */
main()
{
	register c;             /* the current character */
	extern fin,fout;        /* i/o buffers */

	fin = dup(0);
	fout = dup(1);
	if ((signal(2, 1) & 01) == 0)
		signal(2, &restore);
	gtty(1, mode);
	svmode = mode[2];
	mode[2] =& 0353;        /* turn off cr-lf, lcase, & delays  */
	mode[2] =| 061002;      /* turn off tabs and on certain delays */
	stty(1, mode);
	flags = TEXTMD;

	while((c = getchar()) != 0) {

		if(flags & ESCFG) {
			/* immediately following an escape char */
			flags =& ~ESCFG;
			switch(c) {
				case M37HLF:	/* half line-feed */
					putesc(DBOHLF);
					break;

				case M37RHLF:	/* reverse half line-feed */
					putesc(DBORHLF);
					break;

				case M37RLF:	/* reverse line-feed */
					putesc(DBORLF);
					break;

				case DBOAHT:    /* abs. horizontal tab */
				case DBOAVT:    /* abs. vertical tab */
				case DBOHMI:    /* horizontal motion index */
				case DBOVMI:    /* vertical motion index */
					flags =| ONEMORE;

				default:        /* non-M37, -GSI char */
					putesc(c);
			}
			continue;
		}

		if(flags & ONEMORE) {
			/* argument to one of the last above */
			flags =& ~ONEMORE;
			putchar(c);
			continue;
		}

		if(flags & SPECIAL)
			/*
			 * special character gets output and returns non-zero
			 */
			if(special(c)) continue;

		switch(c) {
			case ESC:       /* next char is special */
				flags =| ESCFG;
				break;

			case GREEK:     /* following chars may be greek */
				flags =| SPECIAL;
				break;

			case ENGLISH:   /* no more greek chars */
				flags =& ~SPECIAL;
				break;

			case GSIPLOT:   /* toggle plot mode */
				if(flags & TEXTMD) {
					putesc(DBOPLOT);
					flags =& ~TEXTMD;
				} else {
					putesc(DBOUPLT);
					flags =| TEXTMD;
				}
				break;

			case GSIUPLT:   /* turn off plot mode */
				putesc(DBOUPLT);
				flags =| TEXTMD;
				putchar(c);
				break;

			case LF:        /* carriage return-line feed */
				if(flags & TEXTMD) putchar(CR);
				putchar(c);
				break;

			case CR:        /* carriage return */
				flags =| TEXTMD;
				putchar(c);
				break;

			default:        /* normal print char */
				putchar(c);
		}
	}
	flush();
	restore();
}

/*
 * puts out the escape char before its argument
 */
putesc(c)
char c;
{
	putchar(ESC);
	putchar(c);
}

/*
 * runs through the table: if the char is there, the greek
 * gets plotted and a non-zero return is made; else zero
 * is returned.
 */
special(c)
char c;
{
	register i;

	for(i = 0; tab[i] != 0; i++)
		if(c == tab[i]) {
			plot(trans[i]);
			return(1);
		}
	return(0);
}

/*
 * given an array of ints (s), plot makes sure the Diablo
 * is in plot mode and then proceeds to put out the elements
 * of the array. a negative element means the next char gets
 * output that many times; a zero element ends the array.
 * if the Diablo is not in plot mode, the carriage is advanced
 * by one space.
 */
plot(s)
int *s;         /* array to be output */
{
	register *i, c;

	if(flags & TEXTMD) putesc(DBOPLOT);
	i = s;
	while(c = *i++) {
		if(c < 0) nplot(c,*i++);
		else putchar(c);
	}
	if(flags & TEXTMD) {
		putesc(DBOUPLT);
		putchar(' ');
	}
}

nplot(n,c)
int n;          /* negative the number of times to put out */
char c;         /* the character c */
{
	while(n++) putchar(c);
}

