/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Rstate.c	1.48 89/01/11
*/

/*
**	Read a state file and structure it in memory.
*/

#define	STDIO

#include	"global.h"
#include	"address.h"
#include	"debug.h"
#include	"state.h"

#define	STATE_FILE_DATA
#include	"statefile.h"

#undef	Extern
#define	Extern
#include	"node.h"

#include	<setjmp.h>


#ifndef	DEFUNCT_TIME
#define	DEFUNCT_TIME	(60*60*24*7L)	/* Statefiles time out after 1 week */
#endif

Address *	AddressP;
Entry *		RSource;
char *		RSrcAddr;
Time_t		RSrcDate;

static char *	Buf;
static char *	BufEnd;
static bool	IgnoreCRC;
static int	Last_C;
static FILE *	RStateFd;
static Crc_t	StCRC;
static int	TokenLength;
static int	TokenCount;

extern jmp_buf	NameErrJmp;
extern char *	HomeNode;

Token		Rcheck();
void		SetTop();

static void	RsetFlags();
static Token	Rtoken();



void
Rstate(fd, foreign, sourceaddr, sourcedate, ignoreCRC)
	FILE *		fd;
	bool		foreign;
	char *		sourceaddr;
	Time_t		sourcedate;
	bool		ignoreCRC;
{
	register Node *	np;
	register Data *	dp;
	register Node *	op;
	register Entry *nep;
	register Entry *oep;
	register char *	buf;
	register int	port;
	register Token	token;
	static Token	last_token;
	char *		errs1;
	char *		errs2;
	Data		null_data;
	char		temp[TOKEN_SIZE+2];

	Trace4(1, "Rstate %d %s %s", fileno(fd), foreign?"true":"false", sourceaddr==NULLSTR?"":sourceaddr);

	if ( TokenLength = setjmp(NameErrJmp) )
	{
		Error("Bad home nodename \"%s\" at char %d", HomeNode, TokenLength-1);
		return;
	}

	if ( Home == (Entry *)0 )
	{
		Home = Enter(HomeNode, NodeHash);

		if ( !(Home->e_states & S_FOUND) )
		{
			Home->e_states |= S_FOUND;
			NodeCount++;
			ClearDomains(Home, true);
		}
	}

	if ( foreign )
	{
		RSrcAddr = sourceaddr;
		RSrcDate = sourcedate;

		AddressP = SplitAddress(sourceaddr);

		SetHierarchy(NULLSTR, &ForeignHier);
	}
	else
		SetHierarchy(NULLSTR, &DomHier);

	Buf = temp;
	BufEnd = &temp[TOKEN_SIZE+1];
	Last_C = EOF;
	IgnoreCRC = ignoreCRC;
	RStateFd = fd;
	RSource = (Entry *)0;
	StCRC = 0;
	TokenCount = 0;

	token = t_start;
	last_token = t_start;

	np = (Node *)0;
	oep = (Entry *)0;

	if ( TokenLength = setjmp(NameErrJmp) )
	{
		errs1 = "Bad node name in %s statefile: \"%s\" (char %d)";
		errs2 = foreign?"foreign":"home";

		if ( ignoreCRC )
		{
			Warn(errs1, errs2, &temp[1], TokenLength-1);
			token = last_token;
		}
		else
		{
			Error(errs1, errs2, &temp[1], TokenLength-1);
			return;
		}
	}

	if ( foreign )
		port = 1;
	else
		port = 0;

	buf = &temp[1];

	for ( ;; )
	{
		switch ( token = Rtoken(Accept[(int)token][port]) )
		{
		case t_node:
			last_token = t_nextnode;

			if ( TokenLength == 0 )
				continue;	/* Probably C_EOF next */

			nep = Enter(buf, NodeHash);
			np = nep->e_node;
			oep = (Entry *)0;

			Trace2(2, "NODE \"%s\"", nep->e_name);

			if ( foreign )
			{
				if ( (token = Rcheck(nep)) == t_eof )
				{
					AnteState = true;
					return;
				}
				if ( token != t_node )
					continue;	/* Don't mark it FOUND */
			}

			last_token = token;

			if ( !(nep->e_states & S_FOUND) )
			{
				NodeCount++;
				nep->e_states |= S_FOUND;

				if ( !foreign && strcmp(nep->e_name, buf) != STREQUAL )
					(void)strcpy(nep->e_name, buf);	/* Fix case */
			}

			continue;

		case t_nodehier:
			if ( !foreign )
			{
				if ( np->n_hierarchy != NULLSTR )
					free(np->n_hierarchy);
				np->n_hierarchy = newstr(buf);
			}
			else
			{
				/*
				**	Link hierarchy
				*/

				if
				(
					op->n_hierarchy != NULLSTR
					&&
					strccmp(buf, op->n_hierarchy) != STREQUAL
				)
				{
					register Link **lpp;

					/*
					**	This node thinks that this link
					**	has a different hierarchy
					**	==> different link?
					*/

					if ( (lpp = IsLinked(nep, oep)) != (Link **)0 )
					{
						if
						(
							Links(oep) == 1
							||
							(
								op->n_domains.d_count
								==
								np->n_domains.d_count
								&&
								(
									op->n_domains.d_count == 0
									||
									op->n_domains.d_head->d_entry
									==
									np->n_domains.d_head->d_entry
								)
								&&
								(RSrcDate - op->n_state) > DEFUNCT_TIME
							)
						)
						{
							Warn
							(
								"Changed hierarchy for node \"%s\" linked to \"%s\" from \"%s\" to \"%s\"",
								oep->e_name,
								nep->e_name,
								op->n_hierarchy,
								buf
							);
							free(op->n_hierarchy);
							op->n_hierarchy = NULLSTR;

						}
						else
						{
							dp = &null_data;
							Unlink(nep, lpp);
							Warn
							(
								"Broke putative link between \"%s.%s\" and \"%s.%s\" -- node of same name already exists called \"%s.%s\"",
								nep->e_name, np->n_hierarchy,
								oep->e_name, buf,
								oep->e_name, op->n_hierarchy
							);
						}
					}
				}
				if ( op->n_hierarchy == NULLSTR )
					op->n_hierarchy = newstr(buf);
				SetTop(op);	/* Last domain is top level */
			}
			continue;

		case t_link:
			last_token = t_nextlink;
			dp = &null_data;

			oep = Enter(buf, NodeHash);
			op = oep->e_node;

			Trace2(2, "Link \"%s\"", oep->e_name);

			if ( foreign && oep == Home )
				continue;

			if ( op == np )
			{
				errs1 = "Link to self for";
				errs2 = oep->e_name;
				break;
			}

			last_token = token;

			dp = MakeLink(nep, oep);

			if ( foreign && nep == RSource )
			{
				if ( !(oep->e_states & S_FOUND) )
				{
					NodeCount++;
					oep->e_states |= S_FOUND;
				}

				(void)MakeLink(oep, nep);

				dp->d_states &= ~FOREIGN_FLAGS;
				dp->d_cost = 0;
				dp->d_speed = 0;
			}

			continue;

		case t_alias:
			nep = Enter(buf, AliasHash);
			continue;

		case t_aliasv:
			if ( nep->e_states & S_FOUND )
			{
				errs1 = "Duplicate alias value";
				errs2 = buf;
				break;
			}
			else
			{
				nep->e_value = newstr(buf);
				AliasCount++;
				nep->e_states |= S_FOUND;
			}
			continue;

		case t_xalias:
			nep = Enter(buf, AliasHash);
			if ( foreign )
			{
				if
				(
					(nep->e_states & S_FOUND) 
					||
					Lookup(buf, NodeHash) != (Entry *)0
					||
					Lookup(buf, DomainHash) != (Entry *)0
				)
				{
					if
					(
						nep->e_value == NULLSTR
						||
						strcmp(nep->e_value, RSrcAddr) != STREQUAL
					)
						Warn
						(
							"Rejected export alias \"%s\" for \"%s\" -- already exists",
							buf, RSrcAddr
						);

					continue;	/* Reject if already exists */
				}
				nep->e_value = newstr(RSrcAddr);
				AliasCount++;
				nep->e_states |= S_FOUND;
			}
			else
			{
				if
				(
					!(nep->e_states & S_FOUND)
					||
					strcmp(nep->e_value, Home->e_name) != STREQUAL
				)
				{
					errs1 = "Bad export alias name";
					errs2 = buf;
					break;
				}
				nep->e_next = XAliases;
				XAliases = nep;
			}
			continue;
			
		case t_domain:
			oep = AddDomain(buf, &np->n_domains, S_FOUND);
			if ( nep == Home )
				oep->e_states |= S_OURDOM;
			else
				oep->e_states &= ~S_OTHDOM;
			continue;

		case t_topdom:
			if ( foreign )
			{
				if
				(
					(oep = Lookup(buf, DomainHash)) != (Entry *)0
					&&
					!(oep->e_states & S_OTHDOM)
				)
					continue;
				(void)AddDomain(buf, &np->n_domains, S_TOPDOM|S_OTHDOM|S_FOUND);
				continue;
			}
			oep = AddDomain(buf, &np->n_domains, S_FOUND|S_TOPDOM);
			if ( nep == Home )
				oep->e_states |= S_OURDOM;
			continue;

		case t_othdom:
			if ( foreign )
			{
				if
				(
					(oep = Lookup(buf, DomainHash)) != (Entry *)0
					&&
					!(oep->e_states & S_OTHDOM)
				)
					continue;
				(void)AddDomain(buf, &RSource->e_node->n_domains, S_TOPDOM|S_OTHDOM|S_FOUND);
				continue;
			}
			(void)AddDomain(buf, &np->n_domains, S_TOPDOM|S_OTHDOM|S_FOUND);
			continue;

		case t_domhier:
			if ( foreign )
				(void)AddDomain(buf, &ForeignHier, (States)0);
			else
				(void)AddDomain(buf, &DomHier, S_HIERDOM);
			continue;

		case t_state:
			{
				Time_t	t;

				if ( (t = atol(buf)) > np->n_state )
					np->n_state = t;
			}
			continue;

		case t_date:
#			if	NODE_STATS == 1
			{
				Time_t	t;

				if ( (t = atol(buf)) > np->n_date )
					np->n_date = t;
			}
#			endif	NODE_STATS
			continue;

		case t_recvd:
#			if	NODE_STATS == 1
			np->n_recvd += (ulong)atol(buf);
#			endif	NODE_STATS
			continue;

		case t_sent:
#			if	NODE_STATS == 1
			np->n_sent += (ulong)atol(buf);
#			endif	NODE_STATS
			continue;

		case t_passto:
#			if	NODE_STATS == 1
			np->n_passto += (ulong)atol(buf);
#			endif	NODE_STATS
			continue;

		case t_passfrom:
#			if	NODE_STATS == 1
			np->n_passfrom += (ulong)atol(buf);
#			endif	NODE_STATS
			continue;

		case t_time:
#			if	LINK_STATS == 1
			dp->d_time += (ulong)atol(buf);
#			endif	LINK_STATS
			continue;

		case t_bytes:
#			if	LINK_STATS == 1
			dp->d_bytes += (ulong)atol(buf);
#			endif	LINK_STATS
			continue;

		case t_speed:
			if ( foreign && nep != RSource )
				continue;
			dp->d_speed = (ulong)atol(buf);
			continue;

		case t_cost:
			if ( foreign && nep != RSource )
				continue;
			dp->d_cost = (ulong)atol(buf);
			continue;

		case t_nflags:
			RsetFlags(&nep->e_states, buf, foreign);
			continue;

		case t_lflags:
			RsetFlags(&dp->d_states, buf, foreign);
			continue;

		case t_spooler:
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_spooler != NULLSTR )
				free(np->n_handlers->h_spooler);
			np->n_handlers->h_spooler = newstr(buf);
			continue;

		case t_connector:
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_connector != NULLSTR )
				free(np->n_handlers->h_connector);
			np->n_handlers->h_connector = newstr(buf);
			continue;

		case t_caller:
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_caller != NULLSTR )
				free(np->n_handlers->h_caller);
			np->n_handlers->h_caller = newstr(buf);
			continue;

		case t_filter:
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_filter != NULLSTR )
				free(np->n_handlers->h_filter);
			np->n_handlers->h_filter = newstr(buf);
			continue;

		case t_comment:
#			if	ALL_COMMENTS != 1
			if ( nep != Home )
				continue;
#			endif	ALL_COMMENTS != 1
			if ( np->n_comment != NULLSTR )
				free(np->n_comment);
			np->n_comment = newstr(buf);
			continue;

		case t_handler:
			if ( oep == (Entry *)0 || !(oep->e_states & S_OURDOM) )
			{
				errs1 = "Unexpected domain handler";
				errs2 = buf;
				break;
			}
			if ( oep->e_handler != NULLSTR )
				free(oep->e_handler);
			oep->e_handler = newstr(buf);
			continue;

		case t_eof:
			goto out;

		case t_error:
			errs1 = "Illegal statefile token";
			errs2 = buf;
			break;

		case t_expect:
			errs1 = "Expected field missing";
			errs2 = "";
			break;

		default:
			Fatal2("Rstate: unknown token %d", (int)token);
		}

		if ( foreign )
		{
			Error("%s \"%s\"", errs1, errs2);
			AnteState = true;
		}
		else
			Warn("%s \"%s\" ignored", errs1, errs2);

		token = last_token;
	}

out:
	if
	(
		!ignoreCRC
		&&
		TokenCount > 0
		&&
		(
			getc(fd) != LOCRC(StCRC)
			||
			getc(fd) != HICRC(StCRC)
		)
	)
	{
		Error("bad %sstatefile CRC", foreign?"foreign ":"");
		AnteState = true;
		return;
	}

	if ( foreign )
	{
		FreeAddress(AddressP);

		if ( RSource == (Entry *)0 )
		{
			AnteState = true;
			return;
		}

		if ( sourcedate > 0 )
			RSource->e_node->n_state = sourcedate;

		CheckHierarchy(RSource, &ForeignHier);
		CheckSourceDomains(RSource);
	}
	else
	{
		register char *	cp;

		if ( (cp = Home->e_node->n_hierarchy) != NULLSTR )
			free(cp);
		Home->e_node->n_hierarchy = DomString(&DomHier, DOMAIN_SEP);
	}
}



/*
**	Read a separator followed by a token (name or number)
**	and return token type.
*/

static Token
Rtoken(accept)
	Tokens		accept;
{
	register char *	s = Buf;
	register int	c;
	register int	repeat_c = 0;
	register FILE *	fd = RStateFd;
	register char *	ep = BufEnd;
	register Token	token = t_null;

	Trace2(4, "Rtoken accept %#lx", (long)accept);

	if ( (c = Last_C) != EOF )
	{
		Last_C = EOF;
		goto next;
	}

	for ( ;; )
	{
		if ( (c = getc(fd)) == EOF )
		{
			Trace1(1, "Rtoken at EOF!");
			if ( token == t_null )
				return t_eof;
			goto eof;
		}

next:		*s++ = (char)c;

		if ( repeat_c == 0 || c == repeat_c )
		switch ( c )
		{
		case C_ALIAS:		c = (int)t_alias;	goto found;
		case C_ALIASV:		c = (int)t_aliasv;	goto found;
		case C_BYTES:		c = (int)t_bytes;	goto found;
		case C_CALLER:		c = (int)t_caller;	goto found;
		case C_COMMENT:		c = (int)t_comment;	goto found;
		case C_CONNECTOR:	c = (int)t_connector;	goto found;
		case C_COST:		c = (int)t_cost;	goto found;
		case C_DATE:		c = (int)t_date;	goto found;
		case C_DOMAIN:		c = (int)t_domain;	goto found;
		case C_DOMHIER:		c = (int)t_domhier;	goto found;
		case C_EOF:		c = (int)t_eof;		goto found;
		case C_FILTER:		c = (int)t_filter;	goto found;
		case C_HANDLER:		c = (int)t_handler;	goto found;
		case C_LFLAGS:		c = (int)t_lflags;	goto found;
		case C_LINK:		c = (int)t_link;	goto found;
		case C_NFLAGS:		c = (int)t_nflags;	goto found;
		case C_NODE:		c = (int)t_node;	goto found;
		case C_NODEHIER:	c = (int)t_nodehier;	goto found;
		case C_OTHDOM:		c = (int)t_othdom;	goto found;
		case C_PASSFROM:	c = (int)t_passfrom;	goto found;
		case C_PASSTO:		c = (int)t_passto;	goto found;
		case C_RECVD:		c = (int)t_recvd;	goto found;
		case C_SENT:		c = (int)t_sent;	goto found;
		case C_SPARE2:		c = (int)t_null;	goto found;
		case C_SPEED:		c = (int)t_speed;	goto found;
		case C_SPOOLER:		c = (int)t_spooler;	goto found;
		case C_STATE:		c = (int)t_state;	goto found;
		case C_TIME:		c = (int)t_time;	goto found;
		case C_TOPDOM:		c = (int)t_topdom;	goto found;
		case C_XALIAS:		c = (int)t_xalias;	goto found;

found:			if ( token == t_null )
			{
				if ( !(accept & (Tokens)(1L<<c)) )
				{
					if ( accept & T_EXPECT )
					{
						token = t_expect;
						goto backout;
					}
					if ( c == (int)t_eof )
					{
						token = (Token)c;
						goto backout;
					}
					token = t_null;
				}
				else
					token = (Token)c;

				if ( repeat_c )
					repeat_c = 0;
				else
				switch ( (Token)c )
				{
				case t_caller:
				case t_comment:
				case t_connector:
				case t_filter:
				case t_handler:
				case t_spooler:
					repeat_c = s[-1];
				}

				if ( s != &Buf[1] )
				{
					if ( !IgnoreCRC )
						StCRC = acrc(StCRC, Buf, --s - Buf);
					Trace2
					(
						3,
						"Ignore token: \"%s\"",
						ExpandString(Buf, s-Buf)
					);
					Buf[0] = *s;
					s = &Buf[1];
				}

				TokenCount++;
				continue;
			}
			else
			{
				if ( !repeat_c )
backout:				Last_C = *(unsigned char *)--s;
eof:				if ( (c = s - Buf) > 0 )
				{
					if ( !IgnoreCRC )
						StCRC = acrc(StCRC, Buf, c);
					TokenLength = --c;
				}
				else
					TokenLength = 0;
				if ( repeat_c )
				{
					TokenLength--;
					*--s = '\0';
				}
				else
					*s = '\0';
				Trace3
				(
					4,
					"Rtoken \"%s\": \"%s\"",
					TokenNames[(int)token],
					ExpandString(Buf, TokenLength+1)
				);
				return token;
			}
		}

		if ( s > ep )
		{
			TokenLength = (s - Buf) - 1;
			return t_error;
		}
	}
}



/*
**	Decode state flags from string.
*/

static void
RsetFlags(sp, cp, foreign)
	register States *sp;
	register char *	cp;
	bool		foreign;
{
	register int	c;
	register States	accept;

	if ( foreign )
	{
		accept = FOREIGN_FLAGS;

		if ( *sp & S_LAN )
			accept &= ~S_DOWN;
	}
	else
		accept = EXTERNAL_FLAGS;

	*sp &= ~accept;
	accept |= S_DOWN;

	while ( c = *cp++ )
		*sp |= (1 << (c-'A')) & accept;
}



/*
**	Check foreign node integrity.
*/

Token
Rcheck(ep)
	register Entry *ep;
{
	register Node *	np;

	Trace2(2, "Rcheck(%s)", ep->e_name);

	np = ep->e_node;

	if ( RSource == (Entry *)0 )
	{
		register int	i;
		char *		hierarchy;

		/*
		**	This is first node in statefile,
		**	but hierarchy should already have been read.
		*/

		if ( (RSource = Enter(AddressP->ad_node, NodeHash)) == Home )
		{
			Error
			(
				"Foreign node name same as Home: \"%s\"",
				RSrcAddr
			);
			return t_eof;
		}

		if ( RSource != ep )
		{
			Error
			(
				"Foreign node name not same as statefile: \"%s\" != \"%s\"",
				RSrcAddr,
				ep->e_name
			);
			return t_eof;
		}

		if
		(
			ForeignHier.d_head == (Domain *)0
			&&
			AddressP->ad_domains >= 2
		)
		{
			/*
			**	No stated hierarchy, but neighbours think
			**	there is, so use domains from source address.
			*/

			for ( i = AddressP->ad_domains ; i > 0 ; i++ )
			{
				(void)AddDomain
				(
					AddressP->ad_strings[i],
					&RSource->e_node->n_domains,
					S_FOUND
				);
				(void)AddDomain
				(
					AddressP->ad_strings[i],
					&ForeignHier,
					(States)0
				);
			}

			Warn("Made up hierarchy for node from source address \"%s\"", RSrcAddr);

			hierarchy = newstr(strchr(RSrcAddr, DOMAIN_SEP)+1);
		}
		else
		{
			static char	sep[] = {DOMAIN_SEP, '\0'};

			hierarchy = DomString(&ForeignHier, DOMAIN_SEP);

#			ifndef	BUG_FREE_SUN3
			RSrcAddr = concat(AddressP->ad_node, sep, hierarchy, NULLSTR);
#			else	/* BUG_FREE_SUN3 */
			for ( i = AddressP->ad_domains ; i > 0 ; i-- )
			{
				register Entry *	dp;
				register Domain *	hp;

				if ( (dp = Lookup(AddressP->ad_strings[i], DomainHash)) != (Entry *)0 )
				{
					for ( hp = ForeignHier.d_head ; hp != (Domain *)0 ; hp = hp->d_next )
						if ( hp->d_entry == dp )
							break;
				}
				else
					hp = (Domain *)0;

				if ( hp == (Domain *)0 )
				{
					Error("Source address \"%s\" inconsistent with hierarchy \"%s\"", RSrcAddr, hierarchy);
					return t_eof;
				}
			}
#			endif	/* BUG_FREE_SUN3 */
		}

		if ( RSrcDate > 0 )
		{
			extern Time_t	Time;

			if ( np->n_state >= RSrcDate && np->n_state < Time )
			{
				Report2("foreign statefile for \"%s\" date earlier than last", RSource->e_name);

				return t_eof;
			}

#			if	NODE_STATS == 1
			if ( RSrcDate > np->n_date || np->n_date > Time )
				np->n_date = RSrcDate;
#			endif	NODE_STATS == 1

			if
			(
				np->n_hierarchy != NULLSTR
				&&
				hierarchy != NULLSTR
				&&
				strcmp(np->n_hierarchy, hierarchy) != STREQUAL
				&&
				(RSrcDate - np->n_state) < DEFUNCT_TIME
				&&
				(
					Home->e_node->n_hierarchy == NULLSTR
					||
					strcmp(Home->e_node->n_hierarchy, hierarchy) != STREQUAL
				)
				&&
				(
					np->n_domains.d_head->d_entry
					!=
					ForeignHier.d_last->d_entry
				)
			)
			{
				Warn
				(
					"state discarded for node \"%s\" because hierarchy changed from \"%s\" to \"%s\"",
					RSource->e_name,
					np->n_hierarchy,
					hierarchy
				);
				return t_eof;
			}
		}

		/*
		**	Clear all variables which this statefile may legally set
		*/

		{
			register Link **lpp;

			for ( lpp = &np->n_l_first ; *lpp != (Link *)0 ; )
				if ( (*lpp)->l_entry != Home )
					Unlink(RSource, lpp);
				else
					lpp = &(*lpp)->l_next;
		}

		if ( np->n_hierarchy != NULLSTR )
			free(np->n_hierarchy);
		np->n_hierarchy = hierarchy;

		if ( np->n_comment != NULLSTR )
		{
			free(np->n_comment);
			np->n_comment = NULLSTR;
		}
	}

	if
	(
		ep == Home
		||
		(
			ep != RSource
			&&
			(
				np->n_state > 0					/* Had a statefile direct */
				||
				(
					RSource->e_node->n_state > 0		/* Had a statefile from source before */
					&&
					IsLinked(ep, RSource) == (Link **)0	/* This node not linked to source */
				)
			)
		)
	)
		return t_nextnode;		/* Skip line */

	ClearDomains(ep, true);
	ep->e_states &= ~FOREIGN_FLAGS;

	return t_node;
}



/*
**	Mark top domain in hierarchy.
*/

void
SetTop(np)
	register Node *	np;
{
	register char *	cp;
	register Entry *ep;

	Trace2(3, "SetTop(%s)", np->n_hierarchy);

	if ( (cp = strrchr(np->n_hierarchy, DOMAIN_SEP)) == NULLSTR )
		cp = np->n_hierarchy;
	else
		cp++;

	ep = Enter(cp, DomainHash);
	ep->e_states |= S_TOPDOM;
}
