/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Locking.c	1.12 88/11/25
*/

/*
**	Locking primitives for Unices without them.
*/

#define	FILE_CONTROL
#define	STAT_CALL

#include	"global.h"

/*
**	If a lock file is older than LOCK_TIMEOUT it is probably bogus.
*/

#define	LOCK_TIMEOUT	(20*60)
#define	LOCK_SLEEP	(2*60)

#if	SHARE_SPOOL

#include	"debug.h"

#include	<errno.h>


static char *	LockName;


/*
**	Absolute locking: 1 access/time.
*/

int
_Lock(file, fd, type)
	char *		file;
	register int	fd;
	Lock_t		type;
{
	register int	fib_1;
	register int	fib_2;
	register int	total;
	register int	f;
	struct stat	statb;

	if ( LockName != NULLSTR )
	{
		Fatal3("attempt to lock \"%s\" while \"%s\" exists", file, LockName);
		return SYSERROR;
	}

	if ( access(file, 02) == SYSERROR )
		return type==for_writing?SYSERROR:0;	/* If can't write it, don't lock it */

	LockName = concat(file, ".l", NULLSTR);

	fib_1 = 1;
	fib_2 = 1;
	total = 0;

	do
	{
		if ( link(file, LockName) != SYSERROR )
			return 0;

		if ( errno == EACCES )
		{
			free(LockName);
			LockName = NULLSTR;

			if ( type == for_reading )
				return 0;
			else
				return SYSERROR;
		}

		if ( errno != EEXIST )
		{
			SysWarn("could not make lockfile %s", LockName);
			free(LockName);
			LockName = NULLSTR;

			return SYSERROR;
		}

		if ( stat(LockName, &statb) != SYSERROR )
		{
			long		cur_time;

			(void)time(&cur_time);

			if ( cur_time - statb.st_ctime > LOCK_TIMEOUT )
			{
				Warn("timeout on lockfile %s", LockName);

				if ( unlink(LockName) == SYSERROR )
					SysWarn("could not unlink lockfile %s", LockName);
			}
		}

		(void)sleep(fib_1);

		total += fib_1;
		f = fib_1;
		fib_1 = fib_2;
		fib_2 += f;
	}
	while (total < LOCK_SLEEP);

	return SYSERROR;
}



void
_UnLock()
{
	if ( LockName != NULLSTR )
	{
		if (unlink(LockName) == SYSERROR)
			SysWarn("couldn't remove lock %s", LockName);
		free(LockName);
		LockName = NULLSTR;
	}
}

#else	SHARE_SPOOL

#if	AUTO_LOCKING == 0 && FLOCK == 0 && SYS_LOCKING == 0 && LOCKF == 0 && V8 == 0
#if	SYSTEM >= 5 && SYSVREL >= 2

int
_Lock(file, fd, type)
	char *		file;
	register int	fd;
	Lock_t		type;
{
	struct flock l;

	if ( access(file, 02) == SYSERROR )
		return type==for_writing?SYSERROR:0;	/* If can't write it, don't lock it */

	l.l_whence = 1;
	l.l_start = 0L;
	l.l_len = 0L;
	l.l_type = (type == for_reading) ? F_RDLCK : F_WRLCK;
	return fcntl(fd, F_SETLKW, &l);
}



void
_UnLock(fd)
	register int	fd;
{
	struct flock l;

	l.l_whence = 1;
	l.l_start = 0L;
	l.l_len = 0L;
	l.l_type = F_UNLCK;
	(void)fcntl(fd, F_SETLKW, &l);
}

#else	SYSTEM >= 5 && SYSVREL >= 2

#include	"debug.h"

#include	<errno.h>


static char *	LockName;


/*
**	Absolute locking: 1 access/time.
*/

int
_Lock(file, fd, type)
	char *		file;
	register int	fd;
	Lock_t		type;
{
	register int	fib_1;
	register int	fib_2;
	register int	total;
	register int	f;
	struct stat	statb;

	if ( LockName != NULLSTR )
	{
		Fatal3("attempt to lock \"%s\" while \"%s\" exists", file, LockName);
		return SYSERROR;
	}

	if ( access(file, 02) == SYSERROR )
		return type==for_writing?SYSERROR:0;	/* If can't write it, don't lock it */

	LockName = concat(file, ".l", NULLSTR);

	fib_1 = 1;
	fib_2 = 1;
	total = 0;

	do
	{
#if	FCNTL && defined(O_EXCL)
#if	KILL_0
		int	mypid;
#endif	KILL_0

		if ( (f = open(LockName, O_WRONLY | O_CREAT | O_TRUNC | O_EXCL, 0664)) != SYSERROR )
		{
#if	KILL_0
			mypid = getpid();
			(void) write(f, &mypid, sizeof mypid);
#endif	KILL_0
			(void) close(f);
			return 0;
		}
#else
		if ( link(file, LockName) != SYSERROR )
			return 0;
#endif	FCNTL && defined(O_EXCL)

		if ( errno == EACCES )
		{
			free(LockName);
			LockName = NULLSTR;

			if ( type == for_reading )
				return 0;
			else
				return SYSERROR;
		}

		if ( errno != EEXIST )
		{
			SysWarn("could not make lockfile %s", LockName);
			free(LockName);
			LockName = NULLSTR;

			return SYSERROR;
		}

#if	FCNTL && KILL_0 && defined(O_EXCL)
		if
		(
			(f = open(LockName, O_RDONLY, 0)) != SYSERROR
			&&
			read(f, &mypid, sizeof mypid) == sizeof mypid
			&&
			close(f) != SYSERROR
			&&
			kill(mypid, 0) == SYSERROR
			&&
			errno == ESRCH
		)
		{
			Warn("dead proc has lockfile %s", LockName);

			if ( unlink(LockName) == SYSERROR )
				SysWarn("could not unlink lockfile %s", LockName);
		}
		else
#endif	FCNTL && KILL_0 && defined(O_EXCL)
		if ( stat(LockName, &statb) != SYSERROR )
		{
			long		cur_time;

			(void)time(&cur_time);

			if ( cur_time - statb.st_ctime > LOCK_TIMEOUT )
			{
				Warn("timeout on lockfile %s", LockName);

				if ( unlink(LockName) == SYSERROR )
					SysWarn("could not unlink lockfile %s", LockName);
			}
		}

		(void)sleep(fib_1);

		total += fib_1;
		f = fib_1;
		fib_1 = fib_2;
		fib_2 += f;
	}
	while (total < LOCK_SLEEP);

	return SYSERROR;
}



void
_UnLock()
{
	if ( LockName != NULLSTR )
	{
		if (unlink(LockName) == SYSERROR)
			SysWarn("couldn't remove lock %s", LockName);
		free(LockName);
		LockName = NULLSTR;
	}
}

#endif	SYSTEM >= 5 && SYSVREL >= 2

#else	AUTO_LOCKING == 0 && FLOCK == 0 && SYS_LOCKING == 0 && LOCKF == 0 && V8 == 0

#if	V8 == 1

/*
**	Absolute locking: 1 access/time.
*/

int
_Lock(file, fd, type)
	char *		file;
	register int	fd;
	Lock_t		type;
{
	register int	fib_1;
	register int	fib_2;

	if ( access(file, 02) == SYSERROR )
		return type==for_writing?SYSERROR:0;	/* If can't write it, don't lock it */

#	ifdef	FIOABLOCK

	return ioctl(fd, FIOABLOCK, 0);

#	else	FIOABLOCK

	fib_1 = 1;
	fib_2 = 1;

	for ( ;; )
	{
		register int	f;

		if ( ioctl(fd, FIOALOCK, 0) != SYSERROR )
			return 0;

		if ( errno != EPERM )
		{
			if ( type == for_reading )
				return 0;
			else
				return SYSERROR;
		}

		(void)sleep(fib_1);

		f = fib_1;
		fib_1 = fib_2;
		if ( fib_2 < LOCK_SLEEP )
			fib_2 += f;
	}
#	endif	FIOABLOCK
}

#endif	V8 == 1

#if	LOCKF == 1

lockf_all(fd, op)
{
	register long	l;
	register int	r;

	l = lseek(fd, 0L, 1);	/* Remember where we were */
	(void)lseek(fd, 0L, 0);	/* Start of file */
	r = lockf(fd, op, 0);	/* Lock whole file */
	(void)lseek(fd, l, 0);	/* Back where we were */

	return r;
}

#endif	LOCKF == 1

int
_lock()
{
}


#endif	AUTO_LOCKING == 0 && FLOCK == 0 && SYS_LOCKING == 0 && LOCKF == 0 && V8 == 0
#endif	SHARE_SPOOL
