/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "88/09/05 @(#)purge.c	1.13";

/*
**	Purge the work/spool directories by scanning the command directories.
*/

char *	Usage		= "[-[A][T<level>][W]] [message_id ...]";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"command.h"
#include	"debug.h"
#include	"spool.h"
#include	"Stream.h"

#include	<ndir.h>


/*
**	Parameters set from arguments
*/

bool	All;				/* Remove all unreferenced files - no exceptions! */
char *	Name;				/* Program invoked name */
int	Traceflag;			/* Trace level */
bool	Warnings;			/* Whinge */

/*
**	Structure to count references to a file
*/

#define	WORKNAMLEN	14

typedef struct data_file * DF_p;

typedef struct data_file
{
	DF_p	df_great;		/* Branches of binary tree */
	DF_p	df_less;
	char	df_name[WORKNAMLEN+1];	/* Last componenent of path name in WORKDIR */
	short	df_refs;		/* References */
	short	df_length;		/* Length of name */
}
	DF;

#define	HASH_SIZE	127		/* Size of hash-table base of binary trees */
#if	CRC_HASH == 1
#define	hashname(A,L)	(((unsigned)acrc(0,A,L))%HASH_SIZE)
#else	CRC_HASH == 1
#define	HASH(H,C)	H+=H+(C-' ')	/* The hashing function */
#endif	CRC_HASH == 1

/*
**	Structure to remember timed-out files.
*/

typedef struct UnlinkN *UN_p;
typedef struct UnlinkN
{
	UN_p	un_next;
	DF_p	un_dfp;
}
		UnlinkN;

UN_p	Unfreelist;
UN_p	Unlist;

/*
**	Structure for WORKDIR info.
*/

typedef struct Workdir *	WD_p;

typedef struct Workdir
{
	WD_p	next;
	WD_p	great;
	WD_p	less;
	char *	name;
	int	length;
	DF_p	table[HASH_SIZE];
}
	Workdir;

WD_p	Workdirs;
WD_p	WDirsTable[HASH_SIZE];
DF_p	SearchTable[HASH_SIZE];

/*
**	Miscellaneous info.
*/

#if	DEBUG
long	entries, hash_dups;
#endif
char *	Cantopen	= "can't open \"%s\"";
int	Removed;			/* Count of files removed */
bool	SearchFiles;			/* Explicit file name arg used */
char	Spooldir[]	= SPOOLDIR();
int	Spooldlen	= sizeof(Spooldir)-1;
Time_t	Time;
long	Time_to_die;
WD_p	NNworkdirp;			/* The place where daemons create their incoming messages */

#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))
#define	HOUR		(60*60)

/*
**	Routines
*/

void	matchsearch(), purge(), remove(), searchcoms(), searchstatus(), usage();
bool	read_com();
DF_p	enter(), lookup();
WD_p	newworkdir(), workdir();


int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Time = time((long *)0);

	NNworkdirp = newworkdir(WORKDIR());
	(void)newworkdir(WORK2DIR());

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					All = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'W':
					Warnings = true;
					continue;

				default:
					usage("unrecognised flag '%c'", c);
					exit(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			if ( enter(*argv, SearchTable) == (DF_p)0 )
			{
				usage("bad name for file \"%s\"", *argv);
				exit(1);
			}

			SearchFiles = true;
		}
	}

	purge();

	if ( SearchFiles )
		matchsearch();
	else
		(void)fprintf(stdout, "%d file%s removed.\n", Removed, Removed==1?"":"s");

	Trace3(1, "%ld entries, %ld hash duplications", entries, hash_dups);

	exit(0);
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	(void)fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}



/*
**	Lookup a directory in work directory table.
*/

WD_p
workdir(dir)
	register char *	dir;
{
	register char *	cp1;
	register char *	cp2;
	register WD_p	wdp;
	register int	c;
	register int	length = strlen(dir);

	Trace3(2, "workdir(%.*s)", Spooldlen + 4*WORKNAMLEN, dir);

	DODEBUG
	(
		if ( dir[length-1] != '/' )
			Fatal("Bad name passed to workdir(%s)", dir);
	);

	for
	(
		wdp = WDirsTable[hashname(dir, length)] ;
		wdp != (WD_p)0 ;
		wdp = (c&1) ? wdp->great : wdp->less
	)
	{
		if ( (c = length - wdp->length) != 0 )
			continue;

		for
		(
			cp1 = dir+length, cp2 = &wdp->name[length] ;
			cp1 != dir ;
		)
			if ( c = *--cp1 ^ *--cp2 )
				break;

		if ( c == 0 && cp1 == dir )
			return wdp;
	}

	return (WD_p)0;
}



/*
**	Add a directory to work directory table.
*/

WD_p
newworkdir(dir)
	register char *	dir;
{
	register char *	cp1;
	register char *	cp2;
	register WD_p *	wdpp;
	register int	c;
	register int	length = strlen(dir);
	register DF_p *	dfpp;

	Trace2(1, "newworkdir(%s)", dir);

	DODEBUG
	(
		if ( dir[length-1] != '/' )
			Fatal("Bad name passed to newworkdir(%s)", dir);
	);

	for
	(
		wdpp = &WDirsTable[hashname(dir, length)] ;
		*wdpp != (WD_p)0 ;
		wdpp = (c&1) ? &((*wdpp)->great) : &((*wdpp)->less)
	)
	{
		if ( (c = length - (*wdpp)->length) != 0 )
			continue;

		for
		(
			cp1 = dir+length, cp2 = &(*wdpp)->name[length] ;
			cp1 != dir ;
		)
			if ( c = *--cp1 ^ *--cp2 )
				break;

		if ( c == 0 && cp1 == dir )
		{
			free(dir);
			return *wdpp;
		}
	}

	*wdpp = Talloc(Workdir);

	(*wdpp)->next = Workdirs;
	Workdirs = (*wdpp);

	(*wdpp)->great = (WD_p)0;
	(*wdpp)->less = (WD_p)0;
	(*wdpp)->name = dir;
	(*wdpp)->length = length;

	for ( length = 0, dfpp = (*wdpp)->table ; length < HASH_SIZE ; length++ )
		*dfpp++ = (DF_p)0;

	return *wdpp;
}



/*
**	Search all possible command directories
**	for references to files in WORKDIR,
**	then remove any file in WORKDIR for which there is no reference.
*/

void
purge()
{
	register WD_p		wdp;
	register struct direct *direp;
	DIR *			dirp;

	/*
	**	Find the link directories.
	*/

	if ( (dirp = opendir(Spooldir)) == NULL )
	{
		Syserror(Cantopen, Spooldir);
		return;
	}

	while ( (direp = readdir(dirp)) != NULL )
		if ( direp->d_name[0] != WORKFLAG && !dots(direp->d_name) )
		{
			register char *	fp;

			fp = concat(Spooldir, direp->d_name, "/", NULLSTR);

			searchcoms(fp, true);

			free(fp);
		}

	closedir(dirp);

	/*
	**	Search the holding directories.
	*/

	searchcoms(BADDIR(), false);
	searchcoms(REROUTEDIR(), true);

	if ( SearchFiles )
		return;

	/*
	**	Now remove any files unreferenced by the previous searches.
	*/

	for ( wdp = Workdirs ; wdp != (WD_p)0 ; wdp = wdp->next )
	{
		if ( (dirp = opendir(wdp->name)) == NULL )
		{
			SysWarn(Cantopen, wdp->name);
			continue;
		}

		while ( (direp = readdir(dirp)) != NULL )
		{
			register DF_p	dfp;

			if
			(
				!dots(direp->d_name)
				&&
				(
					(dfp = lookup(direp->d_name, wdp->table)) == (DF_p)0
					||
					dfp->df_refs == 0
				)
			)
			{
				register char *	fp;

				fp = concat(wdp->name, direp->d_name, NULLSTR);

				remove(fp);

				free(fp);
			}
		}

		closedir(dirp);
	}
}



/*
**	Search a directory for command files and enter names of data files.
*/

void
searchcoms(dir, timeouts)
	char *			dir;
	bool			timeouts;
{
	register DIR *		dirp;
	register struct direct *direp;

	Trace2(1, "searchcoms \"%s\"", dir);

	if ( (dirp = opendir(dir)) == NULL )
	{
		struct stat	statb;

		if
		(
			stat(dir, &statb) == SYSERROR
			||
			(statb.st_mode & S_IFMT) == S_IFDIR
		)
			SysWarn(Cantopen, dir);

		return;
	}

	while ( (direp = readdir(dirp)) != NULL )
	{
		register UN_p	unp;
		register int	fd;
		register char *	fp;
		Time_t		mtime;

		/*
		**	Find valid command file.
		*/

		switch ( direp->d_name[0] )
		{
		default:
			if ( strcmp(direp->d_name, STATUSFILE) == STREQUAL )
			{
				/*
				**	Check for NNdaemon temporaries.
				*/
	
				fp = concat(dir, direp->d_name, NULLSTR);
				searchstatus(fp);
				free(fp);
			}
			else
			if ( strcmp(direp->d_name, LSPOOLDIR) == STREQUAL )
			{
				/*
				**	Add potential new work directory
				*/

				(void)newworkdir(concat(dir, direp->d_name, "/", NULLSTR));
			}

			continue;

		case SMALL_ID:
		case MEDIUM_ID:
		case LARGE_ID:
			break;
		}

		Trace2(2, "found \"%s\"", direp->d_name);

		fp = concat(dir, direp->d_name, NULLSTR);

		if ( (fd = open(fp, O_READ)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read commands file \"%s\"", fp);

			free(fp);
			continue;
		}

		Time_to_die = 0;

		if ( !ReadCom(fd, &mtime, read_com) && Warnings )
			Warn("bad commands file \"%s\"", fp);
		else
		if ( timeouts && Time_to_die > 0 && Time > (mtime + Time_to_die) )
		{
			Trace5(1, "\"%s\" timed out: time=%lu, mtime=%lu, ttd=%ld", fp, Time, mtime, Time_to_die);

			for ( unp = Unlist ; unp != (UN_p)0 ; unp = unp->un_next )
				unp->un_dfp->df_refs--;

			if ( !Traceflag && !SearchFiles )
			{
				(void)unlink(fp);
				Removed++;
			}
		}

		(void)close(fd);

		while ( (unp = Unlist) != (UN_p)0 )
		{
			Unlist = unp->un_next;
			unp->un_next = Unfreelist;
			Unfreelist = unp;
		}

		free(fp);
	}

	closedir(dirp);
}



/*
**	Search NNdaemon status file for receive temporaries,
**	(which are always made in WORKDIR).
*/

void
searchstatus(statusfile)
	char *		statusfile;
{
	register int	i;
	register int	fd;
	NN_state	NNstate;

	Trace2(1, "searchstatus \"%s\"", statusfile);

	for ( i = 0 ; i < 3 ; i++ )
	{
		if ( (fd = open(statusfile, O_READ)) != SYSERROR )
		{
			if ( read(fd, (char *)&NNstate, sizeof NNstate) == sizeof NNstate )
			{
				register Str_p	strp;

				if ( strcmp(NNstate.version, StreamSCCSID) != STREQUAL )
					Error("status file version, re-compile");

				for ( strp = inStreams ; strp < &inStreams[NSTREAMS] ; strp++ )
					if ( strp->str_recv.rh_id[0] != '\0' )
						(void)enter(strp->str_recv.rh_id, NNworkdirp->table);

				break;
			}

			(void)close(fd);
		}

		(void)sleep(1);
	}

	if ( fd != SYSERROR )
		(void)close(fd);
}



/*
**	Function called from "ReadCom" to process a command.
**	Enter any "unlink" file names that are in a workdir,
**	and remember them in temporary list in case they have timed-out.
*/

bool
read_com(name, base, range)
	char *	name;
	long	base;
	long	range;
{
	Trace5(3, "command \"%.*s %ld %ld\"", Spooldlen + 4*WORKNAMLEN, name, base, range);

	if ( range == 0 )
	{
		if ( base == 0 )
		{
			register WD_p	wdp	= (WD_p)0;
			register char *	cp;
			register char *	dir	= NULLSTR;

			if ( (cp = strrchr(name, '/')) != NULLSTR )
			{
				register int	c;

				c = *++cp;
				*cp = '\0';
				dir = newstr(name);
				*cp = c;

				if ( (wdp = workdir(dir)) != (WD_p)0 )
				{
					register DF_p	dfp;
enterit:
					if ( (dfp = enter(name + wdp->length, wdp->table)) != (DF_p)0 )
					{
						register UN_p	unp;
	
						if ( (unp = Unfreelist) == (UN_p)0 )
							unp = Talloc(UnlinkN);
						else
							Unfreelist = unp->un_next;
						
						unp->un_next = Unlist;
						Unlist = unp;
						unp->un_dfp = dfp;

						if ( dir != NULLSTR )
							free(dir);
						free(name);
						return true;
					}
				}
			}

			if ( wdp == (WD_p)0 && strncmp(name, Spooldir, Spooldlen) == STREQUAL )
			{
				struct stat	statb;

				if
				(
					stat(dir, &statb) != SYSERROR
					&&
					(statb.st_mode & S_IFMT) == S_IFDIR
				)
				{
					wdp = newworkdir(dir);
					dir = NULLSTR;
					goto enterit;
				}
			}

			if ( Warnings )
				Warn
				(
					"strange name in command file: \"%.*s\"",
					Spooldlen + 4*WORKNAMLEN,
					name
				);

			if ( dir != NULLSTR )
				free(dir);
			free(name);
			return false;
		}
		else
			Time_to_die = base;
	}

	free(name);
	return true;
}



/*
**	Enter a file name in hash table,
**	and increment reference count.
*/

DF_p
enter(namep, table)
	register char *	namep;
	DF_p *		table;
{
	register char *	cp1;
	register char *	cp2;
	register DF_p *	dfpp;
	register int	c;
	register int	len;

	if ( (len = strlen(namep)) > WORKNAMLEN )
	{
		Trace2(1, "enter name \"%s\" too long", namep);
		return (DF_p)0;
	}

	Trace2(2, "enter \"%s\"", namep);

#	if	DEBUG == 2
	if ( Traceflag > 3 )
	{
		Traceflag--;
		if ( lookup(namep, table) == (DF_p)0 )
			Trace(3, "\"%s\" hash = %#x", namep, hashname(namep, len));
		Traceflag++;
	}
#	endif	DEBUG == 2

	for
	(
		dfpp = &table[hashname(namep, len)] ;
		*dfpp != (DF_p)0 ;
		dfpp = (c&1) ? &((*dfpp)->df_great) : &((*dfpp)->df_less)
	)
	{
#		if	DEBUG == 2
		if ( Traceflag > 3 )
		{
			Traceflag--;
			if ( lookup(namep, table) == (DF_p)0 )
				Trace(3, "hash duplication \"%s\"", (*dfpp)->df_name);
			Traceflag++;
		}
		hash_dups++;
#		endif	DEBUG == 2

		if ( (c = len - (*dfpp)->df_length) != 0 )
			continue;

		for
		(
			cp1 = namep+len, cp2 = &(*dfpp)->df_name[len] ;
			cp1 != namep ;
		)
			if ( c = *--cp1 ^ *--cp2 )
				break;

		if ( c == 0 && cp1 == namep )
		{
			(*dfpp)->df_refs++;
			return *dfpp;
		}
	}

	DODEBUG(entries++);

	*dfpp = Talloc(DF);

	(*dfpp)->df_great = (DF_p)0;
	(*dfpp)->df_less = (DF_p)0;
	(void)strcpy((*dfpp)->df_name, namep);
	(*dfpp)->df_length = len;
	(*dfpp)->df_refs = 1;

	return *dfpp;
}



/*
**	Look-up file name in hash table
*/

DF_p
lookup(namep, table)
	register char *	namep;
	DF_p *		table;
{
	register char *	cp1;
	register char *	cp2;
	register DF_p	dfp;
	register int	c;
	register int	len = strlen(namep);

	for
	(
		dfp = table[hashname(namep, len)] ;
		dfp != (DF_p)0 ;
		dfp = (c&1) ? dfp->df_great : dfp->df_less
	)
	{
		if ( (c = len - dfp->df_length) != 0 )
			continue;

		for
		(
			cp1 = namep+len, cp2 = &dfp->df_name[len] ;
			cp1 != namep ;
		)
			if ( c = *--cp1 ^ *--cp2 )
				break;

		if ( c == 0 && cp1 == namep )
			return dfp;
	}

	return (DF_p)0;
}



#if	CRC_HASH == 0
/*
**	Calculate hash for a file name.
*/

int
hashname(namep, length)
	char *		namep;
	int		length;
{
	register char *	cp = namep;
	register int	hash = 0;
	register char	c;

	while ( c = *cp++ )
		HASH(hash, c);

	Trace3(4, "\"%s\" hash = %#x", namep, hash);

	if ( hash < 0 )
		hash = ~hash;

	return hash % HASH_SIZE;
}
#endif	CRC_HASH == 0



/*
**	Remove file.
*/

void
remove(name)
	char *		name;
{
	struct stat	statb;

	if ( stat(name, &statb) == SYSERROR )
	{
		if ( Warnings )
			SysWarn("can't stat \"%s\"", name);
		return;
	}

	if ( (statb.st_mode & S_IFMT) != S_IFREG )
	{
		Warn("NOT removing \"%s\" -- not a regular file", name);
		return;
	}

	if ( !All && statb.st_mtime > (Time-HOUR) )
	{
		Trace2(1, "NOT removing \"%s\" -- less than 1 hour old", name);
		return;
	}
	else
		Trace2(1, "remove \"%s\"", name);

	if ( Warnings )
		Warn("removing \"%s\"", name);
	
	if ( !Traceflag )
	{
		if ( unlink(name) == SYSERROR )
			SysWarn("Can't unlink \"%s\"", name);
		else
			Removed++;
	}
}


/*
**	Search explicit file name table for matches.
*/

void
match1search(dfp)
	register DF_p	dfp;
{
	register WD_p	wdp;

	if ( dfp->df_great != (DF_p)0 )
		match1search(dfp->df_great);

	if ( dfp->df_less != (DF_p)0 )
		match1search(dfp->df_less);

	Trace2(1, "match1search(%s)", dfp->df_name);

	for ( wdp = Workdirs ; wdp != (WD_p)0 ; wdp = wdp->next )
		if ( lookup(dfp->df_name, wdp->table) != (DF_p)0 )
		{
			Report("found \"%s%s\"", wdp->name, dfp->df_name);
			break;
		}
}



void
matchsearch()
{
	register DF_p *	dfpp;

	Trace1(1, "matchsearch()");

	for ( dfpp = SearchTable ; dfpp < &SearchTable[HASH_SIZE]; dfpp++ )
		if ( *dfpp != (DF_p)0 )
			match1search(*dfpp);
}
