/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

/*
**	Call a remote host and start a network daemon on it.
**
**	Needs a "call" script as argument.
*/

#ifndef	lint
static char	sccsid[]	= "@(#)call.c	1.12	84/11/04";
#endif	lint

char	Usage[]	= "\nUsage: %s [-&] [-a<call-arg>] [-d<daemon-arg>] \\\n\
	[-l<logfile>] [-s<call-interpreter>] [-t<timeout>] <call-program>\n";


#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"
#include	"debug.h"

#include	<setjmp.h>
#include	<signal.h>
#include	<sgtty.h>
#include	<errno.h>


/*
**	Parameters set from arguments
*/

int		Background;		/* True if call should run in background */
VarArgs		DmnArgs			/* Passed to exec to invoke daemon */
=
{
	2,
	NNDAEMON,			/* Mode-node daemon */
	"-BF"				/* Batch mode, no fork */
}
;
char *		Name;
int		Ondelay;		/* Set to O_NDELAY for line */
unsigned int	TimeOut		= 30;
VarArgs		CallArgs		/* Passed to exec to invoke interpreter */
#ifdef	BINSIFT
=
{
	2,
	BINSIFT,			/* Default call interpreter */
	"-bcp"				/* No input/output bufferring, no pre-processing */
}
#endif	BINSIFT
;
int		Traceflag;

/*
**	Interface to interpreter
*/

int		Ipipes[2];
#define		Callin		Ipipes[1]
#define		Call0		Ipipes[0]
int		Opipes[2];
#define		Callout		Opipes[0]
#define		Call1		Opipes[1]

char		TimeoutMsg[]	= "\nTIMEOUT\n";

/*
**	Commands from script
*/

typedef enum
{
	t_after, t_close, t_daemon, t_error, t_fail,
	t_mode, t_open, t_opendial, t_ondelay, t_read,
	t_retry, t_sleep, t_speed, t_stty, t_succeed,
	t_timeout, t_trace, t_write
}
		Cmdtyp;

struct Command
{
	char *	c_name;
	Cmdtyp	c_type;
}
		Cmds[] =
{
	{"after",	t_after},
	{"close",	t_close},
	{"daemon",	t_daemon},
	{"fail",	t_fail},
	{"mode",	t_mode},
	{"ondelay",	t_ondelay},
	{"open",	t_open},
	{"opendial",	t_opendial},
	{"read",	t_read},
	{"retry",	t_retry},
	{"sleep",	t_sleep},
	{"speed",	t_speed},
	{"stty",	t_stty},
	{"succeed",	t_succeed},
	{"timeout",	t_timeout},
	{"trace",	t_trace},
	{"write",	t_write}
};

#define		CMDZ		(sizeof(struct Command))
#define 	NCMDS		((sizeof Cmds)/CMDZ)
typedef struct Command *	Cmdp;

struct Speed
{
	char *	s_name;
	int	s_val;
}
		Speeds[] =
{
	 {"110",	B110}
	,{"1200",	B1200}
#	ifdef	B19200
	,{"19200",	B19200}
#	endif	B19200
	,{"2400",	B2400}
	,{"300",	B300}
	,{"4800",	B4800}
	,{"600",	B600}
	,{"9600",	B9600}
	,{"EXTA",	EXTA}
	,{"EXTB",	EXTB}
};

#define		SPDZ		(sizeof(struct Speed))
#define 	NSPDS		((sizeof Speeds)/SPDZ)
typedef struct Speed *		Spdp;

/*
**	List of commands to be executed after daemon has terminated.
*/

typedef struct after_list	AL;
struct after_list
{
	AL *	al_next;
	char *	al_command;
};

AL *		AfterList;
AL **		EAfterList	= &AfterList;

/*
**	Miscellaneous
*/

jmp_buf		Alrm_jmp;
char		Buf[BUFSIZ];
char *		CallScript;
int		Call_pid;
char		Errbuf[BUFSIZ];
int		Fd		= SYSERROR;
char *		Fn;
bool		HadArg;
int		Pid;
int		Read_pid;
int		Retries;
unsigned int	Retry;
Time_t		Time;

/*
**	Routines
*/

bool
		command();

int
		alrmcatch(),
		compare(),
		catch();

void
		Daemon(),
		Stty(),
		cleanup(),
		cleartty(),
		doafter(),
		lockalarm(),
		reader(),
		startcall(),
		usage();



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	register int	c;
	register char *	cp;
	char *		path;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			while ( c = *++*argv )
			{
				switch ( c )
				{
				case '&':
					Background++;
					continue;

				case 'O':
#					ifdef	O_NDELAY
					Ondelay = O_NDELAY;
					continue;
#					else	O_NDELAY
					Error("O_NDELAY not defined");
					return 1;
#					endif	O_NDELAY

				case 'T':
					if ( (Traceflag = atoi(++*argv)) <= 0 )
						Traceflag = 1;
					break;

				case 'a':
					if ( NARGS(&CallArgs) == 0 )
					{
						usage("need '-s' before '-a'");
						return 1;
					}
					NEXTARG(&CallArgs) = ++*argv;
					HadArg = true;
					goto break2;

				case 'd':
					FIRSTARG(&DmnArgs) = ++*argv;
					goto break2;

				case 'l':
					if ( freopen(++*argv, "a", stderr) == NULL )
					{
						Syserror("can't open \"%s\"", *argv);
						return 1;
					}
					goto break2;

				case 's':
					if ( HadArg )
					{
						usage("'-s' must come before '-a'");
						return 1;
					}
					FIRSTARG(&CallArgs) = ++*argv;
					goto break2;

				case 't':
					if ( (TimeOut = atoi(++*argv)) == 0 )
					{
						usage("bad timeout %d", TimeOut);
						return 1;
					}
					break;

				default:
					usage("unrecognised flag '%c'", c);
					return 1;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			struct stat	statb;

			if ( CallScript != NULLSTR )
			{
				usage("only one call program needed");
				return 1;
			}

			CallScript = *argv;

			if ( stat(CallScript, &statb) == SYSERROR )
			{
				Syserror(CallScript);
				return 1;
			}

			if ( statb.st_mode & 0111 )
			{
				/*
				**	Executable program
				*/

				FIRSTARG(&CallArgs) = CallScript;
			}
			else
			{
				if ( NARGS(&CallArgs) == 0 )
				{
					usage("need name of interpreter for \"%s\"", CallScript);
					return 1;
				}

				NEXTARG(&CallArgs) = CallScript;
			}
		}
	}

	if ( CallScript == NULLSTR )
	{
		usage("need call program name");
		return 1;
	}

	if ( Background )
	{
		switch ( fork() )
		{
		case SYSERROR:
			Syserror("Can't fork");
			return 1;

		case 0:
			break;

		default:
			return 0;
		}

#		if	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')
		(void)cleartty();
#		else	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')
#		if	PGRP == 1
		(void)setpgrp();
#		endif	PGRP == 1
#		endif	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

		(void)signal(SIGHUP, SIG_IGN);
		(void)signal(SIGINT, SIG_IGN);
		(void)signal(SIGQUIT, SIG_IGN);
	}
	else
		if ( signal(SIGINT, SIG_IGN) != SIG_IGN )
			(void)signal(SIGINT, catch);

	(void)signal(SIGTERM, catch);

#	if	KILL_0 != 1
	(void)signal(SIG0, SIG_IGN);
#	endif	KILL_0 != 1

	if ( (cp = strrchr(CallScript, '/')) != NULLSTR )
	{
		*cp = '\0';
		path = newstr(CallScript);
		*cp = '/';
	}
	else
		path = newstr(".");

	if ( DaemonActive(path) )
		return 0;
	
	(void)SetDaemonActive(path, getpid());

	free(path);

#	if	FCNTL == 1 && O_APPEND != 0
	(void)fcntl
	(
		fileno(stderr),
		F_SETFL,
		fcntl(fileno(stderr), F_GETFL, 0) | O_APPEND
	);
#	endif

	setbuf(stderr, Errbuf);

	return call();
}



int
call()
{
	register char *	cp;
	register int	n;

	startcall();

	for ( cp = Buf ; (n = read(Callout, cp, 1)) == 1 ; )
	{
		if ( *cp != '\0' )
		{
			if ( ++cp >= &Buf[sizeof Buf] )
			{
				Error("callout too long");
				return 1;
			}
			continue;
		}

		if ( !command(Buf) )
			return 0;

		cp = Buf;
	}

	if ( n == SYSERROR )
		Syserror("read from callout");
	else
		Error("unexpected eof from call prog");

	return 1;
}



bool
command(comm)
	char *		comm;
{
	register char *	cp	= comm;
	register int	n;
	Cmdp		cmdp;
	Cmdtyp		cmdt;
	struct Command	cmd;
#	if	OPENDIAL
	bool		open_dial = false;
#	endif	OPENDIAL

	Trace1(1, cp);

	cmd.c_name = cp;

	if ( (cp = strchr(cp, ' ')) != NULLSTR )
	{
		*cp++ = '\0';
		while ( *cp == ' ' )	cp++;
		if ( *cp == '\0' )
			cp = NULLSTR;
	}

	if
	(
		(cmdp = (Cmdp)bsearch((char *)&cmd, (char *)Cmds, NCMDS, CMDZ, compare))
		!=
		(Cmdp)0
	)
		cmdt = cmdp->c_type;
	else
		cmdt = t_error;

	if ( comm != Buf )
		switch ( cmdt )
		{
		case t_after:
		case t_daemon:
		case t_mode:
		case t_succeed:
			Error("Illegal after command");
			return false;
		}

	switch ( cmdt )
	{
	case t_succeed:
			if ( Fd == SYSERROR )
			{
				Error("succeed with no device");
				return false;
			}
			if ( cp == NULLSTR )
			{
				Error("succeed with no host name");
				return false;
			}
			cleanup();
#			if	UUCPLOCKS != 1
			if ( AfterList != (AL *)0 )
#			endif	UUCPLOCKS != 1
			for ( ;; )
			{
				register int	pid;
				int		xpid;
				int		stat;

				switch ( pid = fork() )
				{
				default:
					signal(SIGINT, SIG_IGN);
					signal(SIGHUP, SIG_IGN);
					signal(SIGTERM, SIG_IGN);
					signal(SIGQUIT, SIG_IGN);
#					if	UUCPLOCKS == 1
					lockalarm();
#					endif	UUCPLOCKS == 1
					while
					(
						(xpid = wait(&stat)) != pid
						&&
						(xpid != SYSERROR || errno == EINTR)
					)
						;
					if ( AfterList != (AL *)0 )
						doafter();
#					if	UUCPLOCKS == 1
					rmlock(NULL);
#					endif	UUCPLOCKS == 1
					return false;

				case SYSERROR:	/* help */
					Syserror("Can't fork");
					continue;

				case 0:	;
				}
				break;
			}
			Daemon(cp);
			return false;

	case t_fail:
			Error("fail \"%s\"", cp==NULLSTR?"":cp);
			return false;

	case t_daemon:
			if ( cp == NULLSTR )
			{
				Error("daemon name?");
				return false;
			}
			FIRSTARG(&DmnArgs) = newstr(cp);
			break;

	case t_mode:
			if ( cp == NULLSTR )
			{
				Error("daemon mode?");
				return false;
			}
			SplitArg(&DmnArgs, cp);
			break;

	case t_close:
			if ( Fd == SYSERROR )
			{
				Error("No file for close");
				return false;
			}
			if ( Read_pid )
			{
				int	status;

				(void)kill(Read_pid, SIGKILL);

				while
				(
					(n = wait(&status)) != Read_pid
					&&
					n != SYSERROR
				);

				Read_pid = 0;
			}
			(void)close(Fd);
			free(Fn);
			Fd = SYSERROR;
#			if	UUCPLOCKS == 1
			rmlock(NULL);
#			endif	UUCPLOCKS == 1
			break;

	case t_ondelay:
#			ifdef	O_NDELAY
			Ondelay = O_NDELAY;
			break;
#			else	O_NDELAY
			Error("O_NDELAY not defined");
			return false;
#			endif	O_NDELAY

	case t_opendial:
#			if	OPENDIAL
			open_dial = true;
#			endif	OPENDIAL

	case t_open:
			if ( Fd != SYSERROR )
			{
				Error("File \"%s\" already open", Fn);
				return false;
			}
#			if	UUCPLOCKS == 1
			if ( cp != NULLSTR )
			{
				register char *	l;

				l = strrchr(cp, '/');
				if ( l == NULLSTR )
					l = cp;
				else
					l++;

				if ( mlock(l) != 0 )
				{
					Error("\"%s\" locked (UUCP).", cp);
					return false;
				}
			}
#			endif	UUCPLOCKS == 1
			(void)signal(SIGALRM, alrmcatch);
			{
				auto char *	argp = cp;	/* Preserve from "setjmp()" */

				while
				(
					argp == NULLSTR
					||
					setjmp(Alrm_jmp)
					|| 
					alarm(TimeOut) == SYSERROR
					||
					(
#						if	OPENDIAL
						open_dial ?
						(Fd = opendial()) :
#						endif	OPENDIAL
						(Fd = open(argp, Ondelay|O_EXCL|O_RDWR))
					) == SYSERROR
				)
				{
					(void)alarm((unsigned)0);
					if
					(
						argp != NULLSTR
						&&
						Retry
						&&
						--Retries >= 0
					)
					{
						sleep(Retry);
					}
					else
					{
						Syserror("cannot open \"%s\"", argp==NULLSTR?"(null)":argp);
						return false;
					}
				}
				Fn = newstr(argp);
			}
			(void)alarm((unsigned)0);
			Retry = 0;
			SetRaw(Fd, 0, 0xff, 1, true);
			break;

	case t_retry:
			if ( cp == NULLSTR || (Retry = atoi(cp)) == 0 )
			{
				Error("bad retry value \"%s\"", cp);
				return false;
			}
			if
			(
				(cp = strchr(cp, ' ')) == NULLSTR
				||
				(Retries = atoi(cp+1)) == 0
			)
				Retries = 1;
			break;

	case t_sleep:
			if
			(
				cp == NULLSTR
				||
				(n = atoi(cp)) == 0
			)
			{
				Error("bad sleep value \"%s\"", cp);
				return false;
			}

			if ( (Read_pid && n >= (TimeOut-2)) )
			{
				Error("sleep period %d conflicts with timeout set at %d", n, TimeOut);
				return false;
			}
			sleep((unsigned)n);
			break;

	case t_stty:
			if ( Fd == SYSERROR )
			{
				Error("No file for stty");
				return false;
			}
			Stty(cp);
			break;

	case t_speed:
		{
			Spdp		spdp;
			struct Speed	speed;

			speed.s_name = cp;

			if
			(
				(spdp = (Spdp)bsearch((char *)&speed, (char *)Speeds, NSPDS, SPDZ, compare))
				==
				(Spdp)0
			)
			{
				Error("unrecognised speed \"%s\"", cp);
				return false;
			}

			if ( Fd == SYSERROR )
			{
				Error("can't set speed without device");
				return false;
			}

			SetRaw(Fd, spdp->s_val, 0xff, 1, true);
			break;
		}

	case t_timeout:
			if ( Read_pid != 0 )
			{
				Error("can't set timeout after reader started");
				return false;
			}

			if ( (TimeOut = atoi(cp)) == 0 )
			{
				Error("bad timeout value \"%s\"", cp);
				return false;
			}
			break;

	case t_write:
			if
			(
				cp == NULLSTR
				||
				(n = strlen(cp)) == 0
				||
				Fd == SYSERROR
				||
				write(Fd, cp, n) != n
			)
			{
				Syserror("cannot write to device");
				return false;
			}
			break;

	case t_read:
			if ( Fd == SYSERROR )
			{
				Error("reader with no device");
				return false;
			}
			if ( Read_pid != 0 )
			{
				Error("reader already active");
				return false;
			}
			reader();
			break;

	case t_trace:
			Mesg("trace", cp==NULLSTR?"":cp);
			putc('\n', stderr);
			(void)fflush(stderr);
			break;

	case t_after:
		{
			register AL *	alp;

			alp = Talloc(AL);
			alp->al_next = (AL *)0;
			alp->al_command = newstr(cp);
			*EAfterList = alp;
			EAfterList = &alp->al_next;
			break;
		}

	default:
			Error("unrecognised command from call \"%s\"", Buf);
			return false;
	}

	return true;
}



void
Stty(cp)
	char *	cp;
{
	VarArgs	va;
	FILE *	fd;
	char *	errs;

	if ( cp == NULLSTR || *cp == '\0' )
	{
		Error("No args for stty");
		return;
	}

	FIRSTARG(&va) = STTY;
	SplitArg(&va, cp);

	fd = ExPipeO(&va, Fd);

	if ( (errs = ExOPipeClose(fd)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}
}



void
doafter()
{
	register AL *	alp;

	if ( (alp = AfterList) == (AL *)0 )
		return;
	
	do
	{
		if ( !command(alp->al_command) )
			return;
	}
	while
		( (alp = alp->al_next) != (AL *)0 );
}



int
pipecatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	Error("SIGPIPE on callin");
}

void
startcall()
{
	if
	(
		pipe(Ipipes) == SYSERROR
		||
		pipe(Opipes) == SYSERROR
	)
		Syserror("cannot make pipes for call");

	switch ( Call_pid = fork() )
	{
	case SYSERROR:
			Syserror("cannot fork");
			break;

	case 0:
			(void)close(Callin);
			(void)close(Callout);
			(void)close(0);
			(void)dup(Call0);
			(void)close(Call0);
			(void)close(1);
			(void)dup(Call1);
			(void)close(Call1);

			NEXTARG(&CallArgs) = NULLSTR;
			(void)execve(ARG(&CallArgs, 0), &ARG(&CallArgs, 0), (char **)0);

			Syserror("cannot exec %s", ARG(&CallArgs, 0));
	}

	(void)signal(SIGPIPE, pipecatch);
	(void)close(Call0);
	(void)close(Call1);
}

int
alrmcatch(sig)
	int	sig;
{
	(void)signal(sig, alrmcatch);
	(void)longjmp(Alrm_jmp, 1);
}

void
reader()
{
	register int	n;

	switch ( Read_pid = fork() )
	{
	case SYSERROR:
		Syserror("cannot fork");
		return;

	case 0:
		DODEBUG(Name = "reader");

		Call_pid = 0;
		(void)close(Callout);

		for ( ;; )
		{
			if ( setjmp(Alrm_jmp) )
			{
				if
				(
					write(Callin, TimeoutMsg, sizeof TimeoutMsg -1)
					!=
					(sizeof TimeoutMsg -1)
				)
				{
					Syserror("write to callin");
					return;
				}
			}

			(void)signal(SIGALRM, alrmcatch);
			(void)alarm(TimeOut);

			while ( (n = read(Fd, Buf, sizeof Buf -1)) > 0 )
			{
				register char *	cp;

				for ( cp = Buf ; cp < &Buf[n] ; cp++ )
				{
					*cp &= 0177;

					if ( *cp == '\r' )
						*cp = ' ';
				}

				DODEBUG(Buf[n] = '\0'; Trace1(1, Buf));

				if ( write(Callin, Buf, n) != n )
				{
					Syserror("write to callin");
					return;
				}

				(void)alarm(TimeOut);
			}

			if ( n == 0 )
			{
				if ( Ondelay )
				{
					sleep(1);
					continue;
				}
				Error("eof from device");
				return;
			}
			Syserror("cannot read device");
		}
	}
}



void
Daemon(args)
	register char *	args;
{
	if ( *args != '\0' )
		SplitArg(&DmnArgs, args);

	if ( NARGS(&DmnArgs) > MAXVARARGS )
		Error("too many args for \"%s\"", ARG(&DmnArgs, 0));

	if ( NARGS(&DmnArgs) < 3 )
		Error
		(
			"too few args for daemon: \"%s %s\", need at least path name of daemon, flag \"-B\", and name of link",
			ARG(&DmnArgs, 0),
			ARG(&DmnArgs, 1)
		);

	(void)close(Callin);
	(void)close(Callout);
	(void)close(1);
	(void)dup(Fd);
	(void)close(Fd);

	Trace5
	(
		1,
		"%s %s %s %s", 
		ARG(&DmnArgs, 0),
		ARG(&DmnArgs, 1),
		ARG(&DmnArgs, 2),
		ARG(&DmnArgs, 3) == NULLSTR ? "" : ARG(&DmnArgs, 3)
	);

	(void)fflush(stderr);

	(void)execve(ARG(&DmnArgs, 0), &ARG(&DmnArgs, 0), (char **)0);
	Syserror("cannot exec \"%s\"", ARG(&DmnArgs, 0));
}

void
cleanup()
{
	register int	pid;
	int		status;

	if ( Read_pid )
		(void)kill(Read_pid, SIGKILL);

	if ( Call_pid )
		(void)kill(Call_pid, SIGKILL);
	
	while
	(
		(Read_pid || Call_pid)
		&&
		(pid = wait(&status)) != SYSERROR
	)
	{
		if ( pid == Read_pid )
			Read_pid = 0;
		if ( pid == Call_pid )
			Call_pid = 0;
	}
}

int
catch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	finish(0);
}

int
compare(p1, p2)
	char *	p1;
	char *	p2;
{
	return strcmp(((Cmdp)p1)->c_name, ((Cmdp)p2)->c_name);
}

finish(reason)
	int	reason;
{
	cleanup();
#	if	UUCPLOCKS == 1
	rmlock(NULL);
#	endif	UUCPLOCKS == 1
	exit(reason);
}

#if	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

void
cleartty()
{
	register fd;

	(void)setpgrp(0, 0);

	if ( (fd = open("/dev/tty", 0)) < 0 )
		return;

	(void)ioctl(fd, TIOCNOTTY, (char *)0);
	(void)close(fd);
}
#endif	BSD4 > 1 || (BSD4 == 1 && BSD4V >= 'c')

#if	UUCPLOCKS == 1
void
lockalarm()
{
	ultouch();
	(void)signal(SIGALRM, lockalarm);
	(void)alarm(1800);
}
#endif	UUCPLOCKS == 1

/*VARARGS1*/
void
usage(s, a1, a2)
	char *	s;
	char *	a1;
	char *	a2;
{
	Mesg("argument error", s, a1, a2);
	(void)fprintf(stderr, Usage, Name);
}
