/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)WriteRoute.c	1.9 84/10/02
*/

/*
**	Make up route table in memory, and write it to RouteFile.
*/

#define	FILE_CONTROL
#define	LOCKING

#include	"global.h"
#include	"debug.h"
#include	"state.h"

#include	"route.h"
#include	"node.h"


/*
**	Routing parameters -- DON'T CHANGE!
*/

#define	MSG_ROUTING	(S_MSG|S_CHEAP|S_FAST)
#define	CON_ROUTING	(S_CON|S_FAST)


static int	String_size;
static int	Route_size;

int		al_comp();

DODEBUG(extern char * HomeNode);



/*
**	Create routing tables in memory.
*/

void
MakeRoute()
{
	register Entry **	epp;
	register int		i;
	register char *		nep;
	register int		j;
	register Link *		lp;
	register char **	strings;
	register Index *	save_indices;
	register int		string_count;

	DODEBUG(TraceNode(HomeNode));

	LinkCount = Home->e_node->n_fromlinks;

	/*
	**	Allocate enough memory to hold entire routing table
	*/

	if ( RouteBase != NULLSTR )
		free(RouteBase);

	LevelCount = DomHier.d_count;

	RouteBase = Malloc
		    (
			Route_size = ROUTE_HEADER_SIZE
					+ LINK_ENTRY_SIZE * LinkCount
					+ NODE_ENTRY_SIZE * NodeCount
					+ ALIAS_ENTRY_SIZE * AliasCount
					+ DOMAIN_ENTRY_SIZE * DomainCount
					+ LEVEL_ENTRY_SIZE * LevelCount
					+ SIZE_FORW_TABLE
					+ SIZE_MEMBER_TABLE
					+ SIZE_DOMFORW_TABLE
		    );

	(void)strclr(RouteBase, Route_size);

	NODE_COUNT = NodeCount;
	LINK_COUNT = LinkCount;
	ALIAS_COUNT = AliasCount;
	DOMAIN_COUNT = DomainCount;
	LEVEL_COUNT = LevelCount;
	HOME_FLAGS = HomeFlags = Home->e_states & EXTERNAL_FLAGS;

	/*
	**	These assignments must be in routefile order.
	*/

	LinkTable = LINK_TABLE_ADDR;
	NodeTable = NODE_TABLE_ADDR;
	AliasTable = ALIAS_TABLE_ADDR;
	DomainTable = DOMAIN_TABLE_ADDR;
	LevelTable = LEVEL_TABLE_ADDR;
	ForwTable = FORW_TABLE_ADDR;
	MemberTable = MEMBER_TABLE_ADDR;
	DomForwTable = DOMFORW_TABLE_ADDR;
	
	/*
	**	Make sorted list of nodes in 'NodeList',
	**	and set index for each node.
	**
	**	NB: 'Home' is not included in 'NodeList',
	**	    which contains 'NodeCount'-1 nodes.
	*/

	MakeList(NodeCount, &NodeList, NodeHash);
	Home->e_index = NodeCount - 1;	/* Home will go at end */

	/*
	**	Allocate enough memory to contain:-
	**	pointers to handlers for each link,
	**	pointers to addresses for aliases,
	**	pointers to handlers for each domain.
	*/

	strings = (char **)Malloc(sizeof(char *) * (PT_N*LinkCount + AliasCount + DomainCount));

	String_size = 1;
	string_count = 0;

	/*
	**	Allocate enough memory to contain
	**	node indexes for the link nodes while they are changed
	**	to be the index in link entry table.
	*/

	save_indices = (Index *)Malloc(sizeof(Index) * LinkCount);

	/*
	**	Set up index for link nodes to be index for link entry
	**	(old index for node entry is saved to be restored later).
	*/

	for ( lp = Home->e_node->n_l_first, i = 0 ; lp != (Link *)0 ; i++, lp = lp->l_next )
	{
		save_indices[i] = lp->l_entry->e_index;
		lp->l_entry->e_index = i;
	}

	/*
	**	Make domains list.
	*/

	MakeList(DomainCount, &DomainList, DomainHash);

	/*
	**	Find all shortest paths for MSG and set in node entry table;
	**	(ignore interruptions, take best route).
	**	Also set primary domain index for node,
	**	and set domain membership and forwarding tables.
	*/

	Paths(Home, MSG_ROUTING|S_DOMAINS|S_DOMROUTE);

	for
	(
		epp = NodeList, nep = NodeTable, i = NodeCount, j = 0 ;
		--i >= 0 ;
		nep += NODE_ENTRY_SIZE, epp++, j += DomainCount
	)
	{
		register Domain *	dp;
		register int		k;

		if ( i == 0 )
			epp = &Home;	/* Last entry in table will be home */

		if ( (dp = (*epp)->e_node->n_domains.d_head) != (Domain *)0 )
		{
			((NodeEntry *)nep)->ne_primary = dp->d_entry->e_index;

			do
			{
				k = j + dp->d_entry->e_index;

				MemberTable[k/8] |= 1 << (k%8);
			}
				while ( (dp = dp->d_next) != (Domain *)0 );
		}
		else
			((NodeEntry *)nep)->ne_primary = LINK_N_A;

		if ( (*epp)->e_route == (Entry *)0 )
			k = LINK_N_A;
		else
			k = (*epp)->e_route->e_index;
		((NodeEntry *)nep)->ne_shortest[PT_MSG] = k;

		(void)strcpy(((NodeEntry *)nep)->ne_name, (*epp)->e_name);
	}

	/*
	**	The shortest paths calculation above will also have
	**	found the shortest path to each domain.
	*/

	for
	(
		epp = DomainList, nep = DomainTable, i = DomainCount ;
		--i >= 0 ;
		nep += DOMAIN_ENTRY_SIZE, epp++
	)
	{
		if ( (*epp)->e_route == (Entry *)0 )
			j = LINK_N_A;
		else
			j = (*epp)->e_route->e_index;
		((DomainEntry *)nep)->de_shortest = j;

		((DomainEntry *)nep)->de_hierarchy = LINK_N_A;

		if ( (*epp)->e_handler != NULLSTR )
		{
			((DomainEntry *)nep)->de_handler = String_size;
			String_size += strlen((*epp)->e_handler) + 1;
			strings[string_count++] = (*epp)->e_handler;
		}
		else
			((DomainEntry *)nep)->de_handler = 0;

		(void)strcpy(((DomainEntry *)nep)->de_name, (*epp)->e_name);
	}

	/*
	**	Set hierarchy order
	*/

	{
		register Domain *	dp;

		for ( dp = DomHier.d_head, i = 0 ; dp != (Domain *)0 ; i++, dp = dp->d_next )
		{
			RT_DOMAIN(dp->d_entry->e_index)->de_hierarchy = i;
			*RT_LEVEL(i) = dp->d_entry->e_index;
		}
	}

	/*
	**	Find all shortest paths for CON and set in node entry table;
	**	(bypass interruptions, take fastest route).
	*/

	ClearRoute(NodeHash);

	Paths(Home, CON_ROUTING);

	for
	(
		epp = NodeList, nep = NodeTable, i = NodeCount ;
		--i > 0 ;
		nep += NODE_ENTRY_SIZE, epp++
	)
	{
		if ( (*epp)->e_route == (Entry *)0 )
			j = LINK_N_A;
		else
			j = (*epp)->e_route->e_index;
		((NodeEntry *)nep)->ne_shortest[PT_CON] = j;
	}

	/*
	**	Restore saved index for each link node
	*/

	for ( lp = Home->e_node->n_l_first, i = 0 ; lp != (Link *)0 ; lp = lp->l_next )
		lp->l_entry->e_index = save_indices[i++];

	free((char *)save_indices);

	/*
	**	Set up link entry table,
	**	and calculate "extended reverse path forwarding" table.
	*/

	for ( lp = Home->e_node->n_l_first, i = 0 ; lp != (Link *)0 ; i++, lp = lp->l_next )
	{
		if ( lp->l_entry->e_node->n_handlers != (Hndl *)0 )
		{
			if ( (nep = lp->l_entry->e_node->n_handlers->h_spooler) != NULLSTR )
			{
				RT_LINK(i)->le_handlers[PT_MSG] = String_size;
				String_size += strlen(nep) + 1;
				strings[string_count++] = nep;
			}
			else
				RT_LINK(i)->le_handlers[PT_MSG] = 0;

			if ( (nep = lp->l_entry->e_node->n_handlers->h_connector) != NULLSTR )
			{
				RT_LINK(i)->le_handlers[PT_CON] = String_size;
				String_size += strlen(nep) + 1;
				strings[string_count++] = nep;
			}
			else
				RT_LINK(i)->le_handlers[PT_CON] = 0;
		}
		else
		{
			RT_LINK(i)->le_handlers[PT_MSG] = 0;
			RT_LINK(i)->le_handlers[PT_CON] = 0;
		}

		RT_LINK(i)->le_index = lp->l_entry->e_index;
		RT_LINK(i)->le_flags =	(
						lp->l_entry->e_states
						|
						lp->l_data->d_states
					)
					&
					EXTERNAL_FLAGS;

		if ( LinkCount == 1 )
			break;	/* Why bother? */

		/*
		**	Find all shortest paths from this link node,
		**	(must be same routing parameters as MSG above),
		**	and set bit in ForwTable for each node
		**	whose shortest path from this link node lies via 'Home'.
		*/

		ClearRoute(NodeHash);

		Paths(lp->l_entry, MSG_ROUTING);

		for ( epp = NodeList, j = NodeCount ; --j > 0 ; epp++ )
			if ( (*epp)->e_route == Home )
			{
				register int	k;

				k = (LinkCount * (*epp)->e_index) + i;
				ForwTable[k/8] |= 1 << (k%8);
			}
	}

	/*
	**	Make AliasTable.
	*/

	if ( (i = AliasCount) > 0 )
	{
		MakeList(i, &AliasList, AliasHash);

		for ( epp = AliasList, nep = AliasTable ; --i >= 0 ; epp++ )
		{
			(void)strcpy(((AliasEntry *)nep)->ae_name, (*epp)->e_name);
			((AliasEntry *)nep)->ae_value = String_size;
			strings[string_count++] = (*epp)->e_value;
			String_size += strlen((*epp)->e_value)+1;
			nep += ALIAS_ENTRY_SIZE;
		}
	}

	if ( string_count > 0 )
	{
		Strings = nep = Malloc(String_size);

		nep++;	/* Allow for 1 offset */

		for ( j = 0 ; j < string_count ; j++ )
			nep = strcpyend(nep, strings[j]) + 1;
	}

	free((char *)strings);
}



/*
**	Write the routing table to the routefile.
*/

void
WriteRoute()
{
	register int	fd;
	static char *	format = "Can't %s \"%s\"";

	if ( RouteBase == NULLSTR )
		MakeRoute();

	if ( RouteFile == NULLSTR )
		RouteFile = ROUTEFILE;

	while ( (fd = open(RouteFile, O_WRITE)) == SYSERROR )
		Syserror(format, "open", RouteFile);

#	if	AUTO_LOCKING != 1
	while ( Lock(RouteFile, fd, for_writing) == SYSERROR )
		Syserror(format, "lock", RouteFile);
#	endif	AUTO_LOCKING

	while ( write(fd, RouteBase, Route_size) != Route_size )
	{
		Syserror(format, "write", RouteFile);
		(void)lseek(fd, (long)0, 0);
	}

	if ( String_size > 1 )
		while ( write(fd, Strings, String_size) != String_size )
		{
			Syserror(format, "write", RouteFile);
			(void)lseek(fd, (long)Route_size, 0);
		}

	UnLock(fd);
	
	(void)close(fd);
}
