/*  pokerd - a classical draw poker server
 *  Copyright (C) 2001,2003 Hans P. Rosenfeld <rosenfeld@grumpf.hope-2000.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 */

#include "player.h"

#define BUFSIZE 256

char *stages[]={ "waiting for players",
		 "ante collected, cards dealt (1st betting round)",
		 "waiting for discards",
		 "replacement cards dealt (2nd betting round)",
		 "showdown",
                 "reinitializing"
};

int insert_player(struct player_t *player, struct table_t *table)
{
     if(table->count==5)
	  return(0);
     if(table->count) {
	  player->prev=table->dealer->prev;
	  player->next=table->dealer;
	  player->prev->next=player;
	  table->dealer->prev=player;
     } else {
	  table->dealer=player;
	  table->dealer->dealer=1;
	  player->next=player->prev=player;
     }
     table->count++;
     send_to_all_players(table, player->name);
     send_to_all_players(table, " has joined the game\n");
     return(1);
}

int remove_player(struct player_t *player, struct table_t *table)
{
     if(table->count==0) return(0);
     player->next->prev=player->prev;
     player->prev->next=player->next;
     if(player==table->dealer) {
	  if(table->count==1)
	       table->dealer=NULL;
	  else {
	       table->dealer=player->next;
	       table->dealer->dealer=1;
	  }
     }
     table->count--;
     if(table->count)
	  next_turn(table);
     player->next=player->prev=NULL;
     player->dealer=0;
     return(1);
}

int rotate(struct table_t *table)
{
     if(table->count==0) return(0);
     table->dealer->dealer=0;
     table->dealer=table->dealer->next;
     table->dealer->dealer=1;
     return(1);
}

char *get_stage(struct table_t *table)
{
     return(stages[table->stage]);
}

int next_turn(struct table_t *table)
{
     if(table->count==0)
	  return(0);
     if(table->act_cnt<1)
	  return(0);
     for(table->active=table->active->next; table->active->state==INACTIVE; table->active=table->active->next)
	  if(table->active==table->dealer)
	       table->flags&=~F_RAISE;
     send_to_player(table->active, "it's your turn\n");
     return(1);
}

struct player_t *get_next_active_player(struct player_t *active)
{
     struct player_t *player=active;

     do {
	  player=player->next;
	  if(player->state==ACTIVE)
	       return(player);
     } while(player->next!=active);
     return(NULL);
}

void send_to_player(struct player_t *player, char *buf)
{
     int r=0;

     if(player->broken==0)
	  r=fputs(buf, player->socket);
     if(r==EOF) player->broken=1;
}

void send_to_all_players(struct table_t *table, char *buf)
{
     struct player_t *player=table->dealer;

     do {
	  send_to_player(player, buf);
	  player=player->next;
     } while(player!=table->dealer);
}

char* read_from_player(struct player_t *player, char *buf, int size)
{
     int i=0;
     char *r;

     if(player->broken)
	  return(NULL);
     if((r=fgets(buf, size, player->socket))==NULL) {
	  player->broken=1;    
	  return(NULL);
     }
     while(buf[i++]);
     if(buf[i-2]=='\n')
	  buf[i-2]=buf[i-3]=0;
     return(buf);
}

int check_name(struct table_t *table, char *name)
{
     struct player_t *player=table->dealer;

     do {
	  player=player->next;
	  if(!strcasecmp(name, player->name))
	       return(1);
     } while(player->dealer==0);
     return(0);
}

void clear_raises(struct table_t *table)
{
     struct player_t *player=table->dealer;

     do {
	  player=player->next;
	  player->lastraise=0;
     } while(player->dealer==0);
}

void end_round(struct table_t *table)
{
     struct player_t *player;

     table->stage=S_WAIT;
     table->flags=F_NONE;
     table->active=NULL;
     table->raiser=NULL;
     table->raise=0;
     table->act_cnt=0;
     rotate(table);
     player=table->dealer;
     do {
	  player=player->next;
	  player->hand=NULL;
	  player->lastraise=0;
	  player->state=INACTIVE;
	  while(player->cards->count)
	       push_first_card(pop_first_card(player->cards), table->discard);
     } while(player!=table->dealer);
     while(table->discard->count)
	  push_first_card(pop_first_card(table->discard), table->card_deck);
     send_to_all_players(table, table->dealer->name);
     send_to_all_players(table, " is now the dealer\n");
}

void end_round_win(struct table_t *table)
{
     table->active->money+=table->pot;
     send_to_all_players(table, table->active->name);
     send_to_all_players(table, " wins this round\n");
     table->pot=0;
     end_round(table);
}

struct player_t *new_player(int sock)
{
     struct player_t *player=calloc(1, sizeof(struct player_t));
     struct sockaddr_in addr;
     int l=sizeof(addr);
     char name[15];
     static int count;

     addr.sin_family=AF_INET;
     if(getpeername(sock, (struct sockaddr*) &addr, (socklen_t*) &l))
	  player->addr=strdup("unknown");
     else player->addr=strdup(inet_ntoa(addr.sin_addr));
     player->socket=fdopen(sock,"a+");
     setlinebuf(player->socket);
     fputs("Welcome to the Casino!\n",player->socket);
     snprintf(name,14,"Player%d",count++);
     fputs("Your initial name has been set to ",player->socket);
     fputs(name,player->socket);
     fputs("\n",player->socket);
     player->name=strdup(name);
     player->money=ENTRY;
     player->cards=initialize_empty_card_deck();
     return(player);
}

void ante(struct table_t *table)
{
     struct player_t *player=table->dealer;

     do {
	  player=player->next;
	  if(player->money>=2*MIN_BET) {
	       player->money-=MIN_BET;
	       table->pot+=MIN_BET;
	       player->state=ACTIVE;
	       table->act_cnt++;
	  } else {
	       player->state=INACTIVE;
	       send_to_player(player, "You don't have enough money to play.\n");
	  }
     } while(player->dealer==0);
     table->stage=S_ANTE;
}

void deal(struct table_t *table)
{
     int i;
     struct player_t *player=table->dealer;

     for(i=0; i!=5; i++)
	  do {
	       player=player->next;
	       if(player->state==ACTIVE)
		    push_first_card(pop_first_card(table->card_deck), player->cards);
	  } while(player->dealer==0);
     player=table->dealer;
     do {
	  player=player->next;
	  sort_card_deck_by_number(player->cards);
     } while(player->dealer==0);
}

void check_round(struct table_t *table)
{
     if((table->flags&(F_PASS|F_RAISE|F_RAISED))!=(F_PASS|F_RAISE|F_RAISED)) {
	  if(table->stage==S_ANTE) {
	       struct player_t *tmppl=table->active;
	       send_to_all_players(table, "no outstanding bets remaining, you now have to discard\n");
	       table->stage=S_NEXTROUND;
	       table->raise=0;
	       table->flags=F_PASS;
	       table->dis_cnt=0;
	       clear_raises(table);
	       do {
		    tmppl=tmppl->next;
	       } while(tmppl!=table->active);
	  } else if(table->stage==S_CARDS) {
	       send_to_all_players(table, "no outstanding bets remaining, preparing showdown\n");
	       table->stage=S_SHOWDOWN;
	       showdown(table);
	  }
     }
}

void showdown(struct table_t *table)
{
     struct comb_t *combs[5];
     struct player_t *highest, *tmppl;
     int i=0, j=0, k;
     int pot_split=1;

     send_to_all_players(table, "game ended, comparing hands...\n");
     highest=tmppl=table->active;
     do {
	  tmppl=tmppl->next;
	  if(tmppl->state==ACTIVE)
	       combs[i++]=tmppl->hand=check(tmppl->cards);
     } while(highest!=tmppl);
     k=i;
     for(i=0; i!=k; i++)
	  for(j=i+1; j!=k; j++)
	       if(compare_combs(combs[i], combs[j])==-1) {
		    struct comb_t *tmp=combs[j];
		    combs[j]=combs[i];
		    combs[i]=tmp;
	       }
     for(i=0; i!=k; i++)
	  if(compare_combs(combs[0], combs[1])==0) {
	       pot_split=2;
	       if(i>2 && compare_combs(combs[1], combs[2])==0) {
		    pot_split=3;
		    if(i>3 && compare_combs(combs[2], combs[3])==0) {
			 pot_split=4;
			 if(i>4 && compare_combs(combs[3], combs[4])==0) {
			      pot_split=5;
			 }
		    }
	       }
	  }
     for(i=0; i!=j; i++) {
	  tmppl=highest;
	  do {
	       tmppl=tmppl->next;
	       if(tmppl->state==ACTIVE)
		    if(combs[i]==tmppl->hand) {
			 char deck[100];
			 int k;
			 send_to_all_players(table, tmppl->name);
			 send_to_all_players(table, ":\n");
			 snprint_card_deck(deck, 99, tmppl->cards);
			 send_to_all_players(table, deck);
			 send_to_all_players(table, "hand: ");
			 snprintf(deck, 99, "%s\n", get_combination(tmppl->hand));
			 send_to_all_players(table, deck);
			 if(tmppl->hand->hcards) {
			      send_to_all_players(table, "hand cards: ");
			      for(k=0; k!=tmppl->hand->hcards; k++) {
				   struct card_t tmp;
				   tmp.number=tmppl->hand->cards[k];
				   snprintf(deck, 99, "%s ", get_number(&tmp));
				   send_to_all_players(table, deck);
			      }
			      send_to_all_players(table, "\n");
			 }
			 if(tmppl->hand->kcards) {
			      send_to_all_players(table, "kickers: ");
			      for(k=0; k!=tmppl->hand->kcards; k++) {
				   struct card_t tmp;
				   tmp.number=tmppl->hand->cards[k+tmppl->hand->hcards];
				   snprintf(deck, 99, "%s ", get_number(&tmp));
				   send_to_all_players(table, deck);
			      }
			      send_to_all_players(table, "\n");
			 }
			 if(i==0 && pot_split==1) {
			      char help[15];
			      send_to_all_players(table, tmppl->name);
			      send_to_all_players(table, " takes the pot of $");
			      snprintf(help, 14, "%d\n", table->pot);
			      send_to_all_players(table, help);
			      tmppl->money+=table->pot/pot_split;  
			 }
		    }
	  } while(highest!=tmppl);
     }
     if(pot_split!=1) {
	  char help[15];
	  send_to_all_players(table, "splitting the pot of $");
	  snprintf(help, 14, "%d in %d chunks of $%d\n", table->pot, pot_split, table->pot/pot_split);
	  send_to_all_players(table, help);
     }
     send_to_all_players(table, "preparing next round...\n");
     end_round(table);
     table->pot=0;
     send_to_all_players(table, "ready for next game\n\n");
}
