/* VPort-50 Monitor
   Copyright (c) 2005, 2006 Hans Rosenfeld

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
*/

#include <arm/asm.h>
#include <arm/addr.h>
#include <arm/mach.h>
#include <arm/status.h>
#include <common/break.h>
#include <common/stdio.h>
#include <common/command.h>

#define UNDEFINED            0
#define UNPREDICTABLE        1
#define DATA_PROC_SHIFT      2
#define DATA_PROC_IMMEDIATE  3
#define MULTIPLY             4
#define LONG_MULTIPLY        5
#define SWAP_BYTE            6
#define LOAD_HALF_SBYTE      7
#define BREAKPOINT           8
#define BREAK_UNPREDICTABLE  9
#define MOVE_STATUS_REG_REG 11 
#define MOVE_REG_STATUS_REG 12
#define MOVE_IMM_STATUS_REG 13
#define LOAD_STORE_IMM_OFFS 14
#define LOAD_STORE_REG_OFFS 15
#define LOAD_STORE_MULTIPLE 16
#define BRANCH              17
#define BRANCH_EXCH_INST    18
#define CPR_LOAD_STORE_DREG 19
#define CPR_DATA_PROC       20
#define CPR_REG_TRANS       21
#define SOFTWARE_INTERRUPT  22

const struct opcode {
        unsigned int mask;
        unsigned int value;
        unsigned int type;
} opcodes[] = {
        { 0x0fffffff, 0xf0000000, UNPREDICTABLE       }, /* NV (never) */
        { 0xf040f000, 0x010f0000, MOVE_STATUS_REG_REG },
        { 0xf04f000f, 0x0120f000, MOVE_REG_STATUS_REG },
        { 0xf000000f, 0x012fff10, BRANCH_EXCH_INST    },
        { 0xf000f00f, 0x016f0f10, UNDEFINED           }, /* CLZ, count leading zeros */
        { 0xf000000f, 0x012fff30, UNDEFINED           }, /* BLX, branch and link/exchange */
        { 0xf06ff00f, 0x01000f50, UNDEFINED           }, /* Enhanced DSP add/substract */
        { 0x000fff0f, 0xe1200070, BREAKPOINT          }, /* Software Breakpoint */
        { 0xf00fff0f, 0x01200070, BREAK_UNPREDICTABLE }, /* Software Breakpoint, wrong CC */
        { 0xf06fff6f, 0x01000080, UNDEFINED           }, /* Enhanced DSP multiplies */
        { 0xf03fff0f, 0x00000090, MULTIPLY            },
        { 0xf07fff0f, 0x00800090, LONG_MULTIPLY       },
        { 0xf04ff00f, 0x01000090, SWAP_BYTE           },
        { 0xf1ffff6f, 0x00000090, LOAD_HALF_SBYTE     },
        { 0xf04fffff, 0x03000000, UNDEFINED           }, /* undefined instruction */
        { 0xf04f0fff, 0x0320f000, MOVE_IMM_STATUS_REG },
        { 0xf1ffffff, 0x00000000, DATA_PROC_SHIFT     },
        { 0xf1ffffff, 0x02000000, DATA_PROC_IMMEDIATE },
        { 0xf1ffffff, 0x04000000, LOAD_STORE_IMM_OFFS },
        { 0xf1ffffef, 0x06000000, LOAD_STORE_REG_OFFS },
        { 0xf1ffffef, 0x06000010, UNDEFINED           }, /* undefined instruction */
        { 0xf1ffffff, 0x08000000, LOAD_STORE_MULTIPLE },
        { 0xf1ffffff, 0x0a000000, BRANCH              },
        { 0xf1ffffff, 0x0c000000, CPR_LOAD_STORE_DREG },
        { 0xf0ffffef, 0x0e000000, CPR_DATA_PROC       },
        { 0xf0ffffef, 0x0e000010, CPR_REG_TRANS       },
        { 0xf0ffffff, 0x0f000000, SOFTWARE_INTERRUPT  },
        { 0, 0, 0 },
};

char * const ccodes[] = {
        "EQ",
        "NE",
        "CS",
        "CC",
        "MI",
        "PL",
        "VS",
        "VC",
        "HI",
        "LS",
        "GE",
        "LT",
        "GT",
        "LE",
        "",
        "NV"
};

char * const data[] = {
        "AND",
        "EOR",
        "SUB",
        "RSB",
        "ADD",
        "ADC",
        "SBC",
        "RSC",
        "TST",
        "TEQ",
        "CMP",
        "CMN",
        "ORR",
        "MOV",
        "BIC",
        "MVN"
};

char * const shift[] = {
        "LSL",
        "LSR",
        "ASR",
        "ROR"
};

void disasm(addr_t *addr)
{
        const struct opcode *ptr;
        unsigned int word;
        int tmp, last;
        char s;

        word = readw(addr);

        printf("%A  %r\t", addr, word);

        for(ptr = opcodes; ptr->mask; ptr++)
                if((word & ~ptr->mask) == ptr->value)
                        break;

        word &= ptr->mask;

        switch(ptr->type) {
        case UNDEFINED:
                printf("undefined");
                break;
        case UNPREDICTABLE:
                printf("unpredictable");
                break;
        case BREAKPOINT:
                printf("BKPT\t%0.4r (undefined)", ((word >> 4) & 0xfff0) | (word & 0xf));
                break;
        case BREAK_UNPREDICTABLE:
                printf("unpredictable BKPT (undefined)");
                break;
        case BRANCH:
                tmp = word & 0x00ffffff;
                if(tmp & 0x00800000)
                        tmp |= 0xff000000;
                putchar('B');
                if(word & 0x01000000)
                        putchar('L');
                printf("%C\t%r", ccodes[(word >> 28) & 0xf], ((unsigned int) *addr) + 8 + (tmp << 2));
                break;
        case BRANCH_EXCH_INST:
                printf("BX%C\tR%d", ccodes[(word >> 28) & 0xf], word & 0xf);
                break;
        case DATA_PROC_IMMEDIATE:
        case DATA_PROC_SHIFT:
                if((word & 0x01800000) == 0x01000000)
                        s = ' ';
                else s = (word & 0x00100000) ? 'S' : ' ';
                printf("%C%C%c\tR%d, ", data[(word >> 21) & 0xf], ccodes[(word >> 28) & 0xf], s, (word >> 12) & 0xf);
                if((word & 0x01a00000) != 0x01a00000)
                        printf("R%d, ", (word >> 16) & 0xf);
                switch(ptr->type) {
                case DATA_PROC_IMMEDIATE:
                        printf("#%r", ror(word & 0xff, (word >> 7) & 0x1e));
                        break;
                case DATA_PROC_SHIFT:
                        printf("R%d", word & 0xf);
                        if(word & 0x00000010)
                                printf(", %C R%d", shift[(word >> 5) & 0x3], (word >> 8) & 0xf);
                        else {
                                tmp = (word >> 7) & 0x1f;
                                if(tmp)
                                        printf(", %C #%0.2r", shift[(word >> 5) & 0x3], tmp);
                                else if((tmp = (word >> 5) & 0x3)) {
                                        if(tmp == 3) printf(", RRX");
                                        else printf(", %C #20", shift[tmp]);       
                                }
                        }
                        break;
                }
                break;
        case MULTIPLY:
                if(word & 0x00200000)
                        printf("MLA");
                else printf("MUL");
                printf("%C%c\tR%d, R%d, R%d", ccodes[(word >> 28) & 0xf], (word & 0x00100000) ? 'S' : ' ',
                       (word >> 16) & 0xf, word & 0xf, (word >> 8) & 0xf);
                if(word & 0x00200000)
                        printf(", R%d", (word >> 12) & 0xf);
                break;
        case LONG_MULTIPLY:
                printf("%c%C%L%C%c\tR%d, R%d, R%d, R%d", (word & 0x00400000) ? 'S' : 'U', (word & 0x00200000) ? "MLA" : "MUL",
                       ccodes[(word >> 28) & 0xf], (word & 0x00100000) ? 'S' : ' ',
                       (word >> 12) & 0xf, (word >> 16) & 0xf, word & 0xf, (word >> 8) & 0xf);
                break;
        case MOVE_STATUS_REG_REG:
                printf("MRS%C\tR%d, %cPSR", ccodes[(word >> 28) & 0xf], (word >> 12) & 0xf, (word & 0x00400000) ? 'S' : 'C');
                break;
        case MOVE_REG_STATUS_REG:
        case MOVE_IMM_STATUS_REG:
                printf("MSR%C\t%cPSR_", ccodes[(word >> 28) & 0xf], (word & 0x00400000) ? 'S' : 'C');
                if(word & 0x00080000) putchar('F');
                if(word & 0x00040000) putchar('S');
                if(word & 0x00020000) putchar('X');
                if(word & 0x00010000) putchar('C');
                if(ptr->type == MOVE_IMM_STATUS_REG)
                        printf(", #%r", ror(word & 0xff, (word >> 7) & 0x1e));
                else printf(", R%d", word & 0xf);
                break;
        case LOAD_STORE_IMM_OFFS:
        case LOAD_STORE_REG_OFFS:
                printf("%C%C", (word & 0x00100000) ? "LDR" : "STR", ccodes[(word >> 28) & 0xf]);
                if(word & 0x00400000) putchar('B');
                if((word & 0x01200000) == 0x00200000) putchar('T');
                printf("\tR%d, [R%d", (word >> 12) & 0xf, (word >> 16) & 0xf);
                if((word & 0x01000000) == 0)
                        putchar(']');
                if(ptr->type == LOAD_STORE_IMM_OFFS) {
                        if(word & 0xfff)
                                printf(", #%c%0.3r", (word & 0x00800000) ? 0 : '-', word & 0xfff);
                } else {
                        printf(", %cR%d", (word & 0x00800000) ? 0 : '-', word & 0xf);
                        tmp = (word >> 7) & 0x1f;
                        if(tmp)
                                printf(", %C #%0.2r", shift[(word >> 5) & 0x3], tmp);
                        else if((tmp = (word >> 5) & 0x3)) {
                                if(tmp == 3) printf(", RRX");
                                else printf(", %C #20", shift[tmp]);       
                        }
                }
                if(word & 0x01000000) {
                        putchar(']');
                        if(word & 0x00200000) putchar('!');
                }
                break;
        case LOAD_HALF_SBYTE:
                printf("%C%C", (word & 0x00100000) ? "LDR" : "STR", ccodes[(word >> 28) & 0xf]);
                if(word & 0x00000040) putchar('S');
                if(word & 0x00000020) putchar('H');
                else putchar('B');
                printf("\tR%d, [R%d", (word >> 12) & 0xf, (word >> 16) & 0xf);
                if((word & 0x01000000) == 0)
                        putchar(']');
                if(word & 0x00400000) {
                        tmp = (word & 0x0f) | ((word >> 4) & 0xf0);
                        if(tmp) 
                                printf(", #%c%0.2r", (word & 0x00800000) ? 0 : '-', tmp);
                } else printf(", %cR%d", (word & 0x00800000) ? 0 : '-', (word & 0xf));
                if(word & 0x01000000) {
                        putchar(']');
                        if(word & 0x00200000) putchar('!');
                }
                break;
        case LOAD_STORE_MULTIPLE:
                printf("%C%C%c%c\tR%d", (word & 0x00100000) ? "LDM" : "STM", ccodes[(word >> 28) & 0xf],
                       (word & 0x00800000) ? 'I' : 'D', (word & 0x01000000) ? 'B' : 'A', (word >> 16) & 0xf);
                if(word & 0x00200000) putchar('!');
                printf(", {");
                for(tmp = 0; (word & 1<<tmp) == 0; tmp++);
                if(tmp < 16) {
                        printf("R%d", tmp);
                        last = tmp;
                        for(tmp++; (word & 1<<tmp) != 0; tmp++);
                        if(last != tmp - 1 && tmp < 16)
                                printf("-R%d", tmp - 1);
                }
                while(tmp < 16) {
                        while((word & 1<<tmp) == 0) tmp++;
                        if(tmp < 16) {
                                printf(", R%d", tmp);
                                last = tmp;
                                for(tmp++; (word & 1<<tmp) != 0; tmp++);
                                if(last != tmp - 1 && tmp < 16)
                                        printf("-R%d", tmp - 1);
                        }
                }
                putchar('}');
                if(word & 0x00400000) putchar('^');
                break;
        case SWAP_BYTE:
                printf("SWP%C%c\tR%d, R%d, [R%d]", ccodes[(word >> 28) & 0xf], (word & 0x00400000) ? 'B' : ' ',
                       (word >> 12) & 0xf, word & 0xf, (word >> 16) & 0xf);
                break;
        case SOFTWARE_INTERRUPT:
                printf("SWI%C\t%0.6r", ccodes[(word >> 28) & 0xf], word & 0x00ffffff);
                break;
        case CPR_LOAD_STORE_DREG:
                printf("%C%C%c\tP%d, CR%d, [R%d", (word & 0x00100000) ? "LDC" : "STC", ccodes[(word >> 28) & 0xf],
                       (word & 0x00400000) ? 'L' : ' ', (word >> 8) & 0xf, (word >> 12) & 0xf, (word >> 16) & 0xf);
                if((word & 0x01000000) == 0)
                        putchar(']');
                if((word & 0x01200000) == 0x00200000)
                        printf(", {%d}", word & 0xff);
                else printf(", #%c%0.3r", (word & 0x00800000) ? 0 : '-', (word & 0xff) << 2);
                if(word & 0x01000000) {
                        putchar(']');
                        if(word & 0x00200000) putchar('!');
                }
                break;
        case CPR_DATA_PROC:
                printf("CPD%C\tP%d, %d, CR%d, CR%d, CR%d, %d", ccodes[(word >> 28) & 0xf],
                       (word >> 8) & 0xf, (word >> 20) & 0xf, (word >> 12) & 0xf, (word >> 16) & 0xf, word & 0xf, (word >> 5) & 0x7);
                break;
        case CPR_REG_TRANS:
                printf("%C%C\tP%d, %d, R%d, CR%d, CR%d, %d", (word & 0x00100000) ? "MCR" : "MRC", ccodes[(word >> 28) & 0xf],
                       (word >> 8) & 0xf, (word >> 21) & 0x7, (word >> 12) & 0xf, (word >> 16) & 0xf, word & 0xf, (word >> 5) & 0x7);
                break;
        default:
                printf("unknown!");
        }
        putchar('\n');
        add_addr(addr, 4);
        return;
}

/* shifter "interpretiert" einen Shifter-Operanden, indem es ihn in einem
   Befehl ausfhrt. Dazu wird eine "Subroutine" bestehend aus:
   - einem MOV um den Shifter-Operanden auszuwerten
   - einem MOV fr den Rcksprung
   auf dem Stack angelegt und ausgefhrt.

   Um die Funktionsweise zu verstehen sollte man eine Weile ber
   Kapitel 5.1 des ARM Architecture Reference Manual meditieren.

   Dieser Code funktioniert nur wenn der Stack zu niedrigeren Adressen
   hin wchst und der Compiler nicht zuviel wegoptimiert.

   Streng genommen msste man vor dem Aufruf der "Subroutine" eine
   Instruction-Memory-Barrier-Routine aufrufen.
*/
unsigned int shifter(unsigned int shift)
{
        unsigned int ret = 0xe1a0f00e; /* mov pc, lr */
        unsigned int mov = 0xe1a00001; /* mov r0, r1 */
        unsigned int rm = ((unsigned int *) &status)[(shift & 0xf) + 1];        /* Inhalt von Rm */
        unsigned int rs = ((unsigned int *) &status)[((shift >> 8) & 0xf) + 1]; /* Inhalt von Rs */

        /* Spezialfall PC als Quellregister */
        if((shift & 0xf) == 0xf)
                rm += 8;

        if(((shift >> 8) & 0xf) == 0xf)
                rs += 8;

        mov |= shift & 0xff0; /* shifter_operand ohne Rm bernehmen */ 
        if(mov & 0x10)        /* shifter_operand enthlt Rs         */
                mov &= 0xfffff0ff; /* Rs wird in R0 bergeben       */

        /* MOV ausfhren, Rs in R0, Rm in R1, &ret um den Optimierer auszutricksen */
        return(((unsigned int (*)(unsigned int, unsigned int, unsigned int *)) &mov)(rs, rm, &ret));
}

struct breakpoint tracepoint;

#define FLAG(x) (((status.cpsr & (x)) == (x)) ? 1 : 0)
#define FLAG_N FLAG(0x80000000)
#define FLAG_Z FLAG(0x40000000)
#define FLAG_C FLAG(0x20000000)
#define FLAG_V FLAG(0x10000000)

/* trace_emulation errechnet den PC nach Ausfhrung des nchsten Befehls
   und fgt dort einen zustzlichen Breakpoint ein
*/
void trace_emulation(void)
{
        const struct opcode *ptr;
        addr_t target = status.r15 + 4;
        unsigned int word;
        int tmp;

        /* nix machen wenn trace nicht verlangt wird */
        if(!tflag)
                return;

        word = readw(&status.r15);

        for(ptr = opcodes; ptr->mask; ptr++)
                if((word & ~ptr->mask) == ptr->value)
                        break;

        word &= ptr->mask;

        switch((word >> 28) & 0xf) {
        case 0:  /* EQ: Z == 1 */
                if(FLAG_Z == 1)
                        break;
                goto linear;
        case 1:  /* NE: Z == 0 */
                if(FLAG_Z == 0)
                        break;
                goto linear;
        case 2:  /* CS: C == 1 */
                if(FLAG_C == 1)
                        break;
                goto linear;
        case 3:  /* CC: C == 0 */
                if(FLAG_C == 0)
                        break;
                goto linear;
        case 4:  /* MI: N == 1 */
                if(FLAG_N == 1)
                        break;
                goto linear;
        case 5:  /* PL: N == 0 */
                if(FLAG_N == 0)
                        break;
                goto linear;
        case 6:  /* VS: V == 1 */
                if(FLAG_V == 1)
                        break;
                goto linear;
        case 7:  /* VC: V == 0 */
                if(FLAG_V == 0)
                        break;
                goto linear;
        case 8:  /* HI: C == 1 && Z == 0 */
                if(FLAG_C == 1 && FLAG_Z == 0)
                        break;
                goto linear;
        case 9:  /* LS: C == 0 && Z == 1 */
                if(FLAG_C == 0 && FLAG_Z == 1)
                        break;
                goto linear;
        case 10: /* GE: N == V */
                if(FLAG_N == FLAG_V)
                        break;
                goto linear;
        case 11: /* LT: N != V */
                if(FLAG_N != FLAG_V)
                        break;
                goto linear;
        case 12: /* GT: Z == 0 && N == V */
                if(FLAG_Z == 0 && FLAG_N == FLAG_V)
                        break;
                goto linear;
        case 13: /* LE: Z == 1 || N != V */
                if(FLAG_Z == 1 || FLAG_N != FLAG_V)
                        break;
                goto linear;
        case 14: /* always */
                break;
        case 15: /* never (unpredictable) */
                goto linear;
        }

        switch(ptr->type) {
        case BRANCH:
                tmp = word & 0x00ffffff;
                if(tmp & 0x00800000)
                        tmp |= 0xff000000; /* sign-extend */
                target = status.r15 + 8 + (tmp << 2);
                break;

        case BRANCH_EXCH_INST:
                tmp = ((unsigned int *) &status)[(word & 0xf) + 1];
                if(tmp & 1) /* THUMB */
                        break;
                target = tmp;
                break;

        case DATA_PROC_IMMEDIATE:
        case DATA_PROC_SHIFT:
                if(((word >> 12) & 0xf) != 15)
                        break; /* Rd != PC */
                if(((word & 0x01800000) != 0x01000000) & (word & 0x00100000))
                        break; /* S bit set, unpredictable */

                target = ((unsigned int *) &status)[((word >> 16) & 0xf) + 1];

                if(((word >> 16) & 0xf) == 0xf)
                        target += 8; /* Spezialfall PC als Quellregister */

                if(ptr->type == DATA_PROC_IMMEDIATE)
                        tmp = ror(word & 0xff, (word >> 7) & 0x1e);
                else    tmp = shifter(word);

                switch((word >> 21) & 0xf) {
                         /* Bitmanipulation */
                case 0:  /* AND */
                        target &=  tmp;
                        break;
                case 1:  /* EOR */
                        target ^=  tmp;
                        break;
                case 12: /* ORR */
                        target |=  tmp;
                        break;
                case 14: /* BIC */
                        target &= ~tmp;
                        break;
                         /* Arithmetik */
                case 6:  /* SBC */
                        target -= FLAG_C ? 0 : 1;
                case 2:  /* SUB */
                        target -= tmp;
                        break;
                case 7:  /* RSC */
                        target +=  FLAG_C ? 0 : 1;
                case 3:  /* RSB */
                        target  =  tmp - target;
                        break;
                case 5:  /* ADC */
                        target +=  FLAG_C;
                case 4:  /* ADD */
                        target +=  tmp;
                        break;
                         /* Transfer */
                case 13: /* MOV */
                        target  =  tmp;
                        break;
                case 15: /* MVN */
                        target  = ~tmp;
                        break;
                         /* Vergleich */
                case 8:  /* TST */
                case 9:  /* TEQ */
                case 10: /* CMP */
                case 11: /* CMN */
                        target  =  status.r15+4;
                        break;
                }
                break;

        case LOAD_STORE_IMM_OFFS:
        case LOAD_STORE_REG_OFFS:
                if((word & 0x00100000) == 0)
                        break;
                if(word & 0x00400000)
                        break;
                if(((word >> 12) & 0xf) != 0xf)
                        break;
                
                if(ptr->type == LOAD_STORE_IMM_OFFS)
                        tmp = word & 0x00000fff;
                else    tmp = shifter(word);
                
                if((word & 0x00800000) == 0)
                        tmp = -tmp;

                if(((word >> 16) & 0xf) == 0xf)
                        tmp += 8; /* Spezialfall PC als Quellregister */

                /* falls die Ausfhrung des Befehls zu einem Data Abort fhren wrde tritt auch hier
                   logischerweise ein Data Abort auf den man irgendwie auffangen sollte */
                target = *((unsigned int *) (((unsigned int *) &status)[((word >> 16) & 0xf) + 1] + tmp));
                break;

        case LOAD_STORE_MULTIPLE:
                if((word & 0x00100000) == 0)
                        break;
                if(((word >> 16) & 0xf) == 0xf)
                        break;
                if(word & 0x00400000)
                        break;
                /* s.o. bzgl. Data Abort */
                if(word & 0x00008000) {
                        tmp  = (word & 0x01000000) ? 16 : 15;
                        tmp *= (word & 0x00800000) ?  1 : -1;
                        target = ((unsigned int *) ((unsigned int *) &status)[((word >> 16) & 0xf) + 1])[tmp];
                }
                break;
        }
        
linear:
        tracepoint.addr = target & 0xfffffffc;
        tracepoint.inst = readw(&tracepoint.addr);
        write_break(&target, BRKINST);
}
