/* 
 * Copyright (c) 2003, Hans Rosenfeld <rosenfeld@grumpf.hope-2000.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * --------------------------------------------------------------
 * BASED ON
 *	$NetBSD: if_tun.c,v 1.39 2000/03/30 09:45:37 augustss Exp $
 * 
 * Copyright (c) 1988, Julian Onions <jpo@cs.nott.ac.uk>
 * Nottingham University 1987.
 *
 * AND ON
 *	$Id: if_tap.h,v 1.1 2000/04/24 20:47:33 maxk Exp $
 *
 * Copyright (c) 1999 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * WHICH ITSELF IS BASED ON
 *      $NetBSD: if_tun.c,v 1.14 1994/06/29 06:36:25 cgd Exp $
 *
 * Copyright (c) 1988, Julian Onions <jpo@cs.nott.ac.uk>
 * Nottingham University 1987.
 *
 * TAPIOCTL() CODE TAKEN FROM ETHER_IOCTL() FROM
 *      $NetBSD: if_ethersubr.c,v 1.95.2.3 2003/06/02 14:49:47 tron Exp $
 *
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 *
 * --------------------------------------------------------------
 *
 * This driver takes packets off a virtual ethernet interface and
 * hands them up to a user process and vice-versa. It is basically
 * modified if_tun that processes ethernet frames instead of raw
 * IP packets.
 * 
 */

/*
 * TODO:
 *	- fix the EBUSY on opening unit 0 for writing. for some strange
 *	  reason the kernel returns EBUSY even before invoking tap_open()
 * 	- add BPF support for packets coming from the virtual interface
 *	- create MAC address of virtual interfaces of random data
 *	- add bridging support
 */

#include "tap.h"
#if NTAP > 0

#include "opt_inet.h"
#include "opt_ns.h"

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/select.h>
#include <sys/poll.h>
#include <sys/file.h>
#include <sys/signalvar.h>
#include <sys/conf.h>

#include <machine/cpu.h>

#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_ether.h>
#include <net/if_types.h>
#include <net/netisr.h>
#include <net/route.h>


#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_inarp.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include "bpfilter.h"
#if NBPFILTER > 0
#include <sys/time.h>
#include <net/bpf.h>
#endif

#include <net/if_tap.h>

#define TAPDEBUG	if (tapdebug) printf
int	tapdebug = 0;

struct tap_softc tapctl[NTAP];
extern int ifqmaxlen;
void	tapattach __P((int));

int	tap_ioctl __P((struct ifnet *, u_long, caddr_t));
static void	tap_start __P((struct ifnet *));

static int tapinit __P((struct ifnet *));

void
tapattach(unused)
	int unused;
{
	int i;
	struct ifnet *ifp;
	u_int8_t enaddr[ETHER_ADDR_LEN] = { 0, 222, 173, 190, 239, 0 };

	for (i = 0; i < NTAP; i++) {
		tapctl[i].tap_flags = TAP_INITED;

		/* generate fake MAC address: 00:DE:AD:BE:EF:unit_no */
		enaddr[5] = (u_int8_t) i;

		ifp = &tapctl[i].tap_if;
		sprintf(ifp->if_xname, "tap%d", i);
		ifp->if_softc = &tapctl[i];
		ifp->if_ioctl = tap_ioctl;
		ifp->if_snd.ifq_maxlen = ifqmaxlen;
		ifp->if_collisions = 0;
		ifp->if_ierrors = 0;
		ifp->if_oerrors = 0;
		ifp->if_ipackets = 0;
		ifp->if_opackets = 0;
		ifp->if_start = tap_start;
		ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
		if_attach(ifp);
		ether_ifattach(ifp,enaddr);
#if NBPFILTER > 0
		bpfattach(&tapctl[i].tap_bpf, ifp, DLT_EN10MB, sizeof(u_int32_t));
#endif

	}
}

/*
 * tunnel open - must be superuser
 */
int
tapopen(dev, flag, mode, p)
	dev_t	dev;
	int	flag, mode;
	struct proc *p;
{
	struct ifnet	*ifp;
	struct tap_softc *tp;
	int	unit, error;

	if ((error = suser(p->p_ucred, &p->p_acflag)) != 0)
		return (error);

	if ((unit = minor(dev)) >= NTAP)
		return (ENXIO);

	tp = &tapctl[unit];
	ifp = &tp->tap_if;

	if (tp->tap_flags & TAP_OPEN)
		return EBUSY;

	tp->tap_flags |= TAP_OPEN;
	TAPDEBUG("%s: opened\n", ifp->if_xname);
	return (0);
}

/*
 * tapclose - close the device - mark i/f down
 */
int
tapclose(dev, flag, mode, p)
	dev_t	dev;
	int	flag;
	int	mode;
	struct proc *p;
{
	int	unit = minor(dev), s;
	struct tap_softc *tp = &tapctl[unit];
	struct ifnet	*ifp = &tp->tap_if;
	struct mbuf	*m;

	tp->tap_flags &= ~TAP_OPEN;

	/*
	 * junk all pending output
	 */
	do {
		s = splimp();
		IF_DEQUEUE(&ifp->if_snd, m);
		splx(s);
		if (m) {
			m_freem(m);
			m = NULL;
		}
	} while (m);

	if (ifp->if_flags & IFF_UP) {
		s = splimp();
		if_down(ifp);
		if (ifp->if_flags & IFF_RUNNING)
			ifp->if_flags &= ~IFF_RUNNING;
		ifp->if_flags &= ~IFF_UP;
		splx(s);
	}

	tp->tap_pgrp = 0;
	selwakeup(&tp->tap_rsel);
		
	TAPDEBUG ("%s: closed\n", ifp->if_xname);
	return (0);
}

static int
tapinit(ifp)
	struct ifnet *ifp;
{
	TAPDEBUG("%s: tapinit\n", ifp->if_xname);

	ifp->if_flags |= IFF_UP | IFF_RUNNING;

	return(0);
}

/*
 * Process an ioctl request.
 */
int
tap_ioctl(ifp, cmd, data)
	struct ifnet *ifp;
	u_long cmd;
	caddr_t	data;
{
	int		error = 0, s;
	struct ethercom *ec = (void *)ifp;
	struct ifreq *ifr = (struct ifreq*)data;
	struct ifaddr *ifa = (struct ifaddr*)data;

	s = splimp();
	switch(cmd) {
	case SIOCSIFADDR:
		TAPDEBUG("%s: tap_ioctl SIOCSIFADDR\n", ifp->if_xname);
                ifp->if_flags |= IFF_UP;
                switch (ifa->ifa_addr->sa_family) {
                case AF_LINK:
                    {   
                        struct sockaddr_dl *sdl =
                            (struct sockaddr_dl *) ifa->ifa_addr;

                        if (sdl->sdl_type != IFT_ETHER ||
                            sdl->sdl_alen != ifp->if_addrlen) {
                                error = EINVAL;
                                break;
                        }

                        memcpy(LLADDR(ifp->if_sadl), LLADDR(sdl),
                            ifp->if_addrlen);

                        /* Set new address. */
                        error = tapinit(ifp);
                        break;
                    }
#ifdef INET
                case AF_INET:
                        if ((ifp->if_flags & IFF_RUNNING) == 0 &&
                            (error = tapinit(ifp)) != 0)
                                break;
                        arp_ifinit(ifp, ifa);
                        break;
#endif /* INET */
#ifdef NS
                case AF_NS:
                    {
                        struct ns_addr *ina = &IA_SNS(ifa)->sns_addr;

                        if (ns_nullhost(*ina))
                                ina->x_host = *(union ns_host *)
                                    LLADDR(ifp->if_sadl);
                        else
                                memcpy(LLADDR(ifp->if_sadl),
                                    ina->x_host.c_host, ifp->if_addrlen);
                        /* Set new address. */
                        error = tapinit(ifp);
                        break;
                    }
#endif /* NS */
                default:
                        if ((ifp->if_flags & IFF_RUNNING) == 0)
                                error = tapinit(ifp);
                        break;
                }
                break;

	case SIOCGIFADDR:
		TAPDEBUG("%s: tap_ioctl SIOCGIFADDR\n", ifp->if_xname);
                memcpy(((struct sockaddr *)&ifr->ifr_data)->sa_data,
                    LLADDR(ifp->if_sadl), ETHER_ADDR_LEN);
                break;

	case SIOCSIFMTU:
	    {
                int maxmtu = ETHERMTU;
		TAPDEBUG("%s: tap_ioctl SIOCSIFMTU\n", ifp->if_xname);

                if (ifr->ifr_mtu < ETHERMIN || ifr->ifr_mtu > maxmtu)
                        error = EINVAL;
                else {
                        ifp->if_mtu = ifr->ifr_mtu;

                        /* Make sure the device notices the MTU change. */
                        if (ifp->if_flags & IFF_UP)
                                error = tapinit(ifp);
                }
                break;
	    }
	case SIOCADDMULTI:
		TAPDEBUG("%s: tap_ioctl SIOCADDMULTI\n", ifp->if_xname);
                error = ether_addmulti(ifr, ec);
                break;

	case SIOCDELMULTI:
		TAPDEBUG("%s: tap_ioctl SIOCDELMULTI\n", ifp->if_xname);
                error = ether_delmulti(ifr, ec);
                break;

	case SIOCSIFFLAGS:
		TAPDEBUG("%s: tap_ioctl SIOCSIFFLAGS\n", ifp->if_xname);
                if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) == IFF_RUNNING) {
                        /*
                         * If interface is marked down and it is running,
                         * then stop and disable it.
                         */
			 ifp->if_flags &= ~IFF_RUNNING;
                } else if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) == IFF_UP) {
                        /*
                         * If interface is marked up and it is stopped, then
                         * start it.
                         */
                        error = tapinit(ifp);
                } else if ((ifp->if_flags & IFF_UP) != 0) {
                        /*
                         * Reset the interface to pick up changes in any other
                         * flags that affect the hardware state.
                         */
                        error = tapinit(ifp);
                }
                break;

	default:
		error = EINVAL;
	}
	splx(s);
	return (error);
}

/*
 * tap_start - wake up processes waiting for data
 */
void
tap_start(ifp)
	struct ifnet   *ifp;
{
	struct tap_softc *tp = ifp->if_softc;
	struct proc	*p;

	TAPDEBUG ("%s: tap_start\n", ifp->if_xname);

	if ((tp->tap_flags & TAP_READY) != TAP_READY) {
		TAPDEBUG ("%s: not ready 0%o\n", ifp->if_xname,
			  tp->tap_flags);
		return;
	}

	if (tp->tap_flags & TAP_RWAIT) {
		tp->tap_flags &= ~TAP_RWAIT;
		wakeup((caddr_t)tp);
	}
	if (tp->tap_flags & TAP_ASYNC && tp->tap_pgrp) {
		if (tp->tap_pgrp > 0)
			gsignal(tp->tap_pgrp, SIGIO);
		else if ((p = pfind(-tp->tap_pgrp)) != NULL)
			psignal(p, SIGIO);
	}
	selwakeup(&tp->tap_rsel);
	return;
}

/*
 * the cdevsw interface is now pretty minimal.
 */
int
tapioctl(dev, cmd, data, flag, p)
	dev_t		dev;
	u_long		cmd;
	caddr_t		data;
	int		flag;
	struct proc	*p;
{
	int		unit = minor(dev), s;
	struct tap_softc *tp = &tapctl[unit];
	struct ifnet *ifp = &tp->tap_if;
	struct tapinfo *tapp;

	switch (cmd) {
	case TAPSDEBUG:
		tapdebug = *(int *)data;
		break;

	case TAPGDEBUG:
		*(int *)data = tapdebug;
		break;

	case TAPSIFINFO:
		tapp = (struct tapinfo *)data;
		ifp->if_mtu = tapp->mtu;
		ifp->if_type = tapp->type;
		ifp->if_baudrate = tapp->baudrate;
		break;

	case TAPGIFINFO:
		tapp = (struct tapinfo *)data;
		tapp->mtu = ifp->if_mtu;
		tapp->type = ifp->if_type;
		tapp->baudrate = ifp->if_baudrate;
		break;

	case FIONBIO:
		if (*(int *)data)
			tp->tap_flags |= TAP_NBIO;
		else
			tp->tap_flags &= ~TAP_NBIO;
		break;

	case FIOASYNC:
		if (*(int *)data)
			tp->tap_flags |= TAP_ASYNC;
		else
			tp->tap_flags &= ~TAP_ASYNC;
		break;

	case FIONREAD:
		s = splimp();
		if (tp->tap_if.if_snd.ifq_head)
			*(int *)data = tp->tap_if.if_snd.ifq_head->m_pkthdr.len;
		else	
			*(int *)data = 0;
		splx(s);
		break;

	case TIOCSPGRP:
		tp->tap_pgrp = *(int *)data;
		break;

	case TIOCGPGRP:
		*(int *)data = tp->tap_pgrp;
		break;

	default:
		return (ENOTTY);
	}
	return (0);
}

/*
 * The cdevsw read interface - reads a packet at a time, or at
 * least as much of a packet as can be read.
 */
int
tapread(dev, uio, ioflag)
	dev_t		dev;
	struct uio	*uio;
	int		ioflag;
{
	int		unit = minor(dev);
	struct tap_softc *tp = &tapctl[unit];
	struct ifnet	*ifp = &tp->tap_if;
	struct mbuf	*m, *m0;
	int		error=0, len, s;

	TAPDEBUG ("%s: read\n", ifp->if_xname);
	if ((tp->tap_flags & TAP_READY) != TAP_READY) {
		TAPDEBUG ("%s: not ready 0%o\n", ifp->if_xname, tp->tap_flags);
		return EHOSTDOWN;
	}

	tp->tap_flags &= ~TAP_RWAIT;

	s = splimp();
	do {
		IF_DEQUEUE(&ifp->if_snd, m0);
		if (m0 == 0) {
			if (tp->tap_flags & TAP_NBIO) {
				splx(s);
				return (EWOULDBLOCK);
			}
			tp->tap_flags |= TAP_RWAIT;
			if (tsleep((caddr_t)tp, PZERO|PCATCH, "tapread", 0)) {
				splx(s);
				return (EINTR);
			}
		}
	} while (m0 == 0);
	splx(s);

	while (m0 && uio->uio_resid > 0 && error == 0) {
		len = min(uio->uio_resid, m0->m_len);
		if (len == 0)
			break;
		error = uiomove(mtod(m0, caddr_t), len, uio);
		MFREE(m0, m);
		m0 = m;
	}

	if (m0) {
		TAPDEBUG("Dropping mbuf\n");
		m_freem(m0);
	}
	if (error)
		ifp->if_ierrors++;
	return (error);
}

/*
 * the cdevsw write interface - an atomic write is a packet - or else!
 */
int
tapwrite(dev, uio, ioflag)
	dev_t		dev;
	struct uio	*uio;
	int		ioflag;
{
	int		unit = minor (dev);
	struct tap_softc *tp = &tapctl[unit];
	struct ifnet	*ifp = &tp->tap_if;
	struct mbuf	*top, **mp, *m;
	int		error=0, tlen, mlen;

	TAPDEBUG("%s: tapwrite\n", ifp->if_xname);

	if (uio->uio_resid < 0 || uio->uio_resid > ifp->if_mtu) {
		TAPDEBUG("%s: len=%lu!\n", ifp->if_xname,
		    (unsigned long)uio->uio_resid);
		return (EIO);
	}

	tlen = uio->uio_resid;

	/* get a header mbuf */
	MGETHDR(m, M_DONTWAIT, MT_DATA);
	if (m == NULL)
		return (ENOBUFS);
	mlen = MHLEN;

	top = 0;
	mp = &top;
	while (error == 0 && uio->uio_resid > 0) {
		m->m_len = min(mlen, uio->uio_resid);
		error = uiomove(mtod (m, caddr_t), m->m_len, uio);
		*mp = m;
		mp = &m->m_next;
		if (uio->uio_resid > 0) {
			MGET (m, M_DONTWAIT, MT_DATA);
			if (m == 0) {
				error = ENOBUFS;
				break;
			}
			mlen = MLEN;
		}
	}
	if (error) {
		if (top)
			m_freem (top);
		ifp->if_ierrors++;
		return (error);
	}

	top->m_pkthdr.len = tlen;
	top->m_pkthdr.rcvif = ifp;

#if NBPFILTER > 0
        if (tp->tap_bpf)
		bpf_mtap(tp->tap_bpf, top);
#endif

        (*ifp->if_input)(ifp, top);
        return (error);
}

/*
 * tappoll - the poll interface, this is only useful on reads
 * really. The write detect always returns true, write never blocks
 * anyway, it either accepts the packet or drops it.
 */
int
tappoll(dev, events, p)
	dev_t		dev;
	int		events;
	struct proc	*p;
{
	int		unit = minor(dev), s;
	struct tap_softc *tp = &tapctl[unit];
	struct ifnet	*ifp = &tp->tap_if;
	int		revents = 0;

	s = splimp();
	TAPDEBUG("%s: tappoll\n", ifp->if_xname);

	if (events & (POLLIN | POLLRDNORM)) {
		if (ifp->if_snd.ifq_len > 0) {
			TAPDEBUG("%s: tappoll q=%d\n", ifp->if_xname,
			    ifp->if_snd.ifq_len);
			revents |= events & (POLLIN | POLLRDNORM);
		} else {
			TAPDEBUG("%s: tappoll waiting\n", ifp->if_xname);
			selrecord(p, &tp->tap_rsel);
		}
	}

	if (events & (POLLOUT | POLLWRNORM))
		revents |= events & (POLLOUT | POLLWRNORM);

	splx(s);
	return (revents);
}

#endif  /* NTAP */
