/* VPort-50 Monitor
   Copyright (c) 2004, Hans Rosenfeld

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
*/

#include <pdp11/addr.h>
#include <pdp11/mach.h>
#include <common/stdio.h>
#include <common/command.h>

extern struct command c_reg[];

#define SINGLE   1
#define SINGLE_W 2
#define SINGLE_R 3
#define SINGLE_M 4
#define DOUBLE   5
#define DOUBLE_W 6
#define BRANCH   7
#define NOARGS   8
#define CCOND    10
#define SCOND    11
#define TRAP     12
#define SOB      13
#define SINGLE_F 14
#define DOUBLE_F 15
#define DOUBLE_A 16
#define DOUBLE_B 17
#define DOUBLE_R 18

struct opcode {
        unsigned int mask;
        unsigned int value;
        unsigned int type;
        char *name;
} opcodes[] = {
        { 0100077, 0005500, SINGLE,   "ADC"    },
        { 0007777, 0060000, DOUBLE_W, "ADD"    },
        { 0000777, 0072000, SINGLE_R, "ASH"    },
        { 0000777, 0073000, SINGLE_R, "ASHC"   },
        { 0100077, 0006300, SINGLE,   "ASL"    },
        { 0100077, 0006200, SINGLE,   "ASR"    },
        { 0000377, 0103000, BRANCH,   "BCC"    },
        { 0000377, 0103400, BRANCH,   "BCS"    },
        { 0000377, 0001400, BRANCH,   "BEQ"    },
        { 0000377, 0002000, BRANCH,   "BGE"    },
        { 0000377, 0003000, BRANCH,   "BGT"    },
        { 0000377, 0101000, BRANCH,   "BHI"    },
        { 0000377, 0103000, BRANCH,   "BHIS"   },
        { 0107777, 0040000, DOUBLE,   "BIC"    },
        { 0107777, 0050000, DOUBLE,   "BIS"    },
        { 0107777, 0030000, DOUBLE,   "BIT"    },
        { 0000377, 0003400, BRANCH,   "BLE"    },
        { 0000377, 0103400, BRANCH,   "BLO"    },
        { 0000377, 0101400, BRANCH,   "BLOS"   },
        { 0000377, 0002400, BRANCH,   "BLT"    },
        { 0000377, 0100400, BRANCH,   "BMI"    },
        { 0000377, 0001000, BRANCH,   "BNE"    },
        { 0000377, 0100000, BRANCH,   "BPL"    },
        { 0000000, 0000003, NOARGS,   "BPT"    },
        { 0000377, 0000400, BRANCH,   "BR"     },
        { 0000377, 0102000, BRANCH,   "BVC"    },
        { 0000377, 0102400, BRANCH,   "BVS"    },
        { 0100077, 0005000, SINGLE,   "CLR"    },
        { 0000000, 0000240, NOARGS,   "NOP"    },
        { 0000017, 0000240, CCOND,    "C"      },
        { 0107777, 0020000, DOUBLE,   "CMP"    },
        { 0100077, 0005100, SINGLE,   "COM"    },
        { 0100077, 0007000, SINGLE,   "CSM"    },
        { 0100077, 0005300, SINGLE,   "DEC"    },
        { 0000777, 0071000, SINGLE_M, "DIV"    },
        { 0000377, 0104000, TRAP,     "EMT"    },
        { 0000000, 0000000, NOARGS,   "HALT"   },
        { 0100077, 0005200, SINGLE,   "INC"    },
        { 0000000, 0000004, NOARGS,   "IOT"    },
        { 0000077, 0000100, SINGLE_W, "JMP"    },
        { 0000777, 0004000, SINGLE_R, "JSR"    },
        { 0000077, 0006400, TRAP,     "MARK"   },
        { 0000077, 0106500, SINGLE_W, "MFPD"   },
        { 0000077, 0006500, SINGLE_W, "MFPI"   },
        { 0000077, 0106700, SINGLE_W, "MFPS"   },
        { 0000000, 0000007, NOARGS,   "MFPT"   },
        { 0107777, 0010000, DOUBLE,   "MOV"    },
        { 0000077, 0106600, SINGLE_W, "MTPD"   },
        { 0000077, 0006600, SINGLE_W, "MTPI"   },
        { 0000077, 0106400, SINGLE_W, "MTPS"   },
        { 0000777, 0070000, SINGLE_M, "MUL"    },
        { 0100077, 0005400, SINGLE,   "NEG"    },
        { 0000000, 0000005, NOARGS,   "RESET"  },
        { 0100077, 0006100, SINGLE,   "ROL"    },
        { 0100077, 0006000, SINGLE,   "ROR"    },
        { 0000000, 0000002, NOARGS,   "RTI"    },
        { 0000007, 0000200, SINGLE_W, "RTS"    },
        { 0000000, 0000006, NOARGS,   "RTT"    },
        { 0100077, 0005600, SINGLE,   "SBC"    },
        { 0000000, 0000260, NOARGS,   "NOP"    },
        { 0000017, 0000260, SCOND,    "S"      },
        { 0000777, 0077000, SOB,      "SOB"    },
        { 0000007, 0000230, TRAP,     "SPL"    },
        { 0007777, 0160000, DOUBLE_W, "SUB"    },
        { 0000077, 0000377, SINGLE_W, "SWAB"   },
        { 0000077, 0006700, SINGLE_W, "SXT"    },
        { 0000377, 0104400, TRAP,     "TRAP"   },
        { 0100077, 0005700, SINGLE,   "TST"    },
        { 0000077, 0007200, SINGLE_W, "TSTSET" },
        { 0000000, 0000001, NOARGS,   "WAIT"   },
        { 0000077, 0007300, SINGLE_W, "WRTLCK" },
        { 0000777, 0074000, SINGLE_R, "XOR"    },
        { 0000007, 0075000, SINGLE_W, "FADD"   },
        { 0000007, 0075010, SINGLE_W, "FSUB"   },
        { 0000007, 0075020, SINGLE_W, "FMUL"   },
        { 0000007, 0075030, SINGLE_W, "FDIV"   },
        { 0000077, 0170600, SINGLE_F, "ABSF"   },
        { 0000377, 0172000, DOUBLE_F, "ADDF"   },
        { 0000000, 0170000, NOARGS,   "CFCC"   },
        { 0000077, 0170400, SINGLE_F, "CLRF"   },
        { 0000377, 0173400, DOUBLE_F, "CMPF"   },
        { 0000377, 0174400, DOUBLE_F, "DIVF"   },
        { 0000377, 0177400, DOUBLE_F, "LDCDF"  },
        { 0000377, 0177000, DOUBLE_A, "LDCIF"  },
        { 0000377, 0176400, DOUBLE_A, "LDEXP"  },
        { 0000377, 0172400, DOUBLE_F, "LDF"    },
        { 0000077, 0170100, SINGLE_W, "LDFPS"  },
        { 0000377, 0171400, DOUBLE_F, "MODF"   },
        { 0000377, 0171000, DOUBLE_F, "MULF"   },
        { 0000077, 0170700, SINGLE_F, "NEGF"   },
        { 0000000, 0170001, NOARGS,   "SETF"   },
        { 0000000, 0170011, NOARGS,   "SETD"   },
        { 0000000, 0170002, NOARGS,   "SETI"   },
        { 0000000, 0170012, NOARGS,   "SETL"   },
        { 0000000, 0170001, NOARGS,   "SETF"   },
        { 0000377, 0176000, DOUBLE_R, "STCFD"  },
        { 0000377, 0174000, DOUBLE_R, "STF"    },
        { 0000377, 0175400, DOUBLE_B, "STCFI"  },
        { 0000377, 0175000, DOUBLE_B, "STEXP"  },
        { 0000077, 0170200, SINGLE_W, "STFPS"  },
        { 0000077, 0170300, SINGLE_W, "STST"   },
        { 0000377, 0173000, DOUBLE_F, "SUBF"   },
        { 0000077, 0170500, SINGLE_F, "TSTF"   },
        { 0177777, 0000000, NOARGS,   "illegal"},
};

char ccodes[] = "CVZN";

char *modetab[] = {
        "%C",      /* Register Mode               */
        "(%C)",    /* Register Deferred Mode      */
        "(%C)+",   /* Autoincrement Mode          */
        "@(%C)+",  /* Autoincrement Deferred Mode */
        "-(%C)",   /* Autodecrement Mode          */
        "@-(%C)",  /* Autodecrement Deferred Mode */
        "%r(%C)",  /* Index Mode                  */
        "@%r(%C)", /* Index Deferred Mode         */
};

char *pctab[] = {
        "PC",      
        "(PC)",
        "#%r",     /* Immediate Mode         */
        "@#%r",    /* Absolute Mode          */
        "-(PC)",
        "@-(PC)",
        "%r",      /* Relative Mode          */
        "@%r",     /* Relative Deferred Mode */
};


void disasm(addr)
        addr_t *addr;
{
        register struct opcode *ptr;
        register unsigned int word, byte;
        unsigned int tmp, wcnt=0;
        char *cptr = ccodes;

        word = readw(addr);
        byte = word&0100000;

        printf("%A  %r\t\t", addr, word);

        for(ptr=opcodes;; ptr++)
                if((word & ~ptr->mask) == ptr->value)
                        break;
     
        printf(ptr->name);
        word &= ptr->mask;

        switch(ptr->type) {
        case SINGLE:
                if(byte)
                        putchar('B');
        case SINGLE_W:
                putchar('\t');
                wcnt += printreg(addr, (word&070)>>3, (word&07));
                break;

        case SINGLE_R:
                printf("\t%C,", c_reg[(word&0700)>>6].name);
                wcnt += printreg(addr, (word&070)>>3, (word&07));
                break;

        case SINGLE_M:
                putchar('\t');
                wcnt += printreg(addr, (word&070)>>3, (word&07));
                printf(",%C", c_reg[(word&0700)>>6].name);
                break;

        case SINGLE_F:
                putchar('\t');
                if(word&070)
                        if((word&070) == 060) {
                                inc_addr(addr);
                                wcnt++;
                                printf(modetab[(word&070)>>3],
                                       readw(addr),
                                       c_reg[word&7].name);
                        } else printf(modetab[(word&070)>>3], c_reg[word&7].name);
                else printf("AC%d", word&7);
                break;

        case DOUBLE:
                if(byte)
                        putchar('B');
        case DOUBLE_W:
                putchar('\t');
                wcnt += printreg(addr, (word&07000)>>9, (word&0700)>>6);
                putchar(',');
                wcnt += printreg(addr, (word&070)>>3, (word&07));
                break;
	  
        case DOUBLE_F:
                putchar('\t');
                if(word&070)
                        if((word&070) == 060) {
                                inc_addr(addr);
                                wcnt++;
                                printf(modetab[(word&070)>>3],
                                       readw(addr),
                                       c_reg[word&7].name);
                        } else printf(modetab[(word&070)>>3], c_reg[word&7].name);
                else printf("AC%d", word&7);
                printf(",AC%d", (word&0300)>>6);
                break;

        case DOUBLE_R:
                printf("\tAC%d,", (word&0300)>>6);
                if(word&070)
                        if((word&070) == 060) {
                                inc_addr(addr);
                                wcnt++;
                                printf(modetab[(word&070)>>3],
                                       readw(addr),
                                       c_reg[word&7].name);
                        } else printf(modetab[(word&070)>>3], c_reg[word&7].name);
                else printf("AC%d", word&7);
                break;

        case DOUBLE_A:
                putchar('\t');
                wcnt += printreg(addr, (word&070)>>3, (word&07));
                printf(",AC%d", (word&0300)>>6);
                break;

        case DOUBLE_B:
                printf("\tAC%d,", (word&0300)>>6);
                wcnt += printreg(addr, (word&070)>>3, (word&07));
                break;

        case BRANCH:
                tmp = (unsigned int) *addr;
                tmp+=2;
                tmp+= ((word&0200) ? word|0177400 : word)<<1;
                printf("\t%r", tmp);
                break;

        case SOB:
                tmp = (unsigned int) *addr;
                tmp+=2;
                tmp-= word<<1;
                printf("\t%r", tmp);
                break;

        case TRAP:
                printf("\t%o", word);
                break;

        case CCOND:
                if(word == 017)
                        printf("CC");
                else {
                        putchar('L');
                        tmp=1;
                        while(!(word&tmp) && *cptr) {
                                tmp<<=1;
                                cptr++;
                        }
                        if(*cptr) {
                                putchar(*cptr++);
                                do {
                                        tmp<<=1;
                                        if(word&tmp)
                                                printf("|CL%c", *cptr);
                                } while(*++cptr);
                        }
                }
                break;

        case SCOND:
                if(word == 017)
                        printf("CC");
                else {
                        putchar('E');
                        tmp=1;
                        while(!(word&tmp) && *cptr) {
                                tmp<<=1;
                                cptr++;
                        }
                        if(*cptr) {
                                putchar(*cptr++);
                                do {
                                        tmp<<=1;
                                        if(word&tmp)
                                                printf("|SE%c", *cptr);
                                } while(*++cptr);
                        }
                }
                break;

        case NOARGS:
                break;
        }
     
        putchar('\n'); 
        if(wcnt == 2) {
                sub_addr(addr, 2);
                printf("%A  %r\n", addr, readw(addr));
                add_addr(addr, 2);
        }
        if(wcnt >= 1)
                printf("%A  %r\n", addr, readw(addr));
        add_addr(addr, 2);
        return;
}

unsigned int printreg(a, m, r)
        addr_t *a;
        unsigned int m, r;
{
        register unsigned int *addr = a, mode = m, reg = r;
        unsigned int word, wcnt = 0;
     
        if(reg != 7) {
                if((mode&6) == 6) {
                        add_addr(addr, 2);
                        wcnt++;
                        printf(modetab[mode], readw(addr), c_reg[reg].name);
                } else printf(modetab[mode], c_reg[reg].name);
        } else {
                if((mode&6) && (mode&6) != 4) {
                        add_addr(addr, 2);
                        wcnt++;
                        word = readw(addr);
                        if((mode&6) == 6) {
                                word += (unsigned int) *addr;
                                word += 2;
                        }
                }
                printf(pctab[mode], word);
        }
        return(wcnt);
}
