/* VPort-50 Monitor
   Copyright (c) 2005 Hans Rosenfeld

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
*/

#include <arm/asm.h>
#include <arm/addr.h>
#include <arm/mach.h>
#include <arm/setjmp.h>
#include <arm/status.h>
#include <common/stdio.h>
#include <common/break.h>

#define RESET      0x00000000
#define GETCHAR    0x00000010
#define PUTCHAR    0x00000011
#define GETCHARW   0x00000012
#define TERMINATE  0x00000014
#define BREAKPOINT 0x00ffffff
#define UNDEFINED  0x01000000
#define PREFABORT  0x02000000
#define DATAABORT  0x03000000

extern int etext;

/* Die ersten 4 Parameter erwartet der Compiler in den Registern R0-R4.
   
   Der Aufruf aus main.s legt die Parameter jedoch anders ab:
   - R0 enthlt den Syscall-Code, 
   - R1 bis R3 enthalten nichts sinnvolles,
   - die gesichterten Register liegen im Stack, zuflligerweise in der
     richtigen Reihenfolge fr struct jmpbuf

   Um das nun dem Compiler klar zu machen werden d1, d2 und d3 als Dummy-
   Parameter definiert.
*/
unsigned int syscall(unsigned int func, unsigned int d1, unsigned int d2, unsigned int d3, struct jmpbuf r)
{
        int i;
        addr_t tmp;

        switch(func) {
        case GETCHAR:
                return(getchar_nb());
        case GETCHARW:
                return(getchar());
        case PUTCHAR:
                putchar(r.r0);
                return(r.r0);
        case TERMINATE:
                printf("\nterminated\n");
                break;

        case BREAKPOINT:
                tmp = r.r15 - 4;
                if(istrace() && tracepoint.addr != 0xffffffff) {
                        if(tracepoint.addr != tmp) {
                                printf("\nfailed to predict next instruction address\n");
                                printf("expected = %r, current = %r\n", tracepoint.addr, tmp);
                        }
                        if(read_break(&tracepoint.addr) == BRKINST)
                                write_break(&tracepoint.addr, tracepoint.inst);
                        tracepoint.addr = 0xffffffff;
                        tracepoint.inst = 0;
                }
                unbreak();
                for(i = 0; i != brkcnt; i++)
                        if(brktbl[i].addr == tmp)
                                break;
                if(brkcur != NBRK) {
                        brkcur = NBRK;
                        if(trace == 0) {
                                r.r15 -= 4;
                                untrace();
                                dobreak();
                                return(r.r0);
                        }
                } else if(i != brkcnt) {
                        printf("\nbreakpoint %d reached\n", i);
                        settrace();
                        brkcur = i;
                }
                r.r15 -= 4;
                break;

        case RESET:
                reset();
                break;
        case UNDEFINED:
                printf("\nundefined instruction\n");
                break;
        case PREFABORT:
                printf("\nprefetch abort\n");
                break;
        case DATAABORT:
                printf("\ndata abort\n");
                if(r.r15 <= (unsigned int) &etext && r.r15 >= (unsigned int) trace_emulation)
                        printf("while predicting next instruction address\n");
                break;

        default:
                printf("\nunknown SWI function %d\n", func);
                break;
        }

        if(r.r15 >= (unsigned int) &etext) {
                status.cpsr = r.cpsr;
                status.r0   = r.r0;
                status.r1   = r.r1;
                status.r2   = r.r2;
                status.r3   = r.r3;
                status.r4   = r.r4;
                status.r5   = r.r5;
                status.r6   = r.r6;
                status.r7   = r.r7;
                status.r8   = r.r8;
                status.r9   = r.r9;
                status.r10  = r.r10;
                status.r11  = r.r11;
                status.r12  = r.r12;
                status.r13  = r.r13;
                status.r14  = r.r14;
                status.r15  = r.r15;
                showreg();
                set_addr(&tmp, status.r15);
                disasm(&tmp);
        }

        longjmp();
}
