;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)main.s	1.8	07/12/05
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.module main
	.title Main monitor routine
	.sbttl Global reference definitions and warm start routine

	.globl ram_addr, mon_pc
	.globl head_no, sector_no, cylinder_msb, cylinder_lsb, load_addr
	.globl putchar, put, puts, crlf, gets, upper, nextparm
	.globl prhex, scanhexb, scanhexw
	.globl SPACE
	.globl _ide_boot, _ide_address, _ide_read_a_sector, _ide_write_a_sector

	.area CODE
	
warm_start::
	ld h,ram_addr+1(iy)	; get keyboard buffer address
	ld l,ram_addr(iy)	;
	ld de,#0x80		; give the keyboard buffer some room
	add hl,de		; initialize the monitor PC to here 
	ld mon_pc+1(iy),h	;
	ld mon_pc(iy),l		;

	call crlf		; skip a line

	.page
	.sbttl Command parser
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Command input and parsing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
cmd:	ld a,#'>		; send a command prompt
	call putchar		;
	ld h,ram_addr+1(iy)	; get keyboard buffer address
	ld l,ram_addr(iy)	;
	call gets		; get a string from the user

	;
	; Check for Intel hex records
	;
	cp a,#':		; Intel hex record?			7
	jp z,intel_hex		; yes, process the record

	cp a,#0			; blank line?
	jr z,cmd		; yes, start over

	;
	; parse the command line
	;

	call upper		; convert buffer to uppercase
	call nextparm		; get the first non-space character in A
	inc hl			; move the pointer to the next character

	; memory dump
	cp a,#'D		; memory dump?
	jp z,memdump		; yes, dump some memory
	
	; enter data
	cp a,#'E		; enter data?
	jp z,enterd		; yes, do data entry

	; go
	cp a,#'G		; go?
	jp z,gorun		; yes, execute program

	; read an I/O port
	cp a,#'I		; Read I/O?
	jp z,readio		; yes, read from a port

	; write an I/O port
	cp a,#'O		; Write I/O?
	jp z,writeio		; yes, write to a port

	; boot from a disk
	cp a,#'B		; Boot?
	jp z,_ide_boot		; yes, boot

	; load from disk
	cp a,#'L		; Load?
	jp z,load		; yes, load

	; save to disk
	cp a,#'S		; Save
	jp z,save		; yes, save

	; help
	cp a,#'?		; help
	jr nz,cmd		; no, try again
	ld hl,#help		; get help info
	call puts		; print it
	jr cmd			; done


	.page
	.sbttl Monitor routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Memory dump
;
;	Usage: D [addr] [len]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
memdump:
	call nextparm		; find the next command line parameter
	jr nz,1$		; if not EOL, get the specified PC
	ld d,mon_pc+1(iy)	; otherwise, default to the saved monitor PC
	ld e,mon_pc(iy)		;
	ld bc,#16		; dump 16 bytes by default
	jr 3$			; start dumping
1$:	call scanhexw		; get the starting address in DE
	call nextparm		; find the next command line parameter
	jr nz,2$		; if not EOL, get length
	ld bc,#16		; dump 16 bytes by default
	jr 3$			; start dumping
2$:	push de			; save starting address
	call scanhexw		; get a 16-bit word
	ld b,d			; copy number of bytes
	ld c,e			;
	pop de			; restore starting address
3$:	push de			; save the PC
	ld h,b			; get the number of bytes
	ld l,c			;
	add hl,de		; add to the monitor PC value
	ex de,hl		; save as ending address
	pop hl			; get the PC
4$:	ld a,h			; print the PC
	call prhex		;
	ld a,l			;
	call prhex		;
	ld a,#':		;
	call putchar		;
	ld a,#SPACE		;
	call putchar		;
	ld b,#16		; place 16 bytes per line
5$:	ld a,(hl)		; get a byte
	call prhex		; print it
	ld a,#SPACE		; get a space
	call putchar		; print it
	inc hl			; next byte
	push hl			; end of dump?
	or a,a			; clear carry
	sbc hl,de		; set the flags
	pop hl			;
	jr nc,6$		; yes, finish up
	djnz 5$			; no, continue if not EOL
	call crlf		; otherwise, print CRLF
	jr 4$			; keep going
6$:	call crlf		; terminate the line
	ld mon_pc+1(iy),d 	; save the new monitor PC
	ld mon_pc(iy),e		;
	jp cmd			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Enter data
;
;	Usage: E [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
enterd:	
	call nextparm		; find the next command line parameter
	jr nz,1$		; if not EOL, get the specified PC
	ld d,mon_pc+1(iy) 	; yes, default to the saved monitor PC
	ld e,mon_pc(iy)		;
	jr 2$			; go for it
1$:	call scanhexw		; get the starting address in DE
2$:	ld a,d			; get address high byte
	call prhex		; print it
	ld a,e			; get address low byte
	call prhex		; print it
	ld a,#':		; get an colon
	call putchar		; print it
	ld a,#SPACE		; get a space
	call putchar		; print it
	ld a,(de)		; get the original byte
	call prhex		; print it
	ld a,#'-		; command prompt
	call putchar		; print it
	ld h,ram_addr+1(iy)	; get keyboard buffer address
	ld l,ram_addr(iy)	;
	call gets		; get a string from the user
	call nextparm		; find the next parameter
	jr z,3$			; if blank line, save PC and leave
	call upper		; otherwise, convert buffer to uppercase
	call scanhexb		; get an 8-bit byte in C
	ld a,c			; get the byte
	ld (de),a		; store it
	inc de			; skip to the next location
	jr 2$			; back to the command prompt
3$:	ld mon_pc+1(iy),d 	; save the new monitor PC
	ld mon_pc(iy),e		;
	jp cmd			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Execute a subroutine
;
;	Usage: G [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
gorun:	
	call nextparm		; find the next command line parameter
	jr nz,1$		; if not EOL, get the specified PC
	ld h,mon_pc+1(iy) 	; otherwise, default to the saved monitor PC
	ld l,mon_pc(iy)		;
	jr 2$			; go for it
1$:	call scanhexw		; no, get a 16-bit word
	ld mon_pc+1(iy),d 	; save as the new monitor PC
	ld mon_pc(iy),e		;
	ld h,d			; move PC to HL
	ld l,e			;
2$:	push iy			; save the location of the local variables
	ld de,#3$		; get the return address
	push de			; push the return address
	jp (hl)			; load new program counter
3$:				; user's subroutine should return back here
	pop iy			; restore the location of the local variables
	jp cmd			; done!


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read from an I/O port
;
;	Usage: I <port>
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
readio:
	call nextparm		; find the next command line parameter
	jp z,cmd		; if EOL, do nothing and leave
	call scanhexb		; otherwise, get the 8-bit port into C
	in a,(c)		; read port
	ld b,a			; save value in B
	ld a,c			; get the port number
	call prhex		; print it
	ld a,#':		; get a colon
	call putchar		; print it
	ld a,b			; get the value from B
	call prhex		; print it
	call crlf		; terminate the line
	jp cmd			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write to an I/O port
;
;	Usage: O <port> [value]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
writeio:
	call nextparm		; find the next command line parameter
	jp z,cmd		; if EOL, do nothing and leave
	call scanhexb		; otherwise, get the 8-bit port in C
	call nextparm		; find the next command line parameter
	jr z,1$			; if EOL, use 0 as the value
	ld b,c			; otherwise, save the port
	call scanhexb		; get the 8-bit value in C
	ld a,c			; move the value to A
	ld c,b			; move the port back to C
1$:	out (c),a		; write port
	jp cmd			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load a sector from disk
;
;	Usage: L [sect] [head] [cyl] [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
load:
	call lsparms		; parse the command line
	call _ide_read_a_sector	; load the sector into memory
	jp cmd			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Save a sector to disk
;
;	Usage: S [sect] [head] [cyl] [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
save:
	call lsparms		; parse the command line
	call _ide_write_a_sector; save the sector to disk
	jp cmd			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Parse load/save command line parameters
;	and get the controller address.
;
;	Parameters: [sect] [head] [cyl] [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
lsparms:
	call nextparm		; skip to next parameter
	jr z,1$			; if EOL, get the last values
	call scanhexb		; get the sector in C
	ld sector_no(iy),c	; save it
	call nextparm		; skip to next parameter
	jr z,1$			; if EOL, get the last values
	call scanhexb		; get the head in C
	ld head_no(iy),c	; save it
	call nextparm		; skip to next parameter
	jr z,1$			; if EOL, get the last values
	call scanhexw		; get the cylinder in DE
	ld cylinder_msb(iy),d	; save it
	ld cylinder_lsb(iy),e	;
	call nextparm		; skip to next parameter
	jr z,1$			; if EOL, get the last values
	call scanhexw		; get the load address
	ld load_addr+1(iy),d	; save it
	ld load_addr(iy),e	;

1$:	call _ide_address	; get controller address in C
	ld a,head_no(iy)	; get the head
	ld d,cylinder_msb(iy)	; get the cylinder
	ld e,cylinder_lsb(iy)	;
	ld b,sector_no(iy)	; get the sector
	ld h,load_addr+1(iy)	; get the load address
	ld l,load_addr(iy)	;
	ret			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Process an Intel hex record.
;	Does not check for errors.
;
;	We have 5200 T states to complete this
;	@ 9600 bps
;
;	T States: ~609 + ~202 per byte
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
intel_hex:
	inc hl			; skip to the next byte			7

	; get the record length
	call scanhexb		; get the data length into C		~175
	or a,a			; length > 0?				4
	jp z,cmd		; no, leave				5,11
	ld b,a			; yes, copy length to B			4

	; get the address
	call scanhexb		; get the MSB of the load addr into C	~175
	ld d,a			; place in D				4
	call scanhexb		; get the LSB of the load addr into C	~175
	ld e,a			; place in E				4

	; check the record type
	ld a,(hl)		; get first nibble			7
	cp #'0			; should be a 0				7
	jp nz,cmd		; nope, leave				5,11
	inc hl			; next character			6
	ld a,(hl)		; get next nibble			7
	cp #'0			; should be a 0				7
	jp nz,cmd		; nope, leave				5,11
	inc hl			; next character			6

	; load the data	
1$:	call scanhexb		; get a byte of data in A		~175
	ld (de),a		; save the byte				7
	inc de			; next location				7
	djnz 1$			; if not done, then continue		13,8
	jp cmd			; done					10


	.page
	.sbttl Initialized data
help:
	.ascii "B = Boot	-- B"
	.db 13,10
	.ascii "D = Dump	-- D [addr] [len]"
	.db 13,10
	.ascii "E = Enter	-- E [addr]"
	.db 13,10
	.ascii "G = Go exec	-- G [addr]"
	.db 13,10
	.ascii "I = Input	-- I <port>"
	.db 13,10
	.ascii "L = Load	-- L [sector] [head] [cylinder] [addr]"
	.db 13,10
	.ascii "O = Output	-- O <port> [val]"
	.db 13,10
	.ascii "S = Save	-- S [sector] [head] [cylinder] [addr]"
	.db 13,10
	.ascii "? = This screen"
	.db 13,10
	.db 0

