;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)cbios.s	1.9	07/12/27
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.area BIOS (ABS)

	.include "memcfg.s"
;
;	"bias" is address offset from 3400H for memory systems
;	than 16K (referred to as "b" throughout the text).
;
bias	=	(msize-20)*1024
ccp	= 	moff+bias	;base of ccp
bdos	=	ccp+0x800	;base of bdos
bdose	=	bdos+6		;bdos entry point
bios	=	ccp+0x1600	;base of bios
iobyte	=	0x0003		;intel i/o byte
cdisk	=	0x0004		;current disk number 0=A,...,15=P
;
nsects	=	(bios-ccp)/128	; warm start sector count
;

	.org bios		;origin of this program

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Available ROM routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
cold_start		= rom+3*0
warm_start		= rom+3*1
con_rxstat		= rom+3*2
con_txstat		= rom+3*3
getchar			= rom+3*4
putchar			= rom+3*5
gets			= rom+3*6
put			= rom+3*7
puts			= rom+3*8
crlf			= rom+3*9
prhex			= rom+3*10
_spio13_rxstat		= rom+3*11
_spio13_txstat		= rom+3*12
_spio13_sctrl		= rom+3*13
_spio13_sread		= rom+3*14
_spio13_swrite		= rom+3*15
_ide_address		= rom+3*16
_ide_hard_reset		= rom+3*17
_ide_drive_id		= rom+3*18
_ide_read_a_sector	= rom+3*19
_ide_write_a_sector	= rom+3*20

punch_addr	= 0x0c		; punch I/O address
reader_addr	= punch_addr	; reader I/O address
_RTS		= 0B00000010	; Request To Send
_DTR		= 0B00000001	; Data Terminal Ready
MODEM_ON	= _DTR+_RTS	; both lines
											  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	CP/M to host disk constants.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
blksiz	= 2048			; CP/M allocation size
hstsiz	= 512			; host disk sector size
hstspt	= 21			; host disk sectors/track
trkoff	= 3			; tracks to reserve for OS
hstblk	= hstsiz / 128		; CP/M sectors/host buffer
cpmspt	= hstblk * hstspt	; CP/M sectors/track
secmsk	= hstblk-1		; sector mask
secshf	= 2			; log2(hstblk)
dsksiz	= ((63*255)-(hstspt*trkoff))*hstsiz/blksiz

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	BDOS constants on entry to write
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
wrall	= 0			; write to allocated
wrdir	= 1			; write to directory
wrual	= 2			; write to unallocated

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Jump vector for individual subroutines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	jp boot			; cold start
wboote:	jp wboot		; warm start
	jp const		; console status
	jp conin		; console character in
	jp conout		; console character out
	jp list			; list character out
	jp punch		; punch character out
	jp reader		; reader character out
	jp home			; move head to home position
	jp seldsk		; select disk
	jp settrk		; set track number
	jp setsec		; set sector number
	jp setdma		; set dma address
	jp read			; read disk
	jp write		; write disk
	jp listst		; return list status
	jp sectran		; sector translate
	jp punchst		; punch status
	jp readerst		; reader status
	jp monitor		; drop into the monitor

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Fixed data tables for four drives
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
dpbase:
	; disk parameter header for disk 00
	.dw	0x0000,0x0000
	.dw	0x0000,0x0000
	.dw	dirbf,dpblk
	.dw	0x0000,all00
	; disk parameter header for disk 01
	.dw	0x0000,0x0000
	.dw	0x0000,0x0000
	.dw	dirbf,dpblk
	.dw	0x0000,all01
	; disk parameter header for disk 02
	.dw	0x0000,0x0000
	.dw	0x0000,0x0000
	.dw	dirbf,dpblk
	.dw	0x0000,all02
	; disk parameter header for disk 03
	.dw	0x0000,0x0000
	.dw	0x0000,0x0000
	.dw	dirbf,dpblk
	.dw	0x0000,all03

dpblk:	; disk parameter block for all disks
	; set for 2K blocks
	.dw	cpmspt		; sectors per track	SPT
	.db	4		; block shift factor	BSH
	.db	15		; block mask		BLM
	.db	0		; null mask		EXM
	.dw	dsksiz-1	; disk size-1		DSM
	.dw	1023		; directory max		DRM
	.db	0b11111111	; alloc 0		AL0
	.db	0b11111111	; alloc 1		AL1
	.dw	0		; check size		CKS
	.dw	trkoff		; track offset		OFF


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Cold boot.
;
;	Perform parameter initialization then
;	go run the OS.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
boot:
	xor a,a			; A = 0
	ld (hstact),a		; host buffer inactive
	ld (unacnt),a		; clear unalloc count
	ld (iobyte),a		; clear the iobyte
	ld (cdisk),a		; select disk zero
	ld a,#MODEM_ON		; get the modem control lines
	ld c,#punch_addr	; get the punch address
	call _spio13_sctrl	; initialize the punch device
	jr gocpm		; initialize and go to CP/M


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Monitor.
;
;	Halt the operating system and transfer
;	control to the ROM monitor.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
monitor:
	ld sp,#0xdff0		; load the ROM monitor's stack pointer
	jp warm_start		; have the monitor do a warm start


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Warm boot.
;
;	Reload the CCP & BDOS from disk.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
wboot:
	xor a,a			; A = 0
	ld (hstact),a		; host buffer inactive
	ld (unacnt),a		; clear unalloc count
	ld sp,#0x80		; use space below buffer for stack
	ld c,#0			; select disk 0
	call seldsk		; 
	call home		; go to track 00
	ld b,#nsects		; B counts # of sectors to load
	ld c,#0			; C has the current track number
	ld d,#1			; D has the next sector to read
	ld hl,#ccp		; base of cp/m (initial load point)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load one more sector.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
load1:
	push bc			; save sector count, current track
	push de			; save next sector to read
	push hl			; save dma address
	ld c,d			; get sector address to register C
	call setsec		; set sector address from register C
	pop bc			; recall dma address to b,c
	push bc			; replace on stack for later recall
	call setdma		; set dma address from b,c
	call read		; read a sector into memory
	or a,a			; any errors?
	jr nz,wboot		; yes, reboot if an error occurs
	pop hl			; no, recall dma address
	ld de,#128		; dma=dma+sector size
	add hl,de		; new dma address is in HL
	pop de			; recall sector address
	pop bc			; recall number of sectors left and current trk
	dec b			; decrement the sector count
	jr z,gocpm		; transfer to cp/m if all have been loaded
	inc d			; otherwise, skip to the next sector
	ld a,d			; get the sector number
	cp a,#cpmspt+1		; end of track?
	jr c,load1		; no, load another one
	ld d,#1			; yes, begin with first sector of next track
	inc c			; skip to the next track
	push bc			; save these
	push de			;
	push hl			;
	ld b,#0			; convert track to 16 bits
	call settrk		; set track from register c
	pop hl			; restore these
	pop de			;
	pop bc			;
	jr load1		; load another sector


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	End of load operation.  Set parameters
;	and go to CP/M.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
gocpm:
	ld a,#0xc3		; C3 is a jmp instruction
	ld (0),a		; for jmp to wboot
	ld hl,#wboote		; wboot entry point
	ld (1),hl		; set address field for jmp at 0
	ld (5),a		; for jmp to bdos
	ld hl,#bdose		; bdos entry point
	ld (6),hl		; address field of jump at 5 to bdos
	ld bc,#0x80		; set the default DMA address
	call setdma		; set the DMA address
	;ei			; enable the interrupt system
	ld a,(cdisk)		; get current disk number
	ld c,a			; send to the CCP
	jp ccp			; go to CP/M for further processing


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Return the status of the consolek and 
;	receive data register.
;
;	Parameters:
;		None
;
;	Returns:
;		A = 0xff if character ready
;		A = 0x00 no character waiting
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
const:
	call con_rxstat		; get the console status
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Return a character from the console.
;
;	Parameters:
;		None
;
;	Returns:
;		A = new character
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
conin:
	call getchar		; get a character
	and a,#0x7f		; strip parity bit
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Send a character to the console.
;
;	Parameters:
;		C = character
;
;	Returns:
;		Clobbers A
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
conout:
	ld a,c			; move character to A
	call putchar		; print the character
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Send a character to the list device.
;
;	Parameters:
;		C = character
;
;	Returns:
;		Clobbers A
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
list:
	ld a,c			; move character to A
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get the status of the list device.
;
;	Parameters:
;		None.
;
;	Returns:
;		A = 0 not ready
;		A = 1 ready
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
listst:
	xor a,a			; 0 is always ok to return
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Send a character to the punch device.
;
;	Parameters:
;		C = character
;
;	Returns:
;		Clobbers A & C
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
punch:
	ld a,c			; move character to A
	ld c,#punch_addr	; get the I/O address
	jp _spio13_swrite	; write the character


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get the status of the punch device.
;
;	Parameters:
;		None.
;
;	Returns:
;		A = 0x00 for idle,  0xff for transmitting.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
punchst:
	push bc			; save these
	ld c,#punch_addr	; get the I/O address
	call _spio13_txstat	; get the transmitter status
	pop bc			; restore these
	jr nz,1$		; transmitter busy?
	xor a,a			; no, return false
	ret			; done
1$:	ld a,#0xff		; yes return true
	ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get a character from the reader device.
;
;	Parameters:
;		None.
;
;	Returns:
;		A = new character
;		Clobbers C
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
reader:
	ld c,#reader_addr	; get the I/O address
	call _spio13_sread	; read a character
	and a,#0x7f		; strip the parity bit
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get the status of the reader device.
;
;	Parameters:
;		None.
;
;	Returns:
;		A = 0 for no data, 0xFF for data available
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
readerst:
	push bc			; save these
	ld c,#reader_addr	; get the I/O address
	call _spio13_rxstat	; get the transmitter status
	pop bc			; restore these
	jr nz,1$		; transmitter busy?
	xor a,a			; no, return false
	ret			; done
1$:	ld a,#0xff		; yes return true
	ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Position the drive head on track 0.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
home:
	ld a,(hstwrt)		; get host written flag
	or a,a			; host written?
	jr nz,homed		; yes, reset to track 0
	ld (hstact),a		; no, clear host active flag
homed:	ld bc,#0		; track 0
	call settrk		; set to track 0
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Select a new disk.
;
;	Parameters:
;		C = new disk
;
;	Returns:
;		HL = disk parameter block or 0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
seldsk:
	ld hl,#0x0000		; error return code
	ld a,c			; move to A
	ld (diskno),a		; save the disk number
	cp a,#4			; between 0 and 3?
	ret nc			; no, leave.
	ld l,a			; yes, convert to 16 bits
	ld h,#0			;
	add hl,hl		; *2
	add hl,hl		; *4
	add hl,hl		; *8
	add hl,hl		; *16 (size of each header)
	ld de,#dpbase		; get the drive parameter location
	add hl,de		; HL=.dpbase(diskno*16)
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set the current track.
;
;	Parameters:
;		BC = new track
;
;	Returns:
;		Nothing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
settrk:
	ld h,b			; get the track number
	ld l,c			; 
	ld (track),hl		; save it
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set the current sector.
;
;	Parameters:
;		C = new sector
;
;	Returns:
;		Nothing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
setsec:
	ld a,c			; get the sector number
	ld (sector),a		; save it
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Translate a sector.
;
;	Parameters:
;		BC = sector
;		DE = translation table
;
;	Returns:
;		HL = new sector
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
sectran:
	ld h,b			; HL=BC
	ld l,c			;
	ret			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set the DMA address.
;
;	Parameters:
;		BC = new address
;
;	Returns:
;		Clobbers HL
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
setdma:
	ld l,c			; low order address
	ld h,b			; high order address
	ld (dmaadr),hl		; save the address
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read a sector from disk.
;
;	Parameters:
;		None.
;
;	Returns:
;		Clobbers everything
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
read:
	xor a,a			; A = 0
	ld (unacnt),a		; clear unalloc count
	ld a,#1			; true flag
	ld (readop),a		; read operation
	ld (rsflag),a		; must read data
	ld a,#wrual		;
	ld (wrtype),a		; treat as unalloc
	jr rwoper		; perform the read


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write a sector to disk.
;
;	Parameters:
;		None.
;
;	Returns:
;		Clobbers everything
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
write:
	xor a,a			; clear A
	ld (readop),a		; not a read operation
	ld a,c			; write type in C
	ld (wrtype),a		; save it
	cp a,#wrual		; write unallocated?
	jr nz,chkuna		; no, check for unallocated
	
	; write to unallocated, set parameters
	ld a,#blksiz/128	; next unallocated records
	ld (unacnt),a		; save it
	ld a,(diskno)		; disk to seek
	ld (unadsk),a		; set unadsk
	ld hl,(track)		; get the track
	ld (unatrk),hl		; set unatrk
	ld a,(sector)		; get sector
	ld (unasec),a		; set unasec

chkuna:
	; check for write to unallocated sector
	ld a,(unacnt)		; get unallocated count
	or a,a			; any unallocated remaining?
	jr z,alloc		; no, skip this

	; more unallocated records remain
	dec a			; decrement unallocated count
	ld (unacnt),a		; save it
	ld a,(diskno)		; get disk number
	ld hl,#unadsk		; get unadsk address
	cp a,(hl)		; same disk?
	jr nz,alloc		; no, skip this

	; disks are the same
	ld hl,#unatrk		; unallocated track address
	call sektrkcmp		; same track?
	jr nz,alloc		; no, skip this

	; tracks are the same
	ld a,(sector)		; get the sector
	ld hl,#unasec		; unallocated sector address
	cp a,(hl)		; same sector?
	jr nz,alloc		; no skip this

	; match, move to next sector for future reference
	inc (hl)		; next sector
	ld a,(hl)		; get next sector
	cp a,#cpmspt		; end of track?
	jr c,noovf		; skip of no overflow

	; overflow to next track
	ld (hl),#0		; unasec = 0
	ld hl,(unatrk)		; get unallocated track
	inc hl			; next track
	ld (unatrk),hl		; save it

noovf:
	; match found, mark as unnecessary read
	xor a,a			; clear A
	ld (rsflag),a		; clear rsflag
	jr rwoper		; perform the write

alloc:
	; not an unallocated record, requires pre-read
	xor a,a			; clear A
	ld (unacnt),a		; clear unallocated count
	inc a			; true
	ld (rsflag),a		; set rsflag

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Common code for read and write follows.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
rwoper:
	; enter here to perform the read/write
	xor a,a			; clear A
	ld (erflag),a		; no errors (yet)
	ld a,(sector)		; get sector
	or a,a			; clear carry & compute the host sector
	rra			; / 2
	or a,a			; clear carry
	rra			; / 4
	ld (sekhst),a		; host sector to seek

	; active host sector?
	ld hl,#hstact		; get address to host active flag
	ld a,(hl)		; get flag
	ld (hl),#1		; always set flag
	or a,a			; was it already set?
	jr z,filhst		; no, fill host buffer

	; host buffer active, same as seek buffer?
	ld a,(diskno)		; get disk number
	ld hl,#hstdsk		; get host disk address
	cp a,(hl)		; same disk?
	jr nz,nomatch		; no, clear host buffer first

	; same disk, same track?
	ld hl,#hsttrk		; get host track address
	call sektrkcmp		; same track?
	jr nz,nomatch		; no, clear host buffer first

	; same disk, same track, same buffer?
	ld a,(sekhst)		; get host sector to seek
	ld hl,#hstsec		; get host sector address
	cp a,(hl)		; same sector?
	jr z,match		; yes, skip this

nomatch:
	; proper disk, but not correct sector
	ld a,(hstwrt)		; get host written flag
	or a,a			; host written?
	call nz,writehst	; if not, clear host buffer

filhst:
	; may have to fill the host buffer
	ld a,(diskno)		; get disk number
	ld (hstdsk),a		; set host disk
	ld hl,(track)		; get track number
	ld (hsttrk),hl		; set host track
	ld a,(sekhst)		; get host sector to seek
	ld (hstsec),a		; set host sector
	ld a,(rsflag)		; get read flag
	or a,a			; need to read?
	call nz,readhst		; yes, read a sector
	xor a,a			; clear A
	ld (hstwrt),a		; no pending write

match:
	; copy data to or from buffer
	ld a,(sector)		; get sector number
	and a,#secmsk		; mask buffer number LS bits
	ld l,a			; convert to 16 bits
	ld h,#0			;
	add hl,hl		; shift left 7 times
	add hl,hl		;
	add hl,hl		;
	add hl,hl		;
	add hl,hl		;
	add hl,hl		;
	add hl,hl		; HL has relative host buffer address
	ld de,#hstbuf		; get host buffer
	add hl,de		; HL now host address
	ex de,hl		; move to DE
	ld hl,(dmaadr)		; get the DMA address
	ld c,#128		; length of move
	ld a,(readop)		; get read flag
	or a,a			; move which way?
	jr nz,rwmove		; skip this if read

	; write operation, mark and switch direction
	ld a,#1			; write flag
	ld (hstwrt),a		; set host write flag
	ex de,hl		; swap source and destination

rwmove:
	; C initially 128, DE is source, HL is destination
	ld a,(de)		; get a source byte
	inc de			; increment source pointer
	ld (hl),a		; copy byte to dest
	inc hl			; increment destination pointer
	dec c			; done?
	jr nz,rwmove		; if not, do it again

	; data has been moved to/from host buffer
	ld a,(wrtype)		; get write type
	cp a,#wrdir		; to directory?
	ld a,(erflag)		; in case of errors
	ret nz			; no further processing

	; clear host buffer for directory write
	or a,a			; errors?
	ret nz			; yes, leave
	xor a,a			; clear A
	ld (hstwrt),a		; buffer written
	call writehst
	ld a,(erflag)		; get error flag
	ret			; leave


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Utility subroutine for 16-bit compare
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
sektrkcmp:
	; HL = .unatrk or .hsttrk, compare with track
	ex de,hl		; save host track
	ld hl,#track		; get track address
	ld a,(de)		; get LSB of host track
	cp a,(hl)		; same?
	ret nz			; no, leave
	inc de			; high byte
	inc hl			;
	ld a,(de)		; get LSB of host track
	cp a,(hl)		; set flags
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Perform the physical write
;
;	Parameters:
;		hstdsk = host disk #
;		hsttrk = host track #
;		hstsec = host sector #
;		hstsiz = bytes to write
;		hstbuf = buffer to write from
;
;	Returns:
;		erflag = non-zero if error
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
writehst:
	call getlba		; get lba in B,DE,A
	push af			; save A to keep it from getting clobbered
	call _ide_address	; get the controller address in C
	pop af			; restore A
	ld hl,#hstbuf		; get the buffer address
	call _ide_write_a_sector; write the sector to disk
	ld (erflag),a		; store any error codes
	ret			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Perform the physical read
;
;	Parameters:
;		hstdsk = host disk #
;		hsttrk = host track #
;		hstsec = host sector #
;		hstsiz = bytes to read
;		hstbuf = buffer to read to
;
;	Returns:
;		erflag = non-zero if error
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
readhst:
	call getlba		; get lba in B,DE,A
	push af			; save A to keep it from getting clobbered
	call _ide_address	; get the controller address in C
	pop af			; restore A
	ld hl,#hstbuf		; get the buffer address
	call _ide_read_a_sector	; load the sector into memory
	ld (erflag),a		; store any error codes
	ret			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Caclculate the LBA from the track,
;	sector and diskno values.
;	LBA = hsttrk * hstspt + hstsec + lbbase(hstdsk)
;
;	Parameters:
;		None.
;
;	Returns:
;		A,DE,B = LBA
;		Clobbers HL & IX
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
getlba:
	ld a,(hstsec)		; get the sector number
	ld l,a			; move to L
	ld h,#0			; convert to 16 bits
	ld bc,(hsttrk)		; get the track mumber
	ld a,c			; track 0?
	or a,b			; 
	jr z,2$			; yes, we don't need to multiply
	ld d,#0			; no, load the sectors/track in DE
	ld e,#hstspt		; 
1$:	add hl,de		; multiply the sectors/track by the track
	dec bc			;
	ld a,b			;
	or a,c			;
	jr nz,1$		;
2$:	ld a,(hstdsk)		; get the disk number
	sla a			; * 2
	sla a			; * 4
	ld e,a			; convert to 16 bits
	ld d,#0			;
	push hl			; save HL
	ld hl,#lbbase		; get the location of the disk table
	add hl,de		; add the offset to the drive
	ex de,hl		; move back to DE
	pop hl			; restore HL
	push de			; move to IX
	pop ix			;
	ld a,(ix)		; get the LSW of the LBA in DE
	ld e,a			;
	ld a,1(ix)		;
	ld d,a			;
	add hl,de		; add the sector
	ld b,l			; move the LSB to B
	ld e,h			; move the MSB to E
	ld a,2(ix)		; get the MSW of the LBA in D & A
	ld d,a			;
	ld a,3(ix)		;
	jr nc,3$		; if no carry, then we're done with that
	inc d			; otherwise increment the LSB of the MSW
	jr nz,3$		; if we didn't carry, then skip this
	inc a			; otherwise, increment the MSB of the MSW
3$:	ret			; done!


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Hard-coded LBA for each drive.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	.even
lbbase:	.db	0x30,0xc4,0x0b,0x40	; 63 sectors x 255 tracks
	.db	0xf1,0x02,0x0c,0x40	; 63 sectors x 255 tracks
	.db	0xb2,0x41,0x0c,0x40	; 63 sectors x 255 tracks
	.db	0x73,0x80,0x0c,0x40	; 63 sectors x 255 tracks


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	The remainder of the CBIOS is reserved
;	uninitialized data area, and does not
;	need to be a part of the system memory
;	image (the space must be available,
;	however, between "begdat" and "enddat").
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
diskno:	.ds	1		; seek disk number 0-15
track:	.ds	2		; seek track number
sector:	.ds	1		; seek sector number

hstdsk:	.ds	1		; host disk number
hsttrk:	.ds	2		; host track number
hstsec:	.ds	1		; host sector number

sekhst:	.ds	1		; seek shr secshf
hstact: .ds	1		; host active flag
hstwrt:	.ds	1		; host written flag

unacnt:	.ds	1		; unalloc record count
unadsk:	.ds	1		; last unallocated disk
unatrk:	.ds	2		; last unallocated track
unasec:	.ds	1		; last unallocated sector

erflag:	.ds	1		; error reporting
rsflag:	.ds	1		; read sector flag
readop:	.ds	1		; 1 if read operation
wrtype:	.ds	1		; write operation type
dmaadr:	.ds	2		; last DMA address
hstbuf: .ds	hstsiz		; host buffer

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Scratch ram area for BDOS use.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
begdat	=	.		;beginning of data area
dirbf:	.ds	128		;scratch directory area
all00:	.ds	dsksiz/8	;allocation vector 0
all01:	.ds	dsksiz/8	;allocation vector 1
all02:	.ds	dsksiz/8	;allocation vector 2
all03:	.ds	dsksiz/8	;allocation vector 3

enddat	=	.		;end of data area
datsiz	=	.-begdat	;size of data area

