;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)ide.s	1.5	07/12/06
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.module	ide

	.globl ram_addr, _ide_port

	.area CODE

need_features = 0

; PIO Control Register
_MODE_ACTIVE	= 0B10000000
_GA_MODE_2	= 0B01000000
_GA_MODE_1	= 0B00100000
_GA_PA_IN	= 0B00010000
_GA_PCH_IN	= 0B00001000
_GB_MODE_1	= 0B00000100
_GB_PB_IN	= 0B00000010
_GB_PCL_IN	= 0B00000001

_PAB_READ	= _MODE_ACTIVE | _GB_PB_IN | _GA_PA_IN
_PAB_WRITE	= _MODE_ACTIVE

_CONFIG_REG	= _MODE_ACTIVE | _GB_PB_IN | _GA_PA_IN | _GA_PCH_IN | _GB_PCL_IN
;
; Drive bus control lines
;
_IDE_DA0	= 0B00000001
_IDE_DA1	= 0B00000010
_IDE_DA2	= 0B00000100
_IDE_CS0	= 0B00001000	; Command Block Registers
_IDE_CS1	= 0B00010000	; Control Block Registers
_IDE_DIOW	= 0B00100000	; Drive I/O Write
_IDE_DIOR	= 0B01000000	; Drive I/O Read
_IDE_RESET	= 0B10000000	; Reset the bus

;
; Drive bus command combinations
;
_IDE_ALT_STATUS	= _IDE_CS1 | _IDE_DA2 | _IDE_DA1		; read only
_IDE_DEV_CTRL	= _IDE_CS1 | _IDE_DA2 | _IDE_DA1		; write only
_IDE_DRV_ADDR	= _IDE_CS1 | _IDE_DA2 | _IDE_DA1 | _IDE_DA0	; read only

_IDE_DATA	= _IDE_CS0
_IDE_ERROR	= _IDE_CS0 | _IDE_DA0				; read only
_IDE_FEATURES	= _IDE_CS0 | _IDE_DA0				; write only
_IDE_SECTOR_CNT	= _IDE_CS0 | _IDE_DA1
_IDE_SECTOR_NUM	= _IDE_CS0 | _IDE_DA1 | _IDE_DA0
_IDE_CYL_LOW	= _IDE_CS0 | _IDE_DA2
_IDE_CYL_HIGH	= _IDE_CS0 | _IDE_DA2 | _IDE_DA0
_IDE_DRV_HEAD	= _IDE_CS0 | _IDE_DA2 | _IDE_DA1
_IDE_STATUS	= _IDE_CS0 | _IDE_DA2 | _IDE_DA1 | _IDE_DA0	; read only
_IDE_COMMAND	= _IDE_CS0 | _IDE_DA2 | _IDE_DA1 | _IDE_DA0	; write only

;
; Logical interface commands
;
_IDE_RECALIBRATE	= 0x10	; Recalibrate
_IDE_READ_SECTOR	= 0x20	; Read sector(s) (w/retry)
_IDE_READ_SECTOR_N	= 0x21	; Read sector(s) (w/o retry)
_IDE_READ_LONG		= 0x22	; Read long (w/retry)
_IDE_READ_LONG_N	= 0x23	; Read long (w/o retry)
_IDE_WRITE_SECTOR	= 0x30	; Write sector(s) (w/retry)
_IDE_WRITE_SECTOR_N	= 0x31	; Write sector(s) (w/o retry)
_IDE_WRITE_LONG		= 0x32	; Write long (w/retry)
_IDE_WRITE_LONG_N	= 0x33	; Write long (w/o retry)
_IDE_READ_VFY_SECTOR	= 0x40	; Read verify sector(s) (w/retry)
_IDE_READ_VFY_SECTOR_N	= 0x41	; Read verify sector(s) (w/o retry)
_IDE_FORMAT_TRACK	= 0x50	; Format track
_IDE_SEEK		= 0x70	; Seek
_IDE_DRIVE_DIAG		= 0x90	; Execute drive diagnostic
_IDE_INIT_PARAM		= 0x91	; Initialize drive parameters
_IDE_IDENTIFY_DRIVE	= 0xec	; Identify drive

;
; Status register bits
;
_BSY	= 7	; Busy.  Drive has access to the Command registers
_DRDY	= 6	; Drive Ready
_DWF	= 5	; Drive Write Fault
_DSC	= 4	; Drive Seek Complete
_DRQ	= 3	; Data Request
_CORR	= 2	; Corrected Data
_IDX	= 1	; Index
_ERR	= 0	; Error (see Error Register)

;
; Error register bits
;
_BBK	= 7	; Bad Block Detected
_UNC	= 6	; Uncorrectable Data Error
_MC	= 5	; Media Changed
_IDNF	= 4	; Requested sector's ID field not found
_MCR	= 3	; Media Change Requested
_ABRT	= 2	; Aborted Command (drive busy or not ready)
_TK0NF	= 1	; Track 0 Not Found
_AMNF	= 0	; Address Mark Not Found

;
; Drive & Head register bits
;
_LBA	= 6	; 0=CHS, 1=LBA
_DRIVE	= 4	; 0=drive 0, 1=drive 1
		; lower 4 bits = head or LBA 24-27

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load the first sector from a disk and 
;	execute the code contained in it
;
;	Parameters:
;		None
;
;	Returns:
;		Doesn't return - executes code from
;		the boot sector.
;
;	T States: 37650
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_boot::
	call _ide_address	; get the IDE port address in C
_ide_boot1::
	call _ide_hard_reset	; reset the bus				232
	ld h,ram_addr+1(iy)	; get keyboard buffer address		19
	ld l,ram_addr(iy)	; get keyboard buffer address		19
	ld de,#0x80		; give the keyboard buffer some room	10
	add hl,de		; use this as our load address		11
	push hl			; push the boot address			11
	xor a,a			; CHS mode, drive 0 & head 0		4
	ld b,#1			; sector 1				7
	ld de,#0		; cylinder 0				10
	call _ide_read_a_sector	; read the sector into memory		37334
	ret			; boot					4


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get the address of the IDE controller
;
;	Parameters:
;		None
;
;	Returns:
;		C = 8255 base address
;		Clobbers A
;
;	T States: 54
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_address::
	ld a,#_CONFIG_REG	; Ports A, B, C read			7
	out (3),a		; Allow reading of ports A,B,C		11
	out (3),a		;					11
	in a,(_ide_port)	; get the IDE address			11
	ld c,a			; move to C				4
	ret			; done					10

	
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Perform a hard reset on the ATA bus
;
;	Parameters:
;		C = 8255 base address
;
;	Returns:
;		Nothing	
;
;	T States: 215
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_hard_reset::
	push bc			; save these				11
	inc c			; port B (high byte)			4
	inc c			; port C (drive interface)		4
	inc c			; PIO control register			4
	ld b,#_PAB_READ		; control word				7
	out (c),b		; read from PA and PB and write to PC	12
	out (c),b		; 					12
	dec c			; port C (drive interface)		4
	ld b,#_IDE_RESET	; bus reset				7
	out (c),b		; reset the bus				12
	out (c),b		;					12
	ld b,#4			; reset to be held for at least 25 uS	7
1$:	djnz 1$			;					13,8
	out (c),b		; clear signals 			12
	out (c),b		; 					12
	inc c			; PIO control register			4
	out (c),b		; deactivate I/O lines			12
	out (c),b		; 					12
	pop bc			; restore these				10
	ret			; done!					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load the drive identification block
;
;	Parameters:
;		C = 8255 base address
;		HL = buffer address
;
;	Returns:
;		HL = end of data + 1
;
;	T States: 34267+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_drive_id::
	push af			; save these				11
	push bc			;					11
	ld a,#0B10100000	; drive 0				7
	ld b,#_IDE_DRV_HEAD	; IDE drive & head register		7
	call _ide_write_register; select drive 0			721+
	ld a,#_IDE_IDENTIFY_DRIVE	; identify drive		7
	ld b,#_IDE_COMMAND	; IDE command register			7
	call _ide_write_register; get the drive identification		721+
	call _ide_read_data	; read drive data 			32745+
	pop bc			; restore these				10
	pop af			;					10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set the drive and head or LBA bits 24-27.
;
;	Parameters:
;		C = 8255 base address
;		A = LBA/CHS flag, drive, head
;
;	Returns:
;		Nothing
;
;	T States: 777+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_set_drv_head::
	push af			; save these				11
	push bc			;					11
	;and a,#0B00011111	; make sure LBA flag is clear		7
	or a,#0B10100000	; bits 5 & 7 always set			7
	ld b,#_IDE_DRV_HEAD	; IDE drive & head register		7
	call _ide_write_register; write drive & head			704+
	pop bc			; restore these				10
	pop af			;					10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set the cylinder or LBA bits 8-23.
;
;	Parameters:
;		C = 8255 base address
;		DE = Cylinder or LBA bits 8-23
;
;	Returns:
;		Nothing
;
;	T States: 1482+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_set_cylinder::
	push af			; save these				11
	push bc			;					11
	ld a,e			; get the cylinder LSB			4
	ld b,#_IDE_CYL_LOW	; IDE cylinder low register		7
	call _ide_write_register; write cylinder LSB			704+
	ld a,d			; get the cylinder MSB			4
	ld b,#_IDE_CYL_HIGH	; IDE cylinder high register		7
	call _ide_write_register; write cylinder MSB			704+
	pop bc			; restore these				10
	pop af			;					10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set the sector or LBA bits 0-7.
;
;	Parameters:
;		C = 8255 base address
;		B = Sector or LBA bits 0-7
;
;	Returns:
;		Nothing
;
;	T States: 767+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_set_sector::
	push af			; save these				11
	push bc			;					11
	ld a,b			; get the sector 			4
	ld b,#_IDE_SECTOR_NUM	; IDE sector register			7
	call _ide_write_register; write sector number			704+
	pop bc			; restore these				10
	pop af			;					10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set CHS
;
;	Parameters:
;		C = 8255 base address
;		A = Drive & Head
;		B = Sector
;		DE = Cylinder
;		HL = Buffer address
;
;	Returns:
;		Clobbers A & B
;
;	T States: 3795+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_set_chs::
	call _ide_set_drv_head	; set the drive & head			794+
	call _ide_set_cylinder	; set the cylinder number		1499+
	call _ide_set_sector	; set the sector number			784+
	.if need_features
	ld a,#1			; PIO default transfer mode, w/o IORDY	7
	ld b,#_IDE_SECTOR_CNT	; Store in sector count register	7
	call _ide_write_register; write feature				704+
	ld a,#3			; Set transfer mode			7
	ld b,#_IDE_FEATURES	; IDE feaures register			7
	call _ide_write_register; set feature				704+
	.endif
	ld a,#1			; read only 1 sector			7
	ld b,#_IDE_SECTOR_CNT	; IDE sector count register		7
	call _ide_write_register; write sector count			704+
	ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read a sector.
;
;	Parameters:
;		C = 8255 base address
;		A = Drive & Head
;		B = Sector
;		DE = Cylinder
;		HL = Buffer address
;
;	Returns:
;		A = Error Code
;
;	T States: 37317+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_read_a_sector::
	push bc			; save these				11
	call _ide_set_chs	; setup the CHS parameters		3812+
	ld a,#_IDE_READ_SECTOR	; read sector				7
	ld b,#_IDE_COMMAND	; IDE command register			7
	call _ide_write_register; get the drive identification		704+
	call _ide_read_data	; read the sector into memory		32745+
	pop bc			; restore these				10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write a sector.
;
;	Parameters:
;		C = 8255 base address
;		A = Drive & Head
;		B = Sector
;		DE = Cylinder
;		HL = Buffer address
;
;	Returns:
;		A = Error Code
;
;	T States: 37310+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_write_a_sector::
	push bc			; save these				11
	call _ide_set_chs	; setup the CHS parameters		3812+
	ld a,#_IDE_WRITE_SECTOR	; write sector				7
	ld b,#_IDE_COMMAND	; IDE command register			7
	call _ide_write_register; get the drive identification		704+
	call _ide_write_data	; read the sector into memory		32738+
	pop bc			; restore these				10
	ret			; done.					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read 512 bytes from the drive buffer
;
;	Parameters:
;		C = 8255 base address
;		HL = buffer address
;
;	Returns:
;		A = Error value
;		HL = end of data + 1
;
;	T States: 32732+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_read_data::
	call _ide_wait_drq	; wait until ready			285+
	bit _ERR,a		; error?				8
	jr z,1$			; no, start reading			7,12
	call _ide_get_error	; yes, get the error in A		316
	ret			; leave					10
1$:	push bc			; save these				11
	inc c			; port B (high byte)			4
	inc c			; port C (drive interface)		4
	inc c			; PIO control register			4
	ld a,#_PAB_READ		; control word				7
	out (c),a		; read from PA and PB and write to PC	12
	out (c),a		; 					12
	dec c			; port C (drive interface)		4
	ld a,#_IDE_DATA		; IDE data register			7
	out (c),a		; setup the address			12
	out (c),a		; 					12
	ld b,#0			; read 256 words (512 bytes)		7
2$:	or a,#_IDE_DIOR		; tack on the read bit			7
	out (c),a		; assert the read signal		12
	out (c),a		; 					12
	dec c			; port B (high byte)			4
	dec c			; port A (low byte)			4
	ini			; read & store the low byte		16
	inc c			; port B (high byte)			4
	ini			; read & store the high byte		16
	inc c			; port C (drive interface)		4
	xor a,#_IDE_DIOR	; erase the read bit			7
	out (c),a		; clear the read signal			12
	out (c),a		; 					12
	inc b			; reading words				4
	jr nz,2$		; done?					7,12
	xor a			; clear the address			4
	out (c),a		; clear all signals			12
	out (c),a		; 					12
	inc c			; PIO control register			4
	out (c),a		; deactivate I/O lines			12
	out (c),a		; 					12
	pop bc			; restore these				10
	xor a,a			; no error				4
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write 512 bytes to the drive buffer
;
;	Parameters:
;		C = 8255 base address
;		HL = buffer address
;
;	Returns:
;		A = Error value
;		HL = end of data + 1
;
;	T States: 32725+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_write_data::
	call _ide_wait_drq	; wait until ready			285+
	bit _ERR,a		; error?				8
	jr z,1$			; no, start reading			7,12
	call _ide_get_error	; yes, get the error in A		316
	ret			; leave					10
1$:	push bc			; save these				11
	inc c			; port B (high byte)			4
	inc c			; port C (drive interface)		4
	inc c			; PIO control register			4
	ld a,#_PAB_WRITE	; setup for write			7
	out (c),a		; write from PA and PB			12
	out (c),a		; 					12
	dec c			; port C (drive interface)		4
	ld a,#_IDE_DATA		; IDE data register			7
	out (c),a		; setup the address			12
	out (c),a		; 					12
	ld b,#0			; write 256 words (512 bytes)		7
2$:	dec c			; port B (high byte)			4
	dec c			; port A (low byte)			4
	outi			; load & write the low byte		16
	inc c			; port B (high byte)			4
	outi			; load & write the high byte		16
	inc c			; port C (drive interface)		4
	or a,#_IDE_DIOW		; tack on the write bit			7
	out (c),a		; assert the write signal		12
	out (c),a		; 					12
	xor a,#_IDE_DIOW	; erase the write bit			7
	out (c),a		; clear the write signal		12
	out (c),a		; 					12
	inc b			; writing words				4
	jr nz,2$		; done?					7,12
	xor a			; clear the address			4
	out (c),a		; clear all signals			12
	out (c),a		; 					12
	inc c			; PIO control register			4
	out (c),a		; deactivate I/O lines			12
	out (c),a		; 					12
	pop bc			; restore these				10
	xor a,a			; no error				4
	ret			; done					3


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read a drive register
;
;	Parameters:
;		C = 8255 base address
;		B = register address
;
;	Returns:
;		A = register contents
;
;	T States: 261
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_read_register:
	inc c			; port B (high byte)			4
	inc c			; port C (drive interface)		4
	inc c			; PIO control register			4
	ld a,#_PAB_READ		; control word				7
	out (c),a		; read from PA and PB and write to PC	12
	out (c),a		; 					12
	dec c			; port C (drive interface)		4
	ld a,b			; register address			4
	out (c),a		; setup the address			12
	out (c),a		; 					12
	or a,#_IDE_DIOR		; tack on the read bit			7
	out (c),a		; assert the read signal		12
	out (c),a		; 					12
	dec c			; port B (high byte)			4
	dec c			; port A (low byte)			4
	in a,(c)		; get the register contents		12
	push af			; save the result			11
	inc c			; port B (high byte)			4
	inc c			; port C (drive interface)		4
	ld a,b			; get the register address		4
	out (c),a		; clear the read signal			12
	out (c),a		; 					12
	xor a			; clear the address			4
	out (c),a		; clear all signals			12
	out (c),a		; 					12
	inc c			; PIO control register			4
	out (c),a		; deactivate I/O lines			12
	out (c),a		; 					12
	dec c			; port C (drive interface)		4
	dec c			; port B (high byte)			4
	dec c			; port A (low byte)			4
	pop af			; get result				10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write a drive register
;
;	Parameters:
;		C = 8255 base address
;		B = register address
;		A = value to be written
;
;	Returns:
;		A = Error value
;
;	T States: 704+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_write_register:
	push bc			; save these				11
	ld b,a			; save A				4
	call _ide_wait_ready	; wait until ready			378+
	bit _ERR,a		; error?				8
	jr z,1$			; no, start writing			7,12
	call _ide_get_error	; yes, get the error in A		316
	pop bc			; restore these				10
	ret			; leave					10
1$:	ld a,b			; restore A				4
	pop bc			; restore these				10
	inc c			; port B (high byte)			4
	inc c			; port C (drive interface)		4
	inc c			; PIO control register			4
	push af			; save the value			11
	ld a,#_PAB_WRITE	; setup for write			7
	out(c),a		; set the 8255 to write to PA and PB	12
	out(c),a		; 					12
	dec c			; port C (drive interface)		4
	ld a,b			; register address			4
	out (c),a		; load the address onto the ATA bus	12
	out (c),a		; 					12
	dec c			; port B (high byte)			4
	dec c			; port A (low byte)			4
	pop af			; get value				10
	out (c),a		; write the value			12
	out (c),a		; 					12
	inc c			; port B (high byte)			4
	inc c			; port C (drive interface)		4
	ld a,b			; register address			4
	or a,#_IDE_DIOW		; tack on the write bit			7
	out (c),a		; assert the write signal		12
	out (c),a		; 					12
	ld a,b			; get the register address		4
	out (c),a		; clear the write signal		12
	out (c),a		; 					12
	xor a			; clear the address			4
	out (c),a		; clear all signals			12
	out (c),a		; 					12
	inc c			; PIO control register			4
	out (c),a		; deactivate I/O lines			12
	out (c),a		; 					12
	dec c			; port C (drive interface)		4
	dec c			; port B (high byte)			4
	dec c			; port A (low byte)			4
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Wait for the DRDY flag to be set
;
;	Parameters:
;		C = 8255 base address
;
;	Returns:
;		When the drive is ready or on an error
;		A = Status Register
;
;	T States: 361+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_wait_ready:
	push bc			; save these				11
1$:	ld b,#_IDE_STATUS	; IDE status register			7
	call _ide_read_register	; get register value in A		278
	bit _ERR,a		; error?				8
	jr nz,2$		; yes, leave				7,12
	bit _BSY,a		; busy?					8
	jr nz,1$		; yes, check again			7,12
	bit _DRDY,a		; ready?				8
	jr z,1$			; no, check again			7,12
2$:	pop bc			; restore these				10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Wait for the DRQ flag to be set
;
;	Parameters:
;		C = 8255 base address
;
;	Returns:
;		When the drive is ready or on an error
;		A = Status Register
;
;	T States: 376+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_wait_drq:
	push bc			; save these				11
1$:	ld b,#_IDE_STATUS	; IDE status register			7
	call _ide_read_register	; get register value in A		278
	bit _ERR,a		; error?				8
	jr nz,2$		; yes, leave				7,12
	bit _BSY,a		; busy?					8
	jr nz,1$		; yes, check again			7,12
	bit _DRDY,a		; ready?				8
	jr z,1$			; no, check again			7,12
	bit _DRQ,a		; ready to transfer?			8
	jr z,1$			; no, check again			7,12
2$:	pop bc			; restore these				10
	ret			; done					10


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get error register
;
;	Parameters:
;		C = 8255 base address
;
;	Returns:
;		A = Error Register
;
;	T States: 316
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_get_error:
	push bc			; save these				11
	ld b,#_IDE_ERROR	; IDE error register			7
	call _ide_read_register	; get register value in A		278
	pop bc			; restore these				10
	ret			; done					10

