	.title	Initialization Routine
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)startup.s	1.15	14/02/18
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.module startup
	.list	(me)

	.area CODE (CON)

	.include "globals.s"

	.sbttl	Global references & code origin definitions

	.if	mirror
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	The ROM is mirrored at 0, but it's
;	really located here.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	;.org rom		; update CODE in the Makefile to revise this
	.else
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	The ROM starts at location 0, so we'll
;	have to hard-code the various vectors.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	;.org 	0x0		; RESET vector
	jp cold_start		; jump to the cold start routine on reset
	.rept	5
	.db	0
	.endm
	reti			; RST 0x08 vector
	.rept	6
	.db	0
	.endm
	reti			; RST 0x10 vector
	.rept	6
	.db	0
	.endm
	reti			; RST 0x18 vector
	.rept	6
	.db	0
	.endm
	reti			; RST 0x20 vector
	.rept	6
	.db	0
	.endm
	reti			; RST 0x28 vector
	.rept	6
	.db	0
	.endm
	reti			; RST 0x30 vector
	.rept	6
	.db	0
	.endm
	jp warm_start		; RST 0x38 vector
	.rept	43
	.db	0
	.endm
	retn			; NMI vector
	.endif

	.page
	.sbttl	Jump table definition
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Jump table to commonly used routines.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
coldst:	jp cold_start		; cold start routine
warmst:	jp warm_start		; warm start routine
	jp con_rxstat		; console receiver status
	jp con_txstat		; console transmitter status
	jp getchar		; get a character from the console
	jp putchar		; send a character to the console
	jp gets			; get a string from the console
	jp put			; send a string to the console
	jp puts			; send a string to the console w/crlf
	jp crlf			; send a crlf to the console
	jp prhex		; convert a byte to ASCII & send to the console
	jp _spio13_rxstat	; serial port receiver status
	jp _spio13_txstat	; serial port transmitter status
	jp _spio13_sctrl	; serial port modem control lines
	jp _spio13_sread	; read a character from a serial port
	jp _spio13_swrite	; write a character to a serial port
	jp _ide_address		; get the I/O port of the ATA controller
	jp _ide_configure	; configure the IDE drive features
	jp _ide_read_sector	; read a sector from the drive
	jp _ide_write_sector	; write a sector to the drive
	jp _ide_get_chs		; get the CHS of the last sector accessed
	.if id_drive
	jp _ide_drive_id	; request the ID from the drive
	.endif


	.page
	.sbttl	Cold start routine
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Determine where the RAM is located
;	and how much is equipped.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
cold_start:
	.if	mirror		; Un-mirror the ROM from low memory
	ld	b,#100		; move the ROM up to the ORG
0$:	in	a,(0)		; for some reason it takes several tries
	djnz	0$		;
	.endif

	;
	;  Configure interrupts
	;
	di			; disable interrupts
	im	1		; interrupt mode 1

	;
	;  Find the RAM starting point
	;
	ld	hl,#0		; start at address 0x0
	ld	bc,#1		; increment one location at a time
	ld	a,#0b10101010	; use this pattern
1$:	ld	(hl),a		; store pattern
	cp	a,(hl)		; pattern still the same?
	jr	nz,2$		; no, continue to the next location
	cpl			; yes, flip the pattern bits
	ld	(hl),a		; store the new pattern	
	cp	a,(hl)		; pattern still the same?
	jr	z,3$		; yes! found the start of RAM
	cpl			; no, flip the pattern back
2$:	add	hl,bc		; next location
	jr	nc,1$		; if not done, store next location
	halt			; else, no RAM found -- panic!
3$:	ld	d,h		; save the RAM starting address (high byte)
	ld	e,l		; save the RAM starting address (low byte)

	; 
	;  Find the length of the RAM.
	; 
	ld	bc,#0		; clear the byte counter
4$:	xor	a,a		; store a 0 in the location
	ld	(hl),a		;
	ld	a,#0b10101010	; use this pattern again
	inc	bc		; increment number of bytes found
	inc	hl		; next address
	ld	(hl),a		; store pattern
	cp	a,(hl)		; pattern the same?
	jr	nz,5$		; no, found end of RAM
	cpl			; flip the pattern bits
	ld	(hl),a		; store the new pattern
	cp	a,(hl)		; pattern the same?
	jr	z,4$		; yes, keep going

	.page
	.sbttl	Storage initialization
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Initialize the stack and local variable
;	storage.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
5$:	ld	sp,hl		; anchor the stack pointer here
	push	bc		; save the amount of RAM
	push	de		; save the RAM start address

	ld	bc,#16		; allocate variable storage above the stack
	or	a,a		; clear the carry
	sbc	hl,bc		; calculate the new stack pointer
	ld	sp,hl		; move the stack pointer down
	push	hl		; store the variable storage location
	pop	iy		; IY = local variable storage


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Initialize page 0 (if it exists)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	xor	a,a		; clear A
	cp	ram_addr+1(iy)	; if the RAM starts at 0...
	jr	nz,6$		;
	cp	ram_addr(iy)	;
	jr	nz,6$		;
	ld	ram_addr(iy),#0x80; reserve space for the vectors

	;
	; populate the various interrupt vectors
	;
	ld	a,#0xC3		; jump instruction
	ld	(0x00),a	;
	ld	hl,#coldst	; jump to the startup routine
	ld	(0x01),hl	;

	ld	hl,#0x4DED	; RETI instruction
	ld	(0x08),hl	; populate the restart vectors
	ld	(0x10),hl	;
	ld	(0x18),hl	;
	ld	(0x20),hl	;
	ld	(0x28),hl	;
	ld	(0x30),hl	;

	ld	(0x38),a	; jump instruction
	ld	hl,#warmst	; jump to the monitor routine on RST 0x38
	ld	(0x39),hl	;
	ld	hl,#0x45ED	; RETN instruction
	ld	(0x66),hl	; populate the NMI vector


	.page
	.sbttl	Initialize I/O devices
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Now that the stack is established,
;	initialize the console.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
6$:	call	con_init	; initialize the console
	ld	hl,#banner	; get banner address
	call	puts		; write banner to console

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Print the various service messages.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	ld	hl,#ramfound 	; get RAM found string
	call	put		; print it
	ld	a,d		; print the RAM starting address
	call	prhex		; 
	ld	a,e		;
	call	prhex		; 
	call	crlf		; print CRLF

	ld	hl,#ramsize	; get RAM size string
	call	put		; print it
	ld	a,ram_amt+1(iy)	; print the RAM starting address
	call	prhex		; 
	ld	a,ram_amt(iy)	;
	call	prhex		; 
	call	crlf		; 

	ld	hl,#stackat	; get stack starts at string
	call	put		; print it
	ld	hl,#15		; calculate the top of stack
	add	hl,sp		;
	ld	a,h		; stack pointer
	call	prhex		; 
	ld	a,l		; 
	call	prhex		; 
	call	crlf		; 
	
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Configure the hard drive.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	call	_ide_address	; get the controller address
	.if	ide16
	ld	a,c		; see if it looks valid
	cp	a,#0		; same as console?
	jr	z,7$		; yes, skip this
	cp	a,#0xff		; no, all ones?
	jr	z,7$		; yes, skip this
	call	_ide_hard_reset	; no, configure drive
	.endif
	ld	hl,#ide1	; print ATA message
	call	put		;
	ld	a,c		; get controller address
	call	prhex		; print it
	call	crlf		; terminate line

	.page
	.sbttl	Finish up
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Go!
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
7$:	jp	warm_start


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Message strings.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
banner:
	.db	13,10
	.ascii	"Monitor v1.5"
	.db	13,10,0

ramfound:
	.asciz	"RAM starts at 0x"
ramsize:
	.asciz	"RAM size is 0x"
stackat:
	.asciz	"Stack starts at 0x"
ide1:
	.if	ide16
	.asciz	"16-bit ATA at 0x"
	.else
	.asciz	"8-bit ATA at 0x"
	.endif

	.end
