	.title	Command Processor
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)main.s	1.11	14/02/18
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.module main

	.area CODE (CON)
	
	.include "globals.s"

warm_start::
	ld	h,ram_addr+1(iy); get keyboard buffer address
	ld	l,ram_addr(iy)	;
	ld	de,#0x80	; give the keyboard buffer some room
	add	hl,de		; initialize the monitor PC to here 
	ld	mon_pc+1(iy),h	;
	ld	mon_pc(iy),l	;

	call	 crlf		; skip a line

	.page
	.sbttl Command parser
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Command input and parsing
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
cmd:	ld	a,#'>		; send a command prompt			4
	call	putchar		;					12+
	ld	h,ram_addr+1(iy); get keyboard buffer address		19
	ld	l,ram_addr(iy)	;					19
	call	gets		; get a string from the user		191+/c

	;
	; Check for Intel hex records
	;
	cp	 a,#':		; Intel hex record?			7
	jp	 z,intel_hex	; yes, process the record		10

	cp	 a,#0		; blank line?
	jr	 z,cmd		; yes, start over

	;
	; parse the command line
	;

	call	upper		; convert buffer to uppercase
	call	nextparm	; get the first non-space character in A
	inc	hl		; move the pointer to the next character

	; memory dump
	cp	a,#'D		; memory dump?
	jp	z,memdump	; yes, dump some memory
	
	; enter data
	cp	a,#'E		; enter data?
	jp	z,enterd	; yes, do data entry

	; go
	cp	a,#'G		; go?
	jp	z,gorun		; yes, execute program

	; read an I/O port
	cp	a,#'I		; Read I/O?
	jp	z,readio	; yes, read from a port

	; write an I/O port
	cp	a,#'O		; Write I/O?
	jp	z,writeio	; yes, write to a port

	; boot from a disk
	cp	a,#'B		; Boot?
	jp	z,boot		; yes, boot

	; load from disk
	cp	a,#'L		; Load?
	jp	z,load		; yes, load

	.if	print_ptable
	; print partition table
	cp	a,#'P		; Print?
	jp	z,ptable	; yes, print table
	.endif

	; save to disk
	cp	a,#'S		; Save?
	jp	z,save		; yes, save

	.if	id_drive
	; identify drive
	cp	a,#'V		; Identify drive?
	jp	z,idfy		; yes, ID drive
	.endif

	; help
	cp	a,#'?		; help
	jr	nz,cmd		; no, try again
	ld	hl,#help	; get help info
	call	puts		; print it
	jr	cmd		; done


	.page
	.sbttl Monitor routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Memory dump
;
;	Usage: D [addr] [len]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
memdump:
	call	nextparm	; find the next command line parameter
	jr	nz,1$		; if not EOL, get the specified PC
	ld	d,mon_pc+1(iy)	; otherwise, default to the saved monitor PC
	ld	e,mon_pc(iy)	;
	ld	bc,#16		; dump 16 bytes by default
	jr	3$		; start dumping
1$:	call	scanhexw	; get the starting address in DE
	call	nextparm	; find the next command line parameter
	jr	nz,2$		; if not EOL, get length
	ld	bc,#16		; dump 16 bytes by default
	jr	3$		; start dumping
2$:	push	de		; save starting address
	call	scanhexw	; get a 16-bit word
	ld	b,d		; copy number of bytes
	ld	c,e		;
	pop	de		; restore starting address
3$:	ex	de,hl		; move starting address to HL

5$:	ld	d,#16		; 16 bytes per line
	ld	a,h		; get high byte of PC
	call	prhex		; print it
	ld	a,l		; get low byte of PC
	call	prhex		; print it
	ld	a,#':		; get a colon
	call	putchar		; print it
4$:	ld	a,#SPACE	; get a space
	call	putchar		; print it
	ld	a,#8		; at 8th byte?
	cp	a,d		;
	jp	nz,7$		; no, continue
	ld	a,#SPACE	; yes, get another space
	call	putchar		; print it
7$:	ld	a,(hl)		; get a byte from memory
	call	prhex		; print it
	inc	hl		; next memory location
	dec	bc		; decrement number of bytes to dump
	ld	a,b		; set the flags
	or	a,c		;
	jr	z,6$		; if done, then finish up
	dec	d		; else, decrement bytes left in line
	jr	nz,4$		; if not done, then print another byte
	call	crlf		; else, new line
	jr	5$		; print another line

6$:	call	crlf		; terminate the line
	ld	mon_pc+1(iy),h 	; save the new monitor PC
	ld	mon_pc(iy),l	;
	jp	cmd		; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Enter data
;
;	Usage: E [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
enterd:	
	call	 nextparm	; find the next command line parameter
	jr	 nz,1$		; if not EOL, get the specified PC
	ld	 d,mon_pc+1(iy)	; yes, default to the saved monitor PC
	ld	 e,mon_pc(iy)	;
	jr	 2$		; go for it
1$:	call	 scanhexw	; get the starting address in DE
2$:	ld	 a,d		; get address high byte
	call	 prhex		; print it
	ld	 a,e		; get address low byte
	call	 prhex		; print it
	ld	 a,#':		; get an colon
	call	 putchar	; print it
	ld	 a,#SPACE	; get a space
	call	 putchar	; print it
	ld	 a,(de)		; get the original byte
	call	 prhex		; print it
	ld	 a,#'-		; command prompt
	call	 putchar	; print it
	ld	 h,ram_addr+1(iy); get keyboard buffer address
	ld	 l,ram_addr(iy)	;
	call	 gets		; get a string from the user
	call	 nextparm	; find the next parameter
	jr	 z,3$		; if blank line, save PC and leave
	call	 upper		; otherwise, convert buffer to uppercase
	call	 scanhexb	; get an 8-bit byte in C
	ld	 a,c		; get the byte
	ld	 (de),a		; store it
	inc	 de		; skip to the next location
	jr	 2$		; back to the command prompt
3$:	ld	 mon_pc+1(iy),d	; save the new monitor PC
	ld	 mon_pc(iy),e	;
	jp	 cmd		; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Execute a subroutine
;
;	Usage: G [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
gorun:	
	call	nextparm	; find the next command line parameter
	jr	nz,1$		; if not EOL, get the specified PC
	ld	h,mon_pc+1(iy) 	; otherwise, default to the saved monitor PC
	ld	l,mon_pc(iy)	;
	jr	2$		; go for it
1$:	call	scanhexw	; no, get a 16-bit word
	ld	mon_pc+1(iy),d 	; save as the new monitor PC
	ld	mon_pc(iy),e	;
	ld	h,d		; move PC to HL
	ld	l,e		;
2$:	push	iy		; save the location of the local variables
	ld	de,#3$		; get the return address
	push	de		; push the return address
	jp	(hl)		; load new program counter
3$:				; user's subroutine should return back here
	pop	iy		; restore the location of the local variables
	jp	cmd		; done!


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read from an I/O port
;
;	Usage: I <port>
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
readio:
	call	 nextparm	; find the next command line parameter
	jp	 z,cmd		; if EOL, do nothing and leave
	call	 scanhexb	; otherwise, get the 8-bit port into C
	in	 a,(c)		; read port
	ld	 b,a		; save value in B
	ld	 a,c		; get the port number
	call	 prhex		; print it
	ld	 a,#':		; get a colon
	call	 putchar	; print it
	ld	 a,b		; get the value from B
	call	 prhex		; print it
	call	 crlf		; terminate the line
	jp	 cmd		; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write to an I/O port
;
;	Usage: O <port> [value]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
writeio:
	call	nextparm	; find the next command line parameter
	jp	z,cmd		; if EOL, do nothing and leave
	call	scanhexb	; otherwise, get the 8-bit port in C
	call	nextparm	; find the next command line parameter
	jr	z,1$		; if EOL, use 0 as the value
	ld	b,c		; otherwise, save the port
	call	scanhexb	; get the 8-bit value in C
	ld	a,c		; move the value to A
	ld	c,b		; move the port back to C
1$:	out	(c),a		; write port
	jp	cmd		; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Boot from an active partition on a  disk
;
;	Usage: B [num] [drive] [param]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
boot:
	ld	part_num(iy),#0	; clear partition number
	ld	head_no(iy),#0	; clear the drive number
	ld	boot_parm(iy),#0; clear boot parameter
	call	nextparm	; find the next command line parameter
	jr	z,1$		; if EOL, then use default parameters
	call	scanhexb	; otherwise, get the partition type in C
	ld	part_num(iy),c	; save partition number
	call	nextparm	; find the next command line parameter
	jr	z,1$		; if EOL, use default value
	call	scanhexb	; get the drive number in C
	xor	a,a		; clear A
	cp	a,c		; drive 0 selected?
	jr	z,2$		; yes, save it
	ld	c,#(1<<4)	; no, set the drive 1 bit
2$:	ld	head_no(iy),c	; save drive number
	call	nextparm	; find the next command line parameter
	jr	z,1$		; if EOL, use default value
	ld	a,(hl)		; get parameter character
	and	a,#0xdf		; convert to uppercase
	ld	boot_parm(iy),a	; save it
1$:	call	_ide_boot	; boot
	jr	load1		; if we made it back, print the reason


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load a sector from disk
;
;	Usage: L [sect] [head] [cyl] [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
load:
	call	lsparms		; parse the command line
	call	_ide_read_sector; load the sector into memory
load1:	push	af		; save the error code
	ld	hl,#msg1	; get the result message
	call	put		; print it
	pop	af		; restore error code
	call	prhex		; print it
	call	crlf		; end of line
	jp	cmd		; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Save a sector to disk
;
;	Usage: S [sect] [head] [cyl] [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
save:
	call	lsparms		; parse the command line
	call	_ide_write_sector; save the sector to disk
	jr	load1		; print result and leave.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Parse load/save command line parameters
;	and get the controller address.
;
;	Parameters: [sect] [head] [cyl] [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
lsparms:
	call	nextparm	; skip to next parameter
	jr	z,1$		; if EOL, get the last values
	call	scanhexb	; get the sector in C
	ld	sector_no(iy),c	; save it
	call	nextparm	; skip to next parameter
	jr	z,1$		; if EOL, get the last values
	call	scanhexb	; get the head in C
	ld	head_no(iy),c	; save it
	call	nextparm	; skip to next parameter
	jr	z,1$		; if EOL, get the last values
	call	scanhexw	; get the cylinder in DE
	ld	cylinder_msb(iy),d; save it
	ld	cylinder_lsb(iy),e;
	call	nextparm	; skip to next parameter
	jr	z,1$		; if EOL, get the last values
	call	scanhexw	; get the load address
	ld	load_addr+1(iy),d; save it
	ld	load_addr(iy),e	;

1$:	ld	a,head_no(iy)	; get the head
	ld	d,cylinder_msb(iy); get the cylinder
	ld	e,cylinder_lsb(iy);
	ld	b,sector_no(iy)	; get the sector
	ld	c,#1		; only process one sector
	ld	h,load_addr+1(iy); get the load address
	ld	l,load_addr(iy)	;
	ret			; done.


	.if	print_ptable
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Print partition table
;
;	Usage: P [drive]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
ptable:
	call	nextparm	; skip to next parameter
	jr	z,2$		; if EOL, get the last values
	call	scanhexb	; get the drive in C
	xor	a,a		; A = 0
	cp	a,c		; drive 0 selected?
	jr	z,1$		; yes, save it
	ld	c,#(1 << 4)	; drive 1
1$:	ld	head_no(iy),c	; save it
2$:	ld	a,head_no(iy)	; get drive

	call	_ide_load_ptable; load the partition table
	jr	nz,load1	; if error, print and leave

	ld	de,#62		; offset of partition table
	add	hl,de		; add to our load address
	push	hl		; save for IX
	pop	ix		; move partition table to IX
	ld	de,#partlen	; get length of partition entry
	ld	b,#4		; 4 entries in the MBR

3$:	push	bc		; save counter value
	push	hl		; save parition table pointer
	pop	ix		; load pointer into IX

	ld	a,#0x20		; get a space
	call	putchar		; print it
	ld	a,#5		; determine partition number
	sub	a,b		;
	add	a,'0		; convert to ASCII
	call	putchar		; print the partition number
	ld	a,#0x20		; get a space
	call	putchar		; print it
	bit	7,drive(ix)	; partition bootable?
	jr	z,4$		; no, continue
	ld	a,#'*		; yes, get an asterisk
4$:	call	putchar		; print the active flag
	ld	a,#0x20		; get a space
	call	putchar		; print it

	ld	bc,#start_lba+3	; offset to starting LBA
	call	pdword		; print it

	ld	bc,#len_lba+3	; offset to length
	call	pdword		; print it

	ld	a,type(ix)	; get the partition type
	call	prhex		; print it
	call	crlf

	add	hl,de		; go to next partition
	pop	bc		; restore counter
	djnz	3$		; continue

	jp	cmd		; done

pdword:
	push	hl		; save the partition table pointer
	add	hl,bc		; add the offset
	ld	b,#4		; print 4 bytes
1$:	ld	a,(hl)		; get a byte
	call	prhex		; print it
	dec	hl		; next byte
	djnz	1$		; if not done, continue
	pop	hl		; otherwise restore table pointer
	ld	a,#0x20		; get a space
	jp	putchar		; print it and return

	.endif


	.if	id_drive
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Identify the hard disk
;
;	Usage: V [addr]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
idfy:	
	call	nextparm	; find the next command line parameter
	jr	nz,1$		; if not EOL, get the specified PC
	jp	cmd		; otherwise abort
1$:	call	scanhexw	; get address in DE
	push	de		; copy to HL
	pop	hl		;
	call	_ide_drive_id	; identify the hard drive
	jr	nz,load1	; if error, print it and leave
	push	de		; copy address to ix

	;	Print model number
	pop	hl		; get the start of the buffer
	push	hl		; save a copy
	ld	de,#27*2	; location of model
	add	hl,de		; add offset
	ld	b,#20		; 20 words in the field
	call	idfys		; print the field.

	;	Print version number
	pop	hl		; get the start of the buffer
	push	hl		; save a copy
	ld	de,#23*2	; location of version
	add	hl,de		; add offset
	ld	b,#4		; 4 words in the field
	call	idfys		; print the field

	;	Print serial number
	pop	hl		; get the start of the buffer
	push	hl		; save a copy
	ld	de,#10*2	; location seral number
	add	hl,de		; add offset
	ld	b,#10		; 10 words in the field
	call	idfys		; print the field

	;	Print number of sectors contained on disk
	pop	hl		; get the start of the buffer
	ld	de,#60*2+3	; number of sectors
	add	hl,de		; add offset to field
	ld	b,#4		; 4 bytes in the field
2$:	ld	a,(hl)		; get the first byte
	call	prhex		; print it
	dec	hl		; next byte
	djnz	2$		; do it again
	call	crlf		; end of line

	jp	cmd		; leave

	;	Print text string stored as big endian words.
idfys:	inc	hl		; do character in MSB spot first
	ld	a,(hl)		; get character
	call	putchar		; print it
	dec	hl		; point to LSB spot
	ld	a,(hl)		; get character
	call	putchar		; print it
	inc	hl		; skip to the next word
	inc	hl		;
	djnz	idfys		; continue if not done
	call	crlf		; terminate line
	ret			; done
	.endif


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Process an Intel hex record.
;	Does not check for errors.
;
;	We have 5200 T states to complete this
;	@ 9600 bps
;
;	T States: ~609 + ~202 per byte
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
intel_hex:
	inc	hl		; skip to the next byte			7

	; get the record length
	call	scanhexb	; get the data length into C		~175
	or	a,a		; length > 0?				4
	jp	z,cmd		; no, leave				5,11
	ld	b,a		; yes, copy length to B			4

	; get the address
	call	scanhexb	; get the MSB of the load addr into C	~175
	ld	d,a		; place in D				4
	call	scanhexb	; get the LSB of the load addr into C	~175
	ld	e,a		; place in E				4

	; check the record type
	ld	a,(hl)		; get first nibble			7
	cp	#'0		; should be a 0				7
	jp	nz,cmd		; nope, leave				5,11
	inc	hl		; next character			6
	ld	a,(hl)		; get next nibble			7
	cp	#'0		; should be a 0				7
	jp	nz,cmd		; nope, leave				5,11
	inc	hl		; next character			6

	; load the data
1$:	call	scanhexb	; get a byte of data in A		~175
	ld	(de),a		; save the byte				7
	inc	de		; next location				7
	djnz	1$		; if not done, then continue		13,8
	jp	cmd		; done					10


	.page
	.sbttl Initialized data
help:
	.ascii	"B = Boot	-- B [num] [drive] [param]"
	.db	13,10
	.ascii	"D = Dump	-- D [addr] [len]"
	.db	13,10
	.ascii	"E = Enter	-- E [addr]"
	.db	13,10
	.ascii	"G = Go		-- G [addr]"
	.db	13,10
	.ascii	"I = Input	-- I <port>"
	.db	13,10
	.ascii	"L = Load	-- L [sector] [head] [cylinder] [addr]"
	.db	13,10
	.ascii	"O = Output	-- O <port> [val]"
	.db	13,10
	.if	print_ptable
	.ascii	"P = Partitions	-- P [drive]"
	.db	13,10
	.endif
	.ascii	"S = Save	-- S [sector] [head] [cylinder] [addr]"
	.db	13,10
	.if	id_drive
	.ascii	"V = ID Drive	-- V [addr]"
	.db	13,10
	.endif
	.ascii	"? = This screen"
	.db	13,10
	.db	0

msg1:	.asciz	"Result: "

	.end
