/*
 *	@(#)serial.c	1.1 14/12/13
 *
 *	open_serial(), close_serial(), read_serial(), write_serial()
 *
 *	Serial port wrapper routines.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <termios.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "globals.h"

int open_serial(char *devname, speed_t speed)
{
	int fd;
	struct termios termio;

	log_msg(10, "open_serial(\"%s\")", devname);

	if (speed == B0)
		speed = B9600;

	memset(&termio, 0, sizeof(termio));

	log_msg(5, "open_serial: open(\"%s\", O_RDWR)", devname);

	if ((fd = open(devname, O_RDWR)) == ERROR) {
		log_msg(0, "open_serial: %s -- %s", devname, strerror(errno));
		return (ERROR);
	}

	log_msg(10, "open_serial: \"%s\": fd = %d", devname, fd);

	cfmakeraw(&termio);
	termio.c_cflag |= CREAD | CLOCAL;
	termio.c_cc[VMIN] = 0;
	termio.c_cc[VTIME] = 1;
	cfsetispeed(&termio, speed);

	log_msg(10, "open_serial: tcsetattr(%d, TCASNOW, &termio)", fd);

	if (tcsetattr(fd, TCSANOW, &termio) == ERROR) {
		log_msg(0, "open_serial: tcsetatrr -- %s", strerror(errno));
		if (close(fd) == ERROR)
			log_msg(0, "open_serial: close -- %s", strerror(errno));
		return (ERROR);
	}

	return (fd);
}


int close_serial(int fd)
{
	log_msg(5, "close_serial: close(%d)", fd);

	if (close(fd) == ERROR) {
		log_msg(5, "close(%d): %s", fd, strerror(errno));
		return (ERROR);
	}

	return (0);
}


int read_serial(int fd, unsigned char *buf, int count)
{
	int n, c, done;
	
	log_msg(10, "read_serial(%d, 0x%8.8lx, %u)", fd, buf, count);

	c = 0;
	done = 0;

	while (c < count || done) {
		log_msg(10, "read_serial: read(%d, &buf[%d], %d)", fd, c, count-c);

		n = read(fd, &buf[c], count - c);

		if (n == ERROR)
			switch (errno) {
				case EAGAIN:
					log_msg(1, "read_serial: EAGAIN");
					done = 1;
					break;
				case EINTR:
					log_msg(1, "read_serial: EINTR");
					continue;
				default:
					log_msg(0, "read_serial: read(%d, &buf[%d], %d) -- %s",
						fd, c, count-c, strerror(errno));
					return (ERROR);
			}
		else {
			if (n)
				c += n;
			else
				break;
		}
	}	

	log_msg(5, "read_serial: %d bytes read", c);
	hex_dump(10, buf, c);

	return (c);
}


int write_serial(int fd, unsigned char *buf, int count)
{
	int n, c;

	log_msg(10, "write_serial(%d, 0x%8.8lx, %u)", fd, buf, count);

	c = 0;

	while (c < count) {
		log_msg(10, "write_serial: write(%d, &buf[%d], %d)", fd, c, count-c);

		n = write(fd, &buf[c], count - c);

		if (n == ERROR)
			switch (errno) {
				case EAGAIN:
					log_msg(1, "write_serial: EAGAIN");
					break;
				case EINTR:
					log_msg(1, "write_serial: EINTR");
					continue;
				default:
					log_msg(0, "write_serial: write(%d, &buf[%d], %d) -- %s",
						fd, c, count-c, strerror(errno));
					return (ERROR);
			}
		else
			c += n;
	}

	log_msg(5, "write_serial: %d bytes written", c);
	hex_dump(10, buf, c);

	return (c);
}
