/*
 *	@(#)main.c	1.2	14/12/13
 *
 *	Another TU58 tape emulator.
 *
 *	Written by Troy Frain.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>
#include <string.h>
#include <libgen.h>
#include <termio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "globals.h"

char *program;
int debug, foreground;
FILE *logfile;

extern void process(char *line, speed_t speed, FILE *tapeunit[2], char wp[2]);
extern void interrupt(int sig);

static void usage(void)
{
	fprintf(stderr, "usage: %s -l line [-f logfile] [-s speed] [-x debug] -0 tapefile -1 tapefile\n", program);

	exit(1);
}

int main(int argc, char *argv[])
{
	int c;
	speed_t speed;
	char *logname;
	char *serial_line;
	char *tapefile[2];
	FILE *tapeunit[2];
	struct stat st;
	char mode[3];
	char wp[2];

	program = basename(argv[0]);
	foreground = TRUE;
	debug = 0;
	logfile = stderr;

	speed = B9600;
	logname = (char *)0;
	serial_line = (char *)0;
	tapefile[0] = (char *)0;
	tapefile[1] = (char *)0;
	tapeunit[0] = (FILE *)0;
	tapeunit[1] = (FILE *)0;
	wp[0] = FALSE;
	wp[1] = FALSE;
	
	while ((c = getopt(argc, argv, "0:1:f:l:s:x:?")) != EOF)
		switch (c) {
			case '0':
				tapefile[0] = optarg;
				break;
			case '1':
				tapefile[1] = optarg;
				break;
			case 'f':
				logname = optarg;
				break;
			case 'l':
				serial_line = optarg;
				break;
			case 's':
				switch (atoi(optarg)) {
					case 50:
						speed = B50;
						break;
					case 75:
						speed = B75;
						break;
					case 110:
						speed = B110;
						break;
					case 134:
						speed = B134;
						break;
					case 150:
						speed = B150;
						break;
					case 200:
						speed = B200;
						break;
					case 300:
						speed = B300;
						break;
					case 600:
						speed = B600;
						break;
					case 1200:
						speed = B1200;
						break;
					case 1800:
						speed = B1800;
						break;
					case 2400:
						speed = B2400;
						break;
					case 4800:
						speed = B4800;
						break;
					case 9600:
						speed = B9600;
						break;
					case 19200:
						speed = B19200;
						break;
					case 38400:
						speed = B38400;
						break;
					case 57600:
						speed = B57600;
						break;
					case 115200:
						speed = B115200;
						break;
					case 230400:
						speed = B230400;
						break;
					default:
						speed = -1;
						break;
				}
				break;
			case 'x':
				debug = atoi(optarg);
				break;
			case '?':
				usage();
				break;
		}
	
	if (serial_line == (char *)0)
		usage();
	
	if (tapefile[0] == (char *)0 && tapefile[1] == (char *)0)
		usage();

	if (speed < 0) {
		fprintf(stderr, "%s: invalid line speed\n", program);
		exit(1);
	}

	if (logname && (logfile = fopen(logname, "w")) == (FILE *)0) {
		fprintf(stderr, "%s: %s -- %s\n", program, logname, strerror(errno));
		exit(2);
	}

	log_msg(1, "debug=%d", debug);

	for (c = 0; c < 2; c++) {
		if (tapefile[c]) {
			if (stat(tapefile[c], &st) == ERROR) {
				log_msg(0, "%s: %s -- %s", program, tapefile[c],
					strerror(errno));
			} else {
				if (!S_ISREG(st.st_mode))
					log_msg(0, "%s: %s is not a regular file", program,
						tapefile[c]);
				else {
					if (st.st_mode & S_IRUSR) {
						mode[0] = 'r';
						if (st.st_mode & S_IWUSR)
							mode[1] = '+';
						else {
							mode[1] = '\0';
							wp[c] = TRUE;

						}
						mode[2] = '\0';
						log_msg(5, "%s: fopen(\"%s\", \"%s\")", program,
							tapefile[c], mode);
						if ((tapeunit[c] = fopen(tapefile[c], mode))
							== (FILE *)0)
							log_msg(0, "%s: %s -- %s", program, tapefile[c],
								strerror(errno));
						else {
							log_msg(5, "%s: mounted \"%s\" on unit %d as %s.",
								program, tapefile[c], c,
								(wp[c] ? "write-protected " : "read/write"));
							log_msg(1, "UNIT %d: \"%s\" %s", c, tapefile[c],
								(wp[c] ? "WRITE PROTECTED" : "R/W"));
						}
					} else {
						log_msg(0, "%s: %s is not readable", program,
						tapefile[c]);
					}
				}
			}
		}
	}

	if (tapeunit[0] || tapeunit[1]) {
		interrupt(0);

		process(serial_line, speed, tapeunit, wp);

		log_msg(1, "%s: closing files.", program);

		if (tapeunit[0])
			fclose(tapeunit[0]);
	
		if (tapeunit[1])
			fclose(tapeunit[1]);
	}

	if (logname)
		fclose(logfile);
	
	log_msg(1, "Terminating.");

	return 0;
}
