	.title	IDE Controller Support
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)ide8.s	1.3	14/02/18
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.module	ide

	.area CODE (CON)

	.include "globals.s"

;
; Drive command combinations
;
_IDE_DATA	= _ide_port
_IDE_ERROR	= _ide_port+1					; read only
_IDE_FEATURES	= _ide_port+1					; write only
_IDE_SECTOR_CNT	= _ide_port+2
_IDE_SECTOR_NUM	= _ide_port+3
_IDE_CYL_LOW	= _ide_port+4
_IDE_CYL_HIGH	= _ide_port+5
_IDE_DRV_HEAD	= _ide_port+6
_IDE_STATUS	= _ide_port+7					; read only
_IDE_COMMAND	= _ide_port+7					; write only
_IDE_ALT_STATUS	= _ide_port+14					; read only
_IDE_DEV_CTRL	= _ide_port+14					; write only

;
; Logical interface commands
;
_IDE_NOP		= 0x00	; NOP
_IDE_REQ_SENSE		= 0x03	; Request Sense
_IDE_RECALIBRATE	= 0x10	; Recalibrate
_IDE_READ_SECTOR	= 0x20	; Read sector(s) (w/retry)
_IDE_READ_SECTOR_N	= 0x21	; Read sector(s) (w/o retry)
_IDE_READ_LONG		= 0x22	; Read long (w/retry)
_IDE_READ_LONG_N	= 0x23	; Read long (w/o retry)
_IDE_WRITE_SECTOR	= 0x30	; Write sector(s) (w/retry)
_IDE_WRITE_SECTOR_N	= 0x31	; Write sector(s) (w/o retry)
_IDE_WRITE_LONG		= 0x32	; Write long (w/retry)
_IDE_WRITE_LONG_N	= 0x33	; Write long (w/o retry)
_IDE_READ_VFY_SECTOR	= 0x40	; Read verify sector(s) (w/retry)
_IDE_READ_VFY_SECTOR_N	= 0x41	; Read verify sector(s) (w/o retry)
_IDE_FORMAT_TRACK	= 0x50	; Format track
_IDE_SEEK		= 0x70	; Seek
_IDE_DRIVE_DIAG		= 0x90	; Execute drive diagnostic
_IDE_INIT_PARAM		= 0x91	; Initialize drive parameters
_IDE_ERASE_SECTOR	= 0xc0	; Erase one or more sectors
_IDE_READ_BUFFER	= 0xe4	; Read the contents of the sector buffer
_IDE_FLUSH_CACHE	= 0xe7	; Finish writing data from the cache
_IDE_IDENTIFY		= 0xec	; Identify device
_IDE_SET_FEATURES	= 0xef	; Set features

;
; Features
;
_ENABLE_8_BIT		= 0x01	; Enable 8-bit transfers
_ENABLE_CACHE		= 0x02	; Enable write cache
_DISABLE_8_BIT		= 0x81	; Disable 8-bit transfers
_DISABLE_CACHE		= 0x82	; Disable write cache

;
; Status register bits
;
_BSY	= 7	; Busy.  Drive has access to the Command registers
_DRDY	= 6	; Drive Ready
_DWF	= 5	; Drive Write Fault
_DSC	= 4	; Drive Seek Complete
_DRQ	= 3	; Data Request
_CORR	= 2	; Corrected Data
_IDX	= 1	; Index
_ERR	= 0	; Error (see Error Register)

;
; Error register bits
;
_BBK	= 7	; Bad Block Detected
_UNC	= 6	; Uncorrectable Data Error
_MC	= 5	; Media Changed
_IDNF	= 4	; Requested sector's ID field not found
_MCR	= 3	; Media Change Requested
_ABRT	= 2	; Aborted Command (drive busy or not ready)
_TK0NF	= 1	; Track 0 Not Found
_AMNF	= 0	; Address Mark Not Found

;
; Drive & Head register bits
;
_LBA	= 6	; 0=CHS, 1=LBA
_DRIVE	= 4	; 0=drive 0, 1=drive 1
		; lower 4 bits = head or LBA 24-27

	.page
	.sbttl Boot from IDE controller
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load the first sector from a disk and 
;	execute the code contained in it
;
;	Parameters:
;		head_no(iy) = Drive number
;		part_num(iy) = Partition number
;		boot_parm(iy) = Boot parameter
;
;	Returns:
;		Doesn't return, except on error.
;		Executes code from the boot
;		sector, otherwise.
;		Uses lots of expensive instructions.
;		Clobbers everything.
;
;	T States: 23305
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_boot::
	ld a,head_no(iy)	; get the drive/head info		19
	call _ide_load_ptable	; load the drive's partition table	11547
	ret nz			; return on error			5,11

	ld de,#62		; offset of partition table		10
	add hl,de		; add to our load address		11
	push hl			; save for IX				11
	ld de,#partlen		; get length of partition entry		10
	ld b,#4			; 4 entries in the MBR			7
	pop ix			; move partition table to IX		10

1$:	bit 7,drive(ix)		; partition bootable?			20
	jr z,2$			; no, continue				7,12
	xor a,a			; yes, clear A				4
	cp a,part_num(iy)	; boot from first bootable partition?	19
	jr z,3$			; yes, start the boot sequence		7,12
2$:	ld a,#5			; calculate the partition number	7
	sub a,b			;					4
	cp a,part_num(iy)	; is this the correct partition number? 19
	jr z,3$			; yes, start the boot sequence		7,12
	add ix,de		; no, check next partition 		15
	djnz 1$			; continue				8,13
	ld hl,#msg1		; none found, print error message	10
	call puts		;
	xor a,a			; clear A, so it doesn't cause problems	4
	ret			; leave					10

3$:	ld h,ram_addr+1(iy)	; get keyboard buffer address		19
	ld l,ram_addr(iy)	; get keyboard buffer address		19
	ld de,#0x80		; give the keyboard buffer some room	10
	add hl,de		; use this as our load address		11
	ld b,start_lba(ix)	; get the LBA				19
	ld e,start_lba+1(ix)	;					19
	ld d,start_lba+2(ix)	;					19
	ld a,start_lba+3(ix)	;					19
	or a,#(1<<_LBA)		; set the LBA bit			7
	ld c,#1			; load only one sector			7

	push hl			; push the boot address			11
	call _ide_read_sector	; read the sector into memory		11423
	ld b,boot_parm(iy)	; get the boot parameter		19
	ret z			; boot if no error			5,11
	pop hl			; otherwise, toss the boot address	10
	ret			; back to the monitor			10

msg1:	.asciz "No bootable partition."


	.page
	.sbttl Get IDE controller address
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get the address of the IDE controller
;
;	Parameters:
;		None
;
;	Returns:
;		C = IDE controller base address
;
;	T States: 17
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_address::
	ld c,#_IDE_DATA		; base address of IDE controller	7
	ret			; done					10

	
	.page
	.sbttl Set the IDE controller features
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Configure the drive.
;
;	Parameters:
;		A = Drive number
;
;	Returns:
;		Error: Z = 0
;		No Error: Z = 1
;		A = Error Code
;
;	T States: 234
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_configure::
	call _ide_ready		; select drive and wait until ready	127
	ld a,#_ENABLE_8_BIT	; enable 8 bit transfer feature		7
	out (_IDE_FEATURES),a	; place in feature register		11
	ld a,#_IDE_SET_FEATURES	; set features command			7
	call _ide_command	; execute command, abort if error	72
	ret			; done					10


	.page
	.sbttl Find the active partition
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load the drive's partition table into
;	memory.
;
;	Parameters:
;		None
;
;	Returns:
;		Error: Z = 0
;		No Error: Z = 1
;		A = Error Code
;		HL = Location of partition table
;		Clobbers B,C,D,E,Flags
;
;	T States: 11530
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

_ide_load_ptable::
	or a,#(1 << _LBA)	; LBA #0 (Master Boot Record)		7
	ld d,#0			; 					7
	ld e,#0			; 					7
	ld b,#0			; 					7
	ld c,#1			; read only the one sector		7
	ld h,ram_addr+1(iy)	; get the load address			19
	ld l,ram_addr(iy)	; 					19
	call _ide_set_chs	; setup the CHS parameters		326
	ld a,#_IDE_READ_SECTOR	; read sector command			7
	call _ide_command	; execute command, abort if error	72

	call _ide_wait_drq	; wait until the data is ready		73
	ld d,#4			; do this 4 times			7
	ld c,#_IDE_DATA		; get the data register			7
1$:	ld b,#128		; load 128 bytes			7
	push hl			; save load address			11
	inir			; load data into memory			2683
	pop hl			; restore load address			10
	dec d			; done?					4
	jr nz,1$		; no, continue				7,12
	call _ide_get_error	; yes, get results, abort if error	61
	ret			; otherwise return.			10
	

	.if id_drive
	.page
	.sbttl Get the drive identification block
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load the drive identification block
;
;	Parameters:
;		A = Drive number
;		HL = buffer address
;
;	Returns:
;		Error: Z = 0
;		No Error: Z = 1
;		A = Error Code
;		HL = end of data + 1
;
;	T States: 11266
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_drive_id::
	call _ide_configure	; select drive and configure		234
	ret nz			; return on error			5,11
	ld a,#_IDE_IDENTIFY	; identify drive			7
	call _ide_command	; execute command, abort if error	72
	call _ide_read_data	; read drive data 			10877
	call _ide_get_error	; get results, abort if error		61
	ret			; done					10
	.endif


	.page
	.sbttl Set the CHS/LBA
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Set CHS/LBA
;
;	Parameters:
;		A = Drive & Head
;		B = Sector
;		DE = Cylinder
;		C = Sector Count
;		HL = Buffer address
;
;	Returns:
;		Clobbers A
;
;	T States: 309
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_set_chs::
	call _ide_configure	; set drive/head & config drive feat.	234
	ret nz			; return on error			5,11
	ld a,e			; get the cylinder LSB			4
	out (_IDE_CYL_LOW),a	; write cylinder LSB			11
	ld a,d			; get the cylinder MSB			4
	out (_IDE_CYL_HIGH),a	; write cylinder MSB			11
	ld a,b			; get the sector 			4
	out (_IDE_SECTOR_NUM),a	; write sector number			11
	ld a,c			; number of sectors to read		4
	out (_IDE_SECTOR_CNT),a	; write sector count			11
	ret			; done					10


	.page
	.sbttl Get the current CHS/LBA
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get CHS/LBA
;
;	Parameters:
;		None
;
;	Returns:
;		A = Drive & Head
;		B = Sector
;		DE = Cylinder
;
;	T States: 66
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_get_chs::
	in a,(_IDE_CYL_LOW)	; get cylinder LSB			11
	ld e,a			; save in E				4
	in a,(_IDE_CYL_HIGH)	; get cylinder MSB			11
	ld d,a			; save in D				4
	in a,(_IDE_SECTOR_NUM)	; get sector number			11
	ld b,a			; save in B	 			4
	in a,(_IDE_DRV_HEAD)	; get drive & head in A			11
	ret			; done					10


	.page
	.sbttl Read a sector from the disk
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read one or more sectors.
;
;	Selects the drive, loads the CHS/LBA and
;	the number of sectors to be transferred,
;	writes the data, then gets the results
;	of the operation
;
;	Parameters:
;		A = Drive & Head
;		B = Sector
;		DE = Cylinder
;		C = Sector Count
;		HL = Buffer address
;
;	Returns:
;		A = Error Code
;
;	T States: 11406
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_read_sector::
	call _ide_set_chs	; setup the CHS parameters		326
	ret nz			; return on error			5,11
	ld a,#_IDE_READ_SECTOR	; read sector command			7
	call _ide_command	; execute command, abort if error	72
1$:	in a,(_IDE_SECTOR_CNT)	; get the number of sectors remaining	11
	or a,a			; done?					4
	ret z			; yes, leave				5,11
	call _ide_read_data	; read the sector into memory		10877
	call _ide_get_error	; get results, abort if error		61
	jr 1$			; no, continue processing		12


	.page
	.sbttl Write a sector to the disk
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write one or more sectors.
;
;	Selects the drive, loads the CHS/LBA and
;	the number of sectors to be transferred,
;	writes the data, then gets the results
;	of the operation
;
;	Parameters:
;		A = Drive & Head
;		B = Sector
;		DE = Cylinder
;		C = Sector Count
;		HL = Buffer address
;
;	Returns:
;		No error:
;			A = 0
;			Z = 1
;		Error:
;			Routine is aborted
;			A = Error code
;			Z = 0
;
;	T States: 11406
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_write_sector::
	call _ide_set_chs	; setup the CHS parameters		326
	ret nz			; return on error			5,11
	ld a,#_IDE_WRITE_SECTOR	; write sector command			7
	call _ide_command	; execute command, abort if error	72
1$:	in a,(_IDE_SECTOR_CNT)	; get the number of sectors remaining	11
	or a,a			; done?					4
	ret z			; yes, leave				5,11
	call _ide_write_data	; no, write a sector into disk		10877
	call _ide_get_error	; get results, abort if error		61
	jr 1$			; continue processing			12


	.page
	.sbttl Read a block of data from the disk
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Read 512 bytes from the drive buffer
;
;	It is assumed the drive has been selected,
;	configured for 8-bit transfers, the
;	CHS/LBA has been loaded, the drive is
;	expecting to send at least one sector
;	and the drive's DRQ is set.
;
;	Parameters:
;		HL = buffer address
;
;	Returns:
;		HL = HL + 512
;		Z = 1
;		N = 1
;		S, H, P/V are clobbered
;
;	T States: 10860
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_read_data::
	call _ide_wait_drq	; wait until data ready, with abort	73
	push bc			; save these				11
	ld c,#_IDE_DATA		; get the data register			7
	ld b,#0			; set counter to 256			7
	inir			; read initial 256 bytes		5371
	inir			; read remaining 256 bytes		5371
	pop bc			; restore these				10
	ret			; done					10


	.page
	.sbttl Write a block of data to the disk
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Write 512 bytes to the drive buffer.
;
;	It is assumed the drive has been selected,
;	configured for 8-bit transfers, the
;	CHS/LBA has been loaded, the drive is
;	expecting to receive at least one sector
;	and the drive's DRQ is set.
;
;	Parameters:
;		HL = buffer address
;
;	Returns:
;		HL = HL + 512
;		Z = 1
;		N = 1
;		S, H, P/V are clobbered
;
;	T States: 10860
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_write_data::
	call _ide_wait_drq	; wait until data ready, with abort	73
	push bc			; save BC				11
	ld c,#_IDE_DATA		; get the data register			7
	ld b,#0			; set counter to 256			7
	otir			; write the initial 256 bytes		5371
	otir			; write the remaining 256 bytes		5371
	pop bc			; restore BC				10
	ret			; done					10


	.page
	.sbttl Drive ready wait routine
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Select the drive and Wait for the RDY
;	flag to be set.
;
;	Wait until the current drive becomes idle,
;	then select the new drive, then wait
;	until the drive becomes ready.  Any
;	pending interrupts are acknowledged in the
;	process.
;
;	Parameters:
;		A = Drive number
;
;	Returns:
;		When the drive is ready
;		A = Drive's status register
;		Z = 0
;
;	T States: 110+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_ready:
	or a,#0B10100000	; bits 5 & 7 always set			7
	push af			; save drive number			11
1$:	in a,(_IDE_STATUS)	; get status register value in A	11
	bit _BSY,a		; busy?					8
	jr nz,1$		; yes, check again			7,12
	pop af			; restore drive number			10
	out (_IDE_DRV_HEAD),a	; set the drive number			11
2$:	in a,(_IDE_STATUS)	; get status register value in A	11
	bit _BSY,a		; busy?					8
	jr nz,2$		; yes, check again			7,12
	bit _DRDY,a		; ready for command?			8
	ret nz			; yes, leave				5,11
	jr 1$			; no, check again			12


	.page
	.sbttl DRQ wait routine
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Wait for the DRQ flag to be set.
;
;	If the DRQ is set, the interrupt is
;	acknowledged and the status register
;	is returned.
;
;	If the DRQ is not set, the return address
;	is thrown away to abort the routine
;	calling this subroutine.  This
;	trashes the accumulator and flags in
;	the process.
;
;	Parameters:
;		None
;
;	Returns:
;		When DRQ has been updated.
;		If the drive has data to transfer:
;			A = Drive status register
;			Z = 0
;		If the drive has no data to transfer:
;			Throws away immediate return address
;			Clobbers A in the process
;			Clobbers the flags
;
;	T States: 56+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_wait_drq:
1$:	in a,(_IDE_ALT_STATUS)	; get status register value in A	11
	bit _BSY,a		; busy?					8
	jr nz,1$		; yes, check again			7,12
	in a,(_IDE_STATUS)	; get status and clear the interrupt	11
	bit _DRQ,a		; ready to transfer?			8
	ret nz			; yes, leave				5,11
	pop af			; no data to transfer, toss return addr	11
	ret			; abort	command				10


	.page
	.sbttl Issue an ATA command with error handling
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Issue a command, wait for the drive to
;	finish execution, then get the results.
;
;	The routine is careful not to acknowledge
;	any interrupts, unless an error occurs.
;	Upon an error, interrupts are acknowledged
;	and the return address is thrown away to
;	abort the routine calling this subroutine.
;	The extended error value is also looked up
;	to return more detailed information.
;
;	Parameters:
;		A = Command
;
;	Returns:
;		When the drive becomes idle.
;		No error:
;			A = 0, Z = 1
;		Error:
;			A = Extended Error Code, Z = 0
;			Throws away immediate return address
;
;	T States: 55+
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
_ide_command:
	out (_IDE_COMMAND),a	; issue the command			11

_ide_get_error:
1$:	in a,(_IDE_ALT_STATUS)	; get status without INTRQ ack		11
	bit _BSY,a		; busy?					8
	jr nz,1$		; yes, check again			7,12
	and a,#(1<<_ERR|1<<_DWF); error?				7
	ret z			; no, leave with A = 0, & Z = 1		5,11
	pop af			; yes, throw away the return address	10
	ld a,#_IDE_REQ_SENSE	; yes, request extended info		7
	out (_IDE_COMMAND),a	; execute command			11
2$:	in a,(_IDE_STATUS)	; get status register value in A	11
	bit _BSY,a		; busy?					8
	jr nz,2$		; yes, check again			7,12
	in a,(_IDE_ERROR)	; get error code			11
	or a,a			; set the flags				4
	ret			; abort the current command		10

	.end
