/*	LJII.C

		TSR Program to provide access to the HP LaserJet Series II setup options.
		Michael Quinlan
		7/4/87
*/

#include <dos.h>
#include <process.h>
#include <ctype.h>
#include <mem.h>

/* #define DEBUG */

#define KeybdVect		0x09
#define PrtrVect		0x17

#define KB_Data			0x60

#define BIOS_Shift_Bits	((char far *) 0x00000417L)
#define BIOS_Shift_Alt	0x08
#define BIOS_Shift_Ctrl	0x04
#define Shift_Combo			(BIOS_Shift_Alt | BIOS_Shift_Ctrl)
#define Hot_Combo				((*BIOS_Shift_Bits & Shift_Combo) == Shift_Combo)
#define Hot_Key_Scan		0x26						/* Hot key is Alt-Ctrl-L */
#define Our_Printer			0								/* Printer = LPT1 */

void interrupt (*old_keyboard)(void);
void interrupt (*old_printer)(void);

unsigned int LJII_Active		= 0;				/* Non-zero when we are "popped up" */
unsigned int Setup_Pending	= 0;				/* Non-zero when setup is pending */

char Setup_Flags[26] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
unsigned char Setup_Data[8] = {0,0,0,0,0,0,0,0};

#define ESC					"\x1B"
#define ARG0				"\x80"
#define ARG1				"\x81"
#define ARG2				"\x82"
#define ARG3				"\x83"
#define ARG4				"\x84"
#define ARG5				"\x85"
#define ARG6				"\x86"
#define ARG7				"\x87"

char Setup_Reset[]						= ESC "E";
char Setup_ClearMargins[]			=	ESC "9";
char Setup_EjectPage[]				= ESC "&l0H";
char Setup_CourierFont[]			= ESC "(s3T";
char Setup_CourierBoldFont[]	= ESC "(s3t3B";
char Setup_LPrinterFont[]			= ESC "(s0T";
char Setup_Portrait[]					= ESC "&l0O";
char Setup_Landscape[]				= ESC "&l1O";
char Setup_WrapLines[]				= ESC "&s0C";
char Setup_NoWrapLines[]			= ESC "&s1C";
char Setup_ManualFeed[]				= ESC "&l2H";
char Setup_TrayFeed[]					= ESC "&l1H";
char Setup_EnvelopeFeed[]			= ESC "&l3H";
char Setup_PageLength[]				= ESC "&l" ARG0 "P";
char Setup_LeftMargin[]				= ESC "&a" ARG1 "L";
char Setup_RightMargin[]			= ESC "&a" ARG2 "M";
char Setup_TopMargin[]				= ESC "&l" ARG3 "E";
char Setup_TextLength[]				= ESC "&l" ARG4 "F";
char Setup_HMI[]							= ESC "&k" ARG5 "H";
char Setup_VMI[]							= ESC "&l" ARG6 "C";
char Setup_Copies[]						= ESC "&l" ARG7 "X";
char Setup_Null[]							= "";

char *(SetupOptions[]) = {
	Setup_Reset,                          /* Option A */
	Setup_ClearMargins,                   /* Option B */
	Setup_EjectPage,                      /* Option C */
	Setup_CourierFont,                    /* Option D */
	Setup_CourierBoldFont,                /* Option E */
	Setup_LPrinterFont,                   /* Option F */
	Setup_Portrait,                       /* Option G */
	Setup_Landscape,                      /* Option H */
	Setup_WrapLines,                      /* Option I */
	Setup_NoWrapLines,                    /* Option J */
	Setup_ManualFeed,                     /* Option K */
	Setup_TrayFeed,                       /* Option L */
	Setup_EnvelopeFeed,                   /* Option M */
	Setup_PageLength,                     /* Option N */
	Setup_LeftMargin,                     /* Option O */
	Setup_RightMargin,                    /* Option P */
	Setup_TopMargin,                      /* Option Q */
	Setup_TextLength,                     /* Option R */
	Setup_HMI,                            /* Option S */
	Setup_VMI,                            /* Option T */
	Setup_Copies,		                      /* Option U */
	Setup_Null,														/* Option V */
	Setup_Null,														/* Option W */
	Setup_Null,														/* Option X */
	Setup_Null,														/* Option Y */
	Setup_Null};													/* Option Z */

#define SCR_ROW					25
#define SCR_COL					80
#define	WINDOW_ROW			20
#define WINDOW_COL			60
#define WINDOW_UL_ROW		2
#define WINDOW_UL_COL		10

#define W_TO_S_ROW(x)		((x)+WINDOW_UL_ROW)
#define W_TO_S_COL(x)		((x)+WINDOW_UL_COL)
#define RC_TO_OFF(row,col)	(((row)*SCR_COL + (col)) * 2)

#define NORM_ATTR				0x1E
#define BOLD_ATTR				0x1C
#define REV_ATTR				0x4B

char OldScreen[SCR_COL*SCR_ROW*2];
char far *VideoBuffer = (char far *) 0xB8000000L;

char NewScreen[WINDOW_COL*WINDOW_ROW+1] =
	"HP Laserjet II Setup v1.0ͻ"
	"                                                          "
	"  [A] Reset Printer           [K] Manual Feed             "
	"  [B] Clear Margins           [L] Tray Feed               "
	"  [C] Eject Page              [M] Envelope Feed           "
	"  [D] Courier Font            [N] Page Length      [   ]  "
	"  [E] Courier Bold Font       [O] Left Margin      [   ]  "
	"  [F] Line Printer Font       [P] Right Margin     [   ]  "
	"  [G] Portrait Mode           [Q] Top Margin       [   ]  "
	"  [H] Landscape Mode          [R] Text Length      [   ]  "
	"  [I] Wrap Lines              [S] Chars/Inch       [   ]  "
	"  [J] Don't Wrap Lines        [T] Lines/Inch       [   ]  "
	"                              [U] Copies           [   ]  "
	"                                                          "
	"Ķ"
	"  Press [letter] to select/deselect the setup sequence.   "
	"  Press <F10> to send the setup sequences now.            "
	"  Press <F2> to clear all selected setup sequences.       "
	"  Press <ESC> to exit.                                    "
	"ͼ";

unsigned int Opt_Posn[21] = {
		RC_TO_OFF(W_TO_S_ROW(2), W_TO_S_COL(4)),		/* Option A */
		RC_TO_OFF(W_TO_S_ROW(3), W_TO_S_COL(4)),		/* Option B */
		RC_TO_OFF(W_TO_S_ROW(4), W_TO_S_COL(4)),		/* Option C */
		RC_TO_OFF(W_TO_S_ROW(5), W_TO_S_COL(4)),		/* Option D */
		RC_TO_OFF(W_TO_S_ROW(6), W_TO_S_COL(4)),		/* Option E */
		RC_TO_OFF(W_TO_S_ROW(7), W_TO_S_COL(4)),		/* Option F */
		RC_TO_OFF(W_TO_S_ROW(8), W_TO_S_COL(4)),		/* Option G */
		RC_TO_OFF(W_TO_S_ROW(9), W_TO_S_COL(4)),		/* Option H */
		RC_TO_OFF(W_TO_S_ROW(10), W_TO_S_COL(4)),		/* Option I */
		RC_TO_OFF(W_TO_S_ROW(11), W_TO_S_COL(4)),		/* Option J */
		RC_TO_OFF(W_TO_S_ROW(2), W_TO_S_COL(32)),		/* Option K */
		RC_TO_OFF(W_TO_S_ROW(3), W_TO_S_COL(32)),		/* Option L */
		RC_TO_OFF(W_TO_S_ROW(4), W_TO_S_COL(32)),		/* Option M */
		RC_TO_OFF(W_TO_S_ROW(5), W_TO_S_COL(32)),		/* Option N */
		RC_TO_OFF(W_TO_S_ROW(6), W_TO_S_COL(32)),		/* Option O */
		RC_TO_OFF(W_TO_S_ROW(7), W_TO_S_COL(32)),		/* Option P */
		RC_TO_OFF(W_TO_S_ROW(8), W_TO_S_COL(32)),		/* Option Q */
		RC_TO_OFF(W_TO_S_ROW(9), W_TO_S_COL(32)),		/* Option R */
		RC_TO_OFF(W_TO_S_ROW(10), W_TO_S_COL(32)),	/* Option S */
		RC_TO_OFF(W_TO_S_ROW(11), W_TO_S_COL(32)),	/* Option T */
		RC_TO_OFF(W_TO_S_ROW(12), W_TO_S_COL(32))};	/* Option U */

#define Option_Data_Offset	(21*2)

#define ESC_KEY 			0x1B
#define F10_KEY 			0x4400
#define F2_KEY				0x3C00
#define CR_KEY				0x0D
#define BS_KEY				0x08
#define CUR_RIGHT			0x4D00
#define CUR_LEFT			0x4B00

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   exit - Reduce code size by replacing the standard exit function.     */
/*          We don't use files, so there is no need to bring in the file  */
/*          system just to close files.                                   */
/*                                                                        */
/*------------------------------------------------------------------------*/
void cdecl exit(int status)
{
	_exit(status);
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   _setargv - Suppress this library function to conserve code space.    */
/*                                                                        */
/*------------------------------------------------------------------------*/
void cdecl _setargv(void)	{}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   _setenvp - Suppress this library function to conserve code space.    */
/*                                                                        */
/*------------------------------------------------------------------------*/
void cdecl _setenvp(void)	{}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   main - Main Program.  Initialize interrupt vectors, etc.             */
/*                                                                        */
/*------------------------------------------------------------------------*/
cdecl main()
{
	void interrupt PrtrHandler(unsigned bp, unsigned di, unsigned si,
		unsigned ds, unsigned es, unsigned dx, unsigned cx, unsigned bx,
		unsigned ax, unsigned ip, unsigned cs, unsigned flags);
	void interrupt KeybdHandler(void);
	unsigned int prgsize(void);

	old_keyboard = getvect(KeybdVect);
	old_printer	 = getvect(PrtrVect);
	setvect(PrtrVect, PrtrHandler);
	setvect(KeybdVect, KeybdHandler);
	bdosptr(0x09, "LJII Version 1.0 Installed\r\n$", 0);
	keep(0, prgsize());
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   prgsize - Calculate program size.                                    */
/*                                                                        */
/*     __brklvl has the end of initialized and uninitialized data within  */
/*     the data segment at program startup. It is then incremented and    */
/*     decremented as memory is malloc'd and free'd.                      */
/*                                                                        */
/*     This function works in Tiny, Small, and Medium models.             */
/*                                                                        */
/*------------------------------------------------------------------------*/
unsigned int prgsize(void)
{
	extern unsigned __brklvl;
	extern unsigned _psp;

	return (_DS + (__brklvl + 15) / 16 - _psp);
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   KeybdHandler - Handle keyboard interrupts. Check for our popup code. */
/*                  If so, and if we are not already active, then invoke  */
/*                  the popup program. Otherwise pass the stuff on to the */
/*                  original handler.                                     */
/*                                                                        */
/*     Static and Global variables are used to keep the stack size to a   */
/*     minimum.                                                           */
/*                                                                        */
/*------------------------------------------------------------------------*/
void interrupt KeybdHandler()
{
	void LJII(void);
	static char c;

	c = inportb(KB_Data);
	(*old_keyboard)();
	if (!LJII_Active && Hot_Combo && (c == Hot_Key_Scan)) {
		++LJII_Active;
		LJII();
	  --LJII_Active;
	}
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   PrtrHandler - Handle requests for printer services.                  */
/*                                                                        */
/*     If setup requests are pending, transmit them before writing any-   */
/*     thing to the printer. Otherwise invoke the old interrupt.          */
/*                                                                        */
/*     Static and Global variables are used to keep the stack size to a   */
/*     minimum.                                                           */
/*                                                                        */
/*------------------------------------------------------------------------*/
#pragma warn -par
void interrupt PrtrHandler(unsigned bp, unsigned di, unsigned si,
	unsigned ds, unsigned es, unsigned dx, unsigned cx, unsigned bx,
	unsigned ax, unsigned ip, unsigned cs, unsigned flags)
{
	void Print_Setup(void);

	if (Setup_Pending && !LJII_Active && (dx == Our_Printer) && (ax <= 0x00FF))
		Print_Setup();
	_AX = ax;
	_DX = dx;
	(*old_printer)();
	ax = _AX;
}
#pragma warn .par

/*------------------------------------------------------------------------*/
/*                                                                        */
/*   Printer_Setup - Send setup strings to the printer.                   */
/*                                                                        */
/*     Static and Global variables are used to keep the stack size to a   */
/*     minimum.                                                           */
/*                                                                        */
/*------------------------------------------------------------------------*/
void Print_Setup(void)
{
	static int i, c;
	static char *cp, dig[3];
	void ThreeDigit(char *, unsigned int, char);
#if defined(DEBUG)
	void Debug_Print(char);
	void Print_Str(char *);
	void Print_Char(char);
#endif

	for (i=0; i<26; i++) {
		if (Setup_Flags[i]) {
#if defined(DEBUG)
			Print_Str("Option ");
			Print_Char(i+'A');
			Print_Str("\r\n");
#endif
			Setup_Flags[i] = 0;
			cp = SetupOptions[i];
			while ((c = *cp++) != 0) {
				if (c < 0x80) {
#if defined(DEBUG)
					Debug_Print(c);
#else
					_AX = c;
					_DX = Our_Printer;
					(*old_printer)();
#endif
				}
				else {
					c = Setup_Data[c & 0x007F];
					if ((i == ('s'-'a')) && (c != 0)) c = (120 + c/2)/c;
					else if ((i == ('t'-'a')) && (c != 0)) c = (48 + c/2)/c;
					ThreeDigit(dig, c, 0);
					if (dig[0]) {
#if defined(DEBUG)
						Debug_Print(dig[0]);
#else
						_AX = dig[0];
						_DX = Our_Printer;
						(*old_printer)();
#endif
					}
					if (dig[1]) {
#if defined(DEBUG)
						Debug_Print(dig[1]);
#else
						_AX = dig[1];
						_DX = Our_Printer;
						(*old_printer)();
#endif
					}
#if defined(DEBUG)
					Debug_Print(dig[2]);
#else
					_AX = dig[2];
					_DX = Our_Printer;
					(*old_printer)();
#endif
				}
			}
#if defined(DEBUG)
			Print_Str("\r\n");
#endif
		}
	}
	Setup_Pending = 0;
}

void LJII(void)
{
	void SaveWindow(void);
	void RestoreWindow(void);
	void DisplayWindow(void);
	void Print_Setup(void);
	int Get_Option_Data(int);
	void Set_Highlight(int, int);
	unsigned int GetKey(void);
	static unsigned int c;

	SaveWindow();
	DisplayWindow();
	Set_Highlight(0, 25);
	while ((c = GetKey()) != ESC_KEY) {
		if ((c >= 'A') && (c <= 'Z')) c = _tolower(c);
		if ((c >= 'a') && (c <= 'z')) {
			c -= 'a';
			if (Setup_Flags[c]) {
				--Setup_Pending;
				Setup_Flags[c] = 0;
				Set_Highlight(c, c);
			}
			else {
				++Setup_Pending;
				Setup_Flags[c] = 0xFF;
				Set_Highlight(c, c);
			  if ((c >= ('n'-'a')) && (c <= ('u'-'a'))) {
					if (!Get_Option_Data(c)) {
						--Setup_Pending;
						Setup_Flags[c] = 0;
						Set_Highlight(c, c);
					}
				}
			}
		}
		else if (c == F10_KEY) {
			Print_Setup();
			Set_Highlight(0, 25);
		}
		else if (c == F2_KEY) {
			for (c=0; c<26; c++) Setup_Flags[c] = 0;
			for (c=0; c<8;  c++) Setup_Data[c]  = 0;
			Setup_Pending = 0;
			Set_Highlight(0, 25);
		}
	}
	RestoreWindow();
}

void Set_Highlight(int i, int j)
{
	void Display_Option_Data(char far *, int, int);
	static char far *cp;

	if (j>21) j=21;
	for (i=i; i<=j; i++) {
		cp = VideoBuffer+Opt_Posn[i];
		*(cp+1) = Setup_Flags[i]?BOLD_ATTR:NORM_ATTR;
		if (i>=('n'-'a'))
			Display_Option_Data(cp+Option_Data_Offset, Setup_Flags[i], Setup_Data[i-('n'-'a')]);
	}
}

void SaveWindow(void)
{
	movedata(FP_SEG(VideoBuffer), 0, _DS, (int) OldScreen, SCR_COL*SCR_ROW*2);
}

void RestoreWindow(void)
{
	movedata(_DS, (int) OldScreen, FP_SEG(VideoBuffer), 0, SCR_COL*SCR_ROW*2);
}

void DisplayWindow(void)
{
	static int row, col;
	static char *sp, far *dp;

	dp = VideoBuffer + (WINDOW_UL_ROW*SCR_COL + WINDOW_UL_COL)*2;
	sp = NewScreen;
	for (row=0; row<WINDOW_ROW; row++) {
		for (col=0; col<WINDOW_COL; col++) {
			*dp++ = *sp++;
			*dp++ = NORM_ATTR;
		}
		dp += (SCR_COL-WINDOW_COL)*2;
	}
}

int Get_Option_Data(int n)
{
	unsigned int GetKey(void);
	void Display_Option_Data(char far *, int, int);
	static unsigned int c;
	static int cur_pos, opt_n;
	static char far *field;

	opt_n = n - ('n'-'a');
	cur_pos = 0;
	field = VideoBuffer + Opt_Posn[n] + Option_Data_Offset;
	Display_Option_Data(field, 1, Setup_Data[opt_n]);
	*(field+1) = REV_ATTR;
	*(field+3) = *(field+5) = BOLD_ATTR;
	for (;;) {
		c = GetKey();
		if ((c >= '0') && (c <= '9')) {
			*field = c;
			if (cur_pos < 2) {
				*(field+1) = BOLD_ATTR;
				field += 2;
				*(field+1) = REV_ATTR;
				++cur_pos;
			}
		}
		else if ((c == BS_KEY) || (c == CUR_LEFT)) {
			if (cur_pos > 0) {
				*(field+1) = BOLD_ATTR;
				--cur_pos;
				field -= 2;
				*(field+1) = REV_ATTR;
			}
		}
		else if (c == CUR_RIGHT) {
			if (cur_pos < 2) {
				*(field+1) = BOLD_ATTR;
				++cur_pos;
				field += 2;
				*(field+1) = REV_ATTR;
			}
		}
		else if (c == ESC_KEY) return 0;
		else if (c == CR_KEY) {
			field -= cur_pos * 2;
			c = (*field-'0')*100 + (*(field+2)-'0')*10 + (*(field+4)-'0');
			if (c > 255) c = 255;
			Setup_Data[opt_n] = c;
			Display_Option_Data(field, 1, Setup_Data[opt_n]);
			return 1;
		}
	}
}

unsigned int GetKey(void)
{
	static unsigned int c;

	_AX = 0;
	geninterrupt(0x16);
	c = _AX;
	if ((c & 0x00FF) != 0) return (c & 0x00FF);
	else return c;
}

void ThreeDigit(char *s, unsigned int n, char pad)
{
	static int i, d;

	for (i=2; i>=0; i--) {
		d = n % 10;
		n /= 10;
		if (d || (i==2)) s[i] = d + '0';
		else s[i] = pad;
	}
}

void Display_Option_Data(char far *cp, int flag, int n)
{
	void ThreeDigit(char *, unsigned int, char);
	static char dig[3];

	if (flag) {
		ThreeDigit(dig, n, '0');
	}
	else {
		dig[0] = dig[1] = dig[2] = ' ';
	}
	*cp++ = dig[0];
	*cp++ = NORM_ATTR;
	*cp++ = dig[1];
	*cp++ = NORM_ATTR;
	*cp++ = dig[2];
	*cp   = NORM_ATTR;
}

#if defined(DEBUG)
void Print_Char(char c)
{
	_AX = c;
	_DX = Our_Printer;
	(*old_printer)();
}

void Print_Str(char *s)
{
	static char c;

	while ((c = *s++) != 0) Print_Char(c);
}

void Debug_Print(char c)
{
	static char hex[16] = "0123456789ABCDEF";

	Print_Str("0x");
	Print_Char(hex[(c >> 4) & 0x000F]);
	Print_Char(hex[c & 0x000F]);
	Print_Str("\r\n");
}
#endif