/**********************************************************************

	SE2.C           Simple EDIT

			A text editor
			based on Simple Software
			principles and conventions.

			(Part 2)

			Copyright (c) 1988, Ted A. Campbell

			Bywater Software
			Box 4023
			Duke Station,
			Durham, NC  27706

			The use of Simple EDIT is governed by the
			"Simple Software User Agreement" (see file
			SS.DOC).

**********************************************************************/

#include "se.h"

#ifdef  FAST_VIDEO
#include "stdio.h"
#include "fv.h"
#else
#include "curses.h"
#endif

#include "kb.h"

#ifdef  UINCLUDE
#include "ctype.h"
#endif

extern FILE *se_fp;

/**********************************************************************

	FUNCTION:  se_control()

	DESCRIPTION:  This function handles control characters entered
	while editing.

	INPUT:  <character> is the control character fetched by
	se_getch().

	RETURNS:  In case of the KMENU character, this function
	returns the value passed to it from se_kmenu(), which
	may be SE_EXIT.  Otherwise, it returns SS_NOMINAL.

**********************************************************************/

se_control( character )
	unsigned int character;
	{
	register int c;
	switch( character )
		{
		case CR:
		case LF:
			se_newline();
			break;
		case DEL_BACK:
			se_dlback();
			break;
		case KB_RIGHT:
			se_right();
			break;
		case KB_LEFT:
			se_culeft();
			break;
		case KB_UP:
			se_cuup();
			break;
		case KB_DOWN:
			se_cudown();
			break;
		case KB_P_UP:
			se_scup( se_lines - 3 );
			break;
		case KB_P_DOWN:
			se_scdn( se_lines - 3 );
			break;
		case KB_INSERT:
			if ( se_imode == 1 )
				{
				se_imode = 0;
				}
			else
				{
				se_imode = 1;
				}
			se_im();
			break;
		case KB_D_CHAR:
			se_dlch();
			break;
		case KB_HELP:
			se_help();
			break;
		case KB_HOME:
			se_home();
			break;
		case KB_END:
			se_end();
			break;
		case KB_FIND:
			se_find();
			break;
/*               case KB_REPLACE:
			se_replace();
			break;
*/                case KB_W_LEFT:
			se_prword();
			break;
		case KB_W_RIGHT:
			se_nxword();
			break;
		case KB_D_LINE:
			se_dlline();
			break;
		case KB_D_WORD:
			se_dlword();
			break;
		case KB_ABANDON:
			message( "Abandon this file? " );
			move( 0, 19 );
			refresh();
			c = kb_ci();
			if ( toupper( c ) == 'Y' )
				{
				return SE_EXIT;
				}
			else
				{
				message( " " );
				se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
				}
			break;
		case KB_SAVE:
			se_save();
			return SS_NOMINAL;
			break;
		case KB_EXIT:
			se_save();
			message( "Finished editing? " );
			move( 0, 18 );
			refresh();
			c = kb_ci();
			if ( toupper( c ) == 'Y' )
				{
				return SE_EXIT;
				}
			else
				{
				message( " " );
				se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
				}
			break;
		default:
			return 0;
		}
	}

/**********************************************************************

	FUNCTION:  se_culeft()

	DESCRIPTION:  This function handles movement of the cursor
	to the left (left arrow key or ^S).

	INPUT:  None.

	RETURNS:  If the current cursor position is the first
	of the file being edited, the function returns SS_ERROR;
	otherwise it returns SS_NOMINAL.

**********************************************************************/

se_culeft()
	{

	/***    Detect beginning of line.  ***/

	if ( se_bpos == 0 )
		{
		if ( se_current->prev != se_btop )
			{
			if ( se_lpos == 0 )
				{
				se_screen( se_current->prev, 0 );
				se_lpos = 1;
				}
			se_current = se_current->prev;
			se_bpos = se_llen( se_current->characters );
			--se_alc;
			se_edadr( --se_lpos, se_cpos( se_current, se_bpos )  );
			se_lc();
			return SS_NOMINAL;
			}
		else
			{
			return SS_ERROR;
			}
		}

	/***    Move the cursor.  ***/

	--se_bpos;
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();

	}

/**********************************************************************

	FUNCTION:  se_right()

	DESCRIPTION:  This function handles movement of the cursor 
	one character to the right.  

	INPUT:  None.  The current character is presumed to be
	se_current->characters[ se_bpos ] and is presumed
	to be at the screen position indicated by se_lpos
	and se_cpos( se_current, se_bpos ) .

	RETURNS:  The function returns SS_ERROR if the cursor cannot 
	be moved to the right (i.e., if it is the last character
	[LF] in the last line of the file); otherwise it 
	returns SS_NOMINAL (successful). 

***********************************************************************/

se_right()
	{

	/***    If the current character is at the end of
	the line, we must move to the beginning
	of the next line.                          ***/

	if ( se_current->characters[ se_bpos ] == LF )
		{

		/***    As long as this is not the last line
			in the file, we can proceed.  ***/

		if ( se_current->next != se_bbot )
			{
			se_current = se_current->next;
			++se_lpos;
			++se_alc;
			se_bpos = 0;
			}

		/***    But if this is the last character of the
			last line in the file, the cursor cannot
			be moved, and so we return -1.  ***/

		else
			{
			return SS_ERROR;
			}
	}

	/***    Advance the cursor ***/

	else
		{
		++se_bpos;
		}

	/***    If it happens that we are at
		the last character in the last
		line on the screen, we must
		scroll the screen down before
		we can display the next line on
		which the cursor will reside.      ***/

	if ( se_lpos == ( se_lines ) )
		{
		--se_lpos;
		se_screen( se_top(), 0 );
		}

	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();

	}

/**********************************************************************

	FUNCTION:  se_cuup()

	DESCRIPTION:  This function moves the cursor to the next line
	above it.

	INPUT:  None.

	RETURNS:  If the current character is in the top line of the
	fie, the function returns SS_ERROR; otherwise it returns
	SS_NOMINAL.

**********************************************************************/

se_cuup()
	{

	/***    First check to see if there is a previous line.  ***/

	if ( se_current->prev == se_btop )
		{
		return SS_ERROR;
		}

	if ( se_lpos == 0 )
		{
		se_screen( se_current->prev, 0 );
		se_lpos = 1;
		}

	se_current = se_current->prev;
	--se_alc;
	--se_lpos;

	/***    We also need to see if in fact the previous line is
		as long as the current one.  If not, we set the
		bpos and cpos to the length of the previous line,
		i.e., to the last character position in the line.  ***/

	if ( se_llen( se_current->characters ) < se_bpos )
		{
		se_bpos = se_llen( se_current->characters );
		}

	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();

	}

/**********************************************************************

	FUNCTION:  se_cudown()

	DESCRIPTION:  This function moves the cursor to the next
	line below it.

	INPUT:  None.

	RETURNS:  If the current character is in the last line of
	the file, the function returns SS_ERROR, otherwise it returns
	SS_NOMINAL.

**********************************************************************/

se_cudown()
	{

	/***    First check to see if there is a next line      ***/

	if ( se_current->next == se_bbot )
		{
		return SS_ERROR;
		}

	se_current = se_current->next;

	if ( se_lpos == ( se_lines - 1 ) )
		{
		se_screen( se_traverse( se_current, 1 - se_lines ), 0 );
		--se_lpos;
		}

	++se_lpos;
	++se_alc;

	/***    We also need to see if the length of the next line is
		less then the current cursor position, in which case
		we adjust the bpos and cpos data.  ***/

	if ( se_llen( se_current->characters ) < se_bpos )
		{
		se_bpos = se_llen( se_current->characters );
		}

	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();

	}

/**********************************************************************

	FUNCTION: se_prword()

	DESCRIPTION:  This function moves the cursor to the previous
	word.  The present implementation is crude, but simple:
	se_culeft() is called repeatedly until the beginning of
	the previous word is reached.

	INPUT:  None.

	RETURNS:  If the function cannot reach a prior word,
	SS_ERROR is returned, otherwise SS_NOMINAL is returned.

**********************************************************************/

se_prword()
	{

	while ( isspace( se_current->characters[ se_bpos ] ) == 0 )
		{
		if ( se_culeft() == -1 )
			{
			return SS_ERROR;
			}
		}
	while ( isspace( se_current->characters[ se_bpos ] ) != 0 )
		{
		if ( se_culeft() == -1 )
			{
			return SS_ERROR;
			}
		}
	while ( isspace( se_current->characters[ se_bpos ] ) == 0 )
		{
		if ( se_culeft() == -1 )
			{
			return SS_ERROR;
			}
		}
	while ( isspace( se_current->characters[ se_bpos ] ) != 0 )
		{
		if ( se_right() == -1 )
			{
			return SS_ERROR;
			}
		}

	}

/**********************************************************************

	FUNCTION:  se_nxword()

	DESCRIPTION:  This function moves the cursor to the next word.

	INPUT:  None.

	RETURNS:  If the beginning of the next word cannot be found,
	SS_ERROR is returned, otherwise SS_NOMINAL is returned.

**********************************************************************/

se_nxword()
	{

	while( isspace( se_current->characters[ se_bpos ] ) == 0 )
		{
		if ( se_right() == -1 )
			{
			return SS_ERROR;
			}
		}
	while( isspace( se_current->characters[ se_bpos ] ) != 0 )
		{
		if ( se_right() == -1 )
			{
			return SS_ERROR;
			}
		}
	}


/**********************************************************************

	FUNCTION:  se_home()

	DESCRIPTION:  This function moves the cursors to the first 
	character of the first line of the file, and adjusts the 
	screen appropriately.

	INPUT:  None.

	RETURNS:  None.

**********************************************************************/

se_home()
	{
	se_current = se_btop->next;
	se_alc = se_lpos = 0;
	se_screen( se_current, 0 );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();
	}

/**********************************************************************

	FUNCTION:  se_end()

	DESCRIPTION:  This function moves the cursor to the last 
	character of the last line in the file, and adjusts the 
	screen accordingly.  

	INPUT:  None.

	RETURNS:  None.  

**********************************************************************/

se_end()
	{
	int alc;
	struct se_line *current;
	alc = se_alc;
	current = se_current;
	while( current->next != se_bbot )
		{
		++alc;
		current = current->next;
		}
	se_current = current;
	se_alc     = alc;
	current = se_traverse( se_current, 0 - se_lines + 2 );
	se_lpos = 0 - se_trct;
	se_screen( current, 0 );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();
	}

/**********************************************************************

	FUNCTION:  se_find()

	DESCRIPTION:  This function finds a specified character string.

	INPUT:  None.

	RETURNS:  The function returns SS_NOMINAL if the string
	has been successfully found, and SS_ERROR if the string
	was not found.

**********************************************************************/

se_find()
	{
	struct se_line *sline;
	int spos, fpos;
	static char sstring[ 128 ];
	message( "Search string:  " );
	move( 0, 16 );
	refresh();
	se_str( sstring );
	sline = se_current;
	spos = se_bpos;
	while ( sline->next != se_bbot )
		{
		if ( ( fpos = se_search( sline, spos, sstring )) != -1 )
			{
			se_redraw( sline,  fpos );
			return SS_NOMINAL;
			}
		spos = 0;
		sline = sline->next;
		}
	message ( "Match not found." );
	return SS_ERROR;
	}

/**********************************************************************

	FUNCTION:  se_search()

	DESCRIPTION:  This function searches a specific line
	in memory for a specific string.

	INPUT:  <line> is a pointer to an se_line structure
	for the line to be searched; <begin> is the beginning
	character in the line to be searched; <match> is a pointer
	to the string to be matched.

	RETURNS:  The function returns -1 if the search fails,
	and the position in the line of the match if the search
	succeeds.

**********************************************************************/

se_search( line, begin, match )
	struct se_line *line;
	int begin;
	char match[];
	{
	static char sbuffer[ 128 ];
	register int c;
	int spos, lpos;

	/*      Initialize the search buffer                    */
	lpos = begin;
	for ( c = 0; c < strlen( match ); ++c )
		{
		sbuffer[ c ] = line->characters[ lpos ];
		++lpos;
		}
	sbuffer[ c ] = 0;

	/*      Search the line                                 */
	spos = begin;
	while( ( strlen( line->characters ) - spos - 1 ) >= strlen( match ) )
		{
		if ( strcmp( sbuffer, match ) == 0 )
			{
			return spos;    /* Search succeeded     */
			}
		for ( c = 0; c < strlen( match ) - 1; ++c )
			{
			sbuffer[ c ] = sbuffer[ c + 1 ];
			}
		sbuffer[ c ] = line->characters[ lpos ];
		++lpos;
		++c;
		sbuffer[ c ] = 0;
		++spos;
		}
	return -1;                      /* Search has failed    */
	}

/**********************************************************************

	FUNCTION:  se_redraw()

	DESCRIPTION:

	INPUT:  None.

	RETURNS:

**********************************************************************/

se_redraw( line, bcol )
	struct se_line *line;
	int bcol;
	{
	int alc;
	struct se_line *current;

	alc = se_alc;
	for ( current = se_current; current != line; current = current->next )
		{
		++alc;
		}
	se_current = current;
	se_alc     = alc;
	se_bpos    = bcol;      /* *** */
	current = se_traverse( se_current, 0 - ( se_lines / 2 ) );
	se_lpos = 0 - se_trct;
	se_screen( current, 0 );
	se_edadr( se_lpos, se_cpos( se_current, se_bpos )  );
	se_lc();

	}

/**********************************************************************

	FUNCTION:  se_cpos()

	DESCRIPTION:  This function converts the buffer position
	in a given line into the appropriate columnar position,
	taking tab stops into account.

	INPUT:  None.

	RETURNS:

**********************************************************************/

se_cpos( lpointer, bpos )
	struct se_line *lpointer;
	unsigned int bpos;
	{
	register unsigned int p;
	unsigned int columns;
	char last_char;

	last_char = 0;
	for ( p = columns = 0; p < bpos; ++p )
		{
		if ( lpointer->characters[ p ] == TAB )
			{
			if ( last_char == TAB )
				{
				columns += TAB_INTERVAL;
				}
			else
				{
				columns = ( columns -
					( columns % TAB_INTERVAL ) ) + TAB_INTERVAL - 1;
				}
			}
		else
			{
			++columns;
			}
		last_char = lpointer->characters[ p ];
		}
	return columns;
	}

